/*******************************************
 *
 * Copyright 2001, Red Hat, Inc., all rights reserved.
 *
 * Author:  Bob Matthews <bmatthews@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "sysinfo.h"
#include "mem_load.h"
#include "programs.h"
#include "trivial.h"

int *grab_and_touch (char *block[], int i)
{
	block[i] = (char *) malloc (MB);
	if (!block[i])
		return NULL;
	return (memset (block[i], 1, MB));
}

/* OK, this is very tricky and full of assumptions.  
 * 
 * In most cases, we allocate 110% of RAM, minus DEFAULT_RESERVE, which is the minimum amount
 * required for the kernel and its data.  Best guesstimates by Those Who Know puts this at
 * around 64M for UP and SMP kernels, and 256M for enterprise kernels, although it will
 * certainly vary widely depending upon the job mix running on the machine.
 *
 * If the machine doesn't have the proper amount of swap (or has no swap at all) we just 
 * allocate as much as we can, minus the DEFAULT_RESERVE.  
 *
 * Also, we need to take care and make sure we don't tell the program that it can
 * access memory beyond 3G.  That's currently the per process limit on the brk area (I think.)
 */
unsigned long compute_allocable_mem (unsigned long ram, unsigned long swap,  int percent_ram)
{
 	unsigned long total = ram + swap;
 	unsigned long usage = ram * percent_ram / 100 ;

 	/* Leave at least DEFAULT_RESERVE free space and check for maths overflow. */
	usage = MIN(usage, total - DEFAULT_RESERVE);
	usage/=1024;	/* to megabytes */
	return MIN(usage, 2930);
}

int do_mem_load(void)
{
	unsigned long ram, swap, touchable_mem, i;
	char *mem_block[MAX_MEM_IN_MB];
	void *success;
	int iterations=0;
	int j=0, percent_ram=110;

	if(get_ram(&ram)){
		printsys("could not get info on ram size, aborting\n");
	}
	if(get_swap(&swap)){
		printsys("could not get info on ram size, aborting\n");
	}
	/* obviously does nothing, but left as documentation - aggelos */
	if (! (j < 10 || j >110))
		percent_ram=j;

	printd("ram=%lu, swap=%lu\n", ram, swap);
	/* determine if how much ram we want to touch */
	touchable_mem = compute_allocable_mem (ram, swap, percent_ram);
	printd("touchable mem=%lu\n", touchable_mem);

	/* loop until we're killed, frobbing memory in various perverted ways */

	for ( ; ; ) {
		/* grab and touch MB blocks of memory until we have what we want */
		for (i = 0;  i < touchable_mem; i++) {
			if(++iterations == 10){
				report_progress();
				iterations=0;
			}
			success = grab_and_touch(mem_block, i);
			if (!success) {
				/*
				 * freebsd returns NULL too early
				 * (or rather just in time)
				 */
				printd("Can't grab and/or touch mem_block "
								"%lu\n", i);
				touchable_mem=i-1;
				break;
			}
		}

		for (i = 0;  i < touchable_mem; i++) {
			if(++iterations==10){
				report_progress();
				iterations=0;
			}
			printd("Moving block %lu to %ld\n", i, (i + touchable_mem / 2) % touchable_mem);
			memcpy (mem_block[i], mem_block[(i + touchable_mem / 2) % touchable_mem], MB);
		}

		for (i = touchable_mem - 1; i != ~0UL; i--) {
			if(++iterations==10){
				report_progress();
				iterations=0;
			}
			memcpy (mem_block[(i + touchable_mem / 2) % touchable_mem], mem_block[i], MB);
		}

		for (i = 0; i < touchable_mem; i++) {
			if(++iterations==10){
				report_progress();
				iterations=0;
			}
			free (mem_block[i]);
		}
	}
	/* not reached */
	exit (0);
}
