/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
using Gtk;

namespace Clinica {

	public class VisitListStore : ListStore {
	
	    public signal void visit_added   (int visit_id);
	    public signal void visit_removed (int visit_id);
	    
	
		enum Field {
			VISIT,
			DATE,
		}
	
		public signal void error (string message);
		
		private ResourceManager resource_manager;
	
		public VisitListStore (ResourceManager resources) {
		    resource_manager = resources;
			/* Error callback */
			error.connect ((t,l) => resource_manager.error_callback(t,l));
			
			/* Create headers for the liststore with types
			 * Visit, Date */
			Type [] column_headers = { typeof(Visit), typeof(string) };
			set_column_types (column_headers);
			
			/* Fill liststore asynchronously */
			Idle.add(() => {
				foreach (Visit v in Visit.all (resource_manager)) {
					add_visit (v);
				}
				
				/* We don't need to execute any more */
				return false;
			});
		}
		
		public void add_visit (Visit v) {
			TreeIter iter;
			append (out iter);
			set_value (iter, Field.VISIT, v);
			set_value (iter, Field.DATE, "");
			
			visit_added (v.get_id ());
		}
		
		public void reload_visit (Visit v) {
			TreeIter iter = visit_to_iter (v);
			set_value (iter, Field.VISIT, v);
			set_value (iter, Field.DATE, "");
		}
		
		/** @brief Get visits of the given patient */
		public List<TreeIter?> get_visits_of (Patient p) {
			Value value;
			var iters = new List<TreeIter?> ();
			
			TreeIter it;
			Visit v;
			
			if (!get_iter_first (out it)) {
				return iters;
			}
			do {
				get_value (it, Field.VISIT, out value);
				v = value as Visit;
				if (p.get_id () == v.patient.get_id ()) {
					iters.append (it);
				}
			} while (iter_next (ref it));
			return iters;
		}
		
		public void remove_visit (Visit v) {
			TreeIter it;
			Value visit;
			int this_id = v.get_id ();
			
			if (!get_iter_first (out it)) {
				error(_("Visit database seems corrupted. This is likely to be a bug in the application"));
			}
			do {
				get_value (it, Field.VISIT, out visit);
				if ((visit as Visit).get_id () == this_id) {
					remove (it);
				    visit_removed (this_id);
					return;
				}
			} while (iter_next (ref it));
		
			assert_not_reached ();
		}
		
		public void remove_visit_from_iter (TreeIter it) {
			Value visit;
			get_value (it, Field.VISIT, out visit);
			int this_id = (visit as Visit).get_id ();
			(visit as Visit).remove ();
			remove (it);
			visit_removed (this_id);
		}
		
		public Visit iter_to_visit (TreeIter it) {
			Value visit;
			get_value (it, Field.VISIT, out visit);
			return (visit as Visit);
		}
		
		public TreeIter visit_to_iter (Visit v) {
			Value visit;
			TreeIter iter;
			if (!get_iter_first (out iter))
				assert_not_reached ();
			do {
				get_value (iter, Field.VISIT, out visit);
				if ((visit as Visit).get_id () == v.get_id ()) {
					return iter;
				}
			} while (iter_next (ref iter));
			
			assert_not_reached ();
		}
	}

}
