/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */

using Gtk;

namespace Clinica {

	public class VisitActions : Alignment {
	
		/* Buttons */
		private Button show_details_button;
		private Button save_button;
		private Button pdf_button;
		private Button delete_button;
	
		/* Signals */
		public signal void save_visit ();
		public signal void save_as_pdf ();
		public signal void show_details ();
		public signal void delete_visit ();
		
		/* Resource Manager */
		private ResourceManager resource_manager { get; set; }
		
		/**
		 * @brief Id of the connection to the saved signal of the
		 * Visit
		 */
		private ulong visit_connection_id = 0;
		
		/**
		 * @brief The visit to which the VisitAction refers.
		 */
        private VisitTab visit_tab;
	
		public VisitActions (ResourceManager resources) {
			GLib.Object (xalign: 0.5F, yalign: 0F, xscale: 1F, yscale: 0F);
			resource_manager = resources;
			
			/* Create an HBox to hold the buttons */
			var hbox = new HBox (false, resource_manager.PADDING);
			
			/* Show more details on this patient */
			show_details_button = new Button.with_label (_("Edit patient details"));
			show_details_button.clicked.connect ((button) => show_details ());
			hbox.pack_start (show_details_button);
			
			/* Delete Visit button */
			delete_button = new Button.from_stock (Gtk.Stock.DELETE);
			delete_button.clicked.connect ((button) => delete_visit ());
			hbox.pack_start (delete_button);
			
			/* Save the visit as a PDF file */
			pdf_button = new Button.with_label (_("Save as PDF"));
			pdf_button.clicked.connect ((button) => save_as_pdf ());
			hbox.pack_start (pdf_button);
		
			/* The save button */
			save_button = new Button.from_stock (Gtk.Stock.SAVE);
			save_button.clicked.connect ((button) => save_visit ());
			hbox.pack_start (save_button);
			
			add (hbox);
		}
		
		
		
		/**
		 * @brief Set active visit of which the actions refers. This is used mainly
		 * to set the status of the delete buttons.
		 */
		public void set_active_tab (VisitTab visittab) {
		    /* Disconnect the old visit, if still connected */
		    if (visit_connection_id != 0)
                this.visit_tab.disconnect (visit_connection_id);
                
            /* Set the sensitivity of the button according to the changed
             * visit. */
            this.visit_tab = visittab;
            visit_connection_id = visit_tab.saved.connect ((visit) => delete_button.set_sensitive (true));
            if (visit_tab.visit.get_id () != 0)
                delete_button.set_sensitive (true);
            else
                delete_button.set_sensitive (false);
		}
	}
	
}
