/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */

using Sqlite;
using Gee;

namespace Clinica {

    /**
     * @brief Struct representing a column in the database.
     */
    public class SqlColumn {
        public string  type;
        public string  v;
        
        public SqlColumn (string type) {
            this.type = type;
        }
    }
    
    public class SqlDataIterator : GLib.Object {

        protected unowned Database db;
        protected Statement stmt;

        public SqlDataIterator (Database db, string table_name, string? order_by = null, bool desc = false) {
            this.db = db;
            if (order_by == null)
                db.prepare (@"SELECT ID FROM $(table_name);", -1, out stmt, null);
            else {
                if (desc)
                    db.prepare (@"SELECT ID FROM $(table_name) ORDER by $(order_by) DESC;", -1, out stmt, null);
                else
                    db.prepare (@"SELECT ID FROM $(table_name) ORDER by $(order_by);", -1, out stmt, null);
            }
        }
        
        public SqlDataIterator.with_like (Database db, string table_name, string? order_by = null, 
            bool desc = false, string where_stmt) {
            this.db = db;
            
            if (order_by == null)
                db.prepare (@"SELECT ID FROM $(table_name) WHERE $(where_stmt);", -1, out stmt, null);
            else {
                if (desc)
                    db.prepare (@"SELECT ID FROM $(table_name) WHERE $(where_stmt) ORDER by $(order_by) DESC;", -1, out stmt, null);
                else
                    db.prepare (@"SELECT ID FROM $(table_name) WHERE $(where_stmt) ORDER by $(order_by);", -1, out stmt, null);
            }
        }
        
        public SqlDataIterator iterator () {
            return this;
        }
        
        public bool next () {
            return (stmt.step () == ROW);
        }
            
        public new int get () {
            return stmt.column_int (0);
        }
    }

    /**
     * @brief Interface representing an abstract data
     * type interfaced to a SQLite database. 
     */
    public class SqlDataType : GLib.Object {
    
        /**
         * @brief Signal emitted when an error eris
         * encountered
         */
        public signal void error (string message);
        
        /**
         * @brief The table name in the database. Must be set
         * by the class implementing this interface.
         */
        public string table_name;
        
        /**
         * @brief The database object used to interact with
         * the sqlite database. It must be opened by the
         * class implementing this interface. 
         */
        public unowned Database db;
        
        /**
         * @brief Hash mapping column_name -> type, value.
         */
        public HashMap<string, SqlColumn?> columns;
        
        public SqlDataType (Database db) {
            this.db = db;
            columns = new HashMap<string, SqlColumn> ();
            this.columns.set ("ID", new SqlColumn ("INTEGER PRIMARY KEY"));
            set_integer ("ID", 0);
        }
        
        public SqlDataType.with_id (Database db, int ID) {
            this (db);
            load (ID);
        }
        
        protected void add_text_field (string name) {
            this.columns.set (name, new SqlColumn ("TEXT"));
        }
        
        protected void add_integer_field (string name) {
            this.columns.set (name, new SqlColumn ("INTEGER"));
        }
        
        protected void add_date_field (string name) {
        	this.columns.set (name, new SqlColumn ("TEXT"));
        }
        
        /**
         * @brief Escape a string and surround it with
         * double quotes ready to be inserted in the 
         * database.
         */
        protected string quote (string str) {
            return "\"" + str.escape ("") + "\"";
        }
        
        /**
         * @brief Initialize resource, i.e. open the database
         * and check for missing tables.
         */
        protected void init_resources () {
	        lock (db) {
			    Statement stmt;
			    
			    /* Check if the table is present in the database and if it's
			     * not, create it */
			    db.prepare (@"SELECT * from sqlite_master WHERE name='$(table_name)';",
			                -1, out stmt, null);
			                
			    if (stmt.step () == DONE) {
			        init_database ();
			    }
		    }
        }
        
        /**
         * @brief Create table structure in the database.
         */
        protected void init_database () {
            Statement stmt;
            string sql = @"CREATE TABLE $(table_name) (";
            foreach (Map.Entry<string, SqlColumn> e in columns.entries) {
                sql += @"$(e.key) $(e.value.type), ";
            }
            
            /* Remove last 2 characters ", " and add ); */
            sql = sql[0:-2];
            sql += ");";
            
            /* Execute query */
            db.prepare (sql, -1, out stmt, null);
            
            if (stmt.step () != DONE) {
                error (@"Error creating $(table_name) table in the database.");
            }
        }
        
        public string get_db_table_name () {
            return this.table_name;
        }
        
        public void set_text (string field, string val) {
        	if (!columns.has_key (field)) {
        		error (@"Error saving to field $(field) from table $(table_name)\n" + 
        				"Selected field is not present in database");
        		return;
        	}
            SqlColumn col = columns.get (field);
            col.v = val;
            columns.set (field, col);
        }
        
        public unowned string get_text (string field) {
        	if (!columns.has_key (field)) {
        		error (@"Error loading from field $(field) from table $(table_name)\n" + 
        				"Selected field is not present in database");
        		return "";
        	}
            SqlColumn col = columns.get (field);
            if (col.v != null)
                return col.v;
            else
                return "";
        }
        
        public unowned int get_integer (string field) {
        	if (!columns.has_key (field)) {
        		error (@"Error loading from field $(field) from table $(table_name)\n" + 
        				"Selected field is not present in database");
        		return 0;
        	}
            SqlColumn col = columns.get (field);
            return int.parse (col.v);
        }
        
        public DateTime get_date (string field) {
        	if (!columns.has_key (field)) {
        		error (@"Error loading from field $(field) from table $(table_name)\n" + 
        				"Selected field is not present in database");
        		return new DateTime.now_utc ();
        	}
        	SqlColumn col = columns.get (field);
        	string [] fields = col.v.split(" ");
        	
        	int year = int.parse (fields[0]);
        	int month = int.parse (fields[1]);
        	int day = int.parse (fields[2]);
        	int hour = int.parse (fields[3]);
        	int minute = int.parse (fields[4]);
        	int seconds = int.parse (fields[5]);
        	
        	if (year < 1 || year > 9999 || 
        		month < 1 || month > 12 ||
        		day < 1 || day > 31 ||
        		minute < 0 || minute > 59 ||
        		seconds < 0 || seconds > 59)
        		return new DateTime.now_local ();
        	
        	return new DateTime.local (year, month, day, hour, minute, seconds);
        }
        
        public void set_date (string field, DateTime date) {
        	if (!columns.has_key (field)) {
        		error (@"Error saving tofield $(field) from table $(table_name)\n" + 
        				"Selected field is not present in database");
        		return;
        	}
        	SqlColumn col = columns.get (field);
        	col.v = datetime_to_string (date);
        }
        
        public static string datetime_to_string (DateTime date) {
            return string.join(" ", 
        		date.get_year ().to_string (), 
        		"%.2d".printf (date.get_month ()), 
        		"%.2d".printf (date.get_day_of_month()), 
        		"%.2d".printf (date.get_hour ()), 
        		"%.2d".printf (date.get_minute ()), 
        		"%.2d".printf (date.get_second ()));
        }
        
        public int get_id () {
            return get_integer ("ID");
        }
        
        public void set_integer (string field, int64 val) {
        	if (!columns.has_key (field)) {
        		error (@"Error saving tofield $(field) from table $(table_name)\n" + 
        				"Selected field is not present in database");
        		return;
        	}
            SqlColumn col = columns.get (field);
            if (field == "ID" && val == 0)
                col.v = "NULL";
            else
                col.v = val.to_string ();
            columns.set (field, col);
        }
        
        public new string get (string field) {
            return get_text (field);
        }
        
        public new void set (string field, string value) {
            set_text (field, value);
        }
        
        /**
         * @brief Save changes permanently in the database
         */
        public void save () {
            Statement stmt;
            string sql = @"INSERT OR REPLACE INTO $(table_name) (";
            foreach (Map.Entry<string, SqlColumn> e in columns.entries) {
                sql += @"$(e.key), ";
            }
            
            sql = sql[0:-2];
            sql += ") VALUES (";

            foreach (Map.Entry<string, SqlColumn> e in columns.entries) {
                if (e.value.v != null) {
                    if (e.value.type == "TEXT")
                        sql += @"$(quote(e.value.v)), ";
                    else
                        sql += @"$(e.value.v), ";
                }
                else {
                    sql += ", ";
                }
            }
            
            sql = sql[0:-2]; sql += ");";
            db.prepare (sql, -1, out stmt, null);
            if (stmt.step () != DONE) {
                error (@"Error inserting values in the database $(table_name)\n Error $(db.errcode()): $(db.errmsg ())");
            }
            
            /* If we save with ID set to 0 then the ID will be autodetermined
             * by sqlite so we should get it back */
            if (get_id () == 0) {
            	set_integer("ID", db.last_insert_rowid());
            }
        }
        
        /**
         * @brief Load data from database, overwriting local
         * variables.
         */
        public void load (int ID = 0) {
            if (ID == 0)
                ID = get_integer ("ID");
                
            Statement stmt;
            string sql = "SELECT ";
            foreach (Map.Entry<string, SqlColumn> e in columns.entries) {
                sql += @"$(e.key), ";
            }
            
            sql = sql[0:-2];
            sql += @" FROM $(table_name) WHERE ID=$(ID);";
            
            db.prepare (sql, -1, out stmt, null);
            if (stmt.step () !=ROW) {
                error (@"Error loading data from table $(table_name).");
                return;
            }
            
            int i = 0;
            foreach (Map.Entry<string, SqlColumn> e in columns.entries) {
            	if (e.value.type == "TEXT")
            		e.value.v = stmt.column_text (i).compress ();
            	else
	                e.value.v = stmt.column_text (i);
                i++;
            }
        }
        
        /**
         * @brief Delete this item permanently from the database.
         */
        public void remove () {
        	Statement stmt;
        	string sql = @"DELETE from $(table_name) WHERE ID=$(get_id ());";
        	
        	db.prepare (sql, -1, out stmt, null);
        	if (stmt.step () != DONE) {
        		error (@"Error deleting item from the database $(table_name)");
        	}
        }
        
        /**
         * @brief Find IDs of elements in ex_table (that is assumed to be a table
         * associateed to another SqlDataType object and return a vector
         * containing their IDs. 
         */ 
        public GLib.List<int> associated_ids (string ex_table, string foreign_key) {
        	var ids = new GLib.List<int> ();
        	Statement stmt;
        	string sql = @"SELECT ID from $(ex_table) WHERE $(foreign_key)=$(get_id());";
        	
        	db.prepare (sql, -1, out stmt, null);
        	while (stmt.step () == ROW) {
        		ids.append (stmt.column_int (0));
        	}
        	return ids;
        }
    }
}
