/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */

using GLib;
using Sqlite;
using Gtk;

namespace Clinica {
 
    /**
     * @brief Type representing the callback for the error
     * signal.
     */
    public delegate void ErrorCallback(GLib.Object source, string message);

    /**
     * @brief Grant access to resources such as
     * UI files, configuration and database
     */
    public class ResourceManager : GLib.Object {
    
        /**
         * @brief The padding the should be used in windows
         * in Clinica.
         */
        public const int PADDING = 6;
    
        /**
         * @brief Callback that manages errors
         * notification to the end user.
         */
        public unowned ErrorCallback error_callback;
        
        /**
         * @brief PatientListStore that holds the list
         * of the patients
         */
        public PatientListStore patient_list_store;
        
        /**
         * @brief DoctorListStore that holds the list
         * of the doctors.
         */
        public DoctorListStore doctor_list_store;
        
        /**
         * @brief VisitListStore that holds the list
         * of the visits.
         */
        public VisitListStore visit_list_store;
        
        /**
         * @brief EventListStore that holds the list of
         * the events scheduled in clinica.
         */
        public EventListStore event_list_store;
        
        /**
         * @brief Reference to a user interface
         */
        public UserInterface user_interface;
        
        /**
         * @brief Clinica settings
         */
        public Settings settings;
    
        /**
         * @brief Signal emitted when an error
         * occurr. The user should be notified
         * with message by the application
         */
        public signal void error (string message);
    
        /**
         * @brief Local configuration directory
         * obtained at runtime.
         */
        private string local_config_directory;
        
        /**
         * @brief Database path.
         */
        private string database_path;
        
        /**
         * @brief SQLite database with data and
         * configurations.
         */
        public Database db;
        
        /**
         * @brief PluginEngine for Clinica
         */
        public PluginEngine plugin_engine;
        
        /**
         * @brief Array containing the valid medicine search engines that
         * are available at the moments. This should be used by plugins to
         * register themselves as valid search engines
         */
        internal GLib.List<unowned MedicineSearchEngine> medicine_search_engines;
        
        /**
         * @brief A pointer to the patient that is been dragged. Should
         * be set on drag_begin by the treeview and reset to null by the
         * drag destination handler. 
         */
		internal Patient? dragging_patient = null;
        
        /**
         * @brief Path to the UI files directory
         */
        private string ui_files_path;
        private string icons_files_path;
        private string prefix;
        private string data_path;
        
        private bool is_local = false;
        
        public const string doctor_table = "doctors";
        public const string patient_table = "patients";
        public const string visits_table = "visits";
        public const string events_table = "events";
        
        /** SIGNALS **/
        public signal void medicine_search_engines_added (MedicineSearchEngine engine);
        public signal void medicine_search_engines_removed (MedicineSearchEngine engine);
        
        private const string [] supported_db_versions = { "0.1", "0.2" };
        private const string db_version = "0.2";
    
        public ResourceManager (string program_name = "clinica", ErrorCallback? callback = null) {
            if (callback != null)
                error_callback = callback;
            else {
                error_callback = (source, message) => debug (message);
            }
        
            /* Connect error callback */
            error.connect ((t,l) => callback(t,l));
        
            /* Init databases and configuration files
             * if needed. */
            local_config_directory = Environment.get_user_config_dir ();
            local_config_directory = Path.build_filename (local_config_directory,
                                                          "clinica");
            database_path = Path.build_filename (local_config_directory,
                                                 "clinica.db");
                                                 
            /* Check if configuration directory exists */
            if (!FileUtils.test (local_config_directory, 
                                  FileTest.IS_DIR)) {
                DirUtils.create_with_parents (local_config_directory, -1);
            }
                                                 
            /* Read database version if existant, saved actual version otherwise.
             * This may be used in future to perform one-time upgrade of the database */
            var version_db_file = Path.build_filename (local_config_directory, 
            										    ".clinica_db_version");
            										   
        	if (!FileUtils.test (version_db_file, FileTest.IS_REGULAR)) {
        		/* Create the file with the version of the database */
        		try {
	        		FileUtils.set_contents(version_db_file, db_version);
	        	} catch (Error e) {
	        		error (_("Error creating some configuration files, check permission on %s".printf(version_db_file)));
	        	}
        	}
        	else {
        		string content;
        		try {
        			FileUtils.get_contents(version_db_file, out content);
        		} catch (Error e) {
        			error (_("Error reading some configuration files, check permission on %s".printf(version_db_file)));
        		}
        		
        		bool upgraded_correcty = true;
        		if (db_version != content) {
        		    /* Ask the user if he really wants to upgrade */
        		    if (!ask_for_upgrade()) {
        		        debug ("The user doesn't want to upgrade, exiting.");
        		        Posix.exit (0);
        		    }
        		
        		    upgraded_correcty = false;
        		    /* Try to determine if the version of the local database is supported */
        		    if (content in supported_db_versions) {
        		        if (upgrade_database (content)) {
        		            upgraded_correcty = true;
        		            try {
            		            FileUtils.set_contents (version_db_file, db_version);
            		        } catch (GLib.Error e) {
            		            error (_("Failure while settings new database version to %s").printf (db_version));
                            }
        		        } else {
        		            error (_("Failure while upgrading database"));
        		        }
        		    }
        		    else
        			    error (_("Version of the database is not compatible"));
        		}
        		
        		if (!upgraded_correcty) {
      		        debug ("Upgrade failed, exiting.");
        		    Posix.exit (1);
        		}
        	}
                                                
            /* Find prefix */
            prefix = "/usr/local";
            if (!FileUtils.test (Path.build_filename(prefix, "bin", "clinica"), FileTest.IS_EXECUTABLE)) {
            	prefix = "/usr";
            }
            
            /* Check if this is a local build and if that's the case then
             * load local interface files. Since we use libtoo to compie, the real
             * program will be in the dir .libs, so added the .. :) */
            is_local = true;
            string [] source_files = { "AboutDialog.vala", "Page.vala", "StartPage.vala" };
            string src_base = Path.build_filename(Path.get_dirname (program_name), "..", "..");
            foreach(string source in source_files) {
	            if (!FileUtils.test (Path.build_filename(src_base, "libclinica", source), FileTest.IS_REGULAR)) {
		            is_local = false;
	            }
            }
             
            if (is_local) {
                debug ("Local build detected, loading files from here");
            	data_path = src_base;
            } else {
	            data_path = Path.build_filename(prefix, "share", "clinica");
        	}
	       
            /* Find UI files */
            ui_files_path = Path.build_filename(data_path, "ui");
            
            /* Find icons */
            icons_files_path = Path.build_filename (ui_files_path, "icons");


            /* Preparing for the creations of the settings, if the build is local try to
             * use local schema file */
            if (is_local) {
                debug ("Trying to compile my own gsettings schema");
            try {
	            Process.spawn_sync (null, {Environment.find_program_in_path ("glib-compile-schemas"), 
                                               src_base}, 
                                        null, GLib.SpawnFlags.STDERR_TO_DEV_NULL, null);
                } catch (SpawnError error) {
                    debug ("Cannot compile the gsettings schema, clinica is likely to not work: %s", error.message);
                }
                Environment.set_variable ("GSETTINGS_SCHEMA_DIR", src_base, true);
                Environment.set_variable ("GI_TYPELIB_PATH", 
                    Path.build_filename (src_base, "_build_", "libclinica"), true);
            }
            
            /* Create configuration */
            settings = new Settings (this);
            
            /* Load all extensions found... this is quite ugly, but an
             * interface to select loaded plugins is still needed. 
             * For now, do it in the idle cycle to allow clinica to start
             * faster. */
             if (settings.get_boolean ("use-plugins"))
                 load_plugins.begin ();
        }
        
        private bool ask_for_upgrade () {
            var mb = new MessageDialog (null,
                DialogFlags.DESTROY_WITH_PARENT |
                DialogFlags.MODAL,
                MessageType.QUESTION,
                ButtonsType.YES_NO,
                "%s", "Database upgrade");
            mb.format_secondary_markup (_("This is a version of Clinica newer than the one that created the\npatients database installed on the system.\nUsing this version requires upgrading the database, and\n<b>the old version will not be able to use it anymore</b>.\nDo you wish to perform this one-time upgrade?\n"));
            mb.set_title (_("Upgrade database"));
            if (mb.run () == ResponseType.YES) {
                mb.destroy ();
                return true;
            }
            else {
                mb.destroy ();
                return false;
           }
        }
        
        /**
         * @brief Instantiate the plugin engine and load
         * all the plugins present in the directory
         */
        public async void load_plugins () {
            /* Setup plugin engine */
            plugin_engine = new PluginEngine (this);   
            plugin_engine.enable_loader ("python");
        
            unowned GLib.List<weak Peas.PluginInfo> list = 
                plugin_engine.get_plugin_list ();
                
            string [] active_plugins = settings.get_active_plugins ();
            
            foreach (Peas.PluginInfo info in list) {
                if (info.get_module_name () in active_plugins) {
                    debug ("Activating plugin %s".printf (info.get_name ()));
                    plugin_engine.load_plugin (info);
                }
                else {
                    debug ("Plugin %s found but disabled".printf (info.get_name ()));
                }
            }
        }
        
        public void register_medicine_search_engine (MedicineSearchEngine engine) {
            medicine_search_engines.append (engine);
            
            /* If no medicine_search_engine is loaded, load this one */
            if (settings.selected_search_engine == null) {
                debug ("Loading search engine %s", engine.get_name ());
                settings.selected_search_engine = engine;
            }
            medicine_search_engines_added  (engine);
        }
        
        public bool unregister_medicine_search_engine (MedicineSearchEngine engine) {
            if (engine == settings.selected_search_engine) {
                settings.selected_search_engine = null;
                settings.set_string ("medicine-search-engine", "");
            }
            foreach (MedicineSearchEngine e in medicine_search_engines) {
                if (e == engine) {
                    medicine_search_engines.remove (e);
                    medicine_search_engines_removed (e);
                    return true;
                }
            }
            return false;
        }
        
        public string get_ui_file (string filename) {
            return Path.build_filename (ui_files_path, filename);
        }
            
        public string get_image_file (string image_name) {
        	return Path.build_filename (icons_files_path, image_name);
        }
        
        public string get_plugins_dir () {
            if (is_local)
                return Path.build_filename (data_path, "_build_", "plugins");
            else {
                return Path.build_filename (prefix, "lib", "clinica", "plugins");
            }
        }
        
        private bool upgrade_database (string db_version) {
            /* Try to open the database first */
            if (!(Database.open (database_path, out db) == OK)) {
                error (_("Error upgrading database, please check your installation"));
            }
            
            string sql;
            Statement stmt;
            int rc;
            
            /* Performing upgrade from 0.1 */
            if (db_version == "0.1") {
                /* We need to create the new table for the events */
               sql = "CREATE TABLE IF NOT EXISTS events (id INTEGER PRIMARY KEY, title TEXT, description TEXT, venue TEXT, patient INTEGER, visit INTEGER, date TEXT);";
               db.prepare (sql, -1, out stmt, null);
               rc = stmt.step ();
               
               if (rc != DONE) {
                   error (_("Error upgrading the database from version 0.1 to 0.2, sqlite exit code: %d".printf (rc)));
                   return false;
               } else {
                   db_version = "0.2";
               }
            }
            
            return true;
        }
        
        /**
         * @brief Check that all the resources needed by Clinica
         * are actually available, and if they are not create them.
         */
        public void initResources () {
            int rc;
            Statement stmt;
            string sql;
            
            /* Open database and, if it does not exists, create it */
            if (!(Database.open (database_path, out db) == OK)) {
                error ("Error opening database.");
            };
            
            /* Check if the required tables exists */
            sql = "SELECT * from sqlite_master WHERE type='table';";
            db.prepare (sql, -1, out stmt, null);
            rc = stmt.step ();
            
            if (rc == DONE) {
                /* That means no tables in the database */
                initDatabase ();
            }

            /* Load stores */
            doctor_list_store = new DoctorListStore (this);
            patient_list_store = new PatientListStore (this);
            visit_list_store = new VisitListStore (this);
            event_list_store = new EventListStore (this);
        }
        
        
        /**
         * @brief Init the database with the required tables
         */
        private void initDatabase () {
            Statement stmt; 
            string sql;
            string err_msg;
            
            /* Create config table */
            sql = "CREATE TABLE config (id INTEGER PRIMARY KEY, " +
                  "key TEXT, value TEXT);";
            db.prepare (sql, -1, out stmt, null);
            if (!(stmt.step () == DONE)) {
                err_msg = db.errmsg ();
                error ("Error creating config structure in the Database: "
                + err_msg);
            }
        }            
        
        /**
         * @brief Return a Doctor object associated to the given ID
         */
        public Doctor getDoctor (int ID) {
            Doctor doctor = new Doctor.with_id (this, ID);
            return doctor;
        }
        
        /**
         * @brief Return a Patient object associated to the given ID
         */
        public Patient getPatient (int ID) {
            Patient p = new Patient.with_id (this, ID);
            return p;
        }
        
        /**
         * @brief Return a Visit object associated to the given ID
         */
        public Visit getVisit (int ID) {
            Visit visit = new Visit.with_id (this, ID);
            return visit;
        }
        
        
    }
   

}
