/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
namespace Clinica {

    public class EventIterator : SqlDataIterator {
    
        private ResourceManager resource_manager { get; set; }
        
        public EventIterator (ResourceManager resources) {
            /* Grab the resource manager and chain the SqlDataIterator constructor,
             * ordering by date in decreasing order. */
            base (resources.db, resources.events_table, "date", false);
            resource_manager = resources;
        }
        
        public EventIterator.with_day (ResourceManager resources, DateTime date, bool descending = true) {
            base.with_like (resources.db, resources.events_table, "date", descending, 
                "date BETWEEN '%s' AND '%s'".printf (SqlDataType.datetime_to_string (date),
                                                     SqlDataType.datetime_to_string (date.add_days (1).add_minutes(-1))));
            resource_manager = resources;
        }
        
        public new EventIterator iterator () {
            return this;
        }
        
        public new Event get () {
            return new Event.with_id (resource_manager, base.get ());
        }
    }

    public class Event : SqlDataType {
    
        private ResourceManager resource_manager { get; set; }
        
        /* Properties mapped to the database */
        public string title {
            get { return get_text ("title"); }
            set { set_text ("title", value); }
        }
        
        public string description {
            get { return get_text ("description"); }
            set { set_text ("description", value); }
        }
        
        public string venue {
            get { return get_text ("venue"); }
            set { set_text ("venue", value); }
        }
        
        /** 
         * @brief Patient associated to this event, if one exists,
         * or null otherwise
         */
        private Patient? _patient;
        public Patient? patient {
            get { 
                int id = get_integer ("patient");
                if (id == 0) {
                    return null;
                }
                else {
                    _patient = new Patient.with_id (resource_manager, id);
                    return _patient;
                }
            }
            set {
                if (value != null) {
                    set_integer ("patient", value.get_id ());
                    _patient = value;
                }
                else {
                    set_integer ("patient", 0);
                }
            }
        }
        
        
        /** 
         * @brief Visit associated to this event, if one exists,
         * or null otherwise
         */
        private Visit? _visit;
        public Visit? visit {
            get { 
                int id = get_integer ("visit");
                if (id == 0) {
                    return null;
                }
                else {
                    _visit = new Visit.with_id (resource_manager, id);
                    return _visit;
                }
            }
            set {
                if (value != null) {
                    set_integer ("visit", value.get_id ());
                    _visit = value;
                }
                else {
                    set_integer ("visit", 0);
                }
            }
        }
        
        /**
         * @brief Date associated to this event.
         */
        private DateTime _date;
        public DateTime date {
        	get { 
        		_date = get_date ("date");
        		return _date;
        	}
        	set {
        		_date = value;
        		set_date ("date", _date);
        	}
        }
    
        public Event (ResourceManager resources) {
            base (resources.db);
            resource_manager = resources;
            error.connect ((t,l) => resources.error_callback(t,l));
            table_name = resource_manager.events_table;
            
            add_text_field ("title");
            add_text_field ("description");
            add_text_field ("venue");
            add_integer_field ("patient");
            add_integer_field ("visit");
            add_date_field ("date");
            
            init_resources ();
            
            visit = null;
            patient = null;
        }
        
        public Event.with_id (ResourceManager resources, int id) {
            this (resources);
            load (id);
        }
        
        public static new EventIterator all (ResourceManager resources) {
            return new EventIterator (resources);
        }
        
        public static new EventIterator for_day (ResourceManager resources, DateTime date, bool descending = false) {
            return new EventIterator.with_day (resources, date, descending);
        }
    }
}
