/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
 using Gee;
 using Sqlite;
 
 namespace Clinica {
 
         
    public class DoctorIterator : SqlDataIterator {
    
		private ResourceManager resource_manager;
    
    	/**
    	 * @brief Iterator over all the Doctors present
    	 * in the database. 
    	 */
        public DoctorIterator (ResourceManager resources) {
            base (resources.db, resources.doctor_table);
            resource_manager = resources;
        }
        
        public new DoctorIterator iterator () { return this; }
        
        public new Doctor get () {
            return new Doctor.with_id (resource_manager, base.get ());
        }
    }
 
    /**
     * @brief A doctor associated with some patients
     */
    public class Doctor : SqlDataType {
        
        public new static DoctorIterator all (ResourceManager resources) {
            return new DoctorIterator (resources);
        }
        
        /* FIELDS */
        public string given_name {
        	get { return get_text ("given_name"); }
        	set { set_text ("given_name", value); }
        }
        
        public string surname {
        	get { return get_text ("surname"); }
        	set { set_text ("surname", value); }
        }
        
        public string phone {
        	get { return get_text ("phone"); }
        	set { set_text ("phone", value); }
        }
        
        public string mobile {
        	get { return get_text ("mobile"); }
        	set { set_text ("mobile", value); }
        }
        
		private ResourceManager resource_manager;
    
        public Doctor (ResourceManager resources) {
            base (resources.db);
            resource_manager = resources;
            table_name = resources.doctor_table;
            error.connect ((t,l) => resources.error_callback(t,l));
            
            add_text_field ("given_name");
            add_text_field ("surname");
            add_text_field ("phone");
            add_text_field ("mobile");
            
            /* Finishing connecting the object to the database */
            init_resources ();
        }
        
        public Doctor.with_id (ResourceManager resources, int ID) {
            this (resources);
            load (ID);
        }
        
        public string get_complete_name () {
        	return string.join(" ", given_name, surname);
       	}
       	
       	public bool has_patients () {
       		GLib.List<int> associated_ids = associated_ids(resource_manager.patient_table, "doctor");
       		return (associated_ids.length () != 0);
       	}
       	
       	public new void remove () {
       		if (has_patients ()) 
       			error (_("You cannot delete a doctor with associated patients. Delete his patients first."));
       		else
	       		base.remove ();
       	}
        
        

    }   
 }
