/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
using Gtk;
 
namespace Clinica {
    
    /**
     * @brief Main application 
     */
    public class Application {
        
        private static bool version;
        
        private const OptionEntry[] option_entries = {
            { "version", 'v', 0, OptionArg.NONE, ref version, "Print version and exit", null },
            { null }
        };
        
        public static int main (string[] args) {

            /* Init gtk */
            Gtk.init (ref args);
            
            /* Init gdk threads */
            Gdk.threads_init ();
            
            /* Parse options */
            try {
                var opt_context = new OptionContext ("- medical records manager");
                opt_context.set_help_enabled (true);
                opt_context.add_main_entries (option_entries, "clinica");
                opt_context.parse (ref args);
            } catch (OptionError e) {
                stderr.printf ("Option parsing failed: %s\n", e.message);
                return 1;
            }
            
            if (version) {
                stderr.printf (_("Clinica %s\n").printf (Config.PACKAGE_VERSION));
                return 0;
            }
        
            /* Init resource manager and connect error function
             * to a message display for the user */
            Gdk.threads_enter ();
            var resources = new ResourceManager (args[0], Utils.show_error_message);
            
            /* Init gettext translation system */
            Intl.textdomain("clinica");
            
            /* Check where are the translations. At the moment being we are sure that Italian
             * translations are present, so check for them in the /usr/share/locale/ directory.
             * If they are not present there, type with /usr/local, and if not present there, 
             * try with autodetection. */
            if (FileUtils.test("/usr/share/locale/it/LC_MESSAGES/clinica.mo", FileTest.IS_REGULAR))
	            Intl.bindtextdomain("clinica", "/usr/share/locale");
	        else if (FileUtils.test("/usr/local/share/locale/it/LC_MESSAGES/clinica.mo", FileTest.IS_REGULAR))
	        	Intl.bindtextdomain("clinica", "/usr/local/share/locale");
	        else
	        	Intl.bindtextdomain("clinica", null);
                                                            
            /* Init resources */
            resources.initResources();

            /* Start interface and connect error signal */
            var user_interface = new UserInterface (resources);
            
            /* Actually start the application */
            user_interface.start ();
            Gdk.threads_leave ();
            
            return 0;
        }
    } 
 
 }

