/* tolua: tag methods
** Support code for Lua bindings.
** Written by Waldemar Celes
** TeCGraf/PUC-Rio
** Jul 1998
** $Id: tolua_tm.c,v 1.1.1.1 2000/04/09 12:18:02 mbn Exp $
*/

/* This code is free software; you can redistribute it and/or modify it. 
** The software provided hereunder is on an "as is" basis, and 
** the author has no obligation to provide maintenance, support, updates,
** enhancements, or modifications. 
*/

#include "tolua.h"
#include "tolua_tm.h"
#include "tolua_st.h"
#include "tolua_tt.h"

#include <string.h>



/* Global tables created in Lua:
   .tolua_class: indexed by instance tags, stores the class tables.
   .tolua_clone: indexed by memory address, stores flag indicanting 
                 it is a clone.
   .tolua_mate:  indexed by memory address, stores the associate instance
                 table.
*/
static int tbl_class;     /* stores reference to ".tolua_class" table */
static int tbl_clone;     /* stores reference to ".tolua_clone" table */
static int tbl_mate;      /* stores reference to ".tolua_mate" table */


/* general tags */
static int tag_global;
static int tag_module;
static int tag_class;
static int tag_instance;
static int tag_linstance;
static int tag_indirect;

/* internal function prototype */
static void setmethods (void);

void toluaI_tm_global (lua_Object lo)
{
 lua_pushobject(lo);
 lua_settag(tag_global);
}

void toluaI_tm_module (lua_Object lo)
{
 lua_pushobject(lo);
 lua_settag(tag_module);
}

void toluaI_tm_class (lua_Object lo, char* name)
{
 int tag = lua_newtag();
 char type[64] = "class "; strcat(type,name);
 lua_copytagmethods(tag,tag_class);
 toluaI_tt_register(tag,type);
 toluaI_tt_sethierarchy(tag,tag_class);
 lua_pushobject(lo);
 lua_settag(tag);
}

void toluaI_tm_setclass (lua_Object lo)
{
 lua_pushobject(lo);
 lua_settag(tag_class);
}

void toluaI_tm_instance (int tag, lua_Object lo)
{
 lua_pushobject(lua_getref(tbl_class));
 lua_pushnumber(tag);
 lua_pushobject(lo);
 lua_settable();
 lua_copytagmethods(tag,tag_instance);
}

void toluaI_tm_linstance (int tag, lua_Object lo)
{
 lua_pushobject(lua_getref(tbl_class));
 lua_pushnumber(tag);
 lua_pushobject(lo);
 lua_settable();
 lua_copytagmethods(tag,tag_linstance);
}

void* tolua_doclone (void* value, int tag)
{
 lua_pushobject(lua_getref(tbl_clone));
 lua_pushuserdata(value);
 lua_pushnumber(tag);
 lua_settable();
 return value;
}

void* tolua_copy (void* value, unsigned int size)
{
 void* clone = malloc(size);
 if (clone)
  memcpy(clone,value,size);
 else
  tolua_error("insuficient memory");
 return clone;
}

void toluaI_tm_undoclone (void* clone)
{
 lua_Object lo;
 lua_pushobject(lua_getref(tbl_clone));
 lua_pushuserdata(clone);
 lo = lua_gettable();
 if (lua_isnumber(lo))
 {
  int tag = lua_getnumber(lo);
  lua_pushobject(lua_getref(tbl_clone));
  lua_pushuserdata(clone);
  lua_pushnil();
  lua_settable();

  /* get base class */
  lua_pushobject(lua_getref(tbl_class));
  lua_pushnumber(tag);
  lo = lua_rawgettable();

  /* look for destructor */
  lua_pushobject(lo);
  lua_pushstring("delete");
  lo = lua_gettable();
  if (lua_iscfunction(lo))
   lua_getcfunction(lo)();
  else
   free(clone); /* no destructor: use raw free */
 }
}

lua_Object toluaI_tm_getmate (lua_Object lo)
{
 lua_pushobject(lua_getref(tbl_mate)); 
 lua_pushuserdata(lua_getuserdata(lo));
 return lua_rawgettable();
}

lua_Object toluaI_tm_getclass (lua_Object lo)
{
 lua_pushobject(lua_getref(tbl_class)); 
 lua_pushnumber(lua_tag(lo));
 return lua_rawgettable();
}

void toluaI_tm_init (void)
{
 lua_Object lo;
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_class");
 lua_pushobject(lo); tbl_class = lua_ref(1);
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_clone");
 lua_pushobject(lo); tbl_clone = lua_ref(1);
 lo = lua_createtable(); 
 lua_pushobject(lo); lua_setglobal(".tolua_mate");
 lua_pushobject(lo); tbl_mate = lua_ref(1);
 toluaI_st_store(&tbl_class);
 toluaI_st_store(&tbl_clone);
 toluaI_st_store(&tbl_mate);

 tag_global = lua_newtag(); 	toluaI_st_store(&tag_global);
 tag_module = lua_newtag(); 	toluaI_st_store(&tag_module);
 tag_class = lua_newtag(); 	toluaI_st_store(&tag_class);
 tag_instance = lua_newtag(); 	toluaI_st_store(&tag_instance);
 tag_linstance = lua_newtag(); 	toluaI_st_store(&tag_linstance);
 tag_indirect = lua_newtag(); 	toluaI_st_store(&tag_indirect);

 toluaI_tt_register(tag_global,"generic variable");
 toluaI_tt_register(tag_module,"generic module");
 toluaI_tt_register(tag_class,"generic class");
 toluaI_tt_register(tag_instance,"generic instance");
 toluaI_tt_register(tag_linstance,"generic lua instance");
 toluaI_tt_register(tag_indirect,"generic indirect");

 /* allows modules and classes to be used as ordinary tables */
 toluaI_tt_sethierarchy(tag_module,tolua_tag_table);
 toluaI_tt_sethierarchy(tag_class,tolua_tag_table);

 setmethods();
}

static void map (void)
{
 lua_Object m = lua_getparam(1);
 lua_Object n = lua_getparam(2);
 /* do not pass string fields starting with a dot */
 if (!lua_isstring(n) || *lua_getstring(n)!='.')
 {
  lua_pushobject(m);
  lua_rawsetglobal(lua_getstring(n));
 }
}

void toluaI_tm_using (lua_Object module)
{
 lua_Object t = lua_createtable();
 lua_pushobject(t);
 lua_pushstring(".module");
 lua_pushobject(module);
 lua_settable();
 lua_pushobject(t);
 lua_settag(tag_indirect);

 lua_pushobject(module);
 lua_pushobject(t);
 lua_pushcclosure(map,1);
 lua_callfunction(lua_getglobal("foreach"));

 lua_pushobject(module);
 lua_pushstring(".get");
 lua_pushobject(lua_gettable());
 lua_pushobject(t);
 lua_pushcclosure(map,1);
 lua_callfunction(lua_getglobal("foreach"));
}

/********************************************************** tag methods */

#if 1

/* tag methods coded in C */

/* generic gettable */
static void oo_gettable 
(lua_Object table, lua_Object base, lua_Object index)
{
 while (lua_istable(base))
 {
  lua_Object value;
  lua_pushobject(base); lua_pushobject(index); 
  value = lua_rawgettable();
  if (!lua_isnil(value))
  {
   lua_pushobject(value);
  }
  else if (lua_isnumber(index))
  {
   lua_Object f;
   lua_pushobject(base); lua_pushstring("operator_get");
   f = lua_rawgettable();
   if (!lua_isnil(f))
   {
    lua_pushobject(table);
    lua_pushobject(index);
    lua_callfunction(f);
    lua_pushobject(lua_getresult(1));
    return;
   }
  }
  else
  {
   lua_Object get;
   lua_pushobject(base); lua_pushstring(".get");
   get = lua_rawgettable();
   if (!lua_isnil(get))
   {
    lua_Object f;
    lua_pushobject(get); lua_pushobject(index);
    f = lua_rawgettable();
    if (!lua_isnil(f))
    {
     lua_pushobject(table);
     lua_pushobject(index); /* need to access array field */
     lua_callfunction(f);
     lua_pushobject(lua_getresult(1));
    }
   }
  }
  lua_pushobject(base); lua_pushstring(".base");
  base = lua_rawgettable();
 }
 lua_pushnil();
}

/* generic settable */
static int oo_settable 
(lua_Object table, lua_Object base, lua_Object index, lua_Object value) 
{
 while (lua_istable(base))
 {
  lua_Object set;
  lua_pushobject(base); lua_pushstring(".set");
  set = lua_rawgettable();
  if (!lua_isnil(set))
  {
   lua_Object f;
   lua_pushobject(set); lua_pushobject(index);
   f = lua_rawgettable();
   if (!lua_isnil(f))
   {
    lua_pushobject(table);
    lua_pushobject(value);
    lua_callfunction(f);
    return 1;
   }
  }
  lua_pushobject(base); lua_pushstring(".base");
  base = lua_rawgettable();
 }
 return 0;
}

/* class tag methods */
static void class_index (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 oo_gettable(table,table,index);
}
static void class_settable (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 lua_Object value = lua_getparam(3);
 if (oo_settable(table,table,index,value) == 0)
 {
  lua_pushobject(table);
  lua_pushobject(index);
  lua_pushobject(value);
  lua_rawsettable();
 }
}

/* instance tags */
static void instance_gettable (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 lua_Object base = toluaI_tm_getclass(table);
 lua_Object mate = toluaI_tm_getmate(table);
 if (!lua_isnil(mate))
 {
  lua_Object value;
  lua_pushobject(mate);
  lua_pushobject(index);
  value = lua_rawgettable();
  if (!lua_isnil(value))
  {
   lua_pushobject(value);
   return;
  }
 }
 oo_gettable(table,base,index);
}
static void instance_settable (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 lua_Object value = lua_getparam(3);
 lua_Object base = toluaI_tm_getclass(table);
 if (lua_isnumber(index))
 {
  lua_Object f;
  lua_pushobject(base); lua_pushstring("operator_set");
  f = lua_rawgettable();
  if (!lua_isnil(f))
  {
   lua_pushobject(table);
   lua_pushobject(index);
   lua_pushobject(value);
   lua_callfunction(f);
   return;
  }
 }
 if (oo_settable(table,base,index,value) == 0)
 {
  lua_Object mate = toluaI_tm_getmate(table);
  if (lua_isnil(mate))
  {
   mate = lua_createtable();
   lua_pushobject(lua_getref(tbl_mate));
   lua_pushuserdata(lua_getuserdata(table));
   lua_pushobject(mate);
   lua_rawsettable();
  }
  lua_pushobject(mate);
  lua_pushobject(index);
  lua_pushobject(value);
  lua_rawsettable();
 }
}
static void instance_gc (void)
{
 lua_Object obj = lua_getparam(1);
 toluaI_tm_undoclone(lua_getuserdata(obj)); 
}
static lua_Object gen_operator (void)
{
 lua_Object op1 = lua_getparam(1);
 lua_Object op2 = lua_getparam(2);
 lua_Object event = lua_getparam(3);
 lua_Object f;
 char name[16] = "operator_"; strcat(name,lua_getstring(event));
 lua_pushobject(op1);
 lua_pushstring(name); 
 f = lua_gettable();
 lua_pushobject(op1);
 lua_pushobject(op2);
 lua_callfunction(f);
 return lua_getresult(1);
}
static void instance_operator (void)
{
 lua_pushobject(gen_operator());
}
static void instance_relational (void)
{
 lua_Object r = gen_operator();
 if ((int)lua_getnumber(r)==0) lua_pushnil();
 else lua_pushobject(r);
}

/* lua instance tags */
static void linstance_index (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 lua_Object base = toluaI_tm_getclass(table);
 oo_gettable(table,base,index);
}


/* module tag methods */
static void module_index (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 lua_Object get;
 lua_pushobject(table); lua_pushstring(".get");
 get = lua_rawgettable();
 if (!lua_isnil(get))
 {
  lua_Object f;
  lua_pushobject(get); lua_pushobject(index);
  f = lua_rawgettable();
  if (!lua_isnil(f))
  {
   lua_callfunction(f);
   lua_pushobject(lua_getresult(1));
   return;
  }
 }
 lua_pushnil();
}
static void module_settable (void)
{
 lua_Object table = lua_getparam(1);
 lua_Object index = lua_getparam(2);
 lua_Object value = lua_getparam(3);
 lua_Object set;
 lua_pushobject(table); lua_pushstring(".set");
 set = lua_rawgettable();
 if (!lua_isnil(set))
 {
  lua_Object f;
  lua_pushobject(set); lua_pushobject(index);
  f = lua_rawgettable();
  if (!lua_isnil(f))
  {
   lua_pushobject(value);
   lua_callfunction(f);
   return;
  }
 }
 lua_pushobject(table);
 lua_pushobject(index);
 lua_pushobject(value);
 lua_rawsettable();
}

/* global variable tag methods */
static void global_getglobal (void)
{
/* lua_Object varname = lua_getparam(1); */
 lua_Object value = lua_getparam(2);
 lua_pushobject(value); lua_pushstring(".get");
 lua_callfunction(lua_rawgettable());
 lua_pushobject(lua_getresult(1));
}
static void global_setglobal (void)
{
/* lua_Object varname = lua_getparam(1); */
 lua_Object value = lua_getparam(2);
 lua_Object newvalue = lua_getparam(3);
 lua_Object f;
 lua_pushobject(value); lua_pushstring(".set");
 f = lua_rawgettable();
 if (lua_isnil(f))
  lua_error("value of const variable cannot be changed");
 else
 {
  lua_pushobject(newvalue);
  lua_callfunction(f);
 }
}

/* indirect tag methods */
static void indirect_getglobal (void)
{
 lua_Object varname = lua_getparam(1);
 lua_Object value = lua_getparam(2);
 lua_pushobject(value);
 lua_pushstring(".module");
 lua_pushobject(lua_gettable());
 lua_pushobject(varname);
 lua_pushobject(lua_gettable());
}
static void indirect_setglobal (void)
{
 lua_Object varname = lua_getparam(1);
 lua_Object value = lua_getparam(2);
 lua_Object newvalue = lua_getparam(3);
 lua_pushobject(value);
 lua_pushstring(".module");
 lua_pushobject(lua_gettable());
 lua_pushobject(varname);
 lua_pushobject(newvalue);
 lua_settable();
}

static void setmethods (void)
{
 /* global variable */
 lua_pushcfunction(global_getglobal);
 lua_settagmethod(tag_global,"getglobal");
 lua_pushcfunction(global_setglobal);
 lua_settagmethod(tag_global,"setglobal");

 /* module */
 lua_pushcfunction(module_index);
 lua_settagmethod(tag_module,"index");
 lua_pushcfunction(module_settable);
 lua_settagmethod(tag_module,"settable");

 /* class */
 lua_pushcfunction(class_index);
 lua_settagmethod(tag_class,"index");
 lua_pushcfunction(class_settable);
 lua_settagmethod(tag_class,"settable");

 /* instance */
 lua_pushcfunction(instance_gettable);
 lua_settagmethod(tag_instance,"gettable");
 lua_pushcfunction(instance_settable);
 lua_settagmethod(tag_instance,"settable");
 lua_pushcfunction(instance_operator);
 lua_settagmethod(tag_instance,"add");
 lua_pushcfunction(instance_operator);
 lua_settagmethod(tag_instance,"sub");
 lua_pushcfunction(instance_operator);
 lua_settagmethod(tag_instance,"mul");
 lua_pushcfunction(instance_operator);
 lua_settagmethod(tag_instance,"div");
 lua_pushcfunction(instance_relational);
 lua_settagmethod(tag_instance,"lt");
 lua_pushcfunction(instance_relational);
 lua_settagmethod(tag_instance,"gt");
 lua_pushcfunction(instance_relational);
 lua_settagmethod(tag_instance,"le");
 lua_pushcfunction(instance_relational);
 lua_settagmethod(tag_instance,"ge");
 lua_pushcfunction(instance_gc);
 lua_settagmethod(tag_instance,"gc");

 /* lua instance */
 lua_pushcfunction(linstance_index);
 lua_settagmethod(tag_linstance,"index");

 /* indirect */
 lua_pushcfunction(indirect_getglobal);
 lua_settagmethod(tag_indirect,"getglobal");
 lua_pushcfunction(indirect_setglobal);
 lua_settagmethod(tag_indirect,"setglobal");
}

#else

/* tag methods coded in Lua */

static void setmethods (void)
{
 /* set tag methods for general tags */
 lua_dofile("tm.lua");

 /* global variable */
 lua_pushobject(lua_getglobal("global_getglobal"));
 lua_settagmethod(tag_global,"getglobal");
 lua_pushobject(lua_getglobal("global_setglobal"));
 lua_settagmethod(tag_global,"setglobal");

 /* module */
 lua_pushobject(lua_getglobal("module_index"));
 lua_settagmethod(tag_module,"index");
 lua_pushobject(lua_getglobal("module_settable"));
 lua_settagmethod(tag_module,"settable");

 /* class */
 lua_pushobject(lua_getglobal("class_index"));
/* lua_pushcfunction(class_index); */
 lua_settagmethod(tag_class,"index");
 lua_pushobject(lua_getglobal("class_settable"));
 lua_settagmethod(tag_class,"settable");

 /* instance */
 lua_pushobject(lua_getglobal("instance_gettable"));
 lua_settagmethod(tag_instance,"gettable");
 lua_pushobject(lua_getglobal("instance_settable"));
 lua_settagmethod(tag_instance,"settable");
 lua_pushobject(lua_getglobal("instance_operator"));
 lua_settagmethod(tag_instance,"add");
 lua_pushobject(lua_getglobal("instance_operator"));
 lua_settagmethod(tag_instance,"sub");
 lua_pushobject(lua_getglobal("instance_operator"));
 lua_settagmethod(tag_instance,"mul");
 lua_pushobject(lua_getglobal("instance_operator"));
 lua_settagmethod(tag_instance,"div");
 lua_pushobject(lua_getglobal("instance_relational"));
 lua_settagmethod(tag_instance,"lt");
 lua_pushobject(lua_getglobal("instance_relational"));
 lua_settagmethod(tag_instance,"gt");
 lua_pushobject(lua_getglobal("instance_relational"));
 lua_settagmethod(tag_instance,"le");
 lua_pushobject(lua_getglobal("instance_relational"));
 lua_settagmethod(tag_instance,"ge");
 lua_pushobject(lua_getglobal("instance_gc"));
 lua_settagmethod(tag_instance,"gc");

 /* lua instance */
 lua_pushobject(lua_getglobal("linstance_index"));
 lua_settagmethod(tag_linstance,"index");

 /* indirect */
 lua_pushobject(lua_getglobal("indirect_getglobal"));
 lua_settagmethod(tag_indirect,"getglobal");
 lua_pushobject(lua_getglobal("indirect_setglobal"));
 lua_settagmethod(tag_indirect,"setglobal");
}

#endif


