/*
	$Id: init_linux.cpp,v 1.2 2000/04/25 21:40:16 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		Initializes ClanLib under Linux. Home of the legendary main().

*/

#include "Core/precomp.h"

#include <sys/time.h>
#include <unistd.h>
#include <dlfcn.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include <assert.h>
#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include "API/Core/System/system.h"
#include "API/Core/System/clanapp.h"
#include "API/Core/System/cl_assert.h"
#include "API/Core/Input/input.h"
#include "API/Core/Input/keyboard.h"
#include "API/Core/Sound/sound.h"
#include "API/Core/Display/display.h"
#include "API/Core/Display/displaycard.h"
#include "API/Core/System/error.h"
#include "API/core.h"
#include "implementation.h"
#include "appconf.h"
#include "Core/DatafileCompiler/datafile_compiler.h"

extern "C"
{
	#include <Hermes/Hermes.h>
}

#include "init_linux.h"


#define TARGET_DISPLAY 0
#define TARGET_SOUND 1
#define TARGET_NETWORK 2


struct target_structure {
	void *dlhandle;
	clan_id_proc module_id;
	clan_id_proc module_abbr;
	clan_init_proc module_init;
};


CL_DisplayCard *cl_current_displaycard = NULL;
CL_SoundCard *cl_current_soundcard = NULL;
long _begin_time;



int select_display_target(const struct dirent *d)
{ 
	if (strstr(d->d_name,"libclan-display") == d->d_name &&
		strstr(d->d_name,".so") != NULL &&
		strstr(d->d_name,CL_VERSION_STRING) != NULL)
	{
		return 1;
	}
	else
	{
		return 0; 
	}
}

int select_sound_target(const struct dirent *d)
{ 
	if (strstr(d->d_name,"libclan-sound")==d->d_name &&
		strstr(d->d_name,".so") != NULL &&
		strstr(d->d_name,CL_VERSION_STRING) != NULL)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}


int select_network_target(const struct dirent *d)
{ 
	if (strstr(d->d_name,"libclan-network")==d->d_name &&
		strstr(d->d_name,".so") != NULL &&
		strstr(d->d_name,CL_VERSION_STRING) != NULL)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

#ifdef USE_DYN

static int num_targets_display=0;
static int num_targets_sound=0;
static int num_targets_network=0;

target_structure* detect_targets(int target_type,const char* target_dir) 
{
	struct dirent **filenames;
	int res = 0;

//	std:cout << "Scanning " << target_dir << std::endl;

	switch(target_type)
	{
	case TARGET_DISPLAY:
		res = scandir(
			target_dir,
			&filenames,
			select_display_target,
			alphasort);

		num_targets_display = res;
		break;

	case TARGET_SOUND:
		res = scandir(
			target_dir,
			&filenames,
			select_sound_target,
			alphasort);

		num_targets_sound = res;
		break;

	case TARGET_NETWORK:
		res = scandir(
			target_dir,
			&filenames,
			select_network_target,
			alphasort);

		num_targets_network=res;
		break;

	default:
		cl_assert(false);
	}

	if (res > 0)
	{
		char fullname[256];

		target_structure *targets=new target_structure[res];

		for (int i=0;i<res;i++)
		{
			targets[i].dlhandle=0;

			strcpy(fullname,target_dir);
			strcat(fullname,"/");
			strcat(fullname,filenames[i]->d_name);

//			std::cout << "Trying " << filenames[i]->d_name << "... ";
			
			targets[i].dlhandle=dlopen(fullname,RTLD_NOW);
			if (!targets[i].dlhandle)
			{
				std::cout << "dlopen failed!" << std::endl;
				std::cout << dlerror() << std::endl;
				continue;
			}

			char *error;

			targets[i].module_id = (clan_id_proc) dlsym(
					targets[i].dlhandle,
					"clan_module_identify");

			if ((error = dlerror()) != NULL)
			{
				std::cout << "identification failed!" << std::endl;
				continue;
			}

			targets[i].module_abbr = (clan_id_proc) dlsym(
				targets[i].dlhandle,
				"clan_module_abbreviation");

			if ((error = dlerror()) != NULL)
			{
				std::cout << "abbreviation identification failed!" << std::endl;
				continue;
			}
			
			targets[i].module_init = (clan_init_proc) dlsym(
				targets[i].dlhandle,
				"clan_module_init");

			if ((error = dlerror()) != NULL)
			{
				std::cout << "module initialisation unresolved!" << std::endl;
				continue;
			}

//			std::cout << "OK!" << std::endl;
		}

		return targets;
	}

	return 0;
}

#else

#ifdef USE_X11
#include "implementation_xwindow.h"
#endif

#ifdef USE_PTC
#include "implementation_ptc.h"
#endif

#ifdef USE_SVGALIB
#include "implementation_svgalib.h"
#endif

#ifdef USE_GGI
#include "implementation_ggi.h"
#endif

#ifdef USE_FBDEV
#include "implementation_fbdev.h"
#endif

#ifdef USE_OPENGL
#include "implementation_glx.h"
#endif

#ifdef USE_CLANSOUND
#include "implementation_clansound.h"
#endif

#ifdef USE_NETWORK
#include "../../Network/Unix/network_unix.h"
#endif

static int num_targets_display=6;
static int num_targets_sound=1;
static int num_targets_network=1;

target_structure* detect_targets(int target_type,const char* target_dir)
{
	static target_structure disp_targets[6]=
	{
		{(void*)0xff,xwindow_identify,xwindow_abbreviation,xwindow_init},
#ifdef USE_PTC
		{(void*)0xff,ptc_identify,ptc_abbreviation,ptc_init},
#else
		{(void*)0,(clan_id_proc)0,(clan_id_proc)0,(clan_init_proc)0},
#endif
#ifdef USE_SVGALIB
		{(void*)0xff,svgalib_identify,svgalib_abbreviation,svgalib_init},
#else
		{(void*)0,(clan_id_proc)0,(clan_id_proc)0,(clan_init_proc)0},
#endif
#ifdef USE_GGI
		{(void*)0xff,ggi_identify,ggi_abbreviation,ggi_init},
#else
		{(void*)0,(clan_id_proc)0,(clan_id_proc)0,(clan_init_proc)0},
#endif
#ifdef USE_FBDEV
		{(void*)0xff,fbdev_identify,fbdev_abbreviation,fbdev_init},
#else
		{(void*)0,(clan_id_proc)0,(clan_id_proc)0,(clan_init_proc)0},
#endif
#ifdef USE_OPENGL
		{(void*)0xff,glx_identify,glx_abbreviation,glx_init},
#else
		{(void*)0,(clan_id_proc)0,(clan_id_proc)0,(clan_init_proc)0},
#endif
	};

#ifdef USE_CLANSOUND
	static target_structure sound_targets[1]=
	{
		{(void*)0xff,clansound_identify,clansound_abbreviation,clansound_init}
	};
#endif

#ifdef USE_NETWORK
	static target_structure net_targets[1]=
	{
		{(void*)0xff,network_identify,network_abbreviation,network_init}
	};
#endif

	switch (target_type)
	{
	case TARGET_DISPLAY:
		return disp_targets;

#ifdef USE_CLANSOUND
	case TARGET_SOUND:
		return sound_targets;
#endif

#ifdef USE_NETWORK
	case TARGET_NETWORK:
		return net_targets;
#endif
	}

	// Make the compiler happy
	return (target_structure*)0x0;
}


#endif

void CL_SetupCore::init_display()
{
	FileConfig config("clanlib");
	std::string display = config.readEntry("display", "ask");
	std::string scandir = config.readEntry("targetdir", CL_SCANDIR);
	
	// Check if any library forces a particular display target:
	if (strcmp(CL_Force_DispTarget::get_name().c_str(), "") != 0)
	{
		display = CL_Force_DispTarget::get_name();
	}
/*	if (strcmp(CL_Force_DispTarget::name.c_str(), "") != 0)
	{
		display = CL_Force_DispTarget::name;
	}
*/
	target_structure *targets = detect_targets(TARGET_DISPLAY,scandir.c_str());
	target_structure *chosen_display = 0;

	char choice[16];

	if (display == "ask")
	{
		std::cout << std::endl << "Please select a display target by entering its abbreviation: " << std::endl;
		
		for (int i=0;i<num_targets_display;i++) 
		{
			if (targets[i].dlhandle!=0)
			{
				std::cout << "	[" << targets[i].module_abbr() << "] ";
				std::cout << targets[i].module_id() << std::endl;
			}
		}

		std::cout << "Your choice: ";
		std::cin >> choice;
	}
	else
	{
		strcpy(choice, display.c_str());
	}

	for (int i=0;i<num_targets_display;i++) 
	{
		if (targets[i].dlhandle!=0)
		{
			if (!strcmp(targets[i].module_abbr(),choice))
			{
				chosen_display=&targets[i];
				break;
			}
		}
	}

	if (!chosen_display)
	{
		throw CL_Error("Cannot open display target");
	}
	else
	{
		for (int i=0;i<num_targets_display;i++)
		{
			if (targets[i].dlhandle!=0)
			{
				if (!strcmp(targets[i].module_abbr(),display.c_str()))
				{
					chosen_display=&targets[i];
					break;
				}
			}
		}

		if (!chosen_display) throw CL_Error("Cannot open display target");
	}

	chosen_display->module_init();

//	CL_Display::num_cards = CL_Display::cards.size();
	cl_current_displaycard = CL_Display::cards[0];
}

void CL_SetupCore::init_sound()
{
	FileConfig config("clanlib");
	std::string scandir = config.readEntry("targetdir", CL_SCANDIR);

	target_structure* targets=detect_targets(TARGET_SOUND,scandir.c_str());

	if (num_targets_sound<=0 || targets[0].dlhandle==0)
		return;
//		throw CL_Error("Unable to load any sound implementations");

	targets[0].module_init();
	if (CL_Sound::cards.size() > 0) cl_current_soundcard = CL_Sound::cards[0];
}

void CL_SetupCore::init_network()
{
	FileConfig config("clanlib");
	std::string scandir = config.readEntry("targetdir", CL_SCANDIR);
	target_structure* targets=detect_targets(TARGET_NETWORK,scandir.c_str());

	if (num_targets_network<=0 || targets[0].dlhandle==0)
		return;
//		throw CL_Error("Unable to load any network implementations");

	targets[0].module_init();
}

void CL_SetupCore::deinit_display()
{
	int num_cards = CL_Display::cards.size();
	for (int i=0; i<num_cards; i++) delete CL_Display::cards[i];
	CL_Display::cards.clear();
//	CL_Display::num_cards = 0;
}

void CL_SetupCore::deinit_sound()
{
	int num_cards = CL_Sound::cards.size();
	for (int i=0; i<num_cards; i++) delete CL_Sound::cards[i]; 
	CL_Sound::cards.clear();
//	CL_Sound::num_cards = 0;
}

void CL_SetupCore::deinit_network()
{
#ifdef USE_NETWORK
	// TODO: Add deinit code here...
#endif
}

void deinit()
{
	// Deinitialization:
	CL_SetupCore::deinit_display();
	CL_SetupCore::deinit_sound();
	CL_SetupCore::deinit_network();

	int num_keyboards = CL_Input::keyboards.size();
	int num_joysticks = CL_Input::joysticks.size();
	int num_pointers = CL_Input::pointers.size();

	for (int i=0; i<num_keyboards; i++) delete CL_Input::keyboards[i];
	for (int i=0; i<num_joysticks; i++) delete CL_Input::joysticks[i];
	for (int i=0; i<num_pointers; i++) delete CL_Input::pointers[i];
}

int main(int argc, char **argv)
{
	Hermes_Init();

	timeval tv;
	gettimeofday(&tv, NULL);
	_begin_time = (long) tv.tv_sec*(long) 1000+(long) tv.tv_usec/(long) 1000;

	if (CL_ClanApplication::app == NULL)
	{
		std::cout << "ClanLib: No global CL_ClanApplication instance!!!" << std::endl;
		return 255;
	}

	CL_ClanApplication::app->init_modules();

	// Datafile compiler support. Check commandline and run datafile compiler is requested:
	if (argc > 1 && strcmp(argv[1], "-datafile") == 0)
	{
		for (int i=1; i<argc; i++)
		{
			int retval = datafile_main(argc-1, argv+1);

			CL_ClanApplication::app->deinit_modules();
			return retval;
		}
	}
	
	int retval = CL_ClanApplication::app->main(argc, argv);

	CL_ClanApplication::app->deinit_modules();
	deinit();

	return retval;
}

unsigned int CL_System::get_time()
{
	timeval tv;
	gettimeofday(&tv, NULL);

	long tid = (long) tv.tv_sec*(long) 1000 + (long) tv.tv_usec/(long) 1000 - _begin_time;

	return tid;
}

void CL_System::sleep(int millis)
{
	struct timeval tv;
	// select doesn't modify timeval if interrupted on non Linux systems
#ifndef linux 
	int then, now, elapsed;
	then = CL_System::get_time();
#else
	tv.tv_sec = millis/ 1000;
	tv.tv_usec = (millis%1000)*1000;
#endif

	int was_error;	
	do
	{
		errno = 0;
#ifndef linux
		now = CL_System::get_time();
		elapsed = now - then;
		then = now;
		if ( elapsed >= millis )
			break;
		millis -= elapsed;
		tv.tv_sec = millis/1000;
		tv.tv_ysec = (millis%1000)*1000;
#endif
		was_error = select(0, NULL, NULL, NULL, &tv);
	}
	while ( (errno == EINTR) && was_error); 
//	usleep(millis*1000);
}

void CL_Display::select_card(CL_DisplayCard *card)
{
	cl_current_displaycard = card;
}

CL_DisplayCard *CL_Display::get_current_card()
{
	cl_assert(cl_current_displaycard != NULL);
	return cl_current_displaycard;
}

void CL_Sound::select_card(CL_SoundCard *card)
{
	cl_current_soundcard = card;
}

CL_SoundCard *CL_Sound::get_current_card()
{
	cl_assert(cl_current_soundcard != NULL);
	return cl_current_soundcard;
}

// -----------------------------------------------------------------------

std::string CL_Force_DispTarget::name;

CL_Force_DispTarget::CL_Force_DispTarget(const char *_name)
{
	// Make sure there is only one display target force!
	cl_assert( strcmp(name.c_str(), "") == 0 );

	name = _name;
}
