/*
	$Id: provider_pcx.cpp,v 1.1.1.1 2000/04/09 12:18:02 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		PCX SurfaceProvider
*/

#include "Core/precomp.h"
#include "API/Core/Display/pixelformat.h"
#include "API/Core/Display/palette.h"
#include "API/Core/IOData/inputsource.h"
#include "API/Core/IOData/inputsource_provider.h"
#include "API/Core/SurfaceProviders/provider_pcx.h"
#include "API/Core/System/error.h" 
#include "API/Core/System/cl_assert.h"
#include "API/Core/Display/res_surface.h"
#include "API/Core/Resources/resourceoptions.h"

CL_Surface *CL_PCXProvider::create(
	std::string handle, 
	CL_InputSourceProvider *provider, 
	bool transparent, 
	unsigned char trans_col)
{
	return CL_Surface::create(new CL_PCXProvider(
		handle, 
		provider, 
		transparent, 
		trans_col), true);
}

CL_PCXProvider::CL_PCXProvider(
	std::string _name, 
	CL_InputSourceProvider *_provider, 
	bool _transparent, 
	unsigned char _trans_col)
{
	provider = _provider != NULL ? _provider->clone() : 
								CL_InputSourceProvider::create_file_provider(".");
	pitch = height = 0;
	bounding_left = bounding_top = bounding_right = bounding_bottom = 0;
	
	transparent = _transparent;
	if (!transparent) trans_col = -1;
	else trans_col = _trans_col;
	
	name = _name;
	palette = NULL;
	image = NULL;
	pixelformat = PAL8;
}

CL_PCXProvider::~CL_PCXProvider()
{
	perform_unlock();
	delete provider;
}
/*
EPixelFormat CL_PCXProvider::get_pixel_format() const
{
	return pixelformat;
}
*/

unsigned int CL_PCXProvider::get_depth() const
{
	switch (pixelformat)
	{
	case RGBA8888:
		return 32;
	default:
		return 8;
	}
}

unsigned int CL_PCXProvider::get_red_mask() const
{
	return 0xff000000;
}

unsigned int CL_PCXProvider::get_green_mask() const
{
	return 0x00ff0000;
}

unsigned int CL_PCXProvider::get_blue_mask() const
{
	return 0x0000ff00;
}

unsigned int CL_PCXProvider::get_alpha_mask() const
{
	return 0x000000ff;
}

void CL_PCXProvider::perform_lock()
{
	if (image != NULL) return;

	cl_assert(provider != NULL);
	CL_InputSource *datafile = provider->open_source(name.c_str());
	cl_assert(datafile != NULL);

	datafile->seek(4, CL_InputSource::seek_set);

	short xmin = datafile->read_short16();
	short ymin = datafile->read_short16();
	short xmax = datafile->read_short16();
	short ymax = datafile->read_short16();

	pitch = xmax - xmin  + 1;
	height = ymax - ymin + 1;

	if (transparent)
	{
		bounding_left = pitch;
		bounding_top = height;
		bounding_right = 0;
		bounding_bottom = 0;
	}
	else
	{
		bounding_left = 0;
		bounding_top = 0;
		bounding_right = pitch;
		bounding_bottom = height;
	}

	datafile->seek(65, CL_InputSource::seek_set);
	int num_planes = datafile->read_char8();
	int dest_num_planes = num_planes;

	switch (num_planes)
	{
	case 1:
		pixelformat = PAL8;
		break;
	case 3:
		pixelformat = RGBA8888;
		dest_num_planes = 4;
		break;
	default:
		cl_assert(false);
		break;
	}

	image = new unsigned char[pitch*height*dest_num_planes];
	cl_assert(image != NULL);

	if (dest_num_planes == 4)
	{
		memset(image, 255, pitch*height*dest_num_planes);
	}

	datafile->seek(128, CL_InputSource::seek_set);

	int size_data = datafile->size() - 128;
	unsigned char *temp = new unsigned char[size_data];
	int read = datafile->read(temp, size_data);
	if (read != size_data)
		throw CL_Error("Invalid pcx file!?");
//	cl_assert(read == size_data);
	delete datafile;
	
	unsigned char *p = temp;

	unsigned char *cur_line = image;
	for (int y=0;y<height;y++)
	{
		for (int plane=0;plane<num_planes;plane++)
		{
			int x = 0;
			while (x < pitch)
			{
				unsigned char packed_byte = *(p++);
				if (packed_byte<192)
				{
					cur_line[(x*dest_num_planes)+plane] = packed_byte;

					if (transparent && packed_byte != trans_col)
					{
						if (x < bounding_left) bounding_left = x;
						if (y < bounding_top) bounding_top = y;
						if (x > bounding_right) bounding_right = x;
						if (y > bounding_bottom) bounding_bottom = y;
					}
					x++;
				}
				else 
				{ 
					unsigned char counter = packed_byte&0x3f;
					packed_byte = *(p++);
					for(unsigned char j=0;j<counter;j++)
					{
						cur_line[(x+j)*dest_num_planes+plane] = packed_byte;
						if (transparent && packed_byte != trans_col)
						{
							if ((x+j) < bounding_left) bounding_left = x+j;
							if (y < bounding_top) bounding_top = y;
							if ((x+j) > bounding_right) bounding_right = x+j;
							if (y > bounding_bottom) bounding_bottom = y;
						}
					}
					x += counter;
				}
			}
		}
		cur_line += dest_num_planes * pitch;
	}

	if (num_planes == 1)
	{
		palette = new CL_Palette((temp+size_data-768));
	}
	else
	{
		palette = NULL;
	}

	delete[] temp;
}

void CL_PCXProvider::perform_unlock()
{
	delete[] image;
	delete palette;

	image = NULL;
	palette = NULL;
}
