/*
	$Id: cl_assert.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="System"

#ifndef header_cl_assert
#define header_cl_assert

#include "../../stl.h"

enum ErrorChannels
{
	info_display = 1,
	info_sound   = 2,
	info_input   = 4,
	info_network = 8
};

#ifdef WIN32
#define cl_assert(a) CL_Assert::die(a, __FILE__, __LINE__, NULL, #a);
#else
#define cl_assert(a) CL_Assert::die(a, __FILE__, __LINE__, __PRETTY_FUNCTION__, __STRING(a));
#endif

#define cl_info(errchannel, message) \
	CL_Assert::info(errchannel, message, __FILE__, __LINE__);

#ifdef DEBUG
	#define cl_assert_debug(a) cl_assert(a)
	#define cl_info_debug(a,b) cl_info(a,b)
#else
	#define cl_assert_debug(a)
	#define cl_info_debug(a,b)
#endif


class CL_AssertListener;
class CL_Assert
//: This is ClanLib's assertion class.
// It is needed because some
// implementations requires to be notified before ClanLib is exited.
// For instance the svgalib implementation need release the keyboard lock.
{
	static std::list<CL_AssertListener*> listeners;
public:
	static void enable_channels(int channel_mask);
	//: Enabled the error channels specified by the 'channel_mask'.
	//: This can be used to get extra debugging information from limited
	//: parts of ClanLib.
	//!param: channel_mask - Channels to be enabled.

	static void die(bool a, char *file, int line, char *func, char *assert_str);
	//: Produces an assert. This function shouldn't be called directly -
	//: instead, use the cl_assert macro.
	//!param: a - Asserts if false.
	//!param: file - Name of the file where the assert occoured.
	//!param: line - Line in the file where the assert occoured.
	//!param: func - Function in which the assert occoured.
	//!param: assert_str - String that describe the assertion check.

	static void info(int channel, char *text, char *file, int line);
	//: Writes an info statements on the specified channel. Don't call
	//: this one directly, instead use the cl_info macro.
	//!param: channel - Channel to use.
	//!param: text - text to print.
	//!param: file - Name of the file where this was printed from.
	//!param: line - Line in the file where this was printed from.
	
	static void add_listener(CL_AssertListener *listener);
	//: Adds an assertion listener which will be called in case of an
	//: assertion.
	//!param: listener - The listener to be added.

	static void remove_listener(CL_AssertListener *listener);
	//: Removes an assertion listener.
	//!param: listener - The listener to be removed.
};

class CL_AssertListener
//: Interface used by the CL_Assert class.
{
public:
	virtual ~CL_AssertListener() {;}

	virtual void assert_occoured(char *file, int line)=0;
	//: Called by CL_Assert::die() when assertion occours. Remember to
	//: add it as an listener - otherwise it will _NOT_ be called.
	//!param: file - Filename where the assert occoured.
	//!param: line - Line number in the file where the assert occoured.
};

#endif
