/*
	$Id: resource_manager.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Resources"

#ifndef header_resource_manager
#define header_resource_manager

#include "resource.h"
#include "resourcetype.h"
#include "resourceoptions.h"
#include "resourceoption.h"
#include "../IOData/inputsource_provider.h"

#include "../../stl.h"

class CL_ResourceOption;
class CL_ResourceOptions;
class CL_ResourceManager;

class CL_ResourceManager
//: The ClanLib resource manager.
// The resource manager is used to retrieve resources from a given resource source.
// This can either be a resource script file (used as input to the
// datafile compiler), or a datafile with all the resources included into
// one large gzipped resource file.
// <br>
// To speedup loading of resources in a game, you can load entire sections
// of resources at once. When a resource in the section is requested, it is
// returned instantly without having to access the disk. This is especially
// useful to make sure all the game resources are loaded before the game is
// started.
// <br>
// Resources are normally not retrieved using the get_resource() function.
// Instead, you should load the resource using the appropiate resource type
// class. For instance, a surface is easiest loaded like this:
// <code>
// CL_ResourceManager res_manager("my_datafile.dat", true);
// CL_Surface *my_surface = CL_Res_Surface::load("my_surface", res_manager);
// </code>
// <br>
// Getting the same resource twice won't create a new instance of the
// resource, but you will still have to clean up the resource after
// yourself. In other words: If you load a resource twice, you only have to
// delete it once.
{
public:
	static CL_ResourceManager *create(
		const char *config_file,
		CL_InputSourceProvider *provider = NULL,
		bool read_directly_from_source=false,
		bool delete_inputsource_provider=false);
	//: Resource Manager constructor.
	//!param: config_file - the name of the file in which the resources are defined
	//!param: provider - the optional inputprovider in which, the resource file is stored
	//!param: read_directly_from_source - if true, any resources are loaded directly from their source, meaning that any datafile-directive is ignored.

	static CL_ResourceManager *create(
		const char *file_name,
		const bool is_datafile );
	//: Resource Manager constructor.
	//!param: file_name - the name of the file to open
	//!param: is_datafile - indicates if the file is a scriptfile or a datafile

	virtual ~CL_ResourceManager() { return; }

	virtual void load_all_resources()=0;
	//: Loads all resources into memory.

	virtual void unload_all_resources()=0;
	//: Unloads all resources from memory.

	virtual void load_section(const char *section_name)=0;
	//: Loads all resources in a given section into memory.
	//!param: section_name - the section to be loaded

	virtual void unload_section(const char *section_name)=0;
	//: Unloads all resources in a given section into memory.
	//!param: section_name - the section to be unloaded

	virtual CL_Resource *get_resource(std::string res_id)=0;
	//: Returns a pointer to the CL_Resource representing the given resource
	//!param: Name of the resource in the configuration file - note: the name of resources are absolute, eg. including section names with slashes (Graphics/Pacman/sur_pacman)

	virtual std::list<std::string> *get_all_resources()=0;
	//: Returns a list of all resources available. Primarily used by the ClanCompiler to 
	//: build datafiles from resources.
	//!retval: The list of resources available. You'll have to delete all strings in the returned
	//!retval: list manually (including the list-instance itself).

	virtual std::list<std::string> *get_resources_of_type(std::string type_id)=0;
	//: Returns a list of all resources available matching a given type. 
	//: Primarily used by the ClanCompiler to build datafiles from resources.
	//!retval: The list of resources available. You'll have to delete all strings in the 
	//!retval: list manually (including the list-instance itself).

	virtual CL_InputSourceProvider *get_resource_provider()=0;
	//: Returns a pointer to the inputsourceprovider, in which all resources are stored
	//: (this can be a fileprovider or a datafileprovider depending on method used to load the script file)
	//!retval: Pointer to inputsourceprovider containing resource data.
};
#endif
