/*
	$Id: input_events.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Input"

#ifndef header_input_events
#define header_input_events

#include "../System/eventchain.h"

class CL_InputDevice;
class CL_Key;

class CL_Event_ButtonPress : public virtual CL_Event
//: Event callback interface for button presses.
// <p>This interface is used to intercept button presses from the keyboard,
// mouse and joystick.</p>
// <p>To use this interface, inherit it and implement on_button_press().
// Then add it to a CL_EventChain_ButtonPress chain. The topmost chain is
// located in CL_Input.</p>
// <p>ClanLib has some common case inheritance of most of our events, and
// you may want to have a look at them before using this interface. One
// example is the CL_InputBuffer class, which converts button events into
// a poll queue.</p>
{
public:
	virtual ~CL_Event_ButtonPress() { return; }
	
	virtual bool on_button_press(
		CL_InputDevice *device,  
		const CL_Key &key)=0;
	// Called when a button press event occours.
	//!retval: If true is returned, the event chain that called this
	//!retval: function will break the chain (stop sending it the remaining
	//!retval: listeners on the chain.
	//!param: device - Input device where the event originates from.
	//!param: key - The key/button that triggered the event.
};
  
class CL_EventChain_ButtonPress :
	public CL_EventChain<CL_Event_ButtonPress*>,
	public CL_Event_ButtonPress
//: Event sending interface for button presses.
// <p>Event chains are used to route an event to an event listener. The
// toplevel event chain for button presses, located in CL_Input, is called
// by ClanLib, but you can also send false events by calling
// on_button_press() yourself.</p>
// <p>Note that CL_EventChain_ButtonPress is inheriated from
// CL_EventButtonPress, so you can add a chain to the toplevel chain. This
// is useful, if you want to do more advanced routing of events.</p>
{
public:
	virtual bool on_button_press(
		CL_InputDevice *device,  
		const CL_Key &key);
	// Call this function to send a button press event down through
	// the event chain.
	//!param: device - Input device that triggered the event.
	//!param: key - Key/button that triggered the event.
};
  
// ---

class CL_Event_ButtonRelease : public virtual CL_Event
//: Event callback interface for button releases.
// <p>This interface is used to intercept button releases from the keyboard,
// mouse and joystick.</p>
// <p>To use this interface, inherit it and implement on_button_release().
// Then add it to a CL_EventChain_ButtonRelease chain. The topmost chain is
// located in CL_Input.</p>
// <p>ClanLib has some common case inheritance of most of our events, and
// you may want to have a look at them before using this interface. One
// example is the CL_InputBuffer class, which converts button events into
// a poll queue.</p>
{
public:
	virtual ~CL_Event_ButtonRelease() { return; }

	virtual bool on_button_release(
		CL_InputDevice *device,
		const CL_Key &key)=0;  
	// Called when a button release event occours.
	//!retval: If true is returned, the event chain that called this
	//!retval: function will break the chain (stop sending it the remaining
	//!retval: listeners on the chain.
	//!param: device - Input device where the event originates from.
	//!param: key - The key/button that triggered the event.
};
  
class CL_EventChain_ButtonRelease :
	public CL_EventChain<CL_Event_ButtonRelease*>,
	public CL_Event_ButtonRelease
//: Event sending interface for button releases.
// <p>Event chains are used to route an event to an event listener. The
// toplevel event chain for button releases, located in CL_Input, is called
// by ClanLib, but you can also send false events by calling
// on_button_release() yourself.</p>
// <p>Note that CL_EventChain_ButtonRelease is inheriated from
// CL_EventButtonRelease, so you can add a chain to the toplevel chain. This
// is useful, if you want to do more advanced routing of events.</p>
{
public:
	virtual bool on_button_release(
		CL_InputDevice *device,
		const CL_Key &key);
	// Call this function to send a button release event down through
	// the event chain.
	//!param: device - Input device that triggered the event.
	//!param: key - Key/button that triggered the event.
};
  
// ---

class CL_Event_MouseMove : public virtual CL_Event
//: Event callback interface for mouse movement.
// <p>This interface is used to intercept movements from the mosue.</p>
// <p>To use this interface, inherit it and implement on_mouse_move().
// Then add it to a CL_EventChain_MouseMove chain. The topmost chain is
// located in CL_Input.</p>
{
public:
	virtual ~CL_Event_MouseMove() { return; }
	
	virtual bool on_mouse_move(
		CL_InputDevice *device)=0;
	// Called when a mouse move event occours.
	//!param: device - Input device where the event originates from.
};
  
class CL_EventChain_MouseMove :
	public CL_EventChain<CL_Event_MouseMove*>,
	public CL_Event_MouseMove
//: Event sending interface for mouse movement.
// <p>Event chains are used to route an event to an event listener. The
// toplevel event chain for mouse movement, located in CL_Input, is called
// by ClanLib, but you can also send false events by calling
// on_mouse_move() yourself.</p>
// <p>Note that CL_EventChain_MouseMove is inheriated from
// CL_EventMouseMove, so you can add a chain to the toplevel chain. This
// is useful, if you want to do more advanced routing of events.</p>
{
public:
	virtual bool on_mouse_move(
		CL_InputDevice *device);
	// Call this function to send a mouse move event down through
	// the event chain.
	//!param: device - Input device that triggered the event.
};

// ---

class CL_Event_MouseEnter : public virtual CL_Event
//: Event callback interface for when the mouse enters the window.
// <p>To use this interface, inherit it and implement on_mouse_enter().
// Then add it to a CL_EventChain_MouseEnter chain. The topmost chain is
// located in CL_Input.</p>
{
public:
	virtual ~CL_Event_MouseEnter() { return; }
	
	virtual bool on_mouse_enter(
		CL_InputDevice *device)=0;
	// Called when a mouse enter event occours.
	//!param: device - Input device where the event originates from.
};
  
class CL_EventChain_MouseEnter :
	public CL_EventChain<CL_Event_MouseEnter*>,
	public CL_Event_MouseEnter
//: Event sending interface for mouse entering.
// <p>Event chains are used to route an event to an event listener. The
// toplevel event chain for mouse entering, located in CL_Input, is called
// by ClanLib, but you can also send false events by calling
// on_mouse_enter() yourself.</p>
// <p>Note that CL_EventChain_MouseEnter is inheriated from
// CL_EventMouseEnter, so you can add a chain to the toplevel chain. This
// is useful, if you want to do more advanced routing of events.</p>
{
public:
	virtual bool on_mouse_enter(
		CL_InputDevice *device);
	// Call this function to send a mouse enter event down through
	// the event chain.
	//!param: device - Input device that triggered the event.
};

// ---

class CL_Event_MouseExit : public virtual CL_Event
//: Event callback interface for when the mouse exits the window.
// <p>To use this interface, inherit it and implement on_mouse_exit().
// Then add it to a CL_EventChain_MouseExit chain. The topmost chain is
// located in CL_Input.</p>
{
public:
	virtual ~CL_Event_MouseExit() { return; }
	
	virtual bool on_mouse_exit(
		CL_InputDevice *device)=0;
	// Called when a mouse exit event occours.
	//!param: device - Input device where the event originates from.
};
  
class CL_EventChain_MouseExit :
	public CL_EventChain<CL_Event_MouseExit*>,
	public CL_Event_MouseExit
//: Event sending interface for mouse exit.
// <p>Event chains are used to route an event to an event listener. The
// toplevel event chain for mouse exit, located in CL_Input, is called
// by ClanLib, but you can also send false events by calling
// on_mouse_exit() yourself.</p>
// <p>Note that CL_EventChain_MouseExit is inheriated from
// CL_EventMouseExit, so you can add a chain to the toplevel chain. This
// is useful, if you want to do more advanced routing of events.</p>
{
public:
	virtual bool on_mouse_exit(
		CL_InputDevice *device);
	// Call this function to send a mouse exit event down through
	// the event chain.
	//!param: device - Input device that triggered the event.
};

#endif
