# ClamTk, copyright (C) 2004-2010 Dave M
#
# This file is part of ClamTk.
#
# ClamTk is free software; you can redistribute it and/or modify it
# under the terms of either:
#
# a) the GNU General Public License as published by the Free Software
# Foundation; either version 1, or (at your option) any later version, or
#
# b) the "Artistic License".
package ClamTk::Device;
# We can't do realtime monitoring of devices
# (like an option to popup an alert when a device is plugged in)
# because we'd need Net::DBus::GLib - not available on many
# distros. Although, Debian finally got it... :)

use strict;
#use warnings;
$|++;

use Glib qw/TRUE FALSE/;
use Gtk2 '-init';

use ClamTk::GUI;

use encoding 'utf8';

use Locale::gettext;
use POSIX qw/locale_h/;
textdomain('clamtk');
setlocale( LC_MESSAGES, '' );
bind_textdomain_codeset( 'clamtk', 'UTF-8' );

# $devs = holds device information
my $devs;

my $udi = '';

sub look_for_device {
    my $dwin = Gtk2::Window->new;
    $dwin->set_title( 'ClamTk ' . gettext('Virus Scanner') );

    # Can't do size_request here because the window ($dwin)
    # won't expand based on # of devices found

    $dwin->signal_connect( destroy => sub { $dwin->destroy } );
    if ( -e '/usr/share/pixmaps/clamtk.png' ) {
        $dwin->set_default_icon_from_file('/usr/share/pixmaps/clamtk.png');
    } elsif ( -e '/usr/share/pixmaps/clamtk.xpm' ) {
        $dwin->set_default_icon_from_file('/usr/share/pixmaps/clamtk.xpm');
    }
    my $dbox = Gtk2::VBox->new( FALSE, 5 );
    $dwin->add($dbox);
    Gtk2->main_iteration while ( Gtk2->events_pending );

    # $count = for iterating through $devs
    my $count = 0;

    # $mount_point holds the (wait for it) mount point
    # info and returns it for scanning
    my $mount_point;

    # The following are for images/language purposes
    my $cd_label     = gettext('CD/DVD');
    my $usb_label    = gettext('USB device');
    my $floppy_label = gettext('Floppy disk');

    # In a perlfect world, we would use Net::DBus for this.
    # Of course, CentOS doesn't HAVE it... grrrr...

    my %hash = get_lshal();

    for my $d ( keys %hash ) {
        while ( my ( $t, $u ) = each %{ $hash{$d} } ) {
            # See if there's anything mounted.
            $mount_point = $hash{$d}{'volume.mount_point'};

            # Grab its parent device.
            my $parent = $hash{$d}{'info.parent'};

            # First we'll see if it's a CD or DVD.
            # I'd like to ignore ISOs due to size, but
            # I'm not sure we can do that.
            if ( $hash{$parent}{'storage.drive_type'} eq 'cdrom' ) {
                next unless $mount_point;
                next unless ( no_dupes($mount_point) );
                $devs->{$count}->{'device'} = $cd_label;
                $devs->{$count}->{'label'}  = $hash{$d}->{'volume.label'};
                $devs->{$count}->{'mount'}  = $mount_point;
                $count++;
                next;
            }

            # Now see if it's a USB flash device. This is
            # tricky but always seems to work (for me :).
            # The 'storage.removable' field seems to be the
            # difference between a flash drive and a usb hard drive:
            # 'true' eq flash_drive, 'false' eq usb hard drive
            elsif ( $hash{$d}{'info.udi'} =~ /volume/ ) {
                if (   $hash{$parent}{'storage.removable'} eq 'true'
                    && $hash{$parent}{'storage.bus'} eq 'usb' )
                {
                    next unless $mount_point;
                    next unless ( no_dupes($mount_point) );

                    $devs->{$count}->{'device'} = $usb_label;
                    $devs->{$count}->{'label'} =
                        $hash{$parent}{'storage.vendor'};
                    $devs->{$count}->{'mount'} = $mount_point;

                    $devs->{$count}->{'label'} ||= $hash{$d}{'info.product'};

                    next
                        unless ( $devs->{$count}->{'label'}
                        and $devs->{$count}->{'mount'} );
                    $count++;
                }
            }
            # And finally, floppies, assuming anyone uses them.
            # The problem is that the section that shows them
            # identifies them as floppies, but doesn't give a mount.
            elsif ( $hash{$d}{'storage.drive_type'} eq 'floppy' ) {
                my $device = $hash{$d}{'block.device'};
                chomp($device);
                next unless ($device);

                # Now we cheat. Hopefully /proc/mounts has it,
                # so just look for the line with the block
                # device and grab the mount point.
                open( my $P, '<', '/proc/mounts' ) or next;
                while (<$P>) {
                    # $_ will look like this:
                    # /dev/fd0	/media/floppy	vfat	...
                    # We just need the first two values:
                    my ( $dev, $mount ) = ( split(/\s+/) )[ 0, 1 ];
                    if ( $dev eq $device ) {
                        $mount_point = $mount;
                        last;
                    }
                }
                close($P);

                # If we didn't match the device
                # and mount point, just leave
                next unless ($mount_point);

                # Ensure there are no dupes
                if ( no_dupes($mount_point) ) {

                    my $label = $hash{$d}{'info.product'};
                    chomp($label);
                    $label ||= $hash{$d}{'storage.vendor'};
                    chomp($label);
                    $label ||= '';
                    $devs->{$count}->{'device'} = $floppy_label;
                    $devs->{$count}->{'label'}  = $label;
                    $devs->{$count}->{'mount'}  = $mount_point;
                    $count++;
                } else {
                    next;
                }
            } else {
                next;
            }
        }
    }

    # Return unless we have devices.
    # This could get confusing for users not
    # understanding the 'mount' thing.
    if ( !scalar( keys %$devs ) ) {
        my $mount_msg = gettext('No devices were found.');
        $mount_msg .= "\n\n";
        $mount_msg
            .= gettext(
            'If you have connected a device, you may need to mount it first.'
            );
        show_message_dialog( $dwin, 'info', 'ok', gettext($mount_msg) );
        $dwin->destroy;
        return 0;
    }

    my $tt = Gtk2::Tooltips->new();

    my $num = scalar( keys %$devs );

    my $dframe = Gtk2::Frame->new( gettext('Devices available') );
    $dbox->pack_start( $dframe, TRUE, TRUE, 0 );
    $dframe->set_border_width(5);
    $dframe->set_shadow_type('etched-in');

    # This is much better as a tooltip...
    # an extra label skews the display.
    $tt->set_tip( $dframe, gettext('Select a device or press Cancel') );

    # This buttonbox will hold the devices found.
    my $m_bbox = Gtk2::HButtonBox->new;
    $dframe->add($m_bbox);
    $m_bbox->set_layout('start');
    $m_bbox->set_border_width(10);

    # Get the information from the devices for display
    for ( 0 .. $num ) {
        my ($label) = $devs->{$_}->{'label'};
        next if ( !$label );
        my ($mount)  = $devs->{$_}->{'mount'};
        my ($device) = $devs->{$_}->{'device'};

        # We'll use the same technique for images -
        # stock gtk2 icons, but the label is customized
        my $gui_img = Gtk2::Image->new_from_stock(
              ( $device eq $cd_label )     ? 'gtk-cdrom'
            : ( $device eq $usb_label )    ? 'gtk-harddisk'
            : ( $device eq $floppy_label ) ? 'gtk-floppy'
            : '',
            'small-toolbar'
        );
        my $gui_btn = Gtk2::Button->new();
        $gui_btn->set_property( 'image' => $gui_img );
        $gui_btn->set_label($label);
        $gui_btn->set_relief('half');
        $tt->set_tip( $gui_btn, "$device ($mount)" );
        $m_bbox->add($gui_btn);
        $gui_btn->signal_connect(
            clicked => sub {
                $dwin->destroy;
                ClamTk::GUI->getfile( 'device', $mount );
            }
        );
    }

    # This hbuttonbox holds the cancel button.
    # A future option might be 'Help'
    my $bottom_row = Gtk2::HButtonBox->new();
    $dbox->pack_start( $bottom_row, FALSE, FALSE, 5 );
    $bottom_row->set_layout('end');
    $bottom_row->set_border_width(5);

    # Cancel button for devices
    my $cancel = Gtk2::Button->new_from_stock('gtk-cancel');
    $bottom_row->add($cancel);
    $cancel->signal_connect(
        clicked => sub {
            $dwin->destroy;
            return 0;
        }
    );
    # Give the cancel button the focus;
    # looks better than when a device has it.
    # It also allows the user to press enter
    # to kill the window.
    $cancel->grab_focus();

    # If there's only one device, set the size of the window.
    # Otherwise it's too small.
    if ( $num == 1 ) {
        $dwin->set_size_request( 300, 150 );
        $dwin->queue_draw();
    }

    $dwin->show_all();
    return;
}

sub no_dupes {
    # Accepts the mount_point (mp) to search.
    # Returns TRUE (1) if no_dupes are found;
    # Otherwise, returns FALSE (0) if they are.
    # Confused? In other words, 
    # TRUE  = there are no dupes 
    # FALSE = there ARE dupes
    my $mp = shift;
    # A quick look to ensure we haven't seen this before.
    my $keys  = scalar keys %$devs;
    my $total = 0;
    if ($keys) {
        for ( 0 .. $keys - 1 ) {
            $total++ if ( $devs->{$_}->{'mount'} eq $mp );
        }
    }
    if ($total) {
        return 0;
    } else {
        return 1;
    }
}

sub get_lshal {
    my %gather;

    my $cmd = qx| which lshal |;
    chomp($cmd);
    # This return is almost worthless. If you don't have lshal
    # installed, $cmd = 'which: no lshal in (blah:/usr/blah)'
    # and "*** unhandled exception in callback:".  But this way,
    # we return and say we didn't find anything. Much neater.
    return unless ( -e $cmd );

    my $pid = open( my $LIST, "-|", $cmd );
    defined($pid) or die "couldn't fork! $!\n";

    while ( defined( my $line = <$LIST> ) ) {
        next if ( $line =~ /^Dumping/ );
        next if ( $line =~ /^\-/ );
        next if ( $line =~ /^\s*$/ );
        chomp($line);

        my ( $key, $value ) = split( /=/, $line );
        next unless ( defined($key) );
        next unless ( defined($value) );
        chomp( $key, $value );

        $key =~ s/^\s+//g;
        $key =~ s/\s+$//g;
        $key =~ s/^\'//;
        $key =~ s/\'$//;

        if ( $key eq 'udi' ) {
            $udi = $value;
            $udi =~ s/^\s+//g;
            $udi =~ s/^\'//;
            $udi =~ s/\s+$//g;
            $udi =~ s/\'$//;
        } else {
            # This is a pain... storage.removable returns
            # 'true' or 'false', which requires a different regex
            # because of the single quotes we normally look for.
            # storage.removable = true  (bool)
            # vs.
            # info.product = 'Computer'  (string)
            if ( $key eq 'storage.removable' ) {
                if ( $value =~ /\s+(.*?)\s+/ ) {
                    $gather{$udi}{$key} = $1;
                }
            } elsif ( $value =~ /'(.*?)'/ ) {
                $gather{$udi}{$key} = $1;
            }
        }
    }
    close($LIST);
    return %gather;
}

sub show_message_dialog {
    my ( $parent, $type, $button, $message ) = @_;
    # $parent = $dwin
    # $type = info, warning, error, question
    # $button = ok, ok-cancel, close, ...
    # $message = <a message>

    my $dialog;
    $dialog =
        Gtk2::MessageDialog->new_with_markup( $parent,
        [qw(modal destroy-with-parent)],
        $type, $button, $message );

    $dialog->run;
    $dialog->destroy;
    return;
}

1;
