;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wcof -*-
;;; $Id: metadata.lisp,v 1.40 2002/02/22 17:18:21 jesse Exp $
;;;
;;; Copyright onShore, Inc. 2001

(in-package :sql)

(locally-enable-sql-reader-syntax)

(defvar +meta-types+
  '((integer    . meta-integer)
    (local-time:duration   . meta-duration)
    (local-time:local-time . meta-timeval)
    (string     . meta-string)
    (keyword    . meta-keyword)
    (boolean    . meta-boolean)))

(defun sql-and-qualifier (qualifier)
  (cond ((null qualifier)
         nil)
        ((= 1 (length qualifier))
         (car qualifier))
        (t
         (apply #'maisql-sys::sql-and (remove-if-not #'identity
                                                     qualifier )))))

(defun frob-metadata-subtypes (subtypes)
  "given a list of metadata specifications of the form (element-name
 :type TYPE :unique UNIQUE :authority AUTHORITY :default DEFAULT) where
 element-name is the name of the specific meta element, return a
 transformed list of specifications of the form (element-name
 LISP-CLASS UNIQUE AUTHORITY DEFAULT) where LISP-CLASS is the lisp
 class of the meta element, UNIQUE a boolean which is t by default,
 where t means that each non-meta element can have only 1 value for
 this meta element (as a person has one mother (unique t) but several
 children (unique nil), AUTHORITY is a boolean (representing what??
 [2001/04/11 :lh]) and DEFAULT is whatever is passed in as :default
 value for meta elements of the given element-name"
  (flet ((frob-ele (ele)
           (let ((spec (list (car ele))))
             (odcl::when-bind (mtype (position :type (cdr ele)))
               (push (cdr (assoc (nth (+ 2 mtype) ele) +meta-types+)) spec))
             (or (odcl::when-bind (mtype (position :unique (cdr ele)))
                   (push (nth (+ 2 mtype) ele) spec))
                 (push t spec))
             (or (odcl::when-bind (mtype (position :authority (cdr ele)))
                   (push (nth (+ 2 mtype) ele) spec))
                 (push nil spec))
             (odcl::when-bind (mtype (position :default (cdr ele)))
               (push (eval (nth (+ 2 mtype) ele)) spec))
             (nreverse spec))))
    (mapcar #'frob-ele subtypes)))

(defgeneric metadata-info (view-class &optional property)
  (:documentation "return as multiple values the metadata specification
 regarding PROPERTY of VIEW-CLASS.  return values are of the form
 (metadata-classname subtype-data), where metadata-classname is the name of
 the lisp class defined to represents metadata elements for the VIEW-CLASS,
 and subtype-data is a list of metadata subtype specifications as returned
 by frob-metadata-subtypes"))

(def-view-class metadata-entity ()
  ((metadata-cache
    :db-kind :virtual
    :initform nil
    :accessor metadata-cache)))

(defmethod oid ((obj metadata-entity))
  (error "You must define a method OID for this class."))

(def-view-class metadata-mixin ()
  ((meta-oid      :type integer
                  :reader oid
                  :initarg :oid
                  :db-kind :key
                  :documentation "unique OID")
   (timestamp     :type local-time:local-time
                  :reader timestamp
                  :initform (local-time:get-local-time))
   (subtype       :type keyword
                  :initarg :subtype
                  :documentation "metadata subtype")
   (primary-oid   :type integer
                  :initarg :primary-oid
                  :documentation "primary fkey")
   (secondary-oid :type integer
                  :initarg :secondary-oid
                  :initform nil
                  :nulls-ok t
                  :documentation "secondary fkey")
   (tertiary-oid  :type integer
                  :initarg :tertiary-oid
                  :initform nil
                  :nulls-ok t
                  :documentation "tertiary fkey")
   (meta-string   :type string
                  :initarg :meta-string
                  :initform nil
                  :nulls-ok t
                  :documentation "storage for string subtypes")
   (meta-keyword  :type keyword
                  :initarg :meta-keyword
                  :initform nil
                  :nulls-ok t
                  :documentation "storage for keyword subtypes")
   (meta-integer  :type integer
                  :initarg :meta-integer
                  :initform nil
                  :nulls-ok t
                  :documentation "storage for integer subtypes")
   (meta-timeval  :type local-time:local-time
                  :initarg :meta-timeval
                  :initform nil
                  :nulls-ok t
                  :documentation "storage for timeval subtypes")
   (meta-duration :type local-time:duration
                  :initarg :meta-duration
                  :initform nil
                  :nulls-ok t
                  :documentation "storage for duration subtypes")
   (meta-boolean  :type boolean
                  :initarg :meta-boolean
                  :initform nil
                  :nulls-ok t
                  :documentation "storage for boolean subtypes"))
  (:documentation "base class for metadata"))

(defun short-name (name)
  (let ((chars (mapcar (lambda (x) (aref x 0)) (split name #\_))))
    (let ((name (make-string (length chars))))
      (dotimes (i (length chars))
        (setf (aref name i) (nth i chars)))
      name)))

(defun build-index (table &rest columns)
  (let ((idxname (format nil "~a_~{~a~^_~}" table (mapcar #'short-name columns))))
    (ignore-errors (execute-command (format nil "drop index ~a" idxname)))
    (execute-command (format nil "create index ~a on ~a (~{~a~^, ~})"
                             idxname table columns))))

(defun drop-index (table column)
  (ignore-errors (execute-command (format nil "drop index ~a_~a" table column))))

(defun build-metadata-indices (metadata-classname)
  (let ((table (string (view-table (mop:find-class metadata-classname)))))
    (build-index table "primary_oid")
    (build-index table "secondary_oid")
    (build-index table "subtype")
    (build-index table "primary_oid" "subtype")
    (build-index table "primary_oid" "secondary_oid" "subtype")
    ))

(defun drop-metadata-indices (metadata-classname)
  (let ((table (string (view-table (mop:find-class metadata-classname)))))
    (drop-index table "primary_oid")
    (drop-index table "secondary_oid")
    (drop-index table "subtype")))

(defmacro def-metadata (classname metadata-classname schema &rest subtypes)
  `(progn
    (def-view-class ,metadata-classname (metadata-mixin)
      ()
      (:schemas ,schema)
      (:version 0))

    (defmethod sql-sys::%register-class ((self (eql ',metadata-classname)) database)
      (declare (ignore database))
      #+nil
      (let ((trigger_fn_name (sql-escape (symconcat :sql ',classname "_M_D")))
            (trigger_name (sql-escape (symconcat :sql ',classname "_M_D_T")))
            (meta_table (sql-sys::view-table (mop::find-class ',metadata-classname)))
            (view_table (sql-sys::view-table (mop::find-class ',classname)))
            (view_pkey (sql-sys::sql-output
                        (sql-expression :attribute
                                        (sql-sys::view-class-slot-column (car (slot-value (mop::find-class ',classname) 'sql-sys::key-slots)))))))
        (execute-command
         (format nil "CREATE FUNCTION ~A () RETURNS OPAQUE AS
             '
             BEGIN
                 DELETE FROM ~A WHERE PRIMARY_OID = OLD.~A;
                 RETURN OLD;
             END;
             '
             LANGUAGE 'plpgsql'"
                 trigger_fn_name meta_table view_pkey)
         :database database)
        (execute-command
         (format nil "CREATE TRIGGER ~A AFTER DELETE ON ~A
                 FOR EACH ROW EXECUTE PROCEDURE ~A()"
                 trigger_name view_table trigger_fn_name)
         :database database))
      )
    
    (defmethod sql-sys::%deregister-class ((self (eql ',metadata-classname)) database)
      (declare (ignore database))
      #+nil
      (let ((trigger_fn_name (sql-escape (symconcat :sql ',classname "_M_D")))
            (trigger_name (sql-escape (symconcat :sql ',classname "_M_D_T")))
            (view_table (sql-sys::view-table (mop::find-class ',classname))))
        (ignore-errors (execute-command
                        (format nil "DROP TRIGGER ~A ON ~A"
                                trigger_name view_table)
                        :database database))
        (ignore-errors (execute-command
                        (format nil "DROP FUNCTION ~A ()"
                                trigger_fn_name)
                        :database database)))
      )
  
    (defmethod initialize-instance :after ((self ,metadata-classname)
                                           &rest all-keys &key &allow-other-keys)
      (declare (ignore all-keys))
      (unless (or (slot-boundp self 'meta-oid) sql-sys::*db-deserializing*)
        (setf (slot-value self 'meta-oid)
              (sequence-next (sequence-from-class ',metadata-classname)))))
                                    
    (defmethod metadata-info ((self ,classname) &optional property)
      (let ((subtype-data ',(frob-metadata-subtypes (car subtypes))))
        (if (null property)
            (values ',metadata-classname subtype-data)
            (let ((metadata-type (or (cdr (assoc property subtype-data))
                                     (error "unknown metadata type ~s for ~s" property self))))
              (values-list (cons ',metadata-classname metadata-type))))))
    ))

(defun meta-qual (property value subtypes)
  "return a sql :where qualifier for the meta-PROPERTY, optionally
specialized further by VALUE.  Look in SUBTYPES for the specification
of PROPERTY"
  (let ((column (or (cadr (assoc property subtypes))
                    (error "unknown metadata subtype ~s" property))))
    (when (eql column 'meta-keyword)
      (setq value (string value)))
    (if value
        [and [= [subtype] (string property)]
             [= (sql-expression :attribute column) value]]
        [= [subtype] (string property)])))

(defmethod %meta-equal ((p1 metadata-mixin) (p2 metadata-mixin))
  (and (equal
        (slot-value p1 'subtype)
        (slot-value p2 'subtype))
       (equal
        (slot-value p1 'primary-oid)
        (slot-value p2 'primary-oid))
       (equal
        (slot-value p1 'secondary-oid)
        (slot-value p2 'secondary-oid))
       (equal
        (slot-value p1 'tertiary-oid)
        (slot-value p2 'tertiary-oid))
       (equal
        (slot-value p1 'value)
        (slot-value p2 'value))))
       

(defmethod %update-metadata-cache ((self metadata-entity) property pvalues)
  (update-alist property pvalues (slot-value self 'metadata-cache)))

(defmethod %add-pvalue-to-cache ((self metadata-entity) (pvalue metadata-mixin))
  (let ((pvalues (get-alist (slot-value pvalue 'subtype)
                            (slot-value self 'metadata-cache))))
    (unless (member-if #'(lambda (p1)
                           (equal (%safe-oid pvalue) (%safe-oid p1)))
                       pvalues)
      (update-alist (slot-value pvalue 'subtype) (cons pvalue pvalues)
                    (slot-value self 'metadata-cache)))))

(defmethod %remove-pvalue-from-cache ((self metadata-entity) (pvalue metadata-mixin))
  (let ((pvalues (get-alist (slot-value pvalue 'subtype)
                            (slot-value self 'metadata-cache))))
    (update-alist (slot-value pvalue 'subtype)
                  (remove-if #'(lambda (p1)
                                 (equal (%safe-oid pvalue) (%safe-oid p1)))
                             pvalues)
                  (slot-value self 'metadata-cache))))

(defmethod %property-cached-p ((self metadata-entity) property)
  (when-bind (pair (assoc property (slot-value self 'metadata-cache)))
             t))
             

(defmethod %flush-instance-metadata ((self metadata-entity))
  (setf (slot-value self 'metadata-cache) nil))

(defmethod %flush-property-cache ((self metadata-entity) property)
  (setf (slot-value self 'metadata-cache)
        (remove-if #'(lambda (pair)
                       (equal property (car pair)))
                   (slot-value self 'metadata-cache))))


(defun %extract-pvalues (self metadata)
  (multiple-value-bind (metadata-class metadata-type)
      (metadata-info self (slot-value metadata 'subtype))
    (declare (ignore metadata-class))
    (list (slot-value metadata metadata-type)
          (slot-value metadata 'secondary-oid)
          (slot-value metadata 'tertiary-oid))))

(defun %safe-oid (val)
  (typecase val
    (metadata-entity
     (oid val))
    (t val)))
      

(defmethod %get-metadata-cache-pvalue ((self metadata-entity) property secondary tertiary)
  "retrieve the value of PROPERTY from SELF's metadata-cache, further specialized
by SECONDARY and TERTIARY"
  (let ((mvalues (get-alist property (slot-value self 'metadata-cache)))
        (vals nil))
    (dolist (pval mvalues)
      (let ((val (%extract-pvalues self pval)))
        (when (cond
                (tertiary
                 (and (equal (third val) (%safe-oid tertiary))
                      (equal (second val) (%safe-oid secondary))))
                (secondary
                 (equal (second val) (%safe-oid secondary)))
                (t t))
          (setf vals (cons pval vals)))))
    vals))

(defmethod %get-metadata-cache ((self metadata-entity) property secondary tertiary)
  "retrieve the value of PROPERTY from SELF's metadata-cache, further specialized
by SECONDARY and TERTIARY"
  (let ((mvalues (get-alist property (slot-value self 'metadata-cache)))
        (vals nil))
    (dolist (val (mapcar #'(lambda (pval)
                             (%extract-pvalues self pval)) mvalues))
      (when (cond
              (tertiary
               (and (equal (third val) (%safe-oid tertiary))
                    (equal (second val) (%safe-oid secondary))))
              (secondary
               (equal (second val) (%safe-oid secondary)))
              (t t))
        (setf vals (cons val vals))))
    vals))

(defun find-meta (self property value &optional secondary tertiary)
  (multiple-value-bind (metadata-class metadata-subtypes)
      (metadata-info self)
    (if (null property)
        (select metadata-class
                :where (sql-and-qualifier
                        `(,[= [primary_oid] (%safe-oid self)]
                          ,(when secondary
                                 [= [secondary_oid] (%safe-oid secondary)])
                          ,(when tertiary
                                 [= [tertiary_oid] (%safe-oid tertiary)]))))
        (let* ((meta-qualifier (meta-qual property value metadata-subtypes))
               (rows (select metadata-class
                             :where (sql-and-qualifier
                                     `(,meta-qualifier
                                       ,[= [primary_oid] (%safe-oid self)]
                                       ,(when secondary [= [secondary_oid] (%safe-oid secondary)])
                                       ,(when tertiary [= [tertiary_oid] (%safe-oid tertiary)]))))))
          rows))))

(defun %show-meta (self)
  (multiple-value-bind (metadata-class metadata-subtypes)
      (metadata-info self)
    (declare (ignore metadata-class))
    (mapcar (lambda (meta)
              (let ((field (cadr (assoc (slot-value meta 'subtype) metadata-subtypes))))
                (cons (slot-value meta 'subtype)
                      (slot-value meta field))))
            (find-meta self nil nil))))
        
(defun show-meta (self)
  "Given an instance of metadata-entity, print the value of each of its
metadata elements."
  (mapcar (lambda (x)
            (format t "~&;; ~s => ~s~%" (car x) (cdr x)))
          (%show-meta self))
  (values))

(defun delete-meta (self &optional property value secondary tertiary)
  (let ((metadata-class (metadata-info self)))
    (if (not (or property value secondary tertiary))
        (progn
          (dolist (meta (select metadata-class
                                :where [= [primary_oid] (%safe-oid self)]))
            (ec-delete meta))
          (%flush-instance-metadata self))
        (progn
          (dolist (meta (find-meta self property value secondary tertiary))
            (%remove-pvalue-from-cache self meta)
            (ec-delete meta))))))

(defun store-meta (self property value &optional secondary tertiary)
  "Set SELF's meta-PROPERTY to VALUE, further specialized by secondary
and tertiary.  If PROPERTY has a non-nil :authority specification,
then secondary is required.  If no meta-PROPERTY record exists for
SELF, create one.  If PROPERTY has a non-nil :unique specification,
then any existing record will be modified to the new VALUE.  Otherwise,
an additional record will be created and inserted into the database."
  (multiple-value-bind (metadata-class metadata-type unique authority)
      (metadata-info self property)
    (when (and authority (null secondary))
      (error "must supply metadata authority"))
    (let ((metadata (and unique
                         (get-meta-pval self property secondary tertiary))))
      (if metadata
          (when-bind (ec-meta (instance-by-key (list metadata-class (oid metadata))))
            (%remove-pvalue-from-cache self metadata)
            (setq metadata ec-meta))
          (progn
            (setq metadata
                  (make-instance metadata-class
                                 :oid (sequence-next (sequence-from-class metadata-class))
                                 :primary-oid (%safe-oid self)
                                 :subtype property))
            (sql-sys::on-txn-abort
             (lambda () (%remove-pvalue-from-cache self metadata)))))
      (setf (slot-value metadata metadata-type) value)
      ;; (ec-insert metadata)
      (when secondary
        (setf (slot-value metadata 'secondary-oid)
              (%safe-oid secondary))
        (when tertiary
          (setf (slot-value metadata 'tertiary-oid)
                (%safe-oid tertiary))))
      (let ((ec sql-sys::*default-editing-context*)
            (mec (when (slot-boundp metadata 'sql-sys::editing-context)
                   (slot-value metadata 'sql-sys::editing-context))))
        (when mec
          (unless (eql ec mec)
            (ec-insert metadata ec))))
      (store-instance metadata)
      (%add-pvalue-to-cache self metadata))))

(defun lisp-to-sql (test)
  (ecase test
    (=
     #'maisql-sys::sql-=)
    (uplike
     #'maisql-sys::sql-uplike)
    (or
     #'maisql-sys::sql-or)))

(defun lsql (test1 attr1 test2 attr2)
  (flet ((qual1 (value)
           (funcall (lisp-to-sql test2) value attr2))
         (qual2 (value)
           (funcall (lisp-to-sql test2) attr1 value)))
    (if (and (atom attr1)
             (atom attr2))
        (funcall (lisp-to-sql test2) attr1 attr2)
        (apply (lisp-to-sql test1)
               (cond ((and (listp attr1)
                           (atom attr2))
                      (mapcar #'qual1 attr1))
                     ((and (atom attr1)
                           (listp attr2))
                      (mapcar #'qual2 attr2))
                     (t
                      (error "bad args to lsql")))))))
                  
;; (sql-or 'or [state] '= '("checkout" "completed-checkout"))


(defun get-all-meta (instances &optional secondary tertiary)
  (declare (ignore secondary tertiary))
  (multiple-value-bind (metadata-class metadata-types)
      (metadata-info (car instances))
    (let ((recs (select metadata-class
                        :where (lsql 'or [primary_oid] '= (mapcar #'oid instances))))
          (results nil))
      (dolist (rec recs)
        (push (cons (slot-value rec 'subtype)
                    (let ((slot (second (assoc (slot-value rec 'subtype) metadata-types))))
                      (if slot
                          (slot-value rec slot)
                          "No metadata type information."))) results))
      results)))

(defun copy-meta (instance-from instance-to)
  (let ((metadata-class (metadata-info instance-from)))
    (dolist (record (select metadata-class
                            :where [= [slot-value metadata-class 'primary-oid]
                                      [slot-value metadata-class 'primary-oid (%safe-oid instance-from)]]))
      (with-slots (subtype secondary-oid tertiary-oid meta-string meta-keyword
                           meta-integer meta-timeval meta-duration meta-boolean)
        record
        (%add-pvalue-to-cache
         instance-to
         (make-instance metadata-class
                        :oid (sequence-next (sequence-from-class metadata-class))
                        :subtype subtype
                        :primary-oid (%safe-oid instance-to)
                        :secondary-oid secondary-oid
                        :tertiary-oid tertiary-oid
                        :meta-string meta-string
                        :meta-keyword meta-keyword
                        :meta-integer meta-integer
                        :meta-timeval meta-timeval
                        :meta-duration meta-duration
                        :meta-boolean meta-boolean))))))



(defun get-meta-pval (self property &optional secondary tertiary)
  (let ((values nil))
    (with-slots (metadata-cache)
      self
      (multiple-value-bind (metadata-class metadata-type unique)
          (metadata-info self property)
        (declare (ignore metadata-type))
        (unless (%property-cached-p self property)
          (setf values (select metadata-class
                               :where (meta-join self property nil nil)))
          (%update-metadata-cache self property values))
        (setf values (%get-metadata-cache-pvalue self property secondary tertiary))
        (if unique
            (if (eql :authority unique)
                (if (or secondary tertiary)
                    (values (car values))
                    values)
                (values (car values)))
            values)))))

(defun get-meta (self property &optional secondary tertiary)
  (let ((values nil))
    (with-slots (metadata-cache)
      self
      (multiple-value-bind (metadata-class metadata-type unique)
          (metadata-info self property)
        (declare (ignore metadata-type))
        (unless (%property-cached-p self property)
          (setf values (select metadata-class
                               :where (meta-join self property nil nil)))
          (%update-metadata-cache self property values))
        (setf values (%get-metadata-cache self property secondary tertiary))
        (if unique
            (if (eql :authority unique)
                (if (or secondary tertiary)
                    (values-list (car values))
                    values)
                (values-list (car values)))
            values)))))

(defun meta-join (self property &optional secondary tertiary)
  (if secondary
      (if tertiary
          [and [= [primary_oid] (%safe-oid self)]
               [= [secondary_oid] (%safe-oid secondary)]
               [= [tertiary_oid] (%safe-oid tertiary)]
               [= [subtype] (string property)]]
          [and [= [primary_oid] (%safe-oid self)]
               [= [secondary_oid] (%safe-oid secondary)]
               [= [subtype] (string property)]])
      [and [= [primary_oid] (%safe-oid self)]
           [= [subtype] (string property)]]))

(locally-disable-sql-reader-syntax)

(export '(:def-metadata :get-meta :store-meta :delete-meta :meta-join) :sql)
