;;;; -*- Mode: LISP; Syntax: ANSI-Common-Lisp; Base: 10 -*-
;;;; *************************************************************************
;;;; FILE IDENTIFICATION
;;;;
;;;; Name:          bool-comp.lisp
;;;; Purpose:       Boolean Function Comparison.
;;;; Author:        R. Scott McIntire
;;;; Date Started:  Aug 2003
;;;;
;;;; $Id: bool-comp.lisp,v 1.2 2003/09/10 22:19:24 rscottmcintire Exp $
;;;; *************************************************************************

(in-package rsm.bool-comp)

(eval-when (:compile-toplevel)
  (declaim (optimize (speed 3) (debug 0) (safety 1) (space 0))))


;;;; ==========================================================================
;;;; ================>  INTERNAL FUNCTIONS   <=================================
;;;; ==========================================================================

;;; CONSTANTS
(eval-when (:load-toplevel :compile-toplevel :execute)
  ;; The three binary pairs.
  (defconstant k #2r00)                 ; Represents the binary pair (1 0)
  (defconstant h #2r01)                 ; Represents the binary pair (0 1)
  (defconstant e #2r10)                 ; Represents the binary pair (1 1)

  ;; The four projections.
  (defconstant hh 0)       
  (defconstant hk 1)
  (defconstant kh 2)
  (defconstant kk 3)

  ;; The logical operators.
  (defconstant or '+)
  (defconstant and '*)
  (defconstant xor '^)
  (defconstant not '~)

  
  (defconstant zero-term #2r11
    "The representation for the zero term.")

  
  (defconstant maxshortval 255
    "Largest 8 bit number.")
  
  (defconstant maxlongval 4095
    "Largest 12 bit number.")

  
  (defconstant allones  
      115792089237316195423570985008687907853269984665640564039457584007913129639935
    "Bit vector of all ones - 256 bits.")

  (defvar bstrings 
#(38597363079105398474523661669562635951089994888546854679819194669304376546645 
  23158417847463239084714197001737581570653996933128112807891516801582625927987 
  6811299366900952671974763824040465167839410862684739061144563765171360567055 
  450552876409790643671482431940419874915447411150352389258589821042463539455 
  1766820105243087041267848467410591083712559083657179364930612997358944255 
  26959946660873538060741835960174461801791452538186943042387869433855 
  6277101735386680763495507056286727952657427581105975853055 
  340282366920938463463374607431768211455))
  
  (defvar bnstrings 
#(77194726158210796949047323339125271902179989777093709359638389338608753093290 
  92633671389852956338856788006950326282615987732512451231566067206330503711948 
  108980789870415242751596221184647442685430573802955824978313020242741769072880
  115341536360906404779899502576747487978354537254490211650198994186870666100480
  115790322417210952336529717160220497262186272106556906860092653394915770695680
  115792089210356248762697446947946071893095522863849111501270640965525260206080
  115792089237316195417293883273301227089774477609353836086800156426807153786880
115792089237316195423570985008687907852929702298719625575994209400481361428480))


  (defconstant ballones  1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365277945956976543709998340361590134383718314428070011855946226376318839397712745672334684344586617496807908705803704071284048740118609114467977783598029006686938976881787785946905630190260940599579453432823469303026696443059025015972399867714215541693835559885291486318237914434496734087811872639496475100189041349008417061675093668333850551032972088269550769983616369411933015213796825837188091833656751221318492846368125550225998300412344784862595674492194617023806505913245610825731835380087608622102834270197698202313169017678006675195485079921636419370285375124784014907159135459982790513399611551794271106831134090584272884279791554849782954323534517065223269061394905987693002122963395687782878948440616007412945674919823050571642377154816321380631045902916136926708342856440730447899971901781465763473223850267253059899795996090799469201774624817718449867455659250178329070473119433165550807568221846571746373296884912819520317457002440926616910874148385078411929804522981857338977648103126085903001302413467189726673216491511131602920781738033436090243804708340403154190335
      "The Big bit strings. These represent 12 variables.
All ones - 4096 bits.")


  (defvar bbstrings 
 #(348129627137717502230584236905541460859988083015794593461411161094651302657185818949608937311665852780296702238146420945995857812728597869087745362617121759315318992181236666113453863378127906104809356670618648742125439613132570915224111561448195539165602636235267901357094682913372869704822659261199343002228979658960595928648968543396753646866526484477607823101008898814353008338657466622571405180564611853295097162106079304811498911362603957546498825033396347116336139020558364556111283517010990696089850256661205456470644338404598941945729363944552250407106164282122708516741999433470781594954198558164064872341268835304415203608577278460029202874034278090065899400771056339226002225065161693307212139790095125041594671635719711819994263504466537183931423702277044696861424294759930518283260984774511505688407756353798301995897667374321131895927626316146872002470981891639941016857214125718272107126877015300972045642236114285480243482633323967260488587824407950089084353299931998696933156400591541605906149955818553083392776356824373144388516935856073948857248791098961637606506772485667480308872303624716128359470643268174327285779659216034375361967667100804489063242224405497170377200973593912677812030081268236113467718063445 
   208877776282630501338350542143324876515992849809476756076846696656790781594311491369765362386999511668178021342887852567597514687637158721452647217570273055589191395308741999668072318026876743662885614002371189245275263767879542549134466936868917323499361581741160740814256809748023721822893595556719605801337387795376357557189381126038052188119915890686564693860605339288611805003194479973542843108338767111977058297263647582886899346817562374527899295020037808269801683412335018733666770110206594417653910153996723273882386603042759365167437618366731350244263698569273625110045199660082468956972519134898438923404761301182649122165146367076017521724420566854039539640462633803535601335039097015984327283874057075024956802981431827091996558102679922310358854221366226818116854576855958310969956590864706903413044653812278981197538600424592679137556575789688123201482589134983964610114328475430963264276126209180583227385341668571288146089579994380356293152694644770053450611979959199218159893840354924963543689973491131850035665814094623886633110161513644369314349274659376982563904063491400488185323382174829677015682385960904596371467795529620625217180600260482693437945334643298302226320584156347606687218048760941668080630838067 
   61434640083126618040691335924507316622350838179257869434366675487291406351268085696989812466764562255346476865555250755175739614010929035721366828697139133996820998620218235196491858243199042253789886471285643895669195225846924279157196157902622742205694582865047276710075532278830506418498116340211648765099231704522458105055700331187662408270563497260754321723707452731944648530351317639277306796570225621169723028606955171437323337299283051331735086770599355373471083356569123156960814738296057181662914751175506845259525471483164519166893417166685691248312852520374595620601529311788961457933093863205423212766106265053720330048572460904711035801300166721776335188371362883392823922070322651760096259962957963242634353818068184438822517089023506561870251241578302005328486640251752444402928409077854971592071957003611465058099588360174317393398992879320036235730173274995283708857155433950283313022390061523700949230982843697437690026347057170693027397851366108839250179994105646828870557011869095577512849992203274073539901710027830554892091223974601285092455669017463818401148253968058967113330406522008728534024231164971940109255233979300183887406058900141968658219216071558324184211936516572825496240602576747549435479658255 
   4063769966588142049384251792671690204591300579950909651300519390209937385103336407972088762392986608330311699277973785361819351899555617148884187112262121704069871504061128398211523697020948320289603385260139868585121863188317948426740601884609286449403921823757991066425229761634702759200264504994544859948198206135726800723528815681674167084045056239038223615965084421957428112902616341897720683041610255096829927962327773986126446436139345807935783949806183040268515241485117095985734827046820903067196695603049090931563941693438898155008513976006446502806686742592872083853019448639736750135652123247051340922271620645576831170528139437276605481019855386265360693394214665438435823638892938054169791515059476167800716011311903250817053659585212496310483545162767058718226742740388293987742346125772507848502814276503482124465731525770285586333785521200158038939349983170894253115064756331341697748562766715575549170921044135628174048435408450979694419313125384631390089727236560296073149685610017995399682684309165989300304782375381787677686968122833548041135199896096828454550662713840476423838976306903301109254521127644058295164743103689117222124136191838184697236290557262612883780556111991198573681284995349059690284646655 
   15935866478678494692948299597427779461677590506849318406155812491935149731778345924421728366190053837387889386368604953506989539316505082735908511037297485053419548904339685953588989275285468030493127088695789343826789734644517032297363850715543687039333626939069589759544746459864177626599752472398767551256342813630190392998564255766822725187292360856200672433938488127974411782900840744430080954906294697039615659647500464080359136580676745542815455011675680018142551796110213980931898783145901888830272224392078007376168164780411017071840152766126871099093923933752965890263911962714380346748593858041902964997235248880987009030406210772236867855136836203521639657022951447543799787527587242014764734720391311398519676135727285891328299899497987572696252057720541588577204829093180822357596212129385454278731453515745226452045302685856285696442969298998132596966796552709459130728773706107310623333088652912139953567095050778284644253595678348135884550154465780402936555837157575050594312666154609225593457891991633111832679693462824350110098887766730577331457747124477545704251343782245181209494131725195666342347253151723804596752048120116928240320780647609952655594956638486526864696323005046584882373166971401015310483455 
   243165735427752782419323904509027599839246051667290210890803316699608832428219873556077233138242812175920068877404368007691392076782510040905495650820869399962268671156242798823890926155842093787758139945589426504631268869798268162616078397661565854175771133631941063897531243237337881204935278878235898476199023542153361310202987939116375754001364973241337224855481058263121630706091124834870707873912325055797809369517590046142439052449858938387976385818843882421535564269903006357706814118038434062672573306437358973309065690707585572993755927039287244943443510658788637931327652272089545734406693358925520735350038337186725641051075932180744520491305346083569422391882107273162319185685637848041020001128610543440136469776514606942298864845908420790425932574180234576669530548995887163526819970178584229873176440407572888205787193476496311419477352762819096270155592217288033765108233460236010807832847404775691389141375421014396778641618600408584546613248222795552348313934041113163521130578185779563973813854037282382170913435589104424597799401490581791168349930580829964540593873922808227828659276255753149470002106468780170052442422042521812031832800542605264728684306258290820958398305769856024455799306591253037055 
   56616434707392290490761425610936370119200663057043891539907083027152279892742690502750959335860032571495923443099728109402277273405692294847282452620712696198373955467828103231444347691550989956881039873361083850474228161013265115146070366552485600677401940739547054356783250877688704546689207392998315053087939593961308784069307532535721754782897722921346289231329299046757563470527204082830856653921129037588309635743044714170374686514192834507347526310839342974723901381831794187667243111491856294079480009392760755857867884715934089062082661945229990888832568635935111302518841844650723711540669774691911876947962618205253056107851217301624134208820551522382692526813304471786055262096352860841786040707619682753575796562108698827464646765070673370820639112646646680196092704945881215564340259946855246755812233393957290992694806509129003220043116247990069681709548957123439420728826971002078466662797840807960741269905252640399703180731109554629405940085423498233309917867723989461867497573180842062573369966031897113621388799322202765664227325846810823722840503845225189485017368600253197092383286748265612238073868175115906793349517449038481661260050568949070931853605813624012125105249859577319301113184255
   3069183075406921794879924958191326267971064214200306562408783097537753842333707652824825400929871111355283689404045219071256757447404925082379759891585283296393050325605295160957871424097373769158306344424796076516420116198969108591963286875537199853974185067048362394637779430939432059181825962412602909207371232332610974498092730815600379641716836477875780887892694506682601714545514557299280443749116298366999808118367767199824411454244308293284236328430140042818554141176729862212003473443329934253658543598381852557345365121904173598417697319574120953343017491137034836726396178491650633622843466212310429183510510530239159406467061556827771687018283934170672009603659688656288938988832586642033232829141797821717732343335491977380591365309980605924075346344595946236135808869775698239914701090142199479355699096454885099789523282113402052735898158200734918439009434916317713414966294415863452506570616997189211537932078804860184952782983296718976573604867252520342427263370964212214179501646026999883132473348993841767502159721508536222916361037612170553264842837254028344488117291677487317670813230349467433085520842722939411387618889248612042885557080945351623459570089440346818169798655 
   9019518416950528558373478086511232658951474842525520401496114928154304263954888744871685539895273661114526127711752741311647435876817688262218507156038130610401495163901903232972509630902824493743884115860872040713600092818281127439862908173490194403044543500338688943473183626296612162177598595038778150748224640910134102965202156674833151227746254019390992078276485688571590771528014587778063423931391577469267104133004258365873497569236190284521423412766208292961107364265401018354177709155899496583189695117148115526028519719111800527135022301538971034869731071392917706799066727972597420358410510886825464968542381153665088451966387602262225924474381597419700979336568073686526612204777692478073154603295328346708081477208471573764157995367182981345833188792298353665579483696959364851940949180543791256181168605705960494222847926663270043419204265042085837665196634999217142030525936161299038675659121198609236848707046490685842854105983966170352434684285893280866191145680663659930149837851838912623143180573101956428171183341036820723919126551718785709217087517790304345238427565334938096608425513343670260869751275056265908287017530874923325587455 
   77894081334563378764508707451908598634363875792546106220991280972045090113143756247350828053674522774404165906274379945792494917548310574063247841644081924087572642176171338404713080855378219374586978557520215735100683452281251094657399551423437026730064426117955053306539262993393010490154689815476218281895530503728577964048272690268380605021290062575641610262537916618269035510094895537402051806873931103824810636945586616354119104490688688465090673215564856884925064517573583036622430157657071060643708268673115864488857173074444884028539024022834729378929140587462440109211187637496865646435687538463024479584585977542842391773730913382839938376372552164733161868656353147749107588601633182995592222440780081492750197264500634681643735282428696126573186770730858155949835208004119920336672905266683507537223194485299755982883113166759811146892486911202254737577396860795701495521591630513742317977001278150503463727678239482415932020220546888454420114386958035618892222763361072714870715782011337989709044796862015189454860473492532343345049314764369668264521641114841645055 
   5809605995369958062859502533304574370686975176362895236661486152287203730997110225737336044533118407251326157754980517443990529594540047121662885672187032401032111639706440498844049850989051627200244765807041812394729680540024104827976584369381522292361208779044769892743225751738076979568811309579125511333060926513482473809005666703473190248287048465778434758483174104149662948997560847687792971140051020682567112405484016981375524172902727645737457138721455439711003261232285410598575761974797795062263483166628242641362219532106140260933522639896469586945691349003977003574070954066763611828411629194964597959539551299822089652627382839279153212532817237109014238798450006761842998540187791824516338046377217714886708046860110828267745904133950686508912304662040967153253533926746540671626187017359828581776751847850397382690835864130139585559180412500910631140632901294962313162001008210248469182468387417181131957600255 
   32317006071311007300714876688669951960444102669715484032130345427524655138867890893197201411522913463688717960921898019494119559150490921095088152386448283120630877367300996091750197750389652106796057638384067568276792218642619756161838094338476170470581645852036305042887575891541065808607552399123930385521914333389668342420684974786564569494856176035326322058077805659331026192708460314150258592864177116725943603718461857357598351152301645904403697613233287231227125684710820209725157101726931323469678542580656697935045997268352998638215525166389437335543602135433229604645318478604952148193555853611059596230655)
 "The bit string representations of the variables X1, X2, X3, X4, X5, X6, X7, X8, X9, X10, X11, X12.")

  (defvar bbnstrings 
      #(696259254275435004461168473811082921719976166031589186922822322189302605314371637899217874623331705560593404476292841891991715625457195738175490725234243518630637984362473332226907726756255812209618713341237297484250879226265141830448223122896391078331205272470535802714189365826745739409645318522398686004457959317921191857297937086793507293733052968955215646202017797628706016677314933245142810361129223706590194324212158609622997822725207915092997650066792694232672278041116729112222567034021981392179700513322410912941288676809197883891458727889104500814212328564245417033483998866941563189908397116328129744682537670608830407217154556920058405748068556180131798801542112678452004450130323386614424279580190250083189343271439423639988527008933074367862847404554089393722848589519861036566521969549023011376815512707596603991795334748642263791855252632293744004941963783279882033714428251436544214253754030601944091284472228570960486965266647934520977175648815900178168706599863997393866312801183083211812299911637106166785552713648746288777033871712147897714497582197923275213013544971334960617744607249432256718941286536348654571559318432068750723935334201608978126484448810994340754401947187825355624060162536472226935436126890 
        835511105130522005353402168573299506063971399237907024307386786627163126377245965479061449547998046672712085371551410270390058750548634885810588870281092222356765581234967998672289272107506974651542456009484756981101055071518170196537867747475669293997446326964642963257027238992094887291574382226878423205349551181505430228757524504152208752479663562746258775442421357154447220012777919894171372433355068447908233189054590331547597387270249498111597180080151233079206733649340074934667080440826377670615640615986893095529546412171037460669750473466925400977054794277094500440180798640329875827890076539593755693619045204730596488660585468304070086897682267416158158561850535214142405340156388063937309135496228300099827211925727308367986232410719689241435416885464907272467418307423833243879826363458827613652178615249115924790154401698370716550226303158752492805930356539935858440457313901723853057104504836722332909541366674285152584358319977521425172610778579080213802447919836796872639575361419699854174759893964527400142663256378495546532440646054577477257397098637507930255616253965601952741293528699318708062729543843618385485871182118482500868722401041930773751781338573193208905282336625390426748872195043766672322523352268 
        982954241330025888651061374792117065957613410868125910949866807796662501620289371151836999468232996085543629848884012082811833824174864571541869259154226143949135977923491763143869731891184676060638183540570302330707123613550788466515138526441963875291113325840756427361208516461288102695969861443386380241587707272359329680891205299002598532329015956172069147579319243711114376485621082228436908745123609938715568457711282742997173396788528821307761388329589685975537333705105970511373035812736914906606636018808109524152407543730632306670294674666971059973005640325993529929624468988623383326929501811286771404257700240859525280777159374475376572820802667548421363013941806134285182753125162428161540159407327411882149661089090951021160273424376104989924019865252832085255786244028039110446854545245679545473151312057783440929593413762789078294383886069120579771682772399924539341714486943204533008358240984379215187695725499159003040421552914731088438365621857741428002879905690349261928912189905529240205599875252385176638427360445288878273459583593620561479290704279421094418372063488943473813286504352139656544387698639551041748083743668802942198496942402271498531507457144933186947390984265165207939849641227960790967674532080 
        1040325111446564364642368458923952692375372948467432870732932963893743970586454120440854723172604571732559795015161289052625754086286237990114351900739103156241887105039648869942150066437362769994138466626595806357791196976209394797245594082459977331047403986882045713004858818978483906355267713278603484146738740770746060985223376814508586773515534397193785245687061612021101596903069783525816494858652225304788461558355910140448370287651672526831560691150382858308739901820189976572348115723986151185202354074380567278480369073520357927682179577857650304718511806103775253466372978851772608034726943551245143276101534885267668779655203695942811003141082978883932337508918954352239570851556592141867466627855225898956983298895847232209165736853814399055483787561668367031866046141539403260862040608197762009216720454784891423863227270597193110101449093427240457968473595691748928797456577620823474623632068279187340587755787298720812556399464563450801771344160098465635862970172559435794726319516164606822318767183146493260878024288097737645487863839445388298530611173400788084364969654743161964502777934567245083969157408676878923562174234544414008863778865110575282492490382659228898247822364669746834862408958809359280712869543680 
        1044372945546673828197059762417026954800502571456876931065827327471461972821825678502902390206631368287052718825052894233034066448646477102180500179340327980460903556994805658654408001145108432846397576884767250437032492049663068228640037320493871073809768575078864634481524503993658744936841378031125630239135682634068157595553907065934494117874392161071967268630592757954931050604189499026969785460738929265188251870658590413970416374951231195893953659645177365668990274509878983454352918652249826186380720497759224291404556847049016414820116251680890624350219398922434372584335734388449630404515847080634152714058809270664364623816701429169315371754247697433994176562656146066230462875407957492679621654635564983813385495231023408174091462213500113564221574854773413548995695679450698374027425358111405131610944537607879160761240956820277539402086435979141617874815978878367113591440913603448709010757297957250003996973141247805662445803646376223433329878923069384486850123343958838515748874889108470208492856409563667617066496390779656608815440708680455115994414915549760435273816066113220195745407416742423189412069582551371258052742225599983009157662680521765857237071078259853024604738224458732986851207870637736939387843706880 
        1044388881169986771263999928297300478070936649208137728716943272393150591271948624420606938378920325202647294538519193960583205430494401530480726046945869627125087576581441327184118791310492792158585976224097806280786892334766443875874066521728508219835242054530032570439342984842587365877130096578662750128451040500682764243793544319987273001483203699431458496061689471587577966752850769161623090706823127685972966430520428544916906687945372820189637536712212655530164534640139529398430844193326157970231116707311043062974574041904731135129602518839900824182031247902924614891437360369084692512773049940085501258098285770563207273639006194329011676441358313778892352118743746625795899402033165894235998571329265373996173471467022665683468183571100746705885850316405201516404038307610261005853895790796714546886639039188218465580120113917176202211286567528963263244593849404764230833283608612046582861144620238070068732151016953715065309433503193260162865354888677237019030264347447682156758356038253494239532670303481845396141046688302205997576446382970422445081164582128534982238690352916408567003808683045489108822658780334520875388558807595660704043381189270580666647121408487807204873312099823339727666234219348909033811901153280 
        1044388881413152506635136276009232092089202823436447410265032820226910016431650373821674532042254867838139147378579230266491649995086065497860958814445672983098674523922997302141987634666555615082983722320304956269495278966036628895198106523331321502350737542153318103393882108000571554757684726905909324459997731583883472732858966036228952156530271920897101714520128973521712735784643100820956652071166631477054434832397108876846187098344767158469121788585996206841660890750835750693609949169201177900602307658491760075332453005821036069979320207117722662159235830901138134661393429664477233482343753829841470905483136731221333733877769217174834552514251616968573563993492617495295314148382180608135581157273932514282997974199539452706406993951290912724329624341760460719763633771633144874653690249377653301500883009114539659231880768729006104695088072439311612787369829426929753368862093420031376900651804074900837670263910502048479989177994719261381762582742114295637847119814372497857489551334050635355850952294274817187604959104441222319544162008246019080907519047450074089096679813611777251441599542273895187986028643056257369619265109472987219292553483853374985528466622647542440199749314968114021310984993945131021102041006080 
        1044388881413152506691752710716624382576895065971976858589353558325762581703586392634626505372588775243352352872105555185162748037255922495907952398447320058874700219096305073257981830242798435018035019686250651065418447415300338976514028339919790540980387792506834595479320761864581409260493792716549666612049159545942355726765079667777658031392208221100212494804933965627458645374255563389838434653801141053202689771772723357135216290338695574272496666981821274149184005607430785375049614222602832045450996628806886507199929541770466891583529548235274898663973127724463951951808300980838223831519578183355159780297410327421594977202888369167777179438592323739958538795846107460850234988177201145750964409766625686468495075918326548817949557684257813730076538763495642113203681518969810948925707607978938570829087460191619207747778301032821196208427179851985730907623422392806420997835744218954081402941621610986598423511742048440577277941329354904592254225541145045407068107116812699371822895596907372297376022604084695037964149568827092433282418334219228004804244213575376376596603956419390270373352068036894356733923812512845494539668164417753658652817480459690527778339054327242899088717363700792681812630673715267993584984391680 
        1044388881413152506691752710716624382579964249047383780384233483283953907971548437330409861406439184862803595481780311363145047917784297492335081783587410389201085291003814724679247064006671965573116422575979128538114100332241674615061933189180684714263835399074900879577540164624257737073754377690779747879247076068708297591543861086689922251656106269806526857140849097848020246865224175226804081438728633403210458335090491660415105742009535386950924884328661026761230353637743702090864583446899967829903677272414380179127411591801030617544226984469391350202964315137212226053642808605295196669336567154773082816496671483611706639790862104308694690915303767542225100781954758506791181210226942698767971330918318987522521788982684753862563089534063043478107744494626356398106199729676496226503074872846326045491459111066027723006347168934171097334117299464743656642561004725739279259315461208549110360886408197976252866883289138591398644610234775266782248621442697914105954021224136874892190232353067578327032607013349675284007976635788833539884684616422541182911816223459033073906897174276429338970188739690807348257688010677971263071629760560585335781557762874848132251630064790978167461342051030462977170181956787177465479828602880 
        1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365200051875641980331233831654138225785083950552277465749725235095346794307599601916087333516532942974033504539897429691338256245201060803893914535756384924762851404239611614608500917109405562380204866454265949087291595759606743764877742468162792104667105495459173531264931375171503341077321717949680998881907145818504688483711045395643582170427950798206975128373353831495314746178286730941650689781849877290214668035731179963609644181307854096174130583818979052166921580848728037242695212949929951551042190561929025086448680160504932230311456540897613584640906445984196552467049924272345293647753175864255808082351549504606730041888017823936400114385158144513058535899526249634545253015374794054599883356218175227331452924722558549936960733419533892684504472716145406068552393021232726327979635228876199079965686627072767760143813112977632709390627732330807247612718081853317533368977597841535037065250244845293595869833157234580037901524982220379728456454033998120376310912300218496266262777387344074565011593368670327711483761631037639070577436688718671720575540186699288312545280 
        1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365277945956976543709998340361590134383718314428070011855946226376318839397712745672334684344586617496807908705803704071284048740118609114467977783598029006681129370886417827884046127656956366228892478256460574066365210290771821284975289641976879497160717152633965328563257396990506204493271825517833589428002008947976305421968653169489800700043920461069306004176574557017203334673772721009211507464275228928957284067323355657482772548674267805293784364913069105690745579399763137016726168676614418373815785804419263443829994913528343726197924232233843448230234354442216902501675118478607266340496883906056813968109678650873269623047506144251207192348736722002959785894766663346330782590857255426849356308544146420467254325915846046997571423088052709552219416707951538967168791556618640795272589062502312550940406613158238821101345989328956470661586832993202111821078441535291621023613008604897804903434271160062834068634843945666266783530255900254990723856788556496635177956672584474648141783972986500343820889912556558586040315196548818440919773527784966907775417291159271196590080 
        1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365277945956976543709998340361590134383718314428070011855946226376318839397712745672334684344586617496807908705803704071284048740118609114467977783598029006686938976881787785946905630190260940599579453432823469303026696443059025015972399867714215541693835559885291486318237914434496734087811872639496475100189041349008417061675093668333850551032972088269550769983616369411933015213796825837188091833656751221318492846368125550225998300412344784862595674492194617023774188907174299818431120503398938670142390167527982718281038672250482020056617189028439217958762461661095296946237237440488670954249120630699182954444685807463642006912490558758032756573144864958427211423010838419416209904320775931621040854102139836942364029067786745528754801263275255572023493503792206541186428523051062105479286926994901193978367674231926737841718190431468443009066164503568191274591482133452385466754657575807952456415920200667342675683651625588293191772291620716891753772421453754942251261942325159403931650834773087264785777247077752391129614356077901998275463259428483942050248854729343557959680))




  (defmacro term-ref (term index)
    "Macros for setting and referencing a term."
    `(aref ,term ,index))
  
  (defmacro term-set (term index val)
    `(setf (aref ,term ,index) ,val))

  (defmacro term-length (term)
    `(length ,term))

  (defmacro make-identity-term (len)
    `(make-array ,len :initial-element e :element-type '(unsigned-byte 2)))

  (defmacro make-zero-term ()
    zero-term)

  (defmacro is-zero-term (term)
    `(eql ,term zero-term))

  (defmacro is-not-zero-term (term)
    `(not (eql ,term zero-term))))



(defun flatten (tree)
  "Flattens a tree into a list."
  (labels ((rec (tree acc)
             (cond ((null tree) acc)
                   ((atom tree) (cons tree acc))
                   (t (rec (car tree) (rec (cdr tree) acc))))))
    (rec tree nil)))


(defun is-not-zero-array (term size)
  "Predicate that checks to see if tensor term, <term>, 
has all entries set to K.
Returns t if at least one entry differs from K; otherwise nil."
  (declare (type fixnum size)
           (type vector (unsigned-byte 2) term))
  (do ((i 0 (1+ i)))
      ((= i size) nil)
    (declare (type fixnum i))
    (when (/= (term-ref term i) k)
      (return-from is-not-zero-array t))))

(defun make-term (&rest args)
  "Makes a term from a list of arguments. The list should consist of 
the numbers 0, 1, 2."
  (let* ((len (term-length args))
         (term (make-array len :initial-element e 
                           :element-type '(unsigned-byte 2))))
    (declare (type vector (unsigned-byte 2) term)
             (type fixnum len))
    (do ((i 0 (1+ i))
         (cur args (cdr cur)))
        ((eql len i) term)
      (declare (type fixnum i))
      (term-set term i (car cur)))))



(defun copy-term (term len)
  "Copy a term."
  (declare (type vector (unsigned-byte 2) term)
           (type fixnum len))
  (let ((new-term (make-identity-term len)))
    (declare (type vector (unsigned-byte 2) new-term))
    (do ((i 0 (1+ i)))
        ((= i len) new-term)
      (declare (type fixnum i))
      (term-set new-term i (term-ref term i)))))



(defun invert-term (term)
  "Finds the logical negation of a term."
  (declare (type vector (unsigned-byte 2) term))
  (let ((len (term-length term))
        (temp-term nil)
        (result nil))
    (declare (type vector (unsigned-byte 2) temp-term)
             (type fixnum len))
    (do ((i 0 (1+ i))
         (new-term (make-identity-term (term-length term))))
        ((= len i) result)
      (declare (type fixnum i)
               (type vector (unsigned-byte 2) new-term)
               (type vector (unsigned-byte 2) temp-term))
      (case (term-ref term i)
        (#.h (setf temp-term (copy-term new-term len))
             (term-set new-term i h)
             (term-set temp-term i k)
             (push temp-term result))
        (#.k (setf temp-term (copy-term new-term len))
             (term-set new-term i k)
             (term-set temp-term i h)
             (push temp-term result))))))


(defun term-mult (t1 t2 size)
  "Takes the logical AND of two terms of the same size.
Assumes both terms are of size <size>."
  (declare (type vector (unsigned-byte 2) t1)
           (type vector (unsigned-byte 2) t2)
           (type fixnum size))
  (let ((prod (make-identity-term size))
        val
        t2-cur)
    (declare (type vector (unsigned-byte 2) prod)
             (type (unsigned-byte 2) val)
             (type (unsigned-byte 2) t2-cur))    
    (do ((i 0 (1+ i)))
        ((= i size) prod)
      (declare (type fixnum i))
      (setf t2-cur (term-ref t2 i))
      (setf val (case (term-ref t1 i)
                  (#.h (case t2-cur
                         (#.h h)
                         (#.e h)
                         (otherwise (return-from term-mult zero-term))))
                  (#.k (case t2-cur
                         (#.k k)
                         (#.e k)
                         (otherwise (return-from term-mult zero-term))))
                  (#.e t2-cur)))
      (term-set prod i val))))


(defun term-list-mult (t1 tl2 t-len)
  "Creates a new list by doing a term-wise AND of <tl2> with <t1>.
Assumes <tl> and each member of <tl2> have size <t-len>."
  (declare (type vector (unsigned-byte 2) t1))
  (if (or (null t1) (null tl2)) 
    (error "term-list-mult: null list")
    (progn
      (when (is-zero-term t1) (return-from term-list-mult zero-term))
      (let ((result nil)
            (prod nil))
        (do ((curj tl2 (cdr curj)))
            ((null curj) (or result zero-term))
          (declare (type vector (unsigned-byte 2) prod))
          (setf prod (term-mult t1 (car curj) t-len))
          (when (not (is-zero-term prod)) (push prod result)))))))


(defun list-mult (tl1 tl2 t-len)
  "Creates a new list by doing a term-wise AND with each pair of 
terms (t1,t2) :t1 a member of tl1, t2 a member of tl2.
Assumes each member is of size <t-len>."
  (declare (type fixnum t-len))
  (let ((prod nil)
        (t1-item nil)
        (result nil))
    (do ((curi tl1 (cdr curi)))
        ((null curi) (or result zero-term))
      (declare (type vector (unsigned-byte 2) t1-item)
               (type vector (unsigned-byte 2) prod))
      (setf t1-item (car curi))
      (do ((curj tl2 (cdr curj)))
          ((null curj))
        (let ((item (car curj)))
          (declare (type vector (unsigned-byte 2) item))
          (setf prod (term-mult t1-item item t-len))
          (when (not (is-zero-term prod)) (push prod result)))))))


(defun compare-by-size (t1 t2 len)
  "Predicate that determines if <t1> is less than <t2>.
Used by term-sort below.
<t1> and <t2> are each assumed to be of length <len>."
  (declare (type vector (unsigned-byte 2) t1)
           (type vector (unsigned-byte 2) t2)
           (type fixnum len))
  (dotimes (i len nil)
    (let ((c1 (term-ref t1 i))
          (c2 (term-ref t2 i)))
      (declare (type (unsigned-byte 2) c1)
               (type (unsigned-byte 2) c2))
      (cond ((< c1 c2) (return-from compare-by-size t))
            ((> c1 c2) (return-from compare-by-size nil))
            (t nil)))))
             

(defun term-sort (tl len)
  "Sorts terms using the <compare-by-size> the term comparator."
  (declare (type fixnum len))
  (sort tl #'(lambda (t1 t2) 
               (declare (type vector (unsigned-byte 2) t1)
                        (type vector (unsigned-byte 2) t2))
               (compare-by-size t1 t2 len))))


(defun is-contained? (t1 t2 len)
  "Whenever term <t1> is true does it follow that term <t2> is true>?
In other words, is <t1> contained in <t2>?
Assumes that <t1> and <t2> are of size <len>."
  (declare (type vector (unsigned-byte 2) t1)
           (type vector (unsigned-byte 2) t2)
           (type fixnum len))
  (do ((i 0 (1+ i))
       (t2-cur))
      ((= i len) t)
    (declare (type fixnum i)
             (type (unsigned-byte 2) t2-cur))
    (setf t2-cur (term-ref t2 i))
    (case (term-ref t1 i)
      (#.h (case t2-cur
             (#.k (return-from is-contained? nil))))
      (#.k (case t2-cur
             (#.h (return-from is-contained? nil))))
      (#.e (case t2-cur
             ((#.h #.k) (return-from is-contained? nil)))))))


(defun combinable? (t1 t2 len)
  "Can the two terms, <t1> <t2>, be combined into one.
Returns the combined result if true.
Assumes <t1> and <t2> are of size <len>."
  (declare (type vector (unsigned-byte 2) t1)
           (type vector (unsigned-byte 2) t2)
           (type fixnum len))
  (let (c1 
        c2
        (index -1)
        term)
    (declare (type vector (unsigned-byte 2) term)
             (type (unsigned-byte 2) c1)
             (type (unsigned-byte 2) c2)
             (type fixnum index))
    (do ((i 0 (1+ i)))
        ((= i len)
         (when (/= index -1)
           (setf term (copy-term t1 len))
           (term-set term index e) 
           term))
      (declare (type fixnum i))
      (setf c1 (term-ref t1 i) 
            c2 (term-ref t2 i))
      (cond ((or 
              (and (eql c1 h) 
                   (eql c2 k))
              (and (eql c1 k)
                   (eql c2 h)))
             (if (/= index -1)
                 (return-from combinable? nil)
               (setf index i)))
            ((not (eql c1 c2)) 
             (return-from combinable? nil))))))



(defun term-eql (t1 t2 len)
  "Predicate: Checks to see if the terms <t1> and <t2> are the same.
Assumes <t1> and <t2> are of size <len>."
  (declare (type vector (unsigned-byte 2) t1)
           (type vector (unsigned-byte 2) t2)
           (type fixnum len))
  (if (or (null t1) (null t2))
    nil
    (do ((i 0 (1+ i)))
        ((= i len) t)
      (declare (type fixnum i))
      (let ((ti1 (term-ref t1 i))
            (ti2 (term-ref t2 i)))
        (declare (type (unsigned-byte 2) ti1)
                 (type (unsigned-byte 2) ti2))
        (when (not (eql ti1 ti2))
          (return-from term-eql nil))))))


(defun remove-successive-term-pairs (tl len)
  "Returns a new list formed by removing successive equal pairs of terms in 
the list of terms <tl> and a second boolean value that indicates if a 
removal took place.
Assumes each member of <tl> has size <len>."
  (if (null tl) 
    nil
    (let ((result nil))
      (do ((curr tl)
           (next (cdr tl))
           (rm-dup-flag nil))
          ((null next) (values (cons (car curr) result) rm-dup-flag))
        (let ((fc (car curr))
              (fn (car next)))
          (declare (type vector (unsigned-byte 2) fc)
                   (type vector (unsigned-byte 2) fn))
          (if (term-eql fc fn len)
            (if (null (cdr next)) 
              (return-from remove-successive-term-pairs (values result t))
              (progn 
                (setf rm-dup-flag t)
                (setf curr (cdr next))
                (setf next (cddr next))))
            (if (null (cdr next))
                (return-from remove-successive-term-pairs 
                  (values (cons fc (cons fn result)) rm-dup-flag))
              (progn
                (push fc result)
                (setf curr next)
                (setf next (cdr next))))))))))


(defun remove-all-pairs (tl len)
  "Returns a new list formed by removing all successive pairs 
repeated (if necessary). Routine first sorts the list.
Assumes each member of <tl> has size <len>."
  (declare (type fixnum len))
  (if (atom tl)
    tl
    (do ((ntl tl)
         (duped t))
        ((not duped) ntl)
      (multiple-value-setq (ntl duped)
        (remove-successive-term-pairs (term-sort ntl len) len)))))


(defun copy-adjusted-terms (tl j size)
  "Uses the assumed symmetry of variables 0 - j to rearrange
the values in the 0 - j slots of each term. 
In these symmetry slots, slot values are placed into
a canonical order by placing the E values first followed by the 
H values followed by the K values.
After the terms are rearranged they are then copied."
  (declare (type fixnum j)
           (type fixnum size))
  (do ((cursor tl (cdr cursor))
       (result nil)
       term c-term 
       (n-list nil nil))
      ((null cursor) result)
    (declare (type vector (unsigned-byte 2) term)
             (type vector (unsigned-byte 2) c-term))
    (setf c-term (car cursor))
    (setf term (copy-term c-term size))
    (do ((i 0 (1+ i)))
        ((= i j))
      (declare (type fixnum i))
      (push (term-ref c-term i) n-list))
    (when n-list
      (setf n-list (sort n-list #'>))
      (do ((cur n-list (cdr cur))
           (i 0 (1+ i)))
          ((null cur))
        (declare (type fixnum i))
        (term-set term i (car cur))))
    (push term result)))

(defun simplify (tl j t-len)
  "Returns a list formed by using the simple notion of combinable?.
Simplify simplifies XOR terms by removing
equal term pairs and combining certain \"orthogonal\" terms.
Assumes each member of <tl> has size <t-len>."
  (declare (type fixnum j)
           (type fixnum t-len))
  (let ((ts (term-sort (copy-adjusted-terms tl j t-len) t-len)))
    (setf ts (remove-successive-term-pairs ts t-len))
    (if (null ts) 
      (return-from simplify nil)
      (let ((combine nil)
            (result nil))
        (do ((c1 ts (cdr c1))
             (temp nil nil))
            ((null c1) result)
          (do ((cur (cdr c1) (cdr cur)))
              ((null cur))
            (setf combine (combinable? (car c1) (car cur) t-len))
            (if combine
              (progn
                (push combine result)
                (if (null (cdr cur))
                    (return-from simplify (simplify 
                                           (nconc result temp) j t-len))
                  (return-from simplify (simplify 
                                         (nconc result temp 
                                                (cdr cur)) j t-len))))
              (push (car cur) temp)))
          (push (car c1) result))
        result))))


(defun var->term (var size)
  "Returns a term given a symbol. To be a proper symbol, <var> must 
satisfy the predicate is-var? - returning a variable index.
Example: (rsm.bool-comp::var->term 'x2 3) 
  #(E H E)"
  (declare (type fixnum size))
  (let* ((term (make-identity-term size))
         (pos (is-var? var)))
    (declare (type vector (unsigned-byte 2) term))
    (when pos
      (do ((i 0 (1+ i)))
          ((= i size) term)
        (declare (type fixnum i))
        (when (= i (1- pos))
          (term-set term i h))))))


(defun rpn-eqn-tree->term-list (eqn term-size)
  "Takes an RPN tree of variables, <eqn> and a <term-size> and converts 
it into a term list.
Example: (rsm.bool-comp::rpn-eqn-tree->term-list '(+ x1 x2 (* x3 x4) x5) 5)
(+ #(1 2 2 2 2) #(2 1 2 2 2) (* #(2 2 1 2 2) #(2 2 2 1 2)) #(2 2 2 2 1))
Assumes that a single variable name is used - in the above case, x."
  (cond ((atom eqn)
         (case eqn
           (#.or  or)
           (#.and and)
           (#.xor xor)
           (#.not not)
           (otherwise 
            (var->term eqn term-size))))
        (t
         (mapcar #'(lambda (x)
                     (rpn-eqn-tree->term-list x term-size)) eqn))))


(defun is-var? (var)
  "Predicate to determine if a symbol is a variable. Returns 
the number that results from the string containing the first digit
to the end of the symbol-name, or nil if no digits are found.
Example: (rsm.bool-comp::is-var? 'xr23) 23"
  (let* ((var-name (symbol-name var))
         (pos (position-if #'digit-char-p var-name)))
    (when pos
      (parse-integer (subseq var-name pos 
                             (position-if #'(lambda (char) 
                                              (not (digit-char-p char)))
                                          var-name :start pos))))))


(defun max-var-size (eqn)
  "Finds the maximum variable size of an equation tree.
Example: (rsm.bool-comp::max-var-size '(+ x1 x2 (* x4 x5 ) x3)) 5
Assumes that a single variable name is used - in the above case, x."
  (cond ((null eqn) 0)
        ((atom eqn)
         (let ((eqn-number (is-var? eqn)))
           (or eqn-number 0)))
        (t
         (apply #'max (mapcar #'max-var-size eqn)))))


(defun rpn-eqn->term-list (eqn)
  "Takes an RPN tree of variables, <eqn> and converts it into a \"term list\".
 Example: (rsm.bool-comp::rpn-eqn-tree->term-list '(+ x1 x2 (* x3 x4) x5))
          (+ #(1 2 2 2 2) #(2 1 2 2 2) 
             (* #(2 2 1 2 2) #(2 2 2 1 2)) #(2 2 2 2 1))
 Assumes that a single variable name is used - in the above case, x."
  (let ((term-size (max-var-size eqn)))
    (declare (type fixnum term-size))
    (rpn-eqn-tree->term-list eqn term-size)))


(defun term-flip (term p1 p2 len)
  "Returns a flipped copy of <term>; locations p1 and p2 are flipped.
 Assumes term has size <len>."
  (declare (type fixnum p1)
           (type fixnum p2)
           (type fixnum len)
           (type vector (unsigned-byte 2) term))
  (let ((copy (copy-term term len)))
    (declare (type vector (unsigned-byte 2) copy))
    (rotatef (term-ref copy p1)
             (term-ref copy p2))
    copy))


(defun flip (tl p1 p2 len)
  "Returns a new flipped version of a term list <tl>.
Flips locations p1 and p2. 
Assumes each member of <tl> has size <len>."
  (declare (type fixnum len)
           (type fixnum p1)
           (type fixnum p2))
  (do ((cur tl (cdr cur))
       (result nil))
      ((null cur) result)
    (let* ((term (car cur))
           (copy (copy-term term len)))
      (declare (type vector (unsigned-byte 2) term)
               (type vector (unsigned-byte 2) copy))
      (rotatef (term-ref copy p1) (term-ref copy p2))
      (push copy result))))


(defun make-term-flip (p1 p2 len projection)
  "Makes a modified (based on <projection>) identity term modified at 
 locations p1 and p2.
 Example: (rsm.bool-comp::make-term-flip 0 2 6 HK)
           #(1 2 0 2 2 2)"
  (declare (type fixnum p1)
           (type fixnum p2)
           (type fixnum len))
  (let ((flip (make-identity-term len)))
    (declare (type vector (unsigned-byte 2) flip))
    (case projection
      (#.hh 
       (term-set flip p1 h)
       (term-set flip p2 h))
      (#.hk
       (term-set flip p1 h)
       (term-set flip p2 k))
      (#.kh
       (term-set flip p1 k)
       (term-set flip p2 h))
      (#.kk
       (term-set flip p1 k)
       (term-set flip p2 k)))
    flip))


(defun sym-diff (tl p1 p2 size projection)
  "Forms a new list of terms that represent a \"projection\" 
(based on one of the 4 types of <projection>)
of the symmetric difference of the \"term list\" <tl> with its 
flipped version. The result is simplified using simplify. 
Recall that simplify simplifies XOR terms by removing
equal term pairs and combining certain \"orthogonal\" terms.
Assumes each member of <tl> has size <size>."
  (declare (type fixnum p1)
           (type fixnum p2)
           (type fixnum size))
  (let* (flipped
         (result nil)
         (flip (make-term-flip p1 p2 size projection)))
    (declare (type (unsigned-byte 2) flip))
    (do ((cursor tl (cdr cursor)))
        ((null cursor) result)
      (let ((term (car cursor)))
        (declare (type vector (unsigned-byte 2) term))
        (unless (= (term-ref term p1) (term-ref term p2))
          (push term result))))
    (when result
      (setf flipped (flip result p1 p2 size))
      (setf result (nconc flipped result))
      (setf result (term-list-mult flip result size))
      (when (not (is-zero-term result))
        (simplify result 0 size)))))



(defun is-function-clearly-non-symmetric? (tl size start-j proj)
 "Predicate: Determines if an XOR \"term list\" <tl> can be \"clearly\" 
(using a non-aggressive simplifier) shown to represent a 
non-symmetric function. This is done by determining if the list <tl> 
is invariant under each of the generators of the symmetric group - 
the transpositions. Actually, the transpositions (0,i) {i=1,n-1} are used
as generators. For each transposition, each of 4 different projections 
(a resolution of the identity) is checked for a non-null status. 
If symmetry can't be shown, one of these sym-diff flipped projections
is returned. The final result is:
(values {sym-diff result} {flip 1 position} {flip t position} 
        {projection - one of 4}).
Assumes <size> > 1.
Assumes each member of <tl> has size <size>."
  (declare (type fixnum size)
           (type fixnum start-j)
           (type fixnum proj))
  (let ((result nil)
        (firsttimeflagk t))
    (do ((j start-j (1+ j)))
        ((= j size))
      (declare (type fixnum j))
      (do ((pj (if firsttimeflagk proj 0) (1+ pj)))
          ((= pj 4))
        (declare (type fixnum pj))
        (setf result (sym-diff tl 0 j size pj)) 
        (when result
          (return-from is-function-clearly-non-symmetric? 
            (values result j pj))))
      (setf firsttimeflagk nil))))


(defun eval-function (tl size val)
  "Evaluates the function represented by a \"term list\", <tl>, 
given the array of values, <val>. The <val> array may take on other 
values than 0 and 1.If a term at index i is 
H and <val> at index i is either E or H, then the term at this 
index evaluates to 1; 0 otherwise.
K and <val> at index i is either E or K, then the term at this 
index evaluates to 1; 0 otherwise.
E, evaluates to 1.
The value of the term is then the AND of these pieces.
This can be useful, as terms can be used as values to term lists.
Assumes each member of <tl> has size <size>.
Example: (rsm.bool-comp::eval-function '(#(E H E) #(H E K)) 3 #(0 1 1)) = 1
Example: (rsm.bool-comp::eval-function '(#(E H E) #(H E K)) 3 #(2 1 0)) = 
           1 * 1 * 1 + 1 * 1 * 1 = 0"
  (declare (type fixnum size)
           (type vector (unsigned-byte 2) val))
  (if (not (= size (term-length val)))
      (error "eval-function: <size=~a> does not match <val term length=~a>" 
             size (term-length val))
    (let ((xor-sum 0)
          (cur-val 0))
      (declare (type fixnum xor-sum)
               (type fixnum cur-val))
      (dolist (term tl xor-sum)
        (declare (type vector (unsigned-byte 2) term))
        (setf cur-val 
              (do ((i 0 (1+ i))
                   (curr 0))
                  ((= i size) 1)
                (declare (type fixnum i)
                         (type (unsigned-byte 2) curr))
                (setf curr (term-ref term i))
                (cond ((= curr h)
                       (when (= (term-ref val i) 0)
                         (return 0)))
                      ((= curr k)
                       (when (= (term-ref val i) 1)
                         (return 0))))))
        (setf xor-sum (logxor xor-sum cur-val))))))


(defun compare-by-weight (t1 t2 size)
  "Returns true if <t1> is larger than <t2>. Here, <t1> is larger than <t2> 
if there are more E's followed by H's followed by K's in <t1> than <t2>."
  (declare (type vector (unsigned-byte 2) t1)
           (type vector (unsigned-byte 2) t2)
           (type fixnum size))
  (let ((sum1 0)
        (sum2 0)
        v1 v2
        (ss (* size size)))
    (do ((i 0 (1+ i)))
        ((= i size) (> sum1 sum2))
      (declare (type fixnum i)
               (type (unsigned-byte 2) v1)
               (type (unsigned-byte 2) v2))
      (setf v1 (term-ref t1 i))
      (setf v2 (term-ref t2 i))
      (case v1
        (#.e (incf sum1 ss))
        (#.h (incf sum1 size)))
      (case v2
        (#.e (incf sum2 ss))
        (#.h (incf sum2 size))))))


(defun is-orthog? (term tl size)
  "Predicate: Returns true if <term> is \"orthogonal\" to each member of 
the \"term list\" <tl>.
Assumes <term> and each member of <tl> has size <size>."
  (declare (type fixnum size))
  (do ((cursor tl (cdr cursor)))
      ((null cursor) t)
    (let ((item (car cursor)))
      (declare (type vector (unsigned-byte 2) item))
      (when (is-zero-term (term-mult item term size))
        (return-from is-orthog? nil)))))


(defun is-non-zero-mult-even? (term tl start-count size)
  "Predicate: Returns true if the number of terms in <tl> that 
are not \"orthogonal\" to <term> plus <start-count> is even.
Assumes <term> and each member of <tl> has size <size>."
  (declare (type fixnum size)
           (type fixnum start-count)
           (type vector (unsigned-byte 2) term))
  (if (null tl)
    (= (mod start-count 2) 0)
    (do ((cursor tl (cdr cursor))
         (prod term)
         (last term)
         (count start-count))
        ((null cursor) (= (mod count 2) 0))
      (declare (type fixnum count)
               (type vector (unsigned-byte 2) prod)
               (type vector (unsigned-byte 2) last))
      (setf prod (term-mult prod (car cursor) size))
      (if  (is-zero-term prod)
        (setf prod last)
        (progn
          (incf count)
          (setf last prod))))))


(defun try-to-find-non-zero-val (tl size)
  "Tries to find a value that when evaluated with the function represented 
by the \"term list\" <tl> yields 1. If no such value could be found, 
returns nil.
Assumes that each member of <tl> has size <size>."
  (declare (type fixnum size))
  (let ((zero-list nil)
        (prod (car tl))
        (last-prod (car tl))
        (last-prod-list (list (list (cons (car tl) (car tl)))))
        (count 1))
    (declare (type fixnum count)
             (type vector (unsigned-byte 2) prod)
             (type vector (unsigned-byte 2) last-prod))
    (do ((cursor (cdr tl) (cdr cursor)))
        ((null cursor) nil)
      (setf prod (term-mult (car cursor) prod size))
      (if (is-zero-term prod)
        (progn
          (push (car cursor) zero-list)
          (setf prod last-prod))
        (progn
          (push (cons (cons prod (car cursor)) zero-list) last-prod-list)
          (incf count)
          (setf last-prod prod)
          (setf zero-list nil))))
    (if (and (= (mod count 2) 1) 
             (eval-function tl size (convert-term->val prod size)))
      (convert-term->val prod size)
      (progn
        (do ((cursor (cdr last-prod-list) (cdr cursor))
             (curr (copy-term prod size))
             (curr-val (caaar last-prod-list))
             (z-list (append (cdar last-prod-list) zero-list))
             (pair nil)
             (val nil)
             (last-val nil))
            ((null cursor) nil)
          (declare (type vector (unsigned-byte 2) curr))
          (setf pair (car cursor))
          (setf last-prod (caar pair))
          (setf last-val (cdar pair))
          (setf z-list (append (cdr pair) z-list))
          (setf val (do ((i 0 (1+ i))
                         (c-last (copy-term last-prod size))
                         (c-last2 nil))
                        ((= i size) nil)
                      (declare (type fixnum i)
                               (type vector (unsigned-byte 2) c-last)
                               (type vector (unsigned-byte 2) c-last2))
                      (when (and (= (term-ref last-prod i) e) 
                                 (/= (term-ref curr i) e))
                        (case (term-ref curr i)
                          (#.h (term-set c-last i k) 
                               (when (is-non-zero-mult-even? 
                                      c-last z-list count size) 
                                 (return c-last)))
                          (#.k (term-set c-last i h) 
                               (when (is-non-zero-mult-even? 
                                      c-last z-list count size) 
                                 (return c-last))))
                        (setf c-last2 (copy-term last-prod size))
                        (case (term-ref curr i)
                          (#.h (term-set c-last2 i k) 
                               (when (is-non-zero-mult-even? 
                                      c-last2 z-list count size) 
                                 (return c-last2)))
                          (#.k (term-set c-last2 i h) 
                               (when (is-non-zero-mult-even? 
                                      c-last2 z-list count size) 
                                 (return c-last2)))))))
          (if (and val (= (eval-function tl size 
                                         (convert-term->val val size)) 1))
              (return-from try-to-find-non-zero-val 
                (convert-term->val val size))
            (progn
              (push curr-val z-list)
              (incf count)
              (setf curr (copy-term last-prod size))
              (setf curr-val last-val))))))))


(defun convert-term->val (term size)
  "Converts all E elements of <term> to 0."
  (declare (type fixnum size))
  (let ((val (copy-term term size)))
    (declare (type vector (unsigned-byte 2) val))
    (do ((i 0 (1+ i)))
        ((= i size) val)
      (declare (type fixnum i))
      (when (eql (term-ref val i) e)
        (term-set val i 0)))))

(defun is-symmetric-function-non-zero? (tl size)
  "Predicate: Determines if the function (assumed to be symmetric) 
represented by the term list <tl> is non-zero. Returns a value that 
generates a non-zero result if true; otherwise, nil.
Assumes that each member of <tl> has size <size>."
  (declare (type fixnum size))
  (let ((val (make-array size :initial-element 0 
                         :element-type '(unsigned-byte 2))))
    (declare (type vector (unsigned-byte 2) val))
    (when (= (eval-function tl size val) 1)
      (return-from is-symmetric-function-non-zero? val))
    (do ((i 0 (1+ i)))
        ((= i size) nil)
      (declare (type fixnum i))
      (setf (aref val (- size 1 i)) 1)
      (when (= (eval-function tl size val) 1)
        (return-from is-symmetric-function-non-zero? val)))))


(defun break-off-pieces (piece term size)
  "Create a \"term list\" of \"orthogonal\" terms that consist of 
<term> minus <piece>. That is, the new list should contain all the 
one's of the term <term> but none of the one's from the term <piece>.
Example: (rsm.bool-comp::break-off-pieces #(1 2 1) #(1 2 2) 3)
          (#(1 2 0))"
  (declare (type vector (unsigned-byte 2) term)
           (type vector (unsigned-byte 2) piece)
           (type fixnum size))
  (if (is-contained? term piece size) 
    nil
    (progn
      (let (prod)
        (declare (type vector (unsigned-byte 2) prod))
        (setf prod (term-mult piece term size))
        (if (is-zero-term prod)
          (list term)
          (let ((cur term)
                break-off t1 p1
                (result nil))
            (declare (type (unsigned-byte 2) t1)
                     (type (unsigned-byte 2) p1)
                     (type vector (unsigned-byte 2) break-off)
                     (type vector (unsigned-byte 2) cur))
            (do ((i 0 (1+ i)))
                ((= i size) result)
              (declare (type fixnum i))
              (setf t1 (term-ref term i))
              (setf p1 (term-ref prod i))
              (when (/= t1 p1)
                (setf break-off (copy-term cur size))
                (term-set break-off i (if (= p1 #.k) #.h #.k))
                (when (is-not-zero-array break-off size) 
                  (push break-off result))
                (term-set cur i p1)))))))))



(defun orthog (tl size)
  "\"Orthogonalize\" an OR \"term list\" <tl> into an 
equivalent XOR \"term list\"."
  (labels ((rec (lst acc)
             (if (null lst) 
               acc
               (rec (rest lst) 
                    (nconc acc
                           (do ((acc-cursor acc (cdr acc-cursor))
                                (pile nil)
                                (break-offs (list (car lst)) pile))
                               ((null acc-cursor) pile)
                             (do ((break-cursor break-offs (cdr break-cursor))
                                  (new-breaks nil))
                                 ((null break-cursor) 
                                  (setf pile (apply #'nconc pile)))
                               (setf pile nil)
                               (setf new-breaks 
                                 (break-off-pieces (car acc-cursor) 
                                                   (car break-cursor) size))
                               (when new-breaks
                                 (push new-breaks pile)))))))))
    (cond ((atom tl) tl)
          (t
           (rec (cdr tl) (list (car tl)))))))

(defun or-tree? (tree)
  "Is this an OR branch of the tree?"
  (and (consp tree)
       (string= (car tree) "+"))) 


(defun and-tree? (tree)
  "Is this an AND branch of the tree?"
  (and (consp tree)
       (string= (car tree) "*"))) 


(defun xor-tree? (tree)
  "Is this an XOR branch of the tree?"
  (and (consp tree)
       (string= (car tree) "^"))) 


(defun not-tree? (tree)
  "Is this a NOT branch of the tree?"
  (and (consp tree)
       (string= (car tree) "~"))) 


(defun is-more-atomic? (x y)
  "Predicate: True when x is an atom."
  (declare (ignore y))
  (when (atom x) t))


(defun list-or->xor (tl1 tl2 size)
  "Convert an OR list of terms to an XOR list of terms."
(let ((result nil)
      (prod nil))
  (setf result (append tl1 tl2 result))
    (do ((c1 tl1 (cdr c1)))
        ((null c1) (or result zero-term))
      (do ((c2 tl2 (cdr c2)))
          ((null c2))
        (setf prod (term-mult (car c1) (car c2) size))
        (when (is-not-zero-term prod) (push prod result))))))


(defun excise-term (term i j size)
  "Snips out the <i>th and <j>th elements (zero based index) from term 
<term> of size <size>.
Returns a term of size <size>-2.
Example: (rsm.bool-comp::excise-term #(1 2 0 2 1) 1 3 5)
        #(1 0 1)"
  (declare (type fixnum i)
           (type fixnum j)
           (type fixnum size))
  (let ((e-term (make-identity-term (- size 2)))
        (count 0))
    (declare (type vector (unsigned-byte 2) e-term)
             (type fixnum count))
    (dotimes (idx size e-term)
      (declare (type fixnum idx))
      (unless (or (= idx i) (= idx j))
        (term-set e-term count (term-ref term idx))
        (incf count)))))


(defun get-invariant-piece (tl size i j proj)
  "Excises the <i>th and <j>th elements (zero index based) from each 
element of the term list <tl>. Places each of the excised terms 
into two lists: invariant or compliment.
If the <i>th/<j>th elements are \"orthogonal\", place into the 
compliment list; otherwise, place in the invariant list. 
Return both lists.
Example: (rsm.bool-comp::get-invariant-piece '(#(1 2 0 2 1) #(0 1 2 1 0)) 
           5 1 3 0)
   (#(0 2 0))
   (#(1 0 1))"
  (declare (type fixnum i)
           (type fixnum j)
           (type fixnum size))
  (let ((invariant nil)
        (complement nil))
    (do ((cursor tl (cdr cursor))
         (term nil)
         (e-term nil))
        ((null cursor) (values invariant complement))
      (declare (type vector (unsigned-byte 2) term)
               (type vector (unsigned-byte 2) e-term))
      (setf term (car cursor))
      (setf e-term (excise-term term i j size))
      (case proj
        (#.hh (if (and (= (term-ref term i) h) 
                       (= (term-ref term j) h))
                (push e-term invariant)
                (push e-term complement)))
        (#.hk (if (and (= (term-ref term i) h) 
                       (= (term-ref term j) k))
                (push e-term invariant)
                (push e-term complement)))
        (#.kh (if (and (= (term-ref term i) k) 
                       (= (term-ref term j) h))
                (push e-term invariant)
                (push e-term complement)))
        (#.kk (if (and (= (term-ref term i) k) 
                       (= (term-ref term j) k))
                (push e-term invariant)
                (push e-term complement)))))))


(defun augment-term (term size i j proj)
  "Project the <i>th/<j>th elements using projection <proj> (one of four).
Example: (rsm.bool-comp::augment-term #(1 2 2) 3 1 2 0)   
(here <proj>=0 = projection (1 1) )
#(1 1 1)
Example: (rsm.bool-comp::augment-term #(1 2 2) 3 1 2 1) 
(here <proj>=1 = projection (1 0).)
#(1 1 0)"
  (declare (type fixnum i)
           (type fixnum j)
           (type fixnum size))
  (let ((a-term (make-identity-term size))
        (count 0)
        ni nj)
    (declare (type fixnum count)
             (type (unsigned-byte 2) ni)
             (type (unsigned-byte 2) nj)
             (type vector (unsigned-byte 2) a-term))
    (case proj
      (#.hh (setf ni h nj h))
      (#.hk (setf ni h nj k))
      (#.kh (setf ni k nj h))
      (#.kk (setf ni k nj k)))
    (dotimes (idx size a-term)
      (declare (type fixnum idx))
      (cond ((= idx i) 
             (term-set a-term idx ni))
            ((= idx j)
             (term-set a-term idx nj))
            (t
             (term-set a-term idx (term-ref term count))
             (incf count))))))


(defun convert-term->bit-string-s (term size)
  "Convert a (short) tensor term of size <size> to a bit string of length 256.
<size> is assumed to be <= 8."
  (declare (type fixnum size))
  (let ((b-string allones))
    (do ((i 0 (1+ i)))
        ((= i size) b-string)
      (declare (type fixnum i))
      (case (term-ref term i)
        (#.h b-string (setf b-string (logand b-string (aref bstrings i))))
        (#.k b-string (setf b-string (logand b-string (aref bnstrings i))))))))


(defun convert-term->bit-string-l (term size)
  "Convert a (long) tensor term of size <size> to a bit string of length 4096.
<size> is assumed to be <= 12."
  (declare (type vector (unsigned-byte 2) term)
           (type fixnum size))
  (let ((b-string ballones))
    (do ((i 0 (1+ i)))
        ((= i size) b-string)
      (declare (type fixnum i))
      (case (term-ref term i)
        (#.h b-string (setf b-string 
                        (logand b-string (aref bbstrings i))))
        (#.k b-string (setf b-string 
                        (logand b-string (aref bbnstrings i))))))))

(defun convert-term-list-s->bit-string (tl size)
  "Convert a (short) XOR tensor term list of term size <size> to a 
bit string of length 256.
<size> is assumed to be <= 8."
  (declare (type fixnum size))
  (let ((result 0))
    (do ((cursor tl (cdr cursor)))
        ((null cursor) result)
      (setf result (logxor result 
                           (convert-term->bit-string-s 
                            (car cursor) size))))))


(defun convert-term-list-l->bit-string (tl size)
  "Convert a (long) XOR tensor term list of term size <size> to a 
bit string of length 4096.
<size> is assumed to be <= 12."
  (declare (type fixnum size))
  (let ((result 0))
    (do ((cursor tl (cdr cursor)))
        ((null cursor) result)
      (setf result (logxor result 
                           (convert-term->bit-string-l 
                            (car cursor) size))))))


(defun make-term-from-number (num size)
  "Converts a number <num> to a bit string of length <size>. 
The low bit is the first bit on the left. Assumes
<size> is <= 12."
  (declare (type fixnum num)
           (type fixnum size))
  (let ((term (make-array size :initial-element 0 
                          :element-type '(unsigned-byte 2))))
    (declare (type vector (unsigned-byte 2) term))
    (multiple-value-bind (d m) (truncate num 2)
      (if (= 1 m)
        (setf (aref term 0) 1)
        (setf (aref term 0) 0))
      (when (or (= d 0) (= size 1)) 
        (return-from make-term-from-number term))
      (multiple-value-bind (d m) (truncate d 2)
        (if (= 1 m)
          (setf (aref term 1) 1)
          (setf (aref term 1) 0))
        (when (or (= d 0) (= size 2)) 
          (return-from make-term-from-number term))
        (multiple-value-bind (d m) (truncate d 2)
          (if (= 1 m)
            (setf (aref term 2) 1)
            (setf (aref term 2) 0))
          (when (or (= d 0) (= size 3)) 
            (return-from make-term-from-number term))
          (multiple-value-bind (d m) (truncate d 2)
            (if (= 1 m)
              (setf (aref term 3) 1)
              (setf (aref term 3) 0))
            (when (or (= d 0) (= size 4)) 
              (return-from make-term-from-number term))
            (multiple-value-bind (d m) (truncate d 2)
              (if (= 1 m)
                (setf (aref term 4) 1)
                (setf (aref term 4) 0))
              (when (or (= d 0) (= size 5)) 
                (return-from make-term-from-number term))
              (multiple-value-bind (d m) (truncate d 2)
                (if (= 1 m)
                  (setf (aref term 5) 1)
                  (setf (aref term 5) 0))
                (when (or (= d 0) (= size 6)) 
                  (return-from make-term-from-number term))
                (multiple-value-bind (d m) (truncate d 2)
                  (if (= 1 m)
                    (setf (aref term 6) 1)
                    (setf (aref term 6) 0))
                  (when (or (= d 0) (= size 7)) 
                    (return-from make-term-from-number term))
                  (multiple-value-bind (d m) (truncate d 2)
                    (if (= 1 m)
                      (setf (aref term 7) 1)
                      (setf (aref term 7) 0))
                    (when (or (= d 0) (= size 8)) 
                      (return-from make-term-from-number term))
                    (multiple-value-bind (d m) (truncate d 2)
                      (if (= 1 m)
                        (setf (aref term 8) 1)
                        (setf (aref term 8) 0))
                      (when (or (= d 0) (= size 9)) 
                        (return-from make-term-from-number term))
                      (multiple-value-bind (d m) (truncate d 2)
                        (if (= 1 m)
                          (setf (aref term 9) 1)
                          (setf (aref term 9) 0))
                        (when (or (= d 0) (= size 10)) 
                          (return-from make-term-from-number term))
                        (multiple-value-bind (d m) (truncate d 2)
                          (if (= 1 m)
                            (setf (aref term 10) 1)
                            (setf (aref term 10) 0))
                          (when (or (= d 0) (= size 11)) 
                            (return-from make-term-from-number term))
                          (multiple-value-bind (d m) (truncate d 2)
                            (declare (ignore d))
                            (if (= 1 m)
                              (setf (aref term 11) 1)
                              (setf (aref term 11) 0))
                            term))))))))))))))


(defun convert-b-string-s->val (b-string size)
  "Treats <b-string> as the bit string representation of a 
Boolean function of length = <size> <= 8.
The bit string representation is just the enumeration of the 
truth table of the Boolean function.
Finds the first non-zero entry in <b-string>. This location, when 
written in binary, yields a bit vector such that when each of 
the corresponding variables are set to, the Boolean function 
evaluates to 1."
  (declare (type fixnum size))
  (labels ((rec (num count)
             (multiple-value-bind (d m) (truncate num 2)
               (if (= m 1)
                 (make-term-from-number (- maxshortval count) size)
                 (rec d (1+ count))))))
    (rec b-string 0)))


(defun convert-b-string-l->val (b-string size)
  "Treats <b-string> as the bit string representation of a 
Boolean function of length = <size> <= 12.
The bit string representation is just the enumeration of the 
truth table of the Boolean function.
Finds the first non-zero entry in <b-string>. This location, 
when written in binary, yields a bit vector such that when each 
of the corresponding variables are set to, the Boolean function 
evaluates to 1."
  (declare (type fixnum size))
  (labels ((rec (num count)
             (multiple-value-bind (d m) (truncate num 2)
               (if (= m 1)
                 (make-term-from-number (- maxlongval count) size)
                 (rec d (1+ count))))))
    (rec b-string 0)))


(defun split (str &optional (delim " ") (reverse nil))  
  "Returns a list of strings formed by splitting <str> at any character 
contained in <delim>. 
Example: (rsm.bool-comp::split \"abc def ghi\" \" \")
         (\"abc\" \"def\" \"ghi\")"
  (labels 
      ((rec (str acc)
         (let ((pos (position-if #'(lambda (ch)
                                     (position ch delim)) str)))
           (if pos
             (rec (subseq str (1+ pos)) (if (> pos 0)
                                          (cons (subseq str 0 pos) acc)
                                          acc))
             (if reverse
               (cons str acc)
               (nreverse (cons str acc)))))))
    (delete "" (rec str nil) :test #'string-equal)))

(defun is-negated? (var)
  "Predicate determining if a string represents a negated variable.
Example: (rsm.bool-comp::is-negated? X34')
  T"
  (let ((len (length var)))
    (declare (type fixnum len))
    (char= #\' (char var (1- len)))))


(defun get-number-from-var (var)
  "Finds the index of a variable (or its negation).
Example: (rsm.bool-comp::get-number-from-var \"X4'\")
   \"4\""
  (let ((pos (position-if #'digit-char-p var)))
    (declare (type fixnum pos))
    (subseq var pos (position-if #'(lambda (char) 
                                     (not (digit-char-p char)))
                                 var :start pos))))


(defun convert-var->term (var size)
  "Converts a string representing a variable to a \"term\".
Example: (rsm.bool-comp::convert-var->term \"X2\" 3)
     #(2 1 2)"
  (declare (type fixnum size))
  (let* ((is-negated (is-negated? var))
         (num (get-number-from-var var))
         (term (make-identity-term size)))
    (declare (type vector (unsigned-byte 2) term)
             (type fixnum num))
    (if (string= var "1")
      term
      (progn
        (term-set term (1- (parse-integer num)) (if is-negated k h))
        term))))

(defun convert-varterm->term (varterm size)
  (declare (type fixnum size))
  (let ((vars (split varterm "*"))
        (term (make-identity-term size)))
    (declare (type vector (unsigned-byte 2) term))
    (dolist (var vars term)
      (let* ((is-negated (is-negated? var))
             (num (get-number-from-var var)))
        (term-set term (1- (parse-integer num)) (if is-negated k h))))))


(defun parse-bool (ls)
  "A recursive decent parser that takes a Boolean expression that 
has been tokenized into a list of strings. 
The result is an rpn Boolean tree. The parser recognizes 
the usual precedence: ~, * ^, +.
BNF:
    expr              = or-term  {\"+\" or-term}*
    or-term           = xor-term {\"^\" xor-term}*
    xor-term          = not-term {\"*\" not-term}*
    not-term          = [~]* atom-or-expr-term
    atom-or-expr-term = VAR | \"(\" expr \")\"   

Example: (rsm.bool-comp::parse-bool 
           '(\"X1\" \"+\" \"(\" \"X2\" \"^\" \"X4\" \")\" \"*\" \"X3'\"))
          (\"+\" \"X1\" (\"*\" (\"^\" \"X2\" \"X4\") \"X3'\"))
Assumes that a single variable name is used - in the above case, x."
  (let ((lst ls)
        (orig-length (length ls)))
    (labels ((expr ()
               (let ((or-list nil))
                 (push (or-term) or-list)
                 (do ()
                     ((null lst))
                   (let ((or-op? (pop lst)))
                     (if (string= or-op? "+")
                       (push (or-term) or-list)
                       (progn 
                         (setf lst (cons or-op? lst))
                         (return)))))
                 (cond ((null or-list) nil)
                       ((= (length or-list) 1) (car or-list))
                       (t
                        (cons "+" (reverse or-list))))))
             (or-term ()
               (let ((xor-list nil))
                 (push (xor-term) xor-list)
                 (do ()
                     ((null lst))
                   (let ((xor-op? (pop lst)))
                     (if (string= xor-op? "^")
                       (push (xor-term) xor-list)
                       (progn
                         (setf lst (cons xor-op? lst))
                         (return)))))
                 (cond ((null xor-list) nil)
                       ((= (length xor-list) 1) (car xor-list))
                       (t
                        (cons "^" (reverse xor-list))))))
             (xor-term ()
               (let ((not-list nil))
                 (push (not-term) not-list)
                 (do ()
                     ((null lst))
                   (let ((and-op? (pop lst)))
                     (if (equal and-op? "*")
                       (push (not-term) not-list)
                       (progn
                         (setf lst (cons and-op? lst))
                         (return)))))
                 (cond ((null not-list) nil)
                       ((= (length not-list) 1) (car not-list))
                       (t
                        (cons "*" (reverse not-list))))))
             (not-term ()
               (let ((atom-list nil) (count 0))
                 (do ()
                     ((null lst))
                   (let ((not-op? (pop lst)))
                     (if (equal not-op? "~")
                       (incf count)
                       (progn
                         (push not-op? lst)
                         (push (atom-or-expr-term) atom-list)
                         (return)))))
                 (cond ((null atom-list) nil)
                       ((if (= (length atom-list) 1)
                          (if (= (mod count 2) 1) 
                            (cons "~" atom-list)
                            (car atom-list))))
                       (t
                        (if (= (mod count 2) 0)
                          (reverse atom-list)
                          (cons "~" (reverse atom-list)))))))
             (atom-or-expr-term ()
               (let (first expr rparen pos)
                 (setf first (pop lst))
                 (if (string= first "(")
                   (progn
                     (setf pos (length lst))
                     (setf expr (expr))
                     (setf rparen (pop lst))
                     (if (not (equal rparen ")"))
                       (progn
                         (setf pos (- orig-length pos))
                         (error "Missing matching right paren for 
left paren at position ~s." pos))
                       expr))
                   first))))
      (expr))))


(defun convert-rpn->xor-terms (tree size)
  "Converts an rpn variable tree into an rpn term tree.
Example: (rsm.bool-comp::convert-rpn->xor-terms 
           '(\"+\" \"X1\" (\"*\" (\"^\" \"X2\" \"X4\") 
              \"X3'\")) 4)
(+ #(1 2 2 2) (* (^ #(2 1 2 2) #(2 2 2 1)) #(2 2 0 2)))
Assumes that a single variable name is used - in the above case, x."
  (cond ((null tree) nil)
        ((atom tree)
         (cond ((string= tree "+") or)
               ((string= tree "^") xor)
               ((string= tree "*") and)
               ((string= tree "~") not)
               (t
                (convert-var->term tree size))))
        (t
         (mapcar #'(lambda (item) (convert-rpn->xor-terms item size)) tree))))



(defun find-max-var-size (str-list)
  "Finds the largest variable index used.
Example: (rsm.bool-comp::find-max-var-size '(\"X1\" \"+\" \"X3\"))
     3
Assumes that a single variable name is used - in the above case, x."
  (do ((size 0)
       (list (cdr str-list) (cdr list))
       (item nil))
      ((null list) size)
    (declare (type fixnum size))
    (setf item (car list))
    (let ((pos (position-if #'digit-char-p item)))
      (when pos
        (let ((pos2 (position #\' item :start pos)))
          (let ((n (parse-integer (subseq item pos pos2))))
            (when (> n size)
              (setf size n))))))))

(defun fluff-string (formula)
  "Returns a new string with spaces added around the characters: 
\"+\", \"*\", \"^\", \"~\", \"(\", \")\"."
  (let ((string (make-string (* 3 (length formula)) 
                             :initial-element #\ )) 
        (pos 0))
    (with-input-from-string (str formula)
      (do ((char (read-char str nil :eof)
                 (read-char str nil :eof)))
          ((eql char :eof))
        (if (or (eql char #\+) (eql char #\*) 
                (eql char #\^) (eql char #\() 
                (eql char #\)) (eql char #\~))
          (progn
            (setf (char string pos) #\ )
            (incf pos)
            (setf (char string pos) char)
            (incf pos)
            (setf (char string pos) #\ )
            (incf pos))
          (progn
            (setf (char string pos) char)
            (incf pos)))))
    string))

(defun formula->rep-helper (str size)
  "Example: (rsm.bool-comp::formula->rep-helper \"X1 + X3\" 3)
    (#(1 2 2) #(0 2 1))
 Assumes that a single variable name is used - in the above case, x."
  (simplify (convert-tree->xor 
             (convert-rpn->xor-terms 
              (parse-bool (split str)) size) size) 0 size))



;;;; ==========================================================================
;;;; ================>  EXTERNAL INTERFACE  <==================================
;;;; ==========================================================================



(defun is-function-non-zero? (tl size)
  "Predicate: Determines if the function represented by the \"term list\", <tl>,
is non-zero. Returns a value that generates a non-zero result if true; 
otherwise, this function is the zero function and nil is returned.
Assumes that each member of <tl> has size <size>.
STRATEGY: - Try to find a non-zero value as though the function, 
             <tl>, is a symmetric function.
             This involves n+1 (n - the number of variables) evaluations.
           - If this fails then if there are 12 or fewer variables use 
             the bit-string representation to find the answer.
          OTHERWISE: 
           - simplify <tl>.
           - Try to show symmetry. If <tl> is completely symmetric, 
             then since we've already tested for a non-zero value for 
             symmetric functions, <tl> must be the zero function. Otherwise,
             find the first index i (and projection) where swapping the 
             variable x1 with xi changes the function, <tl>; that is, we at 
             least can't easily show that it leaves <tl> the same.
           - In the last step we get back (in the case of non-symmetry) 
             a sym-diff flipped version of <tl>.
             Two cases: the flipped version is the same as <tl>, or it 
             is different. In either case, we recurse and move to the next 
             index which is not symmetric - or finish."
  (let ((val (is-function-non-zero-h? tl size 1 0 t)))
    val))


(defun is-function-non-zero-h? (tl size start-j proj cal)
  (let ((result cal)
        next-j next-proj val-f val b-string)
    (when result
      (setf result (is-symmetric-function-non-zero? tl size)))
    (if (or result (null cal))
        (values result result start-j proj)
      (cond ((<= size 8)
             (setf b-string (convert-term-list-s->bit-string tl size))
             (if (= b-string 0)
                 nil
               (progn
                 (setf b-string (convert-b-string-s->val b-string size))
                 (values b-string b-string start-j proj))))
            ((<= size 12)
             (setf b-string (convert-term-list-l->bit-string tl size))
             (if (= b-string 0)
                 nil
               (progn 
                 (setf b-string (convert-b-string-l->val b-string size))
                 (values b-string b-string start-j proj))))
            (t
             ;; Use symmetry
             (setf result (simplify tl (1- start-j) size)) 
             (when result
               (multiple-value-bind (nresult j proj) 
                   (is-function-clearly-non-symmetric? result size start-j proj)
                 (when nresult
                   (setf nresult (sort nresult #'(lambda (t1 t2) 
                                                   (compare-by-weight 
                                                    t1 t2 size))))
                   (setf val (try-to-find-non-zero-val nresult size))
                   (if val
                       (progn  
                         (setf val-f (term-flip val 0 j size))
                         (if (= (eval-function result size val) 1)
                             (values val val j proj)
                           (values val-f val-f j proj)))
                     ;; Use symmetry (If nil means that 
                     ;; nresult = resultl ; that is, nresult is 
                     ;; the same term list.)
                     (if (simplify (append result nresult) (1- j) size)
                         (multiple-value-bind (val d-val jj nproj)
                             (is-function-non-zero-h? nresult size j proj t)
                           (if val
                               (if (= (eval-function result size val) 1)
                                   (values val d-val jj nproj)
                                 (values (term-flip val 0 j size) 
                                         (term-flip d-val 0 j size)
                                         jj nproj))
                             (progn
                               (setf next-j j)
                               (setf next-proj (1+ proj))
                               (when (= (truncate next-proj 4) 0)
                                 (setf next-proj 0)
                                 (incf next-j))
                               (unless (= next-j size)
                                 (is-function-non-zero-h? result size next-j 
                                                          next-proj nil)))))
                       ;; What to do when you have a fixed point 
                       ;; with the above operations 
                       ;; (result = nresult).
                       (multiple-value-bind (invariant complement) 
                           (get-invariant-piece nresult size 0 j proj)
                         (multiple-value-bind (val d-val jj nproj) 
                             (is-function-non-zero-h? invariant 
                                                      (- size 2) 
                                                      1 0 t)
                           (cond (val
                                  (setf val (augment-term val size 0 j proj))
                                  (setf d-val 
                                    (augment-term d-val size 0 j proj))
                                  (if (= (eval-function result size val) 1)
                                      (values val d-val jj nproj)
                                    (values (term-flip val 0 j size)
                                            (term-flip d-val 0 j size)
                                            jj nproj)))
                                 (t
                                  (if complement
                                      (multiple-value-bind
                                          (val d-val jj nproj)
                                          (is-function-non-zero-h? complement 
                                                                   (- size 2) 
                                                                   1 0 t)
                                        (when val
                                          (if (= (eval-function result size val)
                                                 1)
                                              (return-from 
                                                  is-function-non-zero-h?
                                                (values val 
                                                        d-val 
                                                        jj nproj))
                                            (return-from 
                                                is-function-non-zero-h?
                                              (values (term-flip val 
                                                                 0 
                                                                 j 
                                                                 size)
                                                      (term-flip d-val 
                                                                 0 
                                                                 j 
                                                                 size)
                                                      jj nproj))))))
                                  (progn
                                    (setf next-j j)
                                    (setf next-proj (1+ proj))
                                    (when (= (truncate next-proj 4) 0)
                                      (setf next-proj 0)
                                      (incf next-j))
                                    (unless (= next-j size)
                                      (is-function-non-zero-h? 
                                       result 
                                       size 
                                       next-j 
                                       next-proj 
                                       nil)))))))))))))))))


(defun list->eqn (tl)
"Prints a familiar algebraic representation of an XOR term list.
Example: (rsm.bool-comp:list->eqn '(#(1 2 0) #(1 1 1)))
          X1*X3' ^ X1*X2*X3"
  (when (consp tl)
    (let ((len (term-length (car tl))))
      (do ((cursor tl (cdr cursor))
           (result nil))
          ((null cursor) (and result (string-trim " "
                                                  (format nil "~{~A ~^^ ~}" 
                                                          (nreverse result)))))
        (do ((i 0 (1+ i))
             (term nil))
            ((= i len) (if term 
                           (push (format nil "~{~A~^*~}" 
                                         (nreverse term)) result)
                         (push (format nil "1") result)))
          (case (term-ref (car cursor) i)
            (#.k (push (format nil "X~A'" (1+ i)) term))
            (#.h (push (format nil "X~A" (1+ i)) term))))))))


(defun convert-tree->xor (tree size)
  "Convert an RPN tree of logical terms (representing a function) to 
an XOR term list.
Example: (rsm.bool-comp:convert-tree->xor '(#.XOR (#.XOR #(2 1) #(1 2)) 
                                                  (#.XOR #(1 2) #(1 1))) 2)
         (#(2 1) #(1 2) #(1 2) #(1 1))
Example: (rsm.bool-comp:convert-tree->xor '(#.OR (#.XOR #(2 1) #(1 2)) 
                                                 (#.XOR #(1 2) #(1 1))) 2)
         (#(2 1) #(1 2))"
  (labels ((rec (tree)
             (cond ((null tree) nil)
                   ((atom tree) tree)
                   ((or-tree? tree) 
                    (let ((copy (copy-list (cdr tree)))
                          (atoms nil))
                      (setf copy (sort copy #'is-more-atomic?))
                      (do ()
                          ()
                        (if (and (not (null (car copy))) (atom (car copy)))
                          (push (pop copy) atoms)
                          (return)))
                      (and atoms (setf atoms 
                                   (orthog (sort atoms 
                                                 #'(lambda (t1 t2)
                                                     (compare-by-weight 
                                                      t1 t2 size)))
                                           size)))
                      (setf copy (mapcar #'(lambda (branch)
                                             (rec branch)) copy))
                      (cons xor 
                            (if atoms
                              (simplify (reduce #'(lambda (t1 t2)
                                                    (list-or->xor t1 t2 size))
                                                copy :initial-value atoms 
                                                :key #'cdr) 0 size)
                              (simplify (reduce #'(lambda (t1 t2)
                                                    (list-or->xor t1 t2 size))
                                                copy :key #'cdr) 0 size)))))
                   ((and-tree? tree)
                    (let ((copy (copy-list (cdr tree)))
                          (atoms nil))
                      (setf copy (sort copy #'is-more-atomic?))
                      (do ()
                          ()
                        (if (and (not (null (car copy))) (atom (car copy)))
                          (push (pop copy) atoms)
                          (return)))
                      (and atoms 
                           (setf atoms 
                                 (do ((cursor (cdr atoms) (cdr cursor))
                                      (prod (car atoms)))
                                     ((null cursor) prod)
                                   (setf prod (term-mult prod 
                                                         (car cursor) size))
                                   (when (is-zero-term prod) 
                                     (return zero-term)))))
                      (setf copy (mapcar #'(lambda (branch)
                                             (rec branch)) copy))
                      (if atoms
                        (setf atoms (reduce #'(lambda (t1 t2)
                                                (list-mult t1 t2 size))
                                            copy :initial-value (list atoms)
                                            :key #'cdr))
                        (setf atoms (reduce #'(lambda (t1 t2)
                                                (list-mult t1 t2 size))
                                            copy :key #'cdr)))
                      (cons xor atoms)))
                   ((xor-tree? tree)
                    (cons xor (flatten (mapcar #'(lambda (branch)
                                                   (let ((result nil))
                                                     (setf result (rec branch))
                                                     (if (consp result)
                                                       (cdr result)
                                                       result))) 
                                               (cdr tree)))))
                   ((not-tree? tree) 
                    (let ((result nil))
                      (setf result (rec (cadr tree)))
                      (if (consp result)
                        (setf result (cdr result))
                        (setf result (list result)))
                      (cons xor (cons (make-identity-term size) result))))
                   (t
                    (error "Bad tree: ~a~%" tree)))))
    (if (atom tree) 
      tree
      (simplify (cdr (rec tree)) 0 size))))


(defun formula->rep (formula)
  "Convert an algebraic expression, <formula>, into the boolean tensor 
representation.
Example: (rsm.bool-comp:formula->rep \"X1 + X3\")
         (#(1 2 2) #(0 2 1))
Assumes that a single variable name is used - in the above case, x."
  (let* ((fstr (fluff-string formula))
         (size (find-max-var-size (split fstr))))
    (formula->rep-helper fstr size)))


(defun rep-from-file (file-name)
  "Convert an algebraic expression contained in file, <file-name>, \w\
into the boolean tensor representation.
Example: (rsm.bool-comp:rep-from-file \"formula.lsp\")
         (#(1 2 2) #(0 2 1))
The example assumes that the file formula.lsp contains the text: X1 + X3.
This function assumes that a single variable name is used - in the 
above case, x."
  (let ((formula nil)
        size)
    (with-open-file (str file-name :direction :input)
      (do ((line (read-line str nil :eof)
                 (read-line str nil :eof)))
          ((eql line :eof))
        (setf formula (concatenate 'string formula (format nil "~a" line)))))
    (let ((string (fluff-string formula)))
      (setf size (find-max-var-size (split (format nil "~a" string))))
      (values (formula->rep-helper (format nil "~a" string) size) size))))

(defun bool-solve-from-file (file-name)
  "Find an input value such that the equation found in the file, <file-name>,
yeilds the value 1.
Example: (rsm.bool-comp:bool-solve-from-file \"formula.lsp\")
  #(0 0 0 0 0)
In this case, if formula.lsp contains the text
  (x1 + x2 + x3 + x4 + x5) ^ 
   (x1' + x2' + x3' + x4')
then letting x1=x2=x3=x4=0 in the formula represented by the text yields 1. 
Assumes that a single variable name is used - in the above case, x."
  (multiple-value-bind (term-list size) (rep-from-file file-name)
    (is-function-non-zero? term-list size)))

(defun bool-solve (eqn)
  "Find an input value such that equation, <eqn>, yields the value 1.
Example: (rsm.bool-comp:bool-solve \"X1 + X2*X3\")
         #(0 1 1)
That is, if X1=0, X2=1, and X3=1, then \"X1 + X2*X3\" is 1.
Assumes that a single variable name is used - in the above case, x."
  (let* ((eqn-string (fluff-string eqn))
         (size (find-max-var-size (split (format nil "~a" eqn-string)))))
    (is-function-non-zero? (formula->rep-helper eqn-string size) size)))
    

;;;;;;;;;;;;;;;;; Experimental Utilities ;;;;;;;;;;;;;;;;;;;;


(defun make-random-term (size)
  "Makes a random term of size <size>."
  (declare (type fixnum size))
  (let ((term (make-identity-term size)))
    (declare (type vector (unsigned-byte 2) term))
    (dotimes (i size term)
      (declare (type fixnum i))
      (term-set term i (random 3)))))

(defun make-random-function (max-number-of-terms size &optional (max nil))
  "Makes a random function consisting of an XOR term list of size <size>.
The number of terms is a random number uniformly distributed on 
[0,<max-number-of-terms>)
if <max> is nil; otherwise, it is <max-number-of-terms>.
Returns nil if <max-number-of-terms> is zero OR <max> is nil and 
the random number generated is zero."
  (let ((number-of-terms (random max-number-of-terms))
        (result nil))
    (dotimes (i (if max max-number-of-terms number-of-terms) 
               (values result (if max max-number-of-terms number-of-terms)))
      (push (make-random-term-no-repeats size) result))))



(defun perturb-term (term size)
  "(Possibly) perturb a term at one location."
  (let ((j (random size))
        (nterm (copy-term term size)))
    (term-set nterm j (random 3))
    nterm))


(defun make-perturb-of-zero (max-number-of-terms size)
  "Generate a function that is zero or close to zero."
  (let ((term nil)
        (nfunc nil))
    (multiple-value-bind (func num) 
        (make-random-function max-number-of-terms size t)
      (when func
        (setf term (nth (truncate num 2) func))
        (setf nfunc (copy-list func))
        (setf (nth (truncate num 2) nfunc) (perturb-term term size))
        (nconc nfunc func)))))


(defun random-item-except (last-item)
  (if (not last-item)
    (random 3)
    (do ((next-item nil)
         (done nil))
        (done next-item)
      (setf next-item (random 3))
      (unless (eql next-item last-item)
        (setf done 1)))))

(defun make-random-term-no-repeats (size)
  (let ((term (make-identity-term size))
        (last-rand nil)
        (next-rand nil))
    (dotimes (i size term)
      (case (random-item-except last-rand)
        (0 (setf next-rand k))
        (1 (setf next-rand h))
        (2 (setf next-rand e)))
      (term-set term i next-rand)
      (setf last-rand next-rand))))

