;; -*- Mode: Lisp -*-
;; $Id: tree.lisp,v 1.12 2002/03/29 17:13:18 jesse Exp $
;;
;; Rudiments of a threaded AVL tree
;;
;; Knuth ACP 6.2.3, Algorithm A

(in-package :odcl)

(defstruct (avl-node (:conc-name avl/)
                              (:constructor avl/create))
  left                                  ; subtree L
  right                                 ; subtree R
  ;; left-tag                           ; tag L - thread me
  ;; right-tag                          ; tag R - thread me
  (balance 0 :type fixnum)              ; balance factor
  data)                                 ; data

(defclass avl-tree ()
  ((head       :initarg :head
               :accessor tree-head
               :initform (avl/create))
   (test       :initarg :test
               :reader tree-test
               :initform (error "avl-tree constructor must be supplied a test"))
   (size       :initarg :size
               :initform 0)             ; node count
   (height     :initarg :height
               :initform 0)             ; node count
   (generation :initarg :generation
               :initform 0)))           ; generation count

(defmethod tree-root ((self avl-tree))
  (avl/right (tree-head self)))

(defun avl/link (dir node)
  (ecase dir
    (-1 (avl/left node))
    (1  (avl/right node))))

(defsetf avl/link (dir node) (snode)
  `(ecase ,dir
    (-1 (setf (avl/left ,node) ,snode))
    (1  (setf (avl/right ,node) ,snode))))

(defun avl/mapc (map-function node &aux stack (go-left t))
  "stack based iteration until threads are complete"
  (while node
    (if (and go-left (avl/left node))
        (progn
          (push node stack)
          (setq node (avl/left node)))
        (progn
          (funcall map-function node)
          (if (avl/right node)
              (setq node (avl/right node)
                    go-left t)
              (setq node (pop stack)
                    go-left nil))))))

(defun c-iterator (avl-tree)
  (let ((node (tree-root avl-tree))
        (go-left t)
        (stack nil))
    (flet ((get-next ()
             (loop
              (if node
                  (if (and go-left (avl/left node))
                      (progn
                        (push node stack)
                        (setq node (avl/left node)))
                      (let ((data (avl/data node)))
                        (if (avl/right node)
                            (setq node (avl/right node)
                                  go-left t)
                            (setq node (pop stack)
                                  go-left nil))
                        (return-from get-next data)))
                  (return-from get-next nil)))))
      (lambda ()
        (get-next)))))

(defun avl/mapc (map-function node &aux stack (go-left t))
  "stack based iteration until threads are complete"
  (while node
    (if (and go-left (avl/left node))
        (progn
          (push node stack)
          (setq node (avl/left node)))
        (progn
          (funcall map-function node)
          (if (avl/right node)
              (setq node (avl/right node)
                    go-left t)
              (setq node (pop stack)
                    go-left nil))))))

(defun avl/node-copy (root)
  (when root
    (avl/create :left (avl/node-copy (avl/left root))
                :right (avl/node-copy (avl/right root))
                :data (avl/data root)
                :balance (avl/balance root))))

;; public

(defmethod is-empty? ((self avl-tree))
  (null (tree-root self)))

(defmethod tree-member ((self avl-tree) data &aux found)
  (let ((node (tree-root self))
        (test (tree-test self)))
    (while (and node (not found))
      (cond ((funcall test data (avl/data node))
             (setq node (avl/left node)))
            ((funcall test (avl/data node) data)
             (setq node (avl/right node)))
            (t 
             (setq found t))))
    (when node
      (avl/data node))))

(defmethod tree-map ((fn function) (self avl-tree))
  (avl/mapc (lambda (node)
              (funcall fn (avl/data node)))
            (tree-root self)))

(defmethod tree-first ((self avl-tree))
  (when-bind (node (tree-root self))
    (while (avl/left node)
      (setq node (avl/left node)))
    (avl/data node)))

(defmethod tree-last ((self avl-tree))
  (when-bind (node (tree-root self))
    (while (avl/right node)
      (setq node (avl/right node)))
    (avl/data node)))

(defmethod tree->list ((self avl-tree) &aux list)
  (avl/mapc (lambda (node)
               (push (avl/data node) list))
             (tree-root self))
  (nreverse list))

(defmethod get-size ((self avl-tree))
  (slot-value self 'size))

(defmethod clear ((self avl-tree))
  (setf (avl/right (tree-root self)) nil))

;; interface

(defmethod c-probe ((tree avl-tree) item)
  (assert (not (null item)))
  (let ((test (tree-test tree))
        (p (avl/right (tree-head tree)))
        (q nil))
    (unless p
      (return-from c-probe t))
    (loop
     (cond ((funcall test item (avl/data p))
            (setq q (avl/left p))
            (when (null q)
              (return-from c-probe nil))
            (setf p q))
           ((funcall test (avl/data p) item)
            (setq q (avl/right p))
            (when (null q)
              (return-from c-probe nil))
            (setf p q))
           (t
            (return-from c-probe (avl/data p)))))))

(defmethod c-add ((tree avl-tree) item &aux result-data)
  (assert (not (null item)))
  (flet ((merge (data &optional existing-node)
           (unless existing-node
             (setf existing-node (avl/create)))
           (setf result-data (avl/data existing-node)
                 (avl/data existing-node) data)
           existing-node))
    ;; A1 [Initialize.]
    (let ((test (tree-test tree))
          (_t (tree-head tree))
          (s (avl/right (tree-head tree)))
          (p (avl/right (tree-head tree)))
          (q nil)
          (r nil))
      (unless p
        (setf (avl/right (tree-head tree)) (merge item))
        (return-from c-add result-data))
      (loop
       ;; A2 [Compare.]
       (cond ((funcall test item (avl/data p))
              ;; A3 [Move left.]
              (setq q (avl/left p))
              (when (null q)
                (setf q (merge item)
                      (avl/left p) q)
                (return))
              (unless (= 0 (avl/balance q))
                (setf _t p
                      s q))
              (setf p q))
             ((funcall test (avl/data p) item)
              ;; A4 [Move right.]
              (setq q (avl/right p))
              (when (null q)
                (setf q (merge item)
                      (avl/right p) q)
                (return))
              (unless (= 0 (avl/balance q))
                (setf _t p
                      s q))
              (setf p q))
             (t
              (merge item p)
              (return-from c-add result-data))))
      ;; A5 [Insert.] (initialization done above)
      ;; A6 [Adjust balance factors.]
      (let ((a (if (funcall test item (avl/data s)) -1 1)))
        (setf p (avl/link a s)
              r p)
        (while (not (eql p q))
          (cond ((funcall test item (avl/data p))
                 (setf (avl/balance p) -1
                       p (avl/left p)))
                ((funcall test (avl/data p) item)
                 (setf (avl/balance p) 1
                       p (avl/right p)))
                (t
                 (error "logic error 1"))))
        ;; A7 [Balancing act.]
        (cond ((= (avl/balance s) 0)
               ;; i)
               (setf (avl/balance s) a)
               (incf (slot-value tree 'height))
               (return-from c-add result-data))
              ((= (avl/balance s) (- a))
               ;; ii)
               (setf (avl/balance s) 0)
               (return-from c-add result-data))
              ((= (avl/balance s) a)
               (cond ((= (avl/balance r) a)
                      ;; A8 [Single rotation.]
                      (setf p r
                            (avl/link a s) (avl/link (- a) r)
                            (avl/link (- a) r) s
                            (avl/balance s) 0
                            (avl/balance r) 0))
                     ((= (avl/balance r) (- a))
                      ;; A9 [Double rotation.]
                      (setf p (avl/link (- a) r)
                            (avl/link (- a) r) (avl/link a p)
                            (avl/link a p) r
                            (avl/link a s) (avl/link (- a) p)
                            (avl/link (- a) p) s)
                      (cond ((= a (avl/balance p))
                             (setf (avl/balance s) (- a))
                             (setf (avl/balance r) 0))
                            ((= 0 (avl/balance p))
                             (setf (avl/balance s) 0)
                             (setf (avl/balance r) 0))
                            ((= (- a) (avl/balance p))
                             (setf (avl/balance s) 0)
                             (setf (avl/balance r) a))))
                     ;; (t ;; r = 0, a = 1 ?
                     ;;  (error "logic error: a/b r = ~s a = ~s" (avl/balance r) a))
                     )
               (setf (avl/balance p) 0)
               (if (eql s (avl/right _t))
                   (setf (avl/right _t) p)
                   (setf (avl/left _t) p)))
              (t
               (error "logic error 3")))))))

(defun avl/del-l (node branch)
  (let ((br (avl/link branch node)))
    (ecase (avl/balance br)
      (1  (decf (avl/balance br)) t)
      (0  (decf (avl/balance br)) nil)
      (-1 (let ((br-l (avl/left br)))
            (ecase (avl/balance br-l)
              (-1 (setf (avl/left br) (avl/right br-l)
                        (avl/right br-l) br
                        (avl/balance br) 0
                        (avl/balance br-l) 0
                        (avl/link branch node) br-l)
                  t)
              (0  (setf (avl/left br) (avl/right br-l)
                        (avl/right br-l) br
                        (avl/balance br) -1
                        (avl/balance br-l) 1
                        (avl/link branch node) br-l)
                  nil)
              (1  (let ((br-l-r (avl/right br-l)))
                    (setf (avl/right br-l) (avl/left br-l-r)
                          (avl/left br-l-r) br-l
                          (avl/left br) (avl/right br-l-r)
                          (avl/right br-l-r) br)
                    (case (avl/balance br-l-r)
                      (-1 (setf (avl/balance br) 1
                                (avl/balance br-l) 0))
                      (0  (setf (avl/balance br) 0
                                (avl/balance br-l) 0))
                      (1  (setf (avl/balance br) 0
                                (avl/balance br-l) -1)))
                    (setf (avl/link branch node) br-l-r
                          (avl/balance br-l-r) 0)
                    t))))))))

(defun avl/del-r (node branch)
  (let ((br (avl/link branch node)))
    (ecase (avl/balance br)
      (-1 (incf (avl/balance br)) t)
      (0  (incf (avl/balance br)) nil)
      (1  (let ((br-r (avl/right br)))
            (ecase (avl/balance br-r)
              (1  (setf (avl/right br) (avl/left br-r)
                        (avl/left br-r) br
                        (avl/balance br) 0
                        (avl/balance br-r) 0
                        (avl/link branch node) br-r)
                  t)
              (0  (setf (avl/right br) (avl/left br-r)
                        (avl/left br-r) br
                        (avl/balance br) 1
                        (avl/balance br-r) -1
                        (avl/link branch node) br-r)
                  nil)
              (-1 (let ((br-r-l (avl/left br-r)))
                    (setf (avl/left br-r) (avl/right br-r-l)
                          (avl/right br-r-l) br-r
                          (avl/right br) (avl/left br-r-l)
                          (avl/left br-r-l) br)
                    (case (avl/balance br-r-l)
                      (-1 (setf (avl/balance br) 0
                                (avl/balance br-r) 1))
                      (0  (setf (avl/balance br) 0
                                (avl/balance br-r) 0))
                      (1  (setf (avl/balance br) -1
                                (avl/balance br-r) 0)))
                    (setf (avl/link branch node) br-r-l
                          (avl/balance br-r-l) 0)
                    t))))))))

(defmethod c-remove ((self avl-tree) item)
  (with-slots (test)
    self
    (labels ((%%tree-delete (node branch q)
               (let ((br (avl/link branch node)))
                 (if (avl/right br)
                     (when (%%tree-delete br 1 q)
                       (avl/del-l node branch))
                     (progn
                       (setf (avl/data q) (avl/data br)
                             (avl/link branch node) (avl/left br))
                       t))))
             (%tree-delete (node branch)
               (if-bind (br (avl/link branch node))
                   (cond ((funcall test item (avl/data br))
                          (when (%tree-delete br -1)
                            (avl/del-r node branch)))
                         ((funcall test (avl/data br) item)
                          (when (%tree-delete br 1)
                            (avl/del-l node branch)))
                         (t
                          (cond ((null (avl/right br))
                                 (setf (avl/link branch node) (avl/left br))
                                 (avl/data br))
                                ((null (avl/left br))
                                 (setf (avl/link branch node) (avl/right br))
                                 (avl/data br))
                                (t
                                 (when (%%tree-delete br -1 br)
                                   (avl/del-r node branch))
                                 (avl/data br)))))
                   nil)))
      ;; delete to the right
      (%tree-delete (slot-value self 'head) 1))))

(defmethod c-copy ((self avl-tree))
  (with-slots (head test)
    self
    (make-instance 'avl-tree
                   :test test
                   :head (avl/node-copy head))))

; (defun test1 (x &aux foo)
;   (dotimes (x x)
;     (push (random-string :length 3) foo))
;   foo)

; (defvar *troot* nil)

; (defun test2 (x)
;   (setq *troot* (tinkle x)))

; (defun test3 ()
;   (let ((tree (make-instance 'avl-tree :test #'string<)))
;     (do ((x *troot* (cddr x)))
;         ((null x))
;       (c-add tree (car x))
;       (c-remove tree (cadr x)))))

; (defun test4 ()
;   (let ((tree (make-instance 'avl-tree :test #'string<)))
;     (do ((x *troot* (cddr x)))
;         ((null x))
;       (c-add tree (car x))
;       (c-add tree (cadr x)))
;     (do ((x *troot* (cddr x)))
;         ((null x))
;       (c-remove tree (car x))
;       (c-remove tree (cadr x)))))
    
