;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: wcof -*-
;;; $Id: property-sheets.lisp,v 1.8 2002/02/01 17:56:48 jesse Exp $
;;;
;;; Copyright (c) 2000, 2001 onShore Development, Inc.

(in-package :odcl)

;; ------------------------------------------------------------
;; Property Sheet Specifications

(defvar *property-sheet-specifications* (make-hash-table :test #'equal))

(defun %psheet-spec-hash (class spec-name)
  (list class spec-name))

;; psheet-specification are lists destructured as follows: (name &key
;; (required t) (editable t) caption display-type) NAME is the
;; property name.  If REQUIRED is true, than the property a required
;; to have a value.  If EDITABLE is true, it can be edited.  CAPTION
;; and DISPLAY-TYPE default to the values returned from GET-PROPERTY
;; but can be overriden.

;; (def-psheet-spec (resource :cataloging-editing-set)
;;     ((:isbn)
;;      (:record-created :editable nil)
;;      (:director :required nil)
;;      (:rating :required nil :display-type 'media-rating)))

(defmacro def-psheet-spec ((propertied-class spec-name) &body spec)
  "Define a set of psheet property specs on PROPERTIED-CLASS under SPEC-NAME."
  `(setf (gethash (%psheet-spec-hash ,propertied-class ,spec-name)
          *property-sheet-specifications*)
    ,spec))

(defmethod object-property-sheet-spec (self spec-name)
  "Return the psheet spec for an instance with the given name."
  (gethash (%psheet-spec-hash (type-of self)
                              spec-name)
           *property-sheet-specifications*))

(defun property-sheet-null-value (self type)
  "Returns value to represent property values when they are nil in a psheet."
  (declare (ignore self))
  (case type
    (string nil)
    (t nil)))

(defun %convert-old-skool-pspec (old)
  (typecase (second old)
    (string
     (destructuring-bind (name caption criterion display-type &optional (editable t))
         old
       (let ((converted (list name)))
         (setq converted (append converted (list :display-type display-type
                                                 :caption caption)))
         (case criterion
           (:required
            (setq converted (append converted (list :required t))))
           (:optional
            (setq converted (append converted (list :required nil)))))
         (unless editable
           (setq converted (append converted (list :editable nil))))
         converted)))
    (keyword
     old)))

;(%convert-old-skool-pspec '(:foo "Foo:" :required 'fingle))
;(%convert-old-skool-pspec '(:bar "Bar:" :optional 'fingle))
;(%convert-old-skool-pspec '(:foo :required t :caption "Foogle"))
;(%convert-old-skool-pspec '(:foo "Foo:" :required 'fingle))

(defun generate-property-sheet-set (instance specs set-criterion &aux proplist)
  (dolist (property-spec specs)
    (if (eql property-spec :br)
        (push :br proplist)
        (destructuring-bind (name &key (required t) (editable t) caption display-type)
            (%convert-old-skool-pspec property-spec)
          (multiple-value-bind (value type-def caption-def)
              (get-property instance name)
            (unless caption
              (setq caption caption-def))
            (unless display-type
              (setq display-type type-def))
            (ecase set-criterion
              (:all
               (unless value
                 (setq value (property-sheet-null-value instance display-type)))
               (push (list name caption value display-type editable) proplist))
              (:standard
               (when (or value required)
                 (unless value
                   (setq value (property-sheet-null-value instance display-type)))
                 (push (list name caption value display-type editable) proplist))))))))
  (nreverse proplist))

(defmethod object-property-sheet-set (self set-name &optional (set-criterion :standard))
  (let ((proplist nil)
        (pspec (object-property-sheet-spec self set-name)))
    (if (not pspec)
        (progn
          (dolist (prop (object-properties self))
            (destructuring-bind (name value caption display-type)
                prop
              (ecase set-criterion
                (:all
                 (unless value
                   (setq value (property-sheet-null-value self display-type)))
                 (push (list name caption value display-type nil) proplist))
                (:standard
                 (unless value
                   (setq value (property-sheet-null-value self display-type)))
                 (push (list name caption value display-type nil) proplist)))))
          (nreverse proplist))
        (generate-property-sheet-set self pspec set-criterion))))

;; property templates, textual substituion and render to text.

(defun %property-template-resolver (vals ref)
  (let ((ref (read-from-string ref)))
    (etypecase ref
      (symbol
       (format nil "~A" (get-alist ref vals)))
      (list
       (etypecase (car ref)
         (keyword
          (multiple-value-bind (vals dt)
              (get-property (get-alist :root-instance vals) ref)
            (typecase vals
              (null
               (property-template-null-value dt))
              (list
               (string-join
                (mapcar #'(lambda (val)
                            (property-template-display dt val))
                        vals) ", "))
              (t
               (property-template-display dt vals)))))
         
         (symbol
          (eval ref)))))))
                        
            
(defun fill-property-template (root-instance template
                               &key
                               aux-values
                               (sub-delim "$") (escape-char #\\))
  (fill-template (append aux-values
                         (list (cons :root-instance root-instance))) template
                 :sub-delim sub-delim
                 :escape-char escape-char
                 :template-resolver #'%property-template-resolver))


(defmethod property-template-null-value (type)
  (declare (ignore type))
  "")

(defmethod property-template-display (type value)
  (declare (ignore type))
  (princ-to-string value))


