;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: parsing.lisp,v 1.4 2002/03/21 02:13:07 apharris Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :odcl)

(defun lightly-tokenize (bag string)
  (let ((results nil)
        (last 0))
    (dotimes (i (length string))
      (when (member (aref string i) bag :test #'char=)
        (if (= last i)
            (incf last)
            (push (subseq string last i) results))
        (setf last (1+ i))))
    (push (subseq string last) results)))

(defun str-y-or-n (str)
  (not (characterp (or (find #\n str)
		       (find #\N str)))))

(defun str-n-or-y (str)
  (characterp (or (find #\y str)
		  (find #\Y str))))

(defun read-number-or-nil (str)
  (let ((num (read-from-string str)))
    (if (numberp num)
	num nil)))

(defun read-list-or-nil (str)
  (let ((num (read-from-string str)))
    (if (listp num)
	num nil)))

;; ------------------------------------------------------------
;; Parsing and processing utilities

(defun read-comma-sep (line format)
  "Read a line of comma-separated text and parse it into fields as
specified by the format.  Double-quoted field values are returned
without their containing quotes.  If the line has fewer fields than
specified in the format, only the fields found are returned."
  (let ((start 0)
	fields)
    (dotimes (x (length format))
      (if (< start (length line))
          (let (end)
            (setf end (or (position #\, line :start start)
                          (length line)))
            (setq fields (cons (string-trim '(#\Space #\") (subseq line start end))
                               fields))
            (setq start (+ end 1)))
          (format t "~&Field ~A of type ~A not found in line:~%  ~A~%"
                  (cadr(nth x format))
                  (car(nth x format))
                  line)))
    (nreverse fields)))
	    
(defun parse-boolean (string)
  (not (null (position-if (lambda (x) (position x "Yy")) string))))

(let ((str (make-string 80 :element-type 'base-char)))
  (defun read-delimited (del stream)
    "read from a stream, stopping at the given delimiter.  delimiter
can be a single char, or a list of chars."
    (declare (optimize (speed 3) #+cmu (extensions:inhibit-warnings 3))
	     (type (simple-array * (*)) str))
    (let ((dellist (etypecase del
		     (character (list del))
		     (list del))))
      (do ((i 0 (incf i))
	   (c (read-char stream) (read-char stream)))
	  ((member c dellist :test #'char=) (adjust-array str i))
	(if (= i (length str))
	    (setf str (adjust-array str (ash (length str) 1))))
	(setf (aref str i) c)))))

(let ((str (make-string 80 :element-type 'base-char)))
  (defun read-comma-delimited (stream)
    "read from a stream, stopping at the given delimiter."
    ;; (declare (optimize (speed 3)))
    (do ((i 0 (incf i))
	 (c (read-char stream) (read-char stream)))
	((char= c #\,) (adjust-array str i))
      (if (= i (length str))
	  (setf str (adjust-array str (ash (length str) 1))))
      (setf (aref str i) c))))

(defun loop-over-lines (stream func)
  (loop as line = (read-line stream nil)
	while (not (null line))
	do (apply func (list line))))


(defmacro nostr (val)
  `(or (null ,val) (string= ,val "")))

(defun %read-line (delim stream &optional (eof-error t))
  (let ((str nil))
    (do ((ch (read-char stream eof-error :eof)
	     (read-char stream eof-error :eof)))
	((or (equal ch :eof)
	     (equal ch delim)) str)
      (push ch str))
    (if str (concatenate 'string (nreverse str)) nil)))

(defun parse-csv-string (string &key (quote #\"))
  (let ((len (length string))
	(vals (list ""))
	(inq nil))
    (do ((i 0 (incf i)))
	((= i len)
	 (reverse
	  (mapcar #'(lambda (tok)
		      (coerce (reverse tok) 'string))
		  vals)))
      (let ((ch (elt string i)))
	(cond
	  ((char= ch quote)
	   (if inq (setq inq nil) (setq inq t)))
	  ((char= ch #\,)
	   (if inq
	       (push ch (car vals))
	       (push nil vals)))
	   (t
	    (push ch (car vals))))))))


(defun split-tok (string char)
  (labels ((split (list string char)
	     (let ((pos (position char string)))
	       (if (not pos)
		   (nreverse (cons string list))
		   (split (cons :replace (cons (subseq string 0 pos) list))
			  (subseq string (+ 1 pos))
			  char)))))
    (split nil string char)))
