;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: mailer.lisp,v 1.3 2002/03/29 04:23:43 craig Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.
;;;
;;; SMTP mailer

(in-package :odcl)

(defvar *mailer* nil)
(defvar *mailer-debug* nil)
(defvar *mailer-default-sender* nil
  "The address in the Sender field, where mail errors will be returned too.  If NIL, than the mailer backends will use the results of the thunk MAILER-DEFAULT-SENDER.")
;; could be SMTP to handle the SMTP ourselves

(defvar *mailer-default-backend* :sendmail
  "Which backend to use to send messages.  :SENDMAIL means to use
sendmail as the mailer backend, calling out to the sendmail program on
the host.  :SMTP means we should connect to an SMTP server
ourselves.")


(defun mailer-default-sender ()
  (or *mailer-default-sender*
      (format t "webcheckout@~A"
              (local-host-name))))

;; returns a properly mail-formatted date string.

#+cmu
(defun mailer-format-date ()
  (let ((dow-l #("Monday" "Tuesday" "Wednesday" "Thursday" "Friday" "Saturday" "Sunday"))
        (mo-l  #("What" "Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")))
    (multiple-value-bind (ss mm hh dd mo yy dow dst tz)
        (ext::get-decoded-time)
      (declare (ignore dst))
      (format nil "~a, ~d ~d ~d ~2d:~2,'0d:~2,'0d ~a~2,'0d00"
              (aref dow-l dow) dd (aref mo-l mo) yy hh mm ss (if (< 0 tz) "-" "+") tz))))


(defclass mailer ()
  ((sender :initform nil
           :initarg :sender)))
  
(defclass smtp-mailer (mailer)
  ((host     :initform nil
             :initarg :host)
   (port     :initform nil
             :initarg :port)))

(defvar *smtp-mailer-default-port* 25)
(defvar *smtp-mailer-default-host* "prep.onshored.com")

(defmethod make-mailer ((type (eql :smtp)) &rest args)
  (destructuring-bind (&key 
                       (host *smtp-mailer-default-host*)
                       (port *smtp-mailer-default-port*)
                       (default t))
      args
    (let ((mailer (make-instance 'smtp-mailer :host host :port port)))
      (when default
        (setf *mailer* mailer))
      mailer)))

(defclass sendmail-mailer (mailer)
  ((sendmail-program :initform nil
                     :initarg :sendmail-program)
   (sendmail-default-args :initform nil
                          :initarg :default-args)))


(defvar *sendmail-mailer-default-program* "/usr/sbin/sendmail")
(defvar *sendmail-mailer-default-args* nil)

(defmethod make-mailer ((type (eql :sendmail)) &rest args)
  (destructuring-bind (&key
                       (program *sendmail-mailer-default-program*)
                       (defargs *sendmail-mailer-default-args*)
                       (sender (mailer-default-sender))
                       (default t))
      args
    (let ((mailer (make-instance 'sendmail-mailer
                                 :sendmail-program program
                                 :sender sender
                                 :default-args defargs)))
      (when default
        (setf *mailer* mailer))
      mailer)))

(defun ensure-mailer (&rest args)
  (or *mailer*
      (apply #'make-mailer *mailer-default-backend* args)))


(defclass mailer-message ()
  ((from-address  :initarg :from)
   (to-addresses  :initarg :to)
   (subject       :initarg :subject)
   (message       :initarg :message)
   (extra-headers :initarg :headers
                  :initform nil)))

(defun mailer-send (from to subject message &optional headers)
  (let ((message (make-instance 'mailer-message
                                :from from :to (listify to)
                                :subject subject
                                :message message
                                :headers headers))
        (mailer (ensure-mailer)))
    ;;(if (threaded-mailer mailer)
    ;;(%mailer-enqueue mailer message) 
    (%mailer-dispatch mailer message)))

(defun starts-with (sub string)
  (let ((len (length sub)))
    (when (<= len (length string))
      (string= sub (subseq string 0 len)))))

#+cmu
(defmethod %mailer-dispatch ((self smtp-mailer) (message mailer-message))
  (let ((*EOL* (let ((eol (make-string 2)))
                 (setf (aref eol 0) #\Return
                       (aref eol 1) #\Newline)
                 eol)))                 ; \r\n
    (with-slots (host port)
      self
      (with-slots (from-address to-addresses subject message extra-headers)
        message
        (when to-addresses
          (let* ((sock  (ext:connect-to-inet-socket host port))
                 (stream (sys:make-fd-stream sock :input t :output t :element-type 'base-char))
                 (reply nil)
                 (rcpt-ok nil))
            (unwind-protect
                 (labels ((send (expect string &rest format-args)
                            (let ((string (apply #'format `(nil ,string ,@format-args))))
                              (when *mailer-debug*
                                (format t ";; SEND: ~A~%" string))
                              (write-string string stream)
                              (write-string *EOL* stream)
                              (finish-output stream)
                              (when expect
                                (let ((reply (receive)))
                                  (unless (starts-with expect reply)
                                    (error "Mail protocol error: ~a" reply))))))
                          (receive ()
                            (let ((reply (read-line stream)))
                              (when *mailer-debug*
                                (format t ";; RECEIVE: ~A~%" reply))
                              reply)))
                   (setq reply (receive))
                   (unless (starts-with "220" reply)
                     (error "Mail protocol error"))
                   (do ((done nil done))
                       (done)
                     (setq done (not (= (position #\- reply) 3)))
                     (unless done
                       (setq reply (receive))
                       (unless (starts-with "220" reply)
                         (error "Mail protocol error"))))

                   (send "250" "HELO ~a" host)
                   (send "250" "MAIL FROM: ~a" from-address)
                   (dolist (to to-addresses)
                     (send nil "RCPT TO: ~a" to)
                     (setq reply (receive))
                     (if (starts-with "250" reply)
                         (setq rcpt-ok t)
                         (when *mailer-debug*
                           (format t "~&;; RCPT: ~a~%" reply))))
                   (unless rcpt-ok
                     (when *mailer-debug*
                       (format t "~&;; No RCPTs accepted, aborting~%"))
                     (return-from %mailer-dispatch))
                   (send "354" "DATA")
                   (send nil   "From: ~a" from-address)
                   (send nil   "To: ~{~a~^, ~}" to-addresses)
                   (send nil   "Subject: ~a" subject)
                   (send nil   "Date: ~a" (mailer-format-date))
                   (send nil   "X-Mailer: IMHO")

                   (dolist (header extra-headers)
                     (send nil header))

                   (write-string *EOL* stream)
                   ;; End of Headers
                 
                   (send nil message)
                   (send "250" ".")
                   (send nil "QUIT"))
              (close stream))))))))


(defmethod %mailer-dispatch ((self sendmail-mailer) (message mailer-message))
  (with-slots (sendmail-program sender)
    self
    #-cmu
    (error "sendmail interface not supported in this lisp.")
    #+cmu
    (let* ((args (slot-value self 'sendmail-default-args))
           (send nil)
           (send-in nil)
           (send-out nil)
           (send-error nil)
           (*EOL* (let ((eol (make-string 2)))
                    (setf (aref eol 0) #\Return
                          (aref eol 1) #\Newline)
                    eol)))
      (when sender
        (push sender args)
        (push "-f" args))
      (when *mailer-debug* (push "-v" args))
      (unwind-protect
           (progn
             (setq send (ext:run-program sendmail-program args
                                         :wait nil
                                         :input :stream)
                   ;;:error :stream
                   ;;:output :stream)
                   send-in
                   #+cmu (ext:process-input send)
                   
                   send-out
                   #+cmu (ext:process-output send)
                   
                   send-error
                   #+cmu (ext:process-error send))
             
             (labels ((send-string (string &rest format-args)
                        (let ((string (apply #'format `(nil ,string ,@format-args))))
                          (when *mailer-debug*
                            (format t ";; SEND: ~A~%" string))
                          (write-string string send-in)
                          (write-string *EOL* send-in)
                          (finish-output send-in))))
               
               
               (with-slots (from-address to-addresses subject message extra-headers)
                 message
                 (send-string "From: ~a" from-address)
                 (send-string "To: ~{~a~^, ~}" to-addresses)
                 (send-string "Subject: ~a" subject)
                 (send-string "Date: ~a" (mailer-format-date))
                 (send-string "X-Mailer: IMHO")
          
                 (dolist (header extra-headers)
                   (unless (string-nil-or-empty-p header)
                     (send-string header)))
                 
                 (write-string *EOL* send-in)
                 
                 (send-string message)
          
                 (close send-in)
                 (ext:process-wait send))))
        (when (ext:process-alive-p send)
          (ext:process-kill send 11))))))
  
