;;; -*- Mode: Lisp -*-
;;; $Id: list.lisp,v 1.9 2002/02/22 17:37:36 jesse Exp $
;;;
;;; Copyright (c) 2001 onShore Development, Inc.

(in-package :odcl)

(defun drill-car (item)
  (if (consp item)
      (drill-car (car item))
    item))

(defun listify (x)
  (if (not (consp x))
      (list x)
      x))

(defun pairify (list &aux result)
  (do ((list list (cddr list)))
      ((null list) (nreverse result))
    (push (list (car list) (cadr list)) result)))

(defun flatten (list)
  (mapcan #'(lambda (item)
	      (if (atom item)
		  (cons item nil)
                  (flatten item)))
	  list))

(defun break-list (list elt &aux lists)
  "divide LIST into lists, using ELT as seperator (eql)"
  (loop
   (let ((pos (position elt list)))
     (if pos
         (progn
           (push (subseq list 0 pos) lists)
           (setq list (subseq list (1+ pos))))
         (return-from break-list (nreverse (push list lists)))))))

(defun filter-list2 (list &key test key)
  "return elts of list which match pred"
  (declare (ignore key))
  (labels ((rec (lst accum)
             (if (null lst)
                 (reverse accum)
                 (if (funcall test (car lst))
                     (rec (cdr lst) (cons (car lst) accum))
                     (rec (cdr lst) accum)))))
    (rec list nil)))


(defun partition-list (list test &aux result)
  (while list
    (if-bind (matches (remove-if-not (lambda (x)
                                       (funcall test x (car list)))
                                     (cdr list)))
        (progn
          (push (cons (car list) matches) result)
          (setq list (remove-if (lambda (x)
                                  (funcall test x (car list)))
                                (cdr list))))
        (progn
          (push (list (car list)) result)
          (setq list (cdr list)))))
  (nreverse result))

(defun partition-pairwise (list test &aux result)
  (let ((list (copy-list list)))
    (while list
      (destructuring-bind (head &rest tail)
          list
        (if-bind (match (position head tail :test test))
            (progn
              (push (list head (nth match tail)) result)
              (setq list (delete-at-index! match tail)))
            (progn
              (push (list head) result)
              (setq list tail))))))
  (nreverse result))

(defun maybecar (thing)
  "return THING, or (car THING) if THING is a cons"
  (if (listp thing)
      (car thing)
      thing))

;;
;; e.g. "Foo bar baz" -> ("Foo" "bar" "baz")
;;

(defun make-keywords-into-list (keywords)
  (setq keywords (setq keywords (string-trim '(#\space) keywords)))
  (loop for pos1 = 0 then (1+ pos2)
	for pos2 = (position #\space keywords :start pos1)
	collect (subseq keywords pos1 pos2)
	while pos2))

(defun intersect-2 (list-a list-b match-fn &aux list-ab)
  (let ((list-a (copy-list list-a))
        (list-b (copy-list list-b)))
    (flet ((extract-match (a)
             (let ((a-idx (position-if (lambda (c) (eql a c)) list-a))
                   (b-idx (position-if (lambda (b) (funcall match-fn a b)) list-b)))
               (when (and a-idx b-idx)
                 (push (cons (nth a-idx list-a)
                             (nth b-idx list-b)) list-ab)
                 (setq list-a (delete-at-index! a-idx list-a))
                 (setq list-b (delete-at-index! b-idx list-b))))))
      (mapc #'extract-match (copy-list list-a)))
    (values (nreverse list-ab) list-a list-b)))

(defun key-partition (elements key test sort-key sort-test)
  "Break ELEMENTS into a list of sorted lists, where each element
 of each inner list is equal according to KEY and TEST, and each inner
 list is sorted according to SORT-KEY and SORT-TEST.
 e.g.
 (key-partition '((7 8) (2 8) (3 1) (1 1)) #'cadr #'eql #'car #'<)
 ->  (((1 1) (3 1)) ((2 8) (7 8)))"
  (let ((results (make-hash-table :test test)))
    (dolist (element elements)
      (push element (gethash (funcall key element) results)))
    (let ((return nil))
      (maphash (lambda (k v)
                 (declare (ignore k))
                 (push (sort v sort-test :key sort-key) return)) results)
      return)))

(defun delete-at-index (idx list)
  (cond ((= idx 0)
         (cdr list))
        ((= idx (1- (length list)))
         (butlast list))
        (t
         (append (subseq list 0 idx)
                 (subseq list (1+ idx))))))

(defun delete-at-index! (idx list)
  (let ((rplac-head (nthcdr idx list))
        (rplac-tail (nthcdr (1+ idx) list)))
    (cond (rplac-tail
           (rplaca rplac-head (car rplac-tail))
           (rplacd rplac-head (cdr rplac-tail))
           list)
          ((= idx 0)
           nil)
          (t
           (rplacd (nthcdr (1- idx) list) nil)
           list))))

(defun delete-at-positions (list positions)
  "given a list and a list of integral positions of elements to delete,
return the new list. nondestructive"
  (let ((positions (sort positions #'>))
        (newlist (copy-list list)))
    (dolist (pos positions)
      (setq newlist (append (subseq newlist 0 pos)
                            (subseq newlist (1+ pos)))))
    newlist))

(defun print-iterator (out list)
  (if (endp (cdr list))
      (concatenate 'string out (car list))
    (print-iterator (concatenate 'string
		       (car list) ", " out) (cdr list))))

(defun print-list (list)
  (print-iterator "" list))
