;;; -*- Mode: Lisp -*-
;;; $Id: iso-8601.lisp,v 1.13 2002/04/05 00:04:09 craig Exp $
;;;
;;; Copyright (c) 2000 onShore Development, Inc.

(in-package :local-time)

(defun parse-timestring (timestring &key (start 0) end junk-allowed) 
  "parse a timestring and return the corresponding local-time.  If the
timestring starts with P, read a duration; otherwise read an ISO 8601
formatted date string."
  (declare (ignore junk-allowed))  ;; FIXME
  (let ((string (subseq timestring start end)))
    (if (char= (aref string 0) #\P)
        (parse-iso-8601-duration string)
        (parse-iso-8601-time string))))

(defvar *iso-8601-duration-delimiters*
  '((#\D . :days)
    (#\H . :hours)
    (#\M . :minutes)
    (#\S . :seconds)))

(defun iso-8601-delimiter (elt)
  (cdr (assoc elt *iso-8601-duration-delimiters*)))

(defun iso-8601-duration-subseq (string start)
  (let ((pos (position-if #'iso-8601-delimiter string :start start)))
    (when pos
      (values (parse-integer (subseq string start pos))
              (1+ pos)
              (iso-8601-delimiter (aref string pos))))))

(defun parse-iso-8601-duration (string)
  "return a local-time from a duration string"
  (block parse
    (let ((days 0) (secs 0) (index 1))
      (loop
       (multiple-value-bind (duration next-index duration-type)
           (iso-8601-duration-subseq string index)
         (case duration-type
           (:hours
            (incf secs (* duration +seconds/hour+)))
           (:minutes
            (incf secs (* duration +seconds/minute+)))
           (:seconds
            (incf secs duration))
           (:days
            (incf days duration))
           (t
            (return-from parse (make-duration :day days :sec secs))))
         (setq index next-index))))))

(defun syntax-parse-iso-8601 (string)
  (let (year month day hour minute second gmt-sec-offset)
    (handler-case
        (progn
          (setf year   (parse-integer (subseq string 0 4))
                month  (parse-integer (subseq string 5 7))
                day    (parse-integer (subseq string 8 10))
                hour   (if (<= 13 (length string))
                           (parse-integer (subseq string 11 13))
                           0)
                minute (if (<= 16 (length string))
                           (parse-integer (subseq string 14 16))
                           0)
                second (if (<= 19 (length string))
                           (parse-integer (subseq string 17 19))
                           0)
                gmt-sec-offset (when (<= 22 (length string))
                                 (* +seconds/hour+ (parse-integer (subseq string 19 22)))))
          (values year month day hour minute second gmt-sec-offset))
      (simple-error ()
        (error 'iso-8601-syntax-error
               :bad-component
               (car (find-if (lambda (pair) (null (cdr pair)))
                             `((year . ,year) (month . ,month)
                               (day . ,day) (hour ,hour)
                               (minute ,minute) (second ,second)
                               (timezone ,gmt-sec-offset)))))))))

;; e.g. 2000-11-11 00:00:00-06
;; FIXME: use designators
;; FIXME: does not handle fractional seconds

(defun parse-iso-8601-time (string)
  "return the local-time corresponding to the given ISO 8601 datestring"
  (apply #'construct-local-time (multiple-value-list (syntax-parse-iso-8601 string))))

(defun construct-local-time (year month day &optional (hour 0) (minute 0) (seconds 0) seconds-to-utc)
  (when (null seconds-to-utc)
    (setq seconds-to-utc (utc-offset)))
  (make-local-time :day (%lt-date year month day)
                   :sec (+ (* hour +seconds/hour+)
                           (* minute +seconds/minute+)
                           seconds (- seconds-to-utc))))

