/* $Id: lctl.c,v 1.2 2000/05/03 23:11:32 jesse Exp $ */

/*
 * LCTL - tool for maintenance of a long-running lisp process
 *
 * start
 * stop
 * create <imagename>
 * connect <imagename>
 */

#include <stdio.h>
#include <errno.h>

#include "lctl.h"
#include "image.h"

static int lctl_action;
static char *lctl_image;

void usage(char *progname)
{
  printf("Usage: %s <verb> [image]\n", progname);
  printf("  where verb is one of: start, stop, create or connect,\n");
  printf("  and image is the name of the lisp image to operate on.\n");
}

/*
 * Pick off the action, and the target image name.
 */
void process_args(int argc, char *argv[])
{
  paction command;
  int i;
  int found = 0;

  if (argc < 2)
    {
      usage(argv[0]);
      exit(1);
    }

  for (i = 0, command = commands[i]; command.name != NULL; command = commands[++i])
    {
      if (strcmp(command.name, argv[1]) == 0)
        {
          lctl_action = command.number;
          found = 1;
          break;
        }
    }
  
  if (found == 0)
    {
      usage(argv[0]);
      exit(1);
    }    

  if (lctl_action == SVC_CREATE || lctl_action == SVC_CONNECT)
    {
      if (argc < 3)
        {
          usage(argv[0]);
          exit(1);
        }
      else
        {
          lctl_image = argv[2];
        }
    }
}

void shutdown()
{
  if (is_server())
    {
      printf("Shutting down\n");

      image_terminate();
      ipc_terminate();
    }
}

int main(int argc, char *argv[])
{
  int cmd_rtn = -1;
  lisp_image *image;

  process_args(argc, argv);
  ipc_init();
  image_init();
  cmd_rtn = atexit(shutdown);
  

  switch (lctl_action)
    {
    case SVC_TEST:
      printf("Testing lisp process creation.\n");
      image_create("test");
      cmd_rtn = image_destroy("test");
      break;

    case SVC_START:
      printf("Starting Lisp Server.\n");
      if (server_running_p())
        printf("The server is already running.\n");
      else
        cmd_rtn = server_start();
      break;

    case SVC_STOP:
      printf("Stopping Lisp Server.\n");
      if (!server_running_p())
        printf("The server is not running.\n");
      else
        cmd_rtn = server_stop();
      break;

    case SVC_CREATE:
      printf("Creating Lisp Image: %s\n", lctl_image);
      if (!server_running_p())
        printf("The server is not running.\n");
      else
        cmd_rtn = server_create(lctl_image);
      break;

    case SVC_CONNECT:
      printf("Connecting to Image: %s\n", lctl_image);
      if (!server_running_p())
        printf("The server is not running.\n");
      else
        cmd_rtn = server_connect(lctl_image);
      break;

    default:
      printf("Unknown command.\n");
      usage(argv[0]);
      exit(1);
    }

  if (cmd_rtn != 0)
    {
      printf("Command failed.\n");
    }

  return cmd_rtn;
}
