/* cp_remote.c  to initiate and monitor remote routines, pipes, etc. */

#include "cp_head.h"
#include <sys/times.h>

/* prototypes */
Notify_value read_remote();
Notify_value sigchldcatcher();

static char next[256];

int init_remote(char *datastr)
{
	int i=0,nflag=0;
	char command[32];

	while (i< NUM_PROC && remote[i].pid) i++;
	if (i==NUM_PROC) 
	 {
		sprintf(msgbuf,"Can't start remote process; "
			"all slots occupied.");
		msg();
		return 0;
	 }
	if (!grab_next(&datastr,command)) return 0;
	stripsp(command);
	if (strcmp(command,"-n")==0) /* attach moniker to process */
	 {
		if (!grab_next(&datastr,command)) return 0;
		strncpy(remote[i].moniker,command,31);
		remote[i].moniker[31]='\0';
		nflag=1;
		if (!grab_next(&datastr,command)) return 0;
	 }
	if ( !open_service(&remote[i],command, datastr) )
	 {
		sprintf(msgbuf,"Remote process %s failed to start.",command);
		emsg();
		return 0;
	 }
	if (!nflag) /* give default moniker of command name */
		strcpy(remote[i].moniker,command);
	strcpy(remote[i].program,command);
	sprintf(msgbuf,"Remote program '%s' started: moniker=%s, pid=%d.",
		remote[i].program,remote[i].moniker,remote[i].pid);
	msg();
	return 1;
} /* init_remote */

int open_service(struct remote_control *pc,char *command,char *datastr)
{
	int pid,i;
	char *arg[13],*ptr;
	Notify_value read_remote(),sigchldcatcher();

	if (pipe(pc->pipeto) || ( pipe(pc->pipefr) 
		&& (close(pc->pipeto[0]),close(pc->pipeto[1]),1) ) )
	 {
		sprintf(msgbuf,"Pipe opening error.  ");
		emsg();
		return 0;
	 } 
		/* parse datastr into argument pointers, max of 6 */
	ptr=datastr;
	for(i=0;i<11 && grab_next(&ptr,buf);i++)
	 {
		arg[i]=ptr-strlen(buf);
		if (*ptr!='\0') {*ptr='\0';ptr++;}
		
	 }
	arg[i]=(char *)NULL;

/* fork */

	if ((pid = fork())) /* in parent */
	 {
        	if (pid == -1)
		 {
		   sprintf(msgbuf,"Failed to fork for process.  ");
		   emsg();
		   close(pc->pipeto[0]);close(pc->pipeto[1]);
		   close(pc->pipefr[0]);close(pc->pipefr[1]);
		   return 0;
		 }
		pc->pid=pid;
        	(void)close(pc->pipeto[0]);
        	(void)close(pc->pipefr[1]);
        	pc->fp_to_remote=fdopen(pc->pipeto[1], "w");
        	pc->fp_from_remote=fdopen(pc->pipefr[0], "r");
		pc->op_code=1;
	 }
	else /* in child */
	 {
        	(void)close(pc->pipeto[1]);
        	(void)close(pc->pipefr[0]);
	      	(void)dup2(pc->pipeto[0], 0);
        	(void)dup2(pc->pipefr[1], 1);
	/*	(void)dup2(pc->pipefr[1], 2);  */
			/* in child, close out all notifier stuff. */
		for (i=getdtablesize();i>2;i--) 
			(void)close(i);
		for (i=0;i<NSIG;i++)
			(void)signal(i,SIG_DFL);

        	execlp(command, command,
		       arg[0],arg[1],arg[2],arg[3],arg[4],arg[5],arg[6],
		       arg[7],arg[8],arg[9],arg[10],arg[11]);
      		_exit(1);
	 }
	notify_set_input_func(pc->notify_client,read_remote,pc->pipefr[0]);
	notify_set_wait3_func(pc->notify_client,sigchldcatcher,pid);

	return 1;
}

Notify_value read_remote(Notify_client client,register int fd)
{
  char buff[BUFSIZE],keyword[64],*ptr,*dpoint,*inbuffer;
  int i,hit=-1,pnum,opt,nodes,flag,vert,bufinc=65535;
  ssize_t bytes;

  for (i=0;i<NUM_PROC;i++)
    if (remote[i].notify_client==client) {hit=i;break;}
  /* check if the process is really up and running */
  if (hit<0 || !remote[hit].op_code) return NOTIFY_IGNORED;
  inbuffer=ptr=(char *)malloc(bufinc*sizeof(char));
  if ((bytes=read(fd,ptr,bufinc))<1)
    return NOTIFY_DONE;

/* fix??: may need to allow for expansion of buffer */
  if (bytes>=(bufinc-1))
    {
      sprintf(msgbuf,"Incoming buffer from remote overflowed.");
      emsg();
      return NOTIFY_IGNORED;
    }

  while (grab_next(&ptr,keyword))
    {

/* remote asks for a lock */
      if (!strncmp(keyword,"lock",4))
	{
	  if ( !grab_next(&ptr,next) 
	       || sscanf(next,"%d",&pnum)!=1
	       || pnum<0 || pnum >= NUM_PACKS
	       || !reset_lock(&packdata[pnum],hit,1) )
	    {
	      sprintf(msgbuf,"failed lock %d",pnum);
	      msg_remote(hit);
	    }
	  else
	    {
	      sprintf(msgbuf,"succeeded lock %d",pnum);
	      msg_remote(hit);
	    }
	} 

/* remote removes a lock */
      else if (!strncmp(keyword,"unlock",6))
	{
	  if (!grab_next(&ptr,next) 
	      || sscanf(next,"%d",&pnum)!=1
	      || pnum<0 || pnum >= NUM_PACKS
	      || !reset_lock(&packdata[pnum],hit,0) )
	    {
	      sprintf(msgbuf,"failed unlock %d",pnum);
	      msg_remote(hit);
	    }
	  else
	    {
	      sprintf(msgbuf,"succeeded unlock %d",pnum);
	      msg_remote(hit);
	    }
	}
      
/* remote announcing intention to send pack or path data */
      else if (!strncmp(keyword,"sending",7) 
	       && grab_next(&ptr,next))
	{
	  if (!strncmp(next,"pack",4)) /* pack data */
	    {
	      if ( sscanf(ptr,"%d %d",&pnum,&nodes)!=2
		   || pnum<0 || pnum >= NUM_PACKS
		   || nodes<=0 )
		{
		  sprintf(msgbuf,"failed pnum or nodes problem");
		  msg_remote(hit);
		}
	      else
		{
		  if (check_lock(pnum,hit))
				/* locked by other remote process? */
		    {
		      sprintf(msgbuf,
		      "Remote process pid=%d has been locked out of pack %d.",
			      remote[hit].pid,pnum);
		      emsg();
		      sprintf(msgbuf,"failed pack locked");
		      msg_remote(hit);
		    }
		  else
		    {
		      sprintf(msgbuf,"ready");
		      msg_remote(hit);
		      if (!call_read_pack(remote[hit].fp_from_remote,
					  &packdata[pnum],"\0"))
			{
			  sprintf(msgbuf,
			    "Read pack from remote process pid=%d has failed.",
				  remote[hit].pid);
			  emsg();
			  sprintf(msgbuf,"failed readpack");
			  msg_remote(hit);
			}
		      else
			{
			  sprintf(msgbuf,
			     "Pack %d filled by remote routine '%s', pid=%d.",
				  pnum,remote[hit].program,
				  remote[hit].pid);
			  msg();
			  sprintf(msgbuf,"succeeded readpack %d",pnum);
			  msg_remote(hit);
			}
		    }
		}
	    }
	  else if (!strncmp(next,"path",4)) /* path data */
	    {
	      if (!readpath(remote[hit].fp_from_remote,&pathlist,&pathlength))
		{
		  sprintf(msgbuf,"Read path from remote process "
			  "pid=%d has failed.",remote[hit].pid);
		  emsg();
		  sprintf(msgbuf,"failed readpath");
		  msg_remote(hit);
		}
	      else
		{
		  sprintf(msgbuf,"succeeded readpath");
		  msg_remote(hit);
		}
	    }
	}

/* remote asking to get pack or path data */
      else if (!strncmp(keyword,"get",3) && grab_next(&ptr,next))
	{
	  if (!strncmp(next,"pack",4))
	    {
	      if (!grab_next(&ptr,next) 
		  || sscanf(next,"%d",&pnum)!=1
		  || pnum<0 || pnum >= NUM_PACKS
		  || !packdata[pnum].status)
		{
		  sprintf(msgbuf,"failed pack number problem");
		  msg_remote(hit);
		}
	      else
		{
		  if (!grab_next(&ptr,next) 
		      || sscanf(next,"%d",&opt)!=1 
		      || opt<0 || opt > 0777) opt=0777;
		  sprintf(msgbuf,"sending pack %d %d",
			  pnum,packdata[pnum].nodecount);
		  msg_remote(hit);
		  if (!writepack(remote[hit].fp_to_remote,
				 &packdata[pnum],opt,0))
		    {
		      sprintf(msgbuf,
			"Write to remote process pid=%d has failed.",
			      remote[hit].pid);
		      emsg();
		    }
		}
	    }
	  else if (!strncmp(next,"path",4))
	    {
	      if (pathlist==NULL || pathlength<=0)
		{
		  sprintf(msgbuf,"END");
		  msg_remote(hit);
		}
	      else
		{
		  sprintf(msgbuf,"sending path");
		  msg_remote(hit);
		  if (!writepath(remote[hit].fp_from_remote,pathlist))
		    {
		      sprintf(msgbuf,
			"Read path to remote process pid=%d has failed.",
			      remote[hit].pid);
		      emsg();
		    }
		}
	    }	
	}

/* send requested data to child */
      else if (!strncmp(keyword,"data:",5)
	&& grab_line(buff,&ptr,BUFSIZE-1))
	{
	  flag=0;
	  dpoint=buff;
	  stripsp(dpoint);
	  if (strncmp(dpoint,"-p",2)) pnum=current_p;
	  else /* pack number specified */
	    {
	      dpoint +=2;
	      if (!grab_next(&dpoint,next) 
		  || strlen(next)>1) flag=1;
	      else if ((i=atoi(next))>=0 && i<NUM_PACKS 
		       && packdata[i].status) pnum=i;
	      else flag=1; /* no pack specified */
	      dpoint +=1;
	    }
	  if (!flag && grab_next(&dpoint,next))
	    {
	      if (!strcmp(next,"-cent")
		  && (vert=grab_one_vert(&packdata[pnum],&dpoint)))
		{
		  sprintf(msgbuf,"%f %f\n",
			  packdata[pnum].packR_ptr[vert].center.re,
			  packdata[pnum].packR_ptr[vert].center.im);
		}
	      /* future options coded here */
	      else flag=1;
	    }
	  if (flag) sprintf(msgbuf,"no data");
	  msg_remote(hit);
	}
      
/* execute string of commands from remote */
      else if (!strncmp(keyword,"command:",8)
	       && grab_line(buff,&ptr,BUFSIZE-1)) 
	        /* note: have to dump leading linebreak */
	{
	  sort_cmd(buff,&current_p);	
	}
      
/* print message/error message from child */
      else if (!strncmp(keyword,"msg:",4) 
	       && grab_line(buff,&ptr,BUFSIZE-1))
	{
	  strcpy(msgbuf,buff);
	  msg();
	}
      else if (!strncmp(keyword,"emsg:",5)
	       && grab_line(buff,&ptr,BUFSIZE-1))
	{
	  strcpy(msgbuf,buff);
	  emsg();
	}
/* allow child to kill self */
      else if (!strncmp(keyword,"done",4)) 
	{
	  sprintf(msgbuf,"bye");
	  msg_remote(hit);
	}
    } /* end of while */
  return NOTIFY_DONE;
} /* read_remote */

Notify_value sigchldcatcher(Notify_client client,int pid,
			    union wait *status,struct rusage *rusage)
/* catch dead remote processes. Note: catch all SIGCHLD signals; 
currently don't care why it was generated, so this code may orphan a 
process which is only stopped. */
{
	int i,j,k=1,hit=-1;
	struct remote_control *pc;

	for (i=0;i<NUM_PROC;i++)
		if (remote[i].notify_client==client) {hit=i;break;}
	if (hit==-1) return NOTIFY_IGNORED;
	pc=&remote[hit];
		 /* close up shop */
	pc->pid=0;
	pc->op_code=0;
	pc->locks=0;

	strcpy(pc->program,"");
	close(pc->pipeto[1]);
	close(pc->pipefr[0]);
	pc->fp_to_remote=pc->fp_from_remote=(FILE *)NULL;
		/* unlock any packings */
	for (j=0;j<hit;j++) k *=2;
	for (i=0;i<NUM_PACKS;i++) 
		if (packdata[i].locks & k) reset_lock(&packdata[i],hit,0);
	notify_set_input_func(client,
		NOTIFY_FUNC_NULL, pc->pipefr[0]);
	sprintf(msgbuf,"Remote process pid=%d is dying.",pid);
	msg();
	return (notify_default_wait3(client,pid,status,rusage));
} /* sigchldcatcher */

int reset_lock(struct p_data *p,int hit,int flag)
/* set/unset lock, fix cursors, etc. flag: 1=set, 0=unset. */
{
	int k=1,i;

	for (i=0;i<hit;i++) k *=2;
	p->screen->ms_flag=0;
	if (flag) /* set */
		p->locks=p->locks | k;
	else /* unset */
		p->locks=p->locks & (~k);	
	set_cursor(p->screen,p->screen->ms_flag);
	return 1;
} /* reset_lock */

int check_lock(int pnum,int proc)
/* 1 means pnum is locked for process other than proc */
{
	int k=1,i;

	for (i=0;i<proc;i++) k *=2;
	return ((int)( ~(~packdata[pnum].locks | k) ));
} /* check_lock */


