/*
 * plugin_main.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  plugin_main.cpp  --  main source file of the CinePaint plugin 
                         "Bracketing to HDR".
  
  This file contains all CinePaint affairs of the "Bracketing" project. The
    only two services we need from CPaint are loading an image and showing an
    float32 (HDR) image: wrapped and provided to the rest of the program by the
    two functions declared in "plugin_main.hpp":
    
      bool cpaint_load_image (const char* fname); 
      void cpaint_show_image (int W, int H, const uchar* buf, GPrecisionType);

    Explicite dependencies to CPaint (and GTK) I wanted confine to this file.
    At present it succeeds exept for `GPrecisionType' (argument type of
    cpaint_show_image()) and "lib/wire/enums.h" (included hence in
    "plugin_main.hpp").
    
  The communication to CinePaint was adopted from Kai-Uwe Behrmann's plug-in
    "collect".
  
  CinePaint is migrating from GTK to FLTK (Glasgow project).
    "Bracketing"s GUI is written already in FLTK. 
    
  Note for me: 
    `GPrecisionType' is defined in "cinepaint???/lib/wire/precision.h" as an
    enum. There are also precision strings `PRECISION_U8_STRING' and so on.
    (Unfortunately not in an array to have a simple reference to the enums.)
    
 BUGS:
    
 If CPaint headers included *before* "tnt_array2d.hpp" it gives the compiler
   error:
 
   In file included from /usr/include/g++/i586-suse-linux/bits/c++locale.h:44,
                    from /usr/include/g++/iosfwd:46,
                    from /usr/include/g++/ios:44,
                    from /usr/include/g++/ostream:45,
                    from /usr/include/g++/iostream:45,
                    from /usr/local/include/TNT/tnt_array2d.hpp:26,
                    from plugin_main.cpp:35:
   /usr/include/libintl.h:40: error: syntax error before `char'
   
 If included after "tnt_array2d.hpp", but *before* "Br.hpp" (which includes
   "Camera.hpp" and which "response.hpp") it gives the compiler error:

   In file included from Camera.hpp:91,
                    from Br.hpp:11,
                    from plugin_main.cpp:38:
   response.hpp:153: error: variable declaration is not allowed here
   response.hpp:219: error: variable declaration is not allowed here

 I don't know what it is, anyhow CPaint headers thus included behind those.
*/

#include <FL/fl_ask.H>          // fl_alert(), fl_message()

#include "TNT/tnt_array2d.hpp"  // TNT::Array2D<..>
#include "Rgb.hpp"              // Rgb<..>
#include "Br.hpp"               // `Br'
#include "BrGUI.hpp"            // `BrGUI'
#include "Application.hpp"      // Application
#include "plugin_main.hpp"      // cpaint_load_image(), cpaint_show_image()
extern "C" {
#include "lib/plugin_main.h"
#include "lib/wire/libtile.h"
#include "plugin_pdb.h"
}

// Debug output for this file? ...
#define YS_DEBUG
#undef YS_DEBUG
#include "YS/ys_dbg.hpp"


// macros (info strings) for the CinePaint plugin...
#define PLUGIN_NAME          "bracketing_to_HDR"
#define PLUGIN_BRIEF         "Merges bracketed exposures into one HDR image."
#define PLUGIN_DESCRIPTION   "Loads bracketed exposures of a (still) scene and merges them into one High Dynamic Range image (radiance map)."
#define PLUGIN_VERSION       "0.4.1 - November 2005"
#define PLUGIN_AUTHOR        "Hartmut Sbosny <Hartmut.Sbosny@gmx.de>"
#define PLUGIN_COPYRIGHT     "Copyright 2005 Hartmut Sbosny"


// to short the name "TNT::Array2D"...
using namespace TNT;

/// Declare some local functions...

void   query     (void);
void   run       (char    *name,
                  int      nparams,
                  GParam  *param,
                  int     *nreturn_vals,
                  GParam **return_vals);

int cpaint_get_image_ID (const char* fname);

template<class T>
Array2D< Rgb<T> > cpaint_get_image (int W, int H, gint32 layer_ID);

template <class T>
void insert_image (Br_Image &,  
                   const Array2D<Rgb<T> > &,
                   Br_ImageVec &,
                   std::vector<Array2D< Rgb<T> > > &);

template <class T> 
void calc_brightness (Br_Image & img, const Array2D< Rgb<T> > & A);

Br_Precision convert_precision (GPrecisionType p);

//template <class T>
//void cpaint_show_image (const Array2D < Rgb<T> >&, GPrecisionType);


/// file global variables... 

GPlugInInfo  PLUG_IN_INFO =     // required by `MAIN()'
{
  NULL,    /* init_proc  */
  NULL,    /* quit_proc  */
  query,   /* query_proc */
  run,     /* run_proc   */
};

typedef struct { char name[1024]; }  Vals;
Vals vals;      // for communication between CPaint and the plugin

static int n_return_vals_;
static int n_args_;


/// Functions...

MAIN()      // defined in "lib/plugin_main.h"; requires a global GPlugInfo-
            //   object namens "PLUG_IN_INFO" (defined just above)   

/**----------------------------------------------------------------------
 * querry()
 *-----------------------------------------------------------------------*/
void
query ()
{
  static GParamDef args [] =
  {
    { PARAM_INT32,      "run_mode",     "Interactive, non-interactive" },
    { PARAM_STRING,     "directory",    "directory" },
  };
  static GParamDef return_vals [] =
  {
    { PARAM_IMAGE,      "image",        "Output Image" },
  };
  n_args_        = sizeof (args) / sizeof (args[0]);
  n_return_vals_ = sizeof (return_vals) / sizeof (return_vals[0]);

  gimp_install_procedure (PLUGIN_NAME,
                          PLUGIN_BRIEF,
                          PLUGIN_DESCRIPTION,
                          PLUGIN_AUTHOR,
                          PLUGIN_COPYRIGHT,
                          PLUGIN_VERSION,
                          "<Toolbox>/File/New From/Bracketing to HDR",
                          "*",
                          PROC_EXTENSION,
                          n_args_, n_return_vals_,
                          args, return_vals);
}

/**----------------------------------------------------------------------
 * run()
 *-----------------------------------------------------------------------*/
void
run (char    *name,             // I - Name of called function resp. plug-in
     int      nparams,          // I - Number of parameters passed in
     GParam  *param,            // I - Parameter values
     int     *nreturn_vals,     // O - Number of return values
     GParam **return_vals)      // O - Return values
{
  GStatusType    status;        // return status
  GRunModeType   run_mode;
  static GParam  values [1];    // return parameter
  int result = -1;
 
  status                  = STATUS_SUCCESS;
  run_mode                = (GRunModeType) param[0].data.d_int32;
  values[0].type          = PARAM_STATUS;
  values[0].data.d_status = STATUS_CALLING_ERROR;

  *nreturn_vals           = n_return_vals_;
  *return_vals            = values;

  if (strcmp (name, PLUGIN_NAME) == 0)
  {
    sprintf (vals.name, getenv("PWD"));     // default for `vals'
    
    gimp_get_data (PLUGIN_NAME, &vals);     // get data from last call
    printf ("nach gimp_get_data(): vals = \"%s\"\n", vals.name);
     
    Application application;      // init the "Brackeing_to_HDR" appl.
      
    switch (run_mode)
    {
    case RUN_INTERACTIVE:
        result = application.run();
        if (result < 0) {
          status = STATUS_EXECUTION_ERROR;
        }
        break;

    case RUN_NONINTERACTIVE:    // Makes this sense? Not realy.
        if (nparams != n_args_) {
          status = STATUS_CALLING_ERROR;
          break;
        }
        result = application.run();  
        if (result < 0) {
          status = STATUS_EXECUTION_ERROR;
          break;
        }
        *nreturn_vals = n_return_vals_ + 1;
        break;
    
    case RUN_WITH_LAST_VALS:
        break;

    default:
        break;
    }
    values[1].type = PARAM_IMAGE;
    values[1].data.d_image = result;
  }
  values[0].data.d_status = status;
  gimp_set_data (PLUGIN_NAME, &vals, sizeof(Vals));  // data for next call
}

/**----------------------------------------------------------------------
 * converting tools... (helpers for cpaint_load_image())
 *-----------------------------------------------------------------------*/
Array2D< Rgb<uint8> > u8_from_u16 (Array2D<Rgb<uint16> > u16)
{
  Array2D< Rgb<uint8> > u8 (u16.dim1(), u16.dim2());
  
  for (int y=0; y < u8.dim1(); y++)
  for (int x=0; x < u8.dim2(); x++)
  {
    u8 [y][x].r = u16 [y][x].r >> 8;  
    u8 [y][x].g = u16 [y][x].g >> 8;  
    u8 [y][x].b = u16 [y][x].b >> 8;  
  }
  return u8;
}

/**----------------------------------------------------------------------
 * cpaint_load_image()  --  Load image `fname' via CinePaint 
 *
 * @param fname: (I) name of file to load
 * @return: Successful or not?
 *
 * Bilddaten werden eingereiht in den globalen vector `Br.dataVec_u8',
 *   die Metainfos in den globalen vector `Br.imgVec'. 
 *
 * Note: Finally CPaint image always to free, so never leave fnct before.
 *-----------------------------------------------------------------------*/
bool 
cpaint_load_image (const char* fname)        
{
  static int            base_W;             // to save the values
  static int            base_H;             //   of the first image
  static gint           base_gray;          // dito
  static GPrecisionType base_precis;        // dito
  static bool           u16_u8_warning = true; // downsampling warning (y/n)
  char                  tmp[256];

  int image_ID = cpaint_get_image_ID (fname);
  if (image_ID < 0) 
    return false;
    
  gint    nlayers  = -1;             // What for?
  gint32* layers   = gimp_image_get_layers (image_ID, &nlayers);
  gint32  layer_ID = layers[0];
  
  GPrecisionType precis = gimp_drawable_precision (layers[0]);
  gint           gray   = gimp_drawable_gray (layers[0]);

  printf ("image has %d layer(s); layer_ID = %d\n", nlayers, layer_ID);
  printf ("precis = %s, gray = %d\n", 
      BR_PRECISION_SHORT_STR[convert_precision(precis)], gray);
  
  int W = gimp_image_width(image_ID);
  int H = gimp_image_height(image_ID);
  
  bool ok = false;
  // Tests for each image, first or not...
  if (nlayers > 1) {
    fl_alert ("More than one layer in image\n\nImage \"%s\" ignored", 
        fl_filename_name(fname));
  } 
  else if (nlayers < 1) {
    fl_alert ("No layer in image\n\nImage \"%s\" ignored", 
        fl_filename_name(fname));
  } 
  else if (layers[0] <= 0) {
    fl_alert ("No data found in image\n\nImage \"%s\" ignored",
        fl_filename_name(fname));
  } 
  else if (gray) {
    fl_alert ("Not an RGB image\n\nImage \"%s\" ignored",
        fl_filename_name(fname));
  }
  else if (! (precis == PRECISION_U8  ||  
              precis == PRECISION_U16) ) {
    fl_alert ("\"%s\":\n\nSorry, data type '%s' not yet supported as input files.\n\nImage \"%s\" ignored",
        fl_filename_name(fname),
        BR_PRECISION_LONG_STR[convert_precision(precis)],
        fl_filename_name(fname));
  } 
  // Now differentiate: first image or following...
  else if (Br.size() == 0)    // first image  
  {  
    base_W = W; 
    base_H = H; 
    base_precis = precis;
    base_gray = gray;
    ok = true;
  }   
  else  // image behind first: check consistence with the first...
  {
    if (W != base_W || H != base_H) {
      fl_alert ("\"%s\":\n\nOther dimension: %d x %d versus %d x %d in the others\n\nImage \"%s\" ignored",
          fl_filename_name(fname), W, H, base_W, base_H,
          fl_filename_name(fname));
    }
    else if ((base_precis != precis) && (precis != PRECISION_U16)) {
      fl_alert ("\"%s\":\n\nDifferent data type: '%s', whereas '%s' in the others\n\nImage \"%s\" ignored",
          fl_filename_name(fname),
          BR_PRECISION_LONG_STR[convert_precision(precis)],
          BR_PRECISION_LONG_STR[convert_precision(base_precis)],
          fl_filename_name(fname));
      ok = true;
    } 
    else if (base_gray != gimp_drawable_gray(layers[0])) { // erledigt oben
      fl_alert ("Other RGB/Gray layout in image\n\nImage \"%s\" ignored",
          fl_filename_name(fname));
    } 
    else {
      ok = true;
    }
  }
  
  // Take over data...
  if (ok)
  {  
    Br_Precision br_prec = convert_precision(precis); 
    
    Br_Image img (fname, W,H, br_prec, 1.0, true);
    img.list();
    
    // Currently we can work up only 8-bit input data and reduce higher data
    //   (uint16) to uint8. To note, that we must not scale down each image
    //   individually onto the full 8-bit range, because this could change the
    //   relation between the various input images. We can only cut "stupidly"
    //   to the 8 leading bits. Therefore also single image converting of float
    //   data to ints makes no sense, because no reasonable min-max-borders
    //   exist. Would make sense indeed for a *complete* set of float images.
    Array2D< Rgb<uint8> > A;
    
    switch (precis)
    {
    case PRECISION_U8:  
        printf ("uint8\n"); 
        A = cpaint_get_image<uint8> (W,H, layer_ID);
        break;
    
    case PRECISION_U16:  
        printf ("uint16\n"); 
        if (u16_u8_warning) 
        {
          snprintf(tmp, 256, "\"%s\":  16-bit input data  --  not yet fully supported\n\nDownsampling to 8-bit will happen",
              fl_filename_name(fname));
          int choice = fl_choice(tmp, "Abbort", "Continue",
                                      "Nicht mehr anzeigen");
          if (choice == 0) {      // Abbort
            ok=false; 
            break; 
          }  
          else if (choice == 2) { // "Do no more warn"
            u16_u8_warning = false;
          }  
        }
        A = u8_from_u16 (cpaint_get_image<uint16> (W,H, layer_ID));
        // Change precision value accordingly the downsampling...
        img.precision = convert_precision(PRECISION_U8);
        if (Br.size()==0) base_precis = PRECISION_U8;
        break;
    
    case PRECISION_FLOAT:  // currently unreachable
        printf ("f32\n");  ok=false;
        break;
    
    default:  
        // Might never be reached if tests above complete
        printf ("\"%s\":  Not supported data type %d: '%s'\n", 
            fl_filename_name(fname), precis,
            BR_PRECISION_LONG_STR[convert_precision(precis)]); 
        ok=false;
    }
    if (ok) 
    {
      insert_image (img, A, Br.imgVec, Br.dataVec_u8);
      Br.imgVec.list();
    }
  } 
  DBG
  // Hiernach und vor naechstem DBG kam bis inkl. CinePaint 0.19.0:
  //   *** glibc detected *** double free or corruption: 0x081ac58b ***
  //   Also gimp_image_delete(image_ID) der Uebeltaeter?
  
  // Free the image in CinePaint...
  gimp_image_delete (image_ID); DBG
  gimp_image_clean_all (image_ID);
  free (layers); 
  
  return ok;
}

/**----------------------------------------------------------------------
 * cpaint_get_image_ID()  --  Helper for cpaint_load_image()
 *
 * Asks CPaint for loading the file `fname' and returns (only) its image_ID.
 *   Function wraps the `return_vals' stuff.
 *
 * @param fname: name of file the CPaint image_ID is asked for
 * @return: image_ID; -1 if loading failed.
 *-----------------------------------------------------------------------*/
int cpaint_get_image_ID (const char* fname)
{
  int image_ID;
  int n_retvals;
  GimpParam* return_vals;
  
  return_vals = gimp_run_procedure ("gimp_file_load",
                                    &n_retvals,
                                    GIMP_PDB_INT32, GIMP_RUN_INTERACTIVE,
                                    GIMP_PDB_STRING, fname,
                                    GIMP_PDB_STRING, fname,
                                    GIMP_PDB_END);

  if (return_vals[0].data.d_status != GIMP_PDB_SUCCESS)
    image_ID = -1;
  else
    image_ID = return_vals[1].data.d_image;
  
  gimp_destroy_params (return_vals, n_retvals);
  return image_ID;
}

/**----------------------------------------------------------------------
 * cpaint_get_image()  --  Helper for cpaint_load_image()
 *
 * Allocates an Rgb<T>-Array2D and fills in the CPaint buffer related to
 *   `layer_ID', which must be of the same data type <T>!
 *   A possible alpha channel gets removed.
 *
 * @param <T>: type of the target AND the source elements
 * @param W,H: width, height
 * @param layer_ID: as got in cpaint_load_image()
 * @return: allocated and filled Array2D< Rgb<T> >
 *-----------------------------------------------------------------------*/
template <class T>
Array2D< Rgb<T> > cpaint_get_image (int W, int H, gint32 layer_ID)
{
  // RGB-Zielstruktur allokieren; bad_alloc() possibly!
  Array2D< Rgb<T> > A (H,W);            // dim1 = Height, dim2 = Width!

  GDrawable* drawable = gimp_drawable_get (layer_ID);
  GPixelRgn  pixel_rgn;
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0,0, W,H, false,false);

  if (gimp_drawable_has_alpha(layer_ID))      // Umkopieren
  {
    TRACE(("image has alpha channel\n"));
    Rgba<T>* buf = (Rgba<T>*) calloc (W * H, sizeof(Rgba<T>));

    gimp_pixel_rgn_get_rect (&pixel_rgn, (guchar*)buf, 0,0, W,H);

    for (int y=0; y < H; y++)
    for (int x=0; x < W; x++)
    { 
      int wo = y * W + x;
      A [y][x].r = buf[wo].r;     // ginge effektiver, aber Optimierer
      A [y][x].g = buf[wo].g;     //   wird's schon richten
      A [y][x].b = buf[wo].b;    
    }
    free (buf);
  }
  else                            // kein alpha: direkt einschreiben
  {
    gimp_pixel_rgn_get_rect (&pixel_rgn, (guchar*)A[0], 0,0, W,H); DBG
  }
  
  return A;
}


/**----------------------------------------------------------------------
 * insert_image()  --  Helper for cpaint_load_image() 
 *
 * Calculates brightness, and inserts `img' and 'A' at the according
 *   position in their vectors.
 *
 * @param img: image infos of the loaded image
 * @param A: Array2D-data field of the loaded image
 * @param imgVec: std::vector, therein `img' is to insert
 * @param arrVec: std:vector, therein `A' is to insert
 *-----------------------------------------------------------------------*/
template <class T>
void insert_image(Br_Image & img,  
                  const Array2D<Rgb<T> > & A,
                  Br_ImageVec & imgVec,
                  std::vector<Array2D< Rgb<T> > > & arrVec)
{
  calc_brightness (img, A);
  
  for (unsigned i=0; i < imgVec.size(); i++)
  {  
    if (img.brightness < imgVec[i].brightness)
    { 
      imgVec.insert(imgVec.begin()+i, img);
      arrVec.insert(arrVec.begin()+i, A);
      return;
    }
  } 
  // no image had smaller brightness than `img': add it back...
  imgVec.push_back(img);
  arrVec.push_back(A);
}

/**----------------------------------------------------------------------
 * cpaint_show_image()  --  Plot eines untypisierten Bildpuffers ohne alpha
 *                                                      mittels CPaint.
 * @param width,height: Bilddimensionen
 * @param imgbuf: lineares Feld der Bilddaten (untypisiert)
 * @param prec: Datentyp (Konstante aus "cinepaint??/lib/wire/enums.h"
 *
 * Gedacht fuer CinePaint-Plot eines Array2D< Rgb<T> >-Feldes ohne alpha.
 *   Argument ist indes kein Array2D, sondern ein untypisierter Datenpuffer
 *   aus zwei Gruenden: 1) Die Datentyp-Information kann ohnehin nicht T
 *   entnommen werden, sondern ist per `prec' mitzuteilen. 2) So ist es
 *   wenigstesns auch kein Template. -- Bsp. eines Aufrufs:
 * 
 *     cpaint_show_image(A.dim2(), A.dim1(), (uchar*)A[0], PRECISION_FLOAT);
 * 
 * Vorausgesetzt wird, dass urspruengliches Array2D `A' *zusammenhaengend*, dh
 *   keiner `TNT::SubArray'-Aktion entstammt!
 *-----------------------------------------------------------------------*/
void 
cpaint_show_image (int width, int height, const uchar* imgbuf,
                   GPrecisionType prec) 
{
  gint32        image_ID, layer_ID;
  GImageType    image_type;
  GDrawableType drawable_type;

  switch (prec) 
  {
  case PRECISION_U8:                        // 8-bit unigned int
      image_type    = RGB; 
      drawable_type = RGB_IMAGE; break;
  
  case PRECISION_U16:     
      image_type    = U16_RGB; 
      drawable_type = U16_RGB_IMAGE; break;
  
  case PRECISION_FLOAT:                     // 32-bit float
      image_type    = FLOAT_RGB; 
      drawable_type = FLOAT_RGB_IMAGE; break;
  
  case PRECISION_FLOAT16:
      image_type    = FLOAT16_RGB; 
      drawable_type = FLOAT16_RGB_IMAGE; break;
    
  default:
      fl_alert ("%s: Precision %d ('%s') not supported\n",
          PLUGIN_NAME, prec, 
          BR_PRECISION_LONG_STR [convert_precision(prec)]);
      return;
  }       
  
  image_ID = gimp_image_new ((guint)width, (guint)height, image_type);

  if (image_ID == -1) {
    fl_alert ("%s: Can't create a new image\n%d x %d, image_type %d",
        PLUGIN_NAME, width, height, image_type);
    return;
  }

  layer_ID = gimp_layer_new (image_ID,
                             "HDR bracketing",
                             (guint)width, (guint)height,
                             drawable_type,
                             100.0,          // opacity
                             NORMAL_MODE );

  gimp_image_add_layer (image_ID, layer_ID, 0);   // Wozu?

  GDrawable* drawable = gimp_drawable_get (layer_ID);

  GPixelRgn  pixel_rgn;
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0,0, width, height, false,false);

  gimp_pixel_rgn_set_rect (&pixel_rgn, (guchar*)imgbuf, 0,0, width, height);

  gimp_drawable_flush  (drawable);
  gimp_drawable_detach (drawable);
  gimp_display_new (image_ID);
}

/**----------------------------------------------------------------------
 * brightness()  -  Helligkeit bestimmen.
 *
 * Moegl. waere auch, Kanaele zunaechst getrennt und dann Plausibilitaets-Check.
 * Summe mittels uint64-Typ statt double ist genauer, aber nicht schneller,
 * wie Test ergab. Fuer uint-Summe waeren freilich nur uint-Typen geeignet,
 * keine ints und keine floats. Templateparameter sollte dann "Unsign" heissen.
 * 
 * Ergaenzt um: Wieviel Punkte im Arbeitsbereich ("AB")? Dabei gilt (vorerst)
 *   ein Punkt als nicht mehr im AB, sobald *ein* Kanal ausserhalb.
 *-----------------------------------------------------------------------*/
template <class T> 
void calc_brightness (Br_Image & img, const Array2D< Rgb<T> > & A)
{
  Rgb<double> s(0);
  //Rgb<uint64> s(0);   // uint64 statt double beschleunigt nicht

  for (int i=0; i < A.dim1(); i++)
  for (int j=0; j < A.dim2(); j++)
  {
    // brightness...
    s += A[i][j];
    
    // working range...
    if (A[i][j].r == 0 || A[i][j].g == 0 || A[i][j].b == 0) {  
      img.n_low++; 
    }
    else if (A[i][j].r==255 || A[i][j].g==255 || A[i][j].b==255) {
      img.n_high++;    // TODO: so nur fuer 8-Bit!!
    }
  }
  img.brightness = double(s.r + s.g + s.b) / (A.dim1() * A.dim2());
  img.r_wrange = 1.0 - double(img.n_low + img.n_high) / (A.dim1() * A.dim2());
}

/**------------------------------------------
 * convert GPrecisionType in Br_Precision...
 *-------------------------------------------*/
Br_Precision convert_precision (GPrecisionType p)
{
  switch (p)
  {
  case PRECISION_NONE:    return BR_PRECISION_NONE;
  case PRECISION_U8:      return BR_PRECISION_U8; 
  case PRECISION_U16:     return BR_PRECISION_U16; 
  case PRECISION_FLOAT:   return BR_PRECISION_FL32; 
  case PRECISION_FLOAT16: return BR_PRECISION_FL16; 
  case PRECISION_BFP:     return BR_PRECISION_FIX16;
  default:                return BR_PRECISION_UNKNOWN;
  }
}


// END OF FILE
