// Copyright (c) 2009 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#import "chrome/browser/cocoa/restart_browser.h"

#include "app/l10n_util_mac.h"
#include "chrome/browser/browser_list.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/pref_service.h"
#include "chrome/common/pref_names.h"
#include "grit/chromium_strings.h"
#include "grit/generated_resources.h"
#include "grit/app_strings.h"

// Helper to clean up after the notification that the alert was dismissed.
@interface RestartHelper : NSObject {
 @private
  NSAlert* alert_;
}
- (NSAlert*)alert;
- (void)alertDidEnd:(NSAlert*)alert
         returnCode:(int)returnCode
        contextInfo:(void*)contextInfo;
@end

@implementation RestartHelper

- (NSAlert*)alert {
  alert_ = [[NSAlert alloc] init];
  return alert_;
}

- (void)dealloc {
  [alert_ release];
  [super dealloc];
}

- (void)alertDidEnd:(NSAlert*)alert
         returnCode:(int)returnCode
        contextInfo:(void*)contextInfo {
  if (returnCode == NSAlertFirstButtonReturn) {
    // Nothing to do. User will restart later.
  } else if (returnCode == NSAlertSecondButtonReturn) {
    // Set the flag to restore state after the restart.
    PrefService* pref_service = g_browser_process->local_state();
    pref_service->SetBoolean(prefs::kRestartLastSessionOnShutdown, true);
    BrowserList::CloseAllBrowsersAndExit();
  } else {
    NOTREACHED();
  }
  [self autorelease];
}

@end

namespace restart_browser {

void RequestRestart(NSWindow* parent) {
  NSString* title =
      l10n_util::GetNSStringFWithFixup(IDS_PLEASE_RESTART_BROWSER,
          l10n_util::GetStringUTF16(IDS_PRODUCT_NAME));
  NSString* text =
      l10n_util::GetNSStringFWithFixup(IDS_UPDATE_RECOMMENDED,
          l10n_util::GetStringUTF16(IDS_PRODUCT_NAME));
  NSString* notNowButtin = l10n_util::GetNSStringWithFixup(IDS_NOT_NOW);
  NSString* restartButton =
      l10n_util::GetNSStringWithFixup(IDS_RESTART_AND_UPDATE);

  RestartHelper* helper = [[RestartHelper alloc] init];

  NSAlert* alert = [helper alert];
  [alert setAlertStyle:NSInformationalAlertStyle];
  [alert setMessageText:title];
  [alert setInformativeText:text];
  [alert addButtonWithTitle:notNowButtin];
  [alert addButtonWithTitle:restartButton];

  [alert beginSheetModalForWindow:parent
                    modalDelegate:helper
                   didEndSelector:@selector(alertDidEnd:returnCode:contextInfo:)
                      contextInfo:nil];
}

}  // namespace restart_browser
