/*
 * display outputs, per line:
 *
 *   Remaining:	| Empty:	| Degraded:
 *     blue	|  black	|  dimgrey	discharging
 *     green	|  black	|  dimgrey	charging
 *     cyan	|  black	|  dimgrey	charged
 *     grey	|  black	|  dimgrey	charging&discharching!
 *     blue	|  red  	|  dimgrey	discharging - low!
 *     green	|  red  	|  dimgrey	charging - low
 *     cyan	|  red  	|  dimgrey	charged - low [1]
 *     grey	|  red  	|  dimgrey	charging&discharching, low [1]
 *       ...  darkgreen  ...			no batteries present
 *       ...  yellow  ...			error
 *
 * [1] battery must be quite badly degraded
 */

#include <stdio.h>
#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/poll.h>
#include <sys/types.h>
#include <dirent.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>

#define TOP      60
#define BOTTOM 3600

#define TIMEOUT         5000 /* milliseconds */
#define TIMEOUT_ONERROR 3333 /* milliseconds */

static const char program_name[]= "xacpi-simple";

/*---------- general utility stuff and declarations ----------*/

static void fail(const char *m) {
  fprintf(stderr,"error: %s\n", m);
  exit(-1);
}
static void badusage(void) { fail("bad usage"); }

#define CHGST_DISCHARGING 0 /* Reflects order in E(state,charging_state)  */
#define CHGST_CHARGING    1 /*  in fields table.  Also, much code assumes */
#define CHGST_CHARGED     2 /*  exactly these three possible states.      */
#define CHGST_ERROR       8 /* Except that this one is an extra bit.      */

/*---------- structure of and results from /proc/acpi/battery/... ----------*/
/* variables thisbat_... are the results from readbattery();
 * if readbattery() succeeds they are all valid and not VAL_NOTFOUND
 */

typedef struct batinfo_field {
  const char *file;
  const char *label;
  unsigned long *valuep;
  const char *unit;
  const char *enumarray[10];
} batinfo_field;

#define QUANTITY_FIELDS				\
  QF(info,  design_capacity,    "mWh")		\
  QF(info,  last_full_capacity, "mWh")		\
  QF(state, present_rate,       "mW")		\
  QF(state, remaining_capacity, "mWh")		\
  QF(alarm, alarm,              "mWh")

#define QF(f,l,u) static unsigned long thisbat_##f##_##l;
  QUANTITY_FIELDS
#undef QF

static unsigned long thisbat_alarm_present, thisbat_info_present;
static unsigned long thisbat_state_present, thisbat_state_charging_state;

#define VAL_NOTFOUND (~0UL)

static const batinfo_field fields[]= {
#define E(f,l)       #f, #l, &thisbat_##f##_##l, 0
#define QF(f,l,u)  { #f, #l, &thisbat_##f##_##l, u },
  { E(alarm, present),           { "no", "yes" }                          },
  { E(info, present),            { "no", "yes" }                          },
  { E(state,present),            { "no", "yes" }                          },
  { E(state,charging_state),     { "discharging", "charging", "charged" } },
  QUANTITY_FIELDS           /* take care re charging_state values order -  */
  { 0 }                     /* if you must change it, search for CHGST_... */
#undef E
#undef QF
};

static const char *files[]= { "info", "state", "alarm" };

/*---------- parsing of one battery in /proc/acpi/battery/... ----------*/

/* variables private to the parser and its error handlers */
static char batlinebuf[1000];
static FILE *batfile;
static const char *batdirname;
static const char *batfilename;
static const char *batlinevalue;

static int batfailf(const char *why) {
  if (batlinevalue) {
    fprintf(stderr,"battery/%s/%s: %s value `%s': %s\n",
	    batdirname,batfilename, batlinebuf,batlinevalue,why);
  } else {
    fprintf(stderr,"battery/%s/%s: %s: `%s'\n",
	    batdirname,batfilename, why, batlinebuf);
  }
  return -1;
}

static int batfaile(const char *syscall, const char *target) {
  fprintf(stderr,"battery/%s: failed to %s %s: %s\n",
	  batdirname ? batdirname : "*", syscall, target, strerror(errno));
  return -1;
}

static void chdir_base(void) {
  int r;
  
  r= chdir("/proc/acpi/battery");
  if (r) batfaile("chdir","/proc/acpi/battery");
}

static void tidybattery(void) {
  if (batfile) { fclose(batfile); batfile=0; }
  if (batdirname) { chdir_base(); batdirname=0; }
}

static int readbattery(void) { /* 0=>ok, -1=>couldn't */
  const batinfo_field *field;
  const char *const *cfilename, *const *enumsearch;
  char *colon, *ep, *sr, *p;
  int r, l, missing;
  
  r= chdir(batdirname);
  if (r) return batfaile("chdir",batdirname);

  for (field=fields; field->file; field++)
    *field->valuep= VAL_NOTFOUND;

  for (cfilename=files;
       (batfilename= *cfilename);
       cfilename++) {
    batfile= fopen(batfilename,"r");
    if (!batfile) return batfaile("open",batfilename);

    for (;;) {
      batlinevalue= 0;
      
      sr= fgets(batlinebuf,sizeof(batlinebuf),batfile);
      if (ferror(batfile)) return batfaile("read",batfilename);
      if (!sr && feof(batfile)) break;
      l= strlen(batlinebuf);
      assert(l>0);
      if (batlinebuf[l-1] != '\n')
	return batfailf("line too long");
      batlinebuf[l-1]= 0;
      colon= strchr(batlinebuf,':');
      if (!colon)
	return batfailf("line without a colon");
      *colon= 0;

      for (batlinevalue= colon+1;
	   *batlinevalue && isspace((unsigned char)*batlinevalue);
	   batlinevalue++);

      if (!strcmp(batlinebuf,"ERROR"))
	return batfailf("kernel reports error");

      for (p=batlinebuf; p<colon; p++)
	if (*p == ' ')
	  *p= '_';

      for (field=fields; field->file; field++) {
	if (!strcmp(field->file,batfilename) &&
	    !strcmp(field->label,batlinebuf))
	  goto label_interesting;
      }
      continue;

    label_interesting:
      if (field->unit) {

	*field->valuep= strtoul(batlinevalue,&ep,10);
	if (ep==batlinevalue || *ep!=' ')
	  batfailf("value number syntax incorrect");
	if (strcmp(ep+1,field->unit)) batfailf("incorrect unit");

      } else {
	
	for (*field->valuep=0, enumsearch=field->enumarray;
	     *enumsearch && strcmp(*enumsearch,batlinevalue);
	     (*field->valuep)++, enumsearch++);
	if (!*enumsearch)
	  batfailf("unknown enum value");

      }
    }

    fclose(batfile);
    batfile= 0;
  }

  if (!(thisbat_alarm_present==0 ||
	thisbat_info_present==0 ||
	thisbat_state_present==0)) {
    if (thisbat_alarm_present == VAL_NOTFOUND)
      thisbat_alarm_present= 1;
    
    for (field=fields, missing=0;
	 field->file;
	 field++) {
      if (*field->valuep == VAL_NOTFOUND) {
	fprintf(stderr,"battery/%s/%s: %s: not found\n",
		batdirname, field->file,field->label);
	missing++;
      }
    }
    if (missing) return -1;
  }

  r= chdir("..");
  if (r) return batfaile("chdir","..");
  batdirname= 0;

  return 0;
}   

/*---------- data collection and analysis ----------*/

/* These next three variables are the results of the charging state */
static unsigned charging_state_mask; /* 1u<<CHGST_* | ... */
static double nondegraded_norm, fill_norm, ratepersec_norm;
static int alarm_level; /* 0=ok, 1=low */

#define QF(f,l,u) \
  static double total_##f##_##l;
  QUANTITY_FIELDS
#undef QF

static void acquiredata(void) {
  DIR *di;
  struct dirent *de;
  int r;
  
  charging_state_mask= 0;

#define QF(f,l,u) \
  total_##f##_##l= 0;
  QUANTITY_FIELDS
#undef QF

  di= opendir(".");  if (!di) batfaile("opendir","battery");
  while ((de= readdir(di))) {
    if (de->d_name[0]==0 || de->d_name[0]=='.') continue;

    batdirname= de->d_name;
    r= readbattery();
    tidybattery();

    if (r) { charging_state_mask |= CHGST_ERROR; continue; }

    if (!thisbat_info_present || !thisbat_state_present)
      continue;

    charging_state_mask |= 1u << thisbat_state_charging_state;

#define QF(f,l,u) \
    total_##f##_##l += thisbat_##f##_##l;
    QUANTITY_FIELDS
#undef QF

    if (thisbat_state_charging_state == CHGST_DISCHARGING)
      /* negate it */
      total_state_present_rate -= 2.0 * thisbat_state_present_rate;
      
  }
  closedir(di);

  if (total_info_design_capacity < 0.5)
    total_info_design_capacity= 1.0;

  if (total_info_last_full_capacity < total_state_remaining_capacity)
    total_info_last_full_capacity= total_state_remaining_capacity;
  if (total_info_design_capacity < total_info_last_full_capacity)
    total_info_design_capacity= total_info_last_full_capacity;

  alarm_level=
    (total_state_remaining_capacity < total_alarm_alarm &&
     !(charging_state_mask & 1u << CHGST_CHARGING));

  nondegraded_norm= total_info_last_full_capacity / total_info_design_capacity;
  fill_norm= total_state_remaining_capacity / total_info_design_capacity;
  ratepersec_norm=  total_state_present_rate
    / (3600.0 * total_info_design_capacity);
}

static void initacquire(void) {
  chdir_base();
}  

/*---------- argument parsing ----------*/

#define COLOURS					\
  C(blue,      discharging)			\
  C(green,     charging)			\
  C(cyan,      charged)				\
  C(grey,      confusing)			\
  C(black,     normal)				\
  C(red,       low)				\
  C(dimgrey,   degraded)			\
  C(darkgreen, absent)				\
  C(yellow,    error)				\
  C(white,     equilibrium)			\
  GC(remain)					\
  GC(white)					\
  GC(empty)

static XrmDatabase xrm;
static Display *disp;
static int screen;

static const char defaultresources[]=
#define GC(g)
#define C(c,u)					\
  "*" #u "Color: " #c "\n"
  COLOURS
#undef GC
#undef C
  ;

#define S(s) ((char*)(s))
static const XrmOptionDescRec optiontable[]= {
  { S("-display"),      S("*display"),      XrmoptionSepArg },
  { S("-geometry"),     S("*geometry"),     XrmoptionSepArg },
#define GC(g)
#define C(c,u)							\
  { S("-" #u "Color"),  S("*" #u "Color"),  XrmoptionSepArg },	\
  { S("-" #u "Colour"), S("*" #u "Color"),  XrmoptionSepArg },
  COLOURS
#undef GC
#undef C
};

static const char *getresource(const char *want) {
  char name_buf[256], class_buf[256];
  XrmValue val;
  char *rep_type_dummy;
  int r;

  assert(strlen(want) < 128);
  sprintf(name_buf,"xacpi-simple.%s",want);
  sprintf(class_buf,"Xacpi-Simple.%s",want);
  
  r= XrmGetResource(xrm, name_buf,class_buf, &rep_type_dummy, &val);
  if (!r) return 0;
  
  return val.addr;
}

static void more_resources(const char *str, const char *why) {
  XrmDatabase more;

  if (!str) return;

  more= XrmGetStringDatabase((char*)str);
  if (!more) fail(why);
  XrmCombineDatabase(more,&xrm,0);
}

static void parseargs(int argc, char **argv) {
  Screen *screenscreen;
  
  XrmInitialize();

  XrmParseCommand(&xrm, (XrmOptionDescRec*)optiontable,
		  sizeof(optiontable)/sizeof(*optiontable),
		  program_name, &argc, argv);

  if (argc>1) badusage();

  disp= XOpenDisplay(getresource("display"));
  if (!disp) fail("could not open display");

  screen= DefaultScreen(disp);

  screenscreen= ScreenOfDisplay(disp,screen);
  if (!screenscreen) fail("screenofdisplay");
  more_resources(XScreenResourceString(screenscreen), "screen resources");
  more_resources(XResourceManagerString(disp), "display resources");
  more_resources(defaultresources, "default resources");
} 

/*---------- display ----------*/

static Window win;
static int width, height;
static Colormap cmap;
static unsigned long lastbackground;

typedef struct {
  GC gc;
  unsigned long lastfg;
} Gcstate;

#define C(c,u) static unsigned long pix_##c;
#define GC(g) static Gcstate gc_##g;
  COLOURS
#undef C
#undef GC

static void refresh(void);

#define CHGMASK_CHG_DIS (1u<<CHGST_CHARGING | 1u<<CHGST_DISCHARGING)

static void failr(const char *m, int r) {
  fprintf(stderr,"error: %s (code %d)\n", m, r);
  exit(-1);
}

static void setbackground(unsigned long newbg) {
  int r;
  
  if (newbg == lastbackground) return;
  r= XSetWindowBackground(disp,win,newbg);
  if (!r) fail("XSetWindowBackground");
  lastbackground= newbg;
}

static void setforeground(Gcstate *g, unsigned long px) {
  XGCValues gcv;
  int r;
  
  if (g->lastfg == px) return;
  
  memset(&gcv,0,sizeof(gcv));
  g->lastfg= gcv.foreground= px;
  r= XChangeGC(disp,g->gc,GCForeground,&gcv);
  if (!r) fail("XChangeGC");
}

static void show_solid(unsigned long px) {
  setbackground(px);
  XClearWindow(disp,win);
}

static void show(void) {
  double elap, then;
  int i, leftmost_lit, leftmost_nondeg, beyond, first_beyond;

  if (!charging_state_mask)
    return show_solid(pix_darkgreen);

  if (charging_state_mask & CHGST_ERROR)
    return show_solid(pix_yellow);

  setbackground(pix_dimgrey);
  XClearWindow(disp,win);
  
  setforeground(&gc_remain,
		!(charging_state_mask & CHGMASK_CHG_DIS) ? pix_cyan :
		!(~charging_state_mask & CHGMASK_CHG_DIS) ? pix_grey :
		charging_state_mask & (1u<<CHGST_CHARGING)
		? pix_green : pix_blue);
		
  setforeground(&gc_empty, alarm_level ? pix_red : pix_black);

  for (i=0, first_beyond=1; i<height; i++) {
    elap= !i ? 0 :
      height==2 ? BOTTOM :
      TOP * exp( (double)i / (height-2) * log( (double)BOTTOM/TOP ) );
    
    then= fill_norm + ratepersec_norm * elap;

    beyond=
      ((charging_state_mask & (1u<<CHGST_DISCHARGING) && then <= 0.0) ||
       (charging_state_mask & (1u<<CHGST_CHARGING) && then>=nondegraded_norm));

    if (then <= 0.0) then= 0.0;
    else if (then >= nondegraded_norm) then= nondegraded_norm;

    leftmost_lit= width * then;
    leftmost_nondeg= width * nondegraded_norm;

    if (beyond && first_beyond) {
      XDrawLine(disp, win, gc_white.gc, 0,i, leftmost_nondeg,i);
      first_beyond= 0;
    } else {
      if (leftmost_lit < leftmost_nondeg)
	XDrawLine(disp, win, gc_empty.gc,
		  leftmost_lit,i, leftmost_nondeg,i);
      if (leftmost_lit >= 0)
	XDrawLine(disp, win, gc_remain.gc, 0,i, leftmost_lit,i);
    }
  }
}

static void initgc(Gcstate *gc_r) {
  XGCValues gcv;

  memset(&gcv,0,sizeof(gcv));
  gcv.function= GXcopy;
  gcv.line_width= 1;
  gc_r->lastfg= gcv.foreground= pix_white;
  gc_r->gc= XCreateGC(disp,win, GCFunction|GCLineWidth|GCForeground, &gcv);
}

static void colour(unsigned long *pix_r, const char *whichcolour) {
  XColor xc;
  const char *name;
  Status st;

  name= getresource(whichcolour);
  if (!name) fail("get colour resource");
  
  st= XAllocNamedColor(disp,cmap,name,&xc,&xc);
  if (!st) fail(name);
  
  *pix_r= xc.pixel;
}

static void initgraphics(int argc, char **argv) {
  int xwmgr, r;
  const char *geom_string;
  XSizeHints *normal_hints;
  XWMHints *wm_hints;
  XClassHint *class_hint;
  int pos_x, pos_y, gravity;
  char *program_name_silly;
  
  program_name_silly= (char*)program_name;

  normal_hints= XAllocSizeHints();
  wm_hints= XAllocWMHints();
  class_hint= XAllocClassHint();

  if (!normal_hints || !wm_hints || !class_hint)
    fail("could not alloc hint(s)");

  geom_string= getresource("geometry");

  xwmgr= XWMGeometry(disp,screen, geom_string,"128x32", 0,
		 normal_hints,
		 &pos_x, &pos_y,
		 &width, &height,
		 &gravity);

  win= XCreateSimpleWindow(disp,DefaultRootWindow(disp),
			   pos_x,pos_y,width,height,0,0,0);
  cmap= DefaultColormap(disp,screen);
  
#define C(c,u) colour(&pix_##c, #u "Color");
#define GC(g) initgc(&gc_##g);
  COLOURS
#undef C
#undef GC

  r= XSetWindowBackground(disp,win,pix_dimgrey);
  if (!r) fail("init set background");
  lastbackground= pix_dimgrey;

  normal_hints->flags= PWinGravity;
  normal_hints->win_gravity= gravity;
  normal_hints->x= pos_x;
  normal_hints->y= pos_y;
  normal_hints->width= width;
  normal_hints->height= height;
  if ((xwmgr & XValue) || (xwmgr & YValue))
    normal_hints->flags |= USPosition;

  wm_hints->flags= InputHint;
  wm_hints->input= False;

  class_hint->res_name= program_name_silly;
  class_hint->res_class= program_name_silly;

  XmbSetWMProperties(disp,win, program_name,program_name,
		     argv,argc, normal_hints, wm_hints, class_hint);

  XSelectInput(disp,win, ExposureMask|StructureNotifyMask);
  XMapWindow(disp,win);
}
 
static void refresh(void) {
  acquiredata();
  show();
}

static void newgeometry(void) {
  int dummy;
  Window dummyw;
  
  XGetGeometry(disp,win, &dummyw,&dummy,&dummy, &width,&height, &dummy,&dummy);
}

static void eventloop(void) {
  XEvent ev;
  struct pollfd pfd;
  int r, timeout;
  
  newgeometry();
  refresh();

  for (;;) {
    XFlush(disp);

    pfd.fd= ConnectionNumber(disp);
    pfd.events= POLLIN|POLLERR;

    timeout= !(charging_state_mask & CHGST_ERROR) ? TIMEOUT : TIMEOUT_ONERROR;
    r= poll(&pfd,1,timeout);
    if (r==-1 && errno!=EINTR) failr("poll",errno);

    while (XPending(disp)) {
      XNextEvent(disp,&ev);
      if (ev.type == ConfigureNotify) {
	XConfigureEvent *ce= (void*)&ev;
	width= ce->width;
	height= ce->height;
      }
    }
    refresh();
  }
}

int main(int argc, char **argv) {
  parseargs(argc,argv);
  initacquire();
  initgraphics(argc,argv);
  eventloop();
  return 0;
}
