/*
  Copyright 2021 Northern.tech AS

  This file is part of CFEngine 3 - written and maintained by Northern.tech AS.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; version 3.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

  To the extent this program is licensed as part of the Enterprise
  versions of CFEngine, the applicable Commercial Open Source License
  (COSL) may apply to this file if you as a licensee so wish it. See
  included file COSL.txt.
*/

#ifndef CFENGINE_RLIST_H
#define CFENGINE_RLIST_H

#include <cf3.defs.h>
#include <writer.h>
#include <json.h>
#include <fncall.h>

/* NOTE: an empty Rlist is simply NULL. */
struct Rlist_
{
    Rval val;
    Rlist *next;
};

RvalType DataTypeToRvalType(DataType datatype);

bool RlistValueIsType(const Rlist *rlist, RvalType type);
char *RvalScalarValue(Rval rval);
FnCall *RvalFnCallValue(Rval rval);
Rlist *RvalRlistValue(Rval rval);
JsonElement *RvalContainerValue(Rval rval);

const char *RvalTypeToString(RvalType type);

Rval RvalNew(const void *item, RvalType type);

/**
 * Get new secret Rval.
 *
 * @note RvalDestroy() not required to be called on the returned value.
 */
Rval RvalNewSecret();

Rval RvalNewRewriter(const void *item, RvalType type, JsonElement *map);
Rval RvalCopy(Rval rval);
Rval RvalCopyRewriter(Rval rval, JsonElement *map);
void RvalDestroy(Rval rval);
JsonElement *RvalToJson(Rval rval);
char *RvalToString(Rval rval);
char *RlistToString(const Rlist *rlist);
void RvalWrite(Writer *writer, Rval rval);
void RvalWriteQuoted(Writer *writer, Rval rval);
unsigned RvalHash(Rval rval, unsigned seed);

Rlist *RlistCopy(const Rlist *list);
Rlist *RlistCopyRewriter(const Rlist *list, JsonElement *map);
unsigned int RlistHash        (const Rlist *list, unsigned seed);
unsigned int RlistHash_untyped(const void *list, unsigned seed);
void RlistDestroy        (Rlist *list);
void RlistDestroy_untyped(void *rl);
void RlistDestroyEntry(Rlist **liststart, Rlist *entry);
char *RlistScalarValue(const Rlist *rlist);
char *RlistScalarValueSafe(const Rlist *rlist);
FnCall *RlistFnCallValue(const Rlist *rlist);
Rlist *RlistRlistValue(const Rlist *rlist);
Rlist *RlistParseShown(const char *string);
Rlist *RlistParseString(const char *string);
Rlist *RlistKeyIn(Rlist *list, const char *key);
int RlistLen(const Rlist *start);
bool RlistMatchesRegexRlist(const Rlist *list, const Rlist *search);
bool RlistMatchesRegex(const Rlist *list, const char *str);
bool RlistIsInListOfRegex(const Rlist *list, const char *str);
bool RlistIsNullList(const Rlist *list);
bool RlistContainsString(const Rlist *list, const char *string);

Rlist *RlistAppendRval(Rlist **start, Rval rval);

Rlist *RlistPrependScalarIdemp(Rlist **start, const char *scalar);
Rlist *RlistAppendScalarIdemp(Rlist **start, const char *scalar);
Rlist *RlistAppendScalar(Rlist **start, const char *scalar);

Rlist *RlistPrepend(Rlist **start, const void *item, RvalType type);
Rlist *RlistAppend(Rlist **start, const void *item, RvalType type);
Rlist *RlistAppendAllTypes(Rlist **start, const void *item, RvalType type, bool all_types);
Rlist *RlistAppendString(Rlist **start, const char *string);

Rlist *RlistFromSplitString(const char *string, char sep);
Rlist *RlistFromStringSplitLines(const char *string, bool detect_crlf);
Rlist *RlistFromSplitRegex(const char *string, const char *regex, size_t max_entries, bool allow_blanks);
Rlist *RlistFromRegexSplitNoOverflow(const char *string, const char *regex, int max);
Rlist *RlistFromContainer(const JsonElement *container);

void RlistWrite(Writer *writer, const Rlist *list);
Rlist *RlistLast(Rlist *start);
void RlistFilter(Rlist **list, bool (*KeepPredicate)(void *item, void *predicate_data), void *predicate_user_data, void (*DestroyItem)(void *item));
void RlistReverse(Rlist **list);
void ScalarWrite(Writer *w, const char *s, bool quote);
void RlistFlatten(EvalContext *ctx, Rlist **list);
bool RlistEqual        (const Rlist *list1, const Rlist *list2);
bool RlistEqual_untyped(const void *list1, const void *list2);


#endif
