/* cdbackup.c
Copyright (c) 2000-2002 Craig Condit, Stefan Hlswitt.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met: 

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer. 
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution. 

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.
*/

#define _LARGEFILE64_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <getopt.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <linux/cdrom.h>

#include "cdbackup.h"
#include "cdrom.h"
#include "misc.h"
#include "version.h"

/* #define DEBUGOUT */

/* defaults */
char * prg_name ="cdbackup";
char * cd_dev   ="/dev/burner";
char * cdr_dev  =0;			/* no default here, too dangerous */
char * cd_label ="CDBackup Track";
int    cd_speed =4;
long   cd_len   =333000;		/* blocks */
int    padsize  =15;			/* blocks */
int    multidisk=0;
char * multicmd =0;
int    verbose  =0;
int    xamode2  =0;

char **cdrec_opt=0;
int    cdrec_opt_count=0;

long long totalSize;
struct tm curtime;  /* global, so multi-disks get all the same time */

/****************************************************************************/

void usage()
{
  fprintf(stderr,
    "Usage: %s [options ...] [-- cdrecord-options ...]\n"
    "Reads from standard input, block formats and writes to CD-R(W).\n\n"
    "  -d DEVICE      DEVICE for CD queries (default /dev/burner)\n"
    "  -l N           CD-R has a size of N MB (default 650)\n"
    "  -r DEVICE      DEVICE for CD recording (e.g. 0,4,0)\n"
    "  -s N           record CD at speed N (default 4)\n"
    "  -X             enable CDROM XA2 mode in cdrecord\n"
    "  -a LABEL       use LABEL as CD session title\n"
    "  -p N           use a padsize of N sectors for the session (default 15)\n"
    "  -m             enable multi-disk mode\n"
    "  -c COMMAND     call COMMAND on disk change in multi-disk mode\n"
    "  -v             be verbose\n"
    "  -V             prints version & exits\n"
    "  --             pass rest of commandline to cdrecord\n"
    "\n", prg_name);
}

/****************************************************************************/

void parse_cmdline(char argc, char *argv[]) 
{
  int i;

  while ((i=getopt(argc,argv,"d:r:l:s:p:a:c:mvVX"))>0) {
    switch (i) {
       case 'V': fprintf(stderr,"cdbackup "VERSION" (compiled "__DATE__")\n"
	                        "Copyright (C) 2000-2002\n"
			        "This is free software; see the source for copying conditions.\n"
			        "There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n"
			        "PARTICULAR PURPOSE.\n");
                 exit(0);
       case 'v': verbose=1; break;
       case 'm': multidisk=1; break;
       case 'X': xamode2=1; break;
       case 'c': multicmd=optarg; break;
       case 'd': cd_dev=optarg; break;
       case 'r': cdr_dev=optarg; break;
       case 'a': cd_label=optarg; break;
       case 'l': errno=0; cd_len=strtol(optarg,NULL,10);
                 if(errno==ERANGE || cd_len<1) serror("Option -l: length out of range (must be >=1)\n");
	         cd_len = (long long)cd_len * (1024*1024) / CD_FRAMESIZE; /* convert to blocks */
	         break;
       case 's': errno=0; cd_speed=strtol(optarg,NULL,10);
                 if(errno==ERANGE || cd_speed<1) serror("Option -s: speed out of range (must be >=1)\n");
	         break;
       case 'p': errno=0; padsize=strtol(optarg,NULL,10);
                 if(errno==ERANGE || padsize<15) serror("Option -p: padsize out of range (must be >=15)\n");
	         break;
       default:  usage(); exit(0);
       }
    }

  if(optind<argc) { /* save position/count of cdrecord options */
    cdrec_opt_count=argc-optind;
    cdrec_opt=&argv[optind];
    }
    
  if(!cdr_dev) serror("You must specify a device for cdrecord with -r\n");
}

/****************************************************************************/

#define MARG(ptr,len,form,arg) { int l=(len);\
                               if(!(*ptr=(char *)malloc(l+1))) serror("No memory for cdrecord args\n");\
                               snprintf(*ptr++,l,form,arg);\
                             }

void start_cdrecord() 
{
  char **args, **p;
  int l;
  
  if(!(p=args=calloc(cdrec_opt_count+8,sizeof(char *))))
    serror("No memory for cdrecord args\n");

  *p++="cdrecord";
  *p++="-multi";
  MARG(p,16,"speed=%d",cd_speed);
  MARG(p,6+strlen(cdr_dev),"dev=%s",cdr_dev);

  for(l=0 ; l<cdrec_opt_count ; l++) *p++=cdrec_opt[l];

  MARG(p,20,"padsize=%ds",padsize);
  if(xamode2) *p++="-xa2"; else *p++="-data";
  *p++="-";
  *p++=0;

  if(verbose) {
    fprintf(stderr,"%s: cdrecord command:",prg_name);
    for(p=args ; *p ; p++) fprintf(stderr," %s",*p);
    fprintf(stderr,"\n");
    }

  execvp("cdrecord",args);
  error("Unable to launch cdrecord");
}

/****************************************************************************/

int backup(char disk_set)
{
  pid_t childpid;
  int fd[2];
  int outpipe, bytes;
  long long grandTotal=0;
  struct header_block header;

  char buffer[CD_FRAMESIZE];
  struct data_block *db=(struct data_block *)&buffer[0];

  sprintf(buffer, "%04d%02d%02d%02d%02d", curtime.tm_year + 1900,
    curtime.tm_mon + 1, curtime.tm_mday, curtime.tm_hour, curtime.tm_min);
  
  strncpy(header.id_str,HDR_STRING,32); header.id_str[32] = 0;
  strncpy(header.vol_id,cd_label,32); header.vol_id[32] = 0;
  strncpy(header.t_stamp,buffer,12); header.t_stamp[12] = 0;
  header.disk_set = disk_set;

  if(verbose)
    fprintf(stderr,"%s: Recording to device %s, multidisk %s, disk %d\n",prg_name,cdr_dev,multidisk?"enabled":"disabled",disk_set); 

#ifndef DEBUGOUT /* the "real" code */
  /* launch cdrecord */
  if(pipe(fd) == -1) error("Unable to create pipe handles");
  if((childpid=fork()) == -1) error("Fork failed");
  if(childpid == 0) {        /* child */
    close(fd[1]);
    close(0);		     /* stdin */
    dup2(fd[0], 0);
    start_cdrecord();        /* doesn't return */
    }

  close(fd[0]); outpipe=fd[1];
  
  /* output the header block */
  memset(buffer,0,CD_FRAMESIZE);
  memcpy(buffer,&header,sizeof(struct header_block));
  if((bytes=write(outpipe, buffer, CD_FRAMESIZE)) != CD_FRAMESIZE) error("Error writing header block");

  cd_avail-=bytes; grandTotal+=bytes;
  /* account for the padsize */
  cd_avail-=padsize*CD_FRAMESIZE;
#else
  /* debug code; send data to /dev/null.  Don't need the pipe. */
  fprintf(stderr, "DEBUG CODE: sending data to /dev/null!\n");
  outpipe = open("/dev/null", O_WRONLY);
  if (outpipe < 0) { perror("/dev/null"); exit(1); }
#endif

  db->reserved = 0;

  do {
    /* read a block */
    db->status = 0;		         /* this isn't the last block (for now) */
    bytes=full_read(0,&buffer[DBSIZE],DATASIZE);
    if (bytes < 0) error("Error reading input");
    if (bytes != DATASIZE) db->status=1; /* EOF, this is the last block */
    db->datasize = htons(bytes);

    /* check for free space */
    if(cd_avail < (CD_FRAMESIZE*2)) {	/* less than 2 block free */
      if(db->status==0) db->status=2;   /* if not last block, mark disk as full */
      }

    /* write a block */
    bytes = write(outpipe, buffer, CD_FRAMESIZE);
    if(bytes != CD_FRAMESIZE) error("Error writing data block");

    grandTotal+=bytes; cd_avail-=bytes;
    } while(db->status==0);

  /* close pipe and wait for child termination */
  close(outpipe);
  while (wait(0) != childpid);

  totalSize+=grandTotal;
  if(verbose) fprintf(stderr,"%s: Recording finished. %lld kB written (%lld kB on this disk)\n",
                      prg_name,totalSize/1024,grandTotal/1024);

  if(db->status==2) return 1; /* disk was full */
  return 0;
}

/****************************************************************************/

int main(int argc, char *argv[]) 
{
  int cdr;
  int disknum, result, loop;
  time_t curtime_t;

  disknum=1; totalSize=0;
  curtime_t=time(0); curtime=*localtime(&curtime_t);

  parse_cmdline(argc,argv);

  do {
    do {
      cdr=open_cdr(cd_dev); result=read_toc(cdr,0); close_cdr(cdr);
      print_space();
      loop=1;
  
      if(disknum>1 && result!=0) {
        fprintf(stderr,"%s: Can't do multidisk continuation on non-empty disk! Try another disk\n", prg_name);
        if(start_diskchange(multicmd,cd_dev)) serror("Diskchange command failed");
        }
      else if(cd_avail<(padsize+MIN_BLOCKS)*CD_FRAMESIZE) {
        if(multidisk) {
          fprintf(stderr,"%s: Not enough free space on disk! Try another disk\n", prg_name);
          if(start_diskchange(multicmd,cd_dev)) serror("Diskchange command failed");
          }
        else serror("Not enough free space on disk");
        }
      else loop=0;
      } while(loop);

    result = backup(disknum);
    if(result == 1) {
      if(multidisk == 0) serror("Disk full, multi-disk not enabled. Aborting");
      
      disknum++;
      if(start_diskchange(multicmd,cd_dev)) serror("Diskchange command failed");
      }
    } while (result != 0);

  return 0;
}
