;;; @file format1541.s
;;; 1541/1571 program for formatting a disk,
;;; adapted and optimized from Daniel Kahlin's code.
;;; @author Marko Mkel (msmakela@nic.funet.fi)
;;; Original copyright message follows.

;* Written by Daniel Kahlin <daniel@kahlin.net>
;* (loosely based on the formatting routine used in the SpeedDos-Plus
;* 1541-firmware upgrade.)
;*
;* DESCRIPTION
;*   Fast format routine for the 1541/1571 (drive part).
;*   Features:
;*     - soft bump (i.e tries not to bump so much)
;*     - real verify (which can be turned off)
;*     - fairly fast, but very safe!
;*       (24 seconds with verify, 17 seconds without) 
;*
;* REFERENCES
;*   "Inside Commodore DOS", Richard Immers and Gerald G. Neufeld,
;*        Second Printing 1985, ISBN 0-88190-366-3, Datamost, Inc.
;*   "Das grosse FloppyBuch", Englisch & Szczepanowski, 1984, 
;*        ISBN 3-89011-005-3, Data Becker GmbH 
;*   "SpeedDos-Plus", disassembly of the 1541-firmware.
;*
;*
;* fastformat - a faster format for the Commodore 1541/1571 
;*
;* Copyright (c) 1995, 1996, 2002, Daniel Kahlin <daniel@kahlin.net>
;* All rights reserved.
;* 
;* Redistribution and use in source and binary forms, with or without
;* modification, are permitted provided that the following conditions
;* are met:
;* 1. Redistributions of source code must retain the above copyright
;*    notice, this list of conditions and the following disclaimer.
;* 2. Redistributions in binary form must reproduce the above copyright
;*    notice, this list of conditions and the following disclaimer in the
;*    documentation and/or other materials provided with the distribution.
;* 3. Neither the names of the copyright holders nor the names of their
;*    contributors may be used to endorse or promote products derived 
;*    from this software without specific prior written permission.
;* 
;* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;* ``AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;* A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR
;* CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;* PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

FMT_TRACKS	= 35		; number of tracks to format
FMT_HDRBUFFER	= $0300	; buffer for disk header

ftnum		= $51		; track being formatted

id1		= $12		; first ID byte
id2		= $13		; second ID byte

*		= $400

;**************************************************************************
;*
;* main format routine...
;* started by an execute job buffer 1.
;* just to put us running in FDC mode.
;*
;******
job	lda	ftnum
	bpl	format

	;; track number $ff means that we have to initialize
	lda	#%01100000	;headstep+drivemotor
	sta	$20	;DRVST


;*
;* Determine how many tracks to bump.
;* If we know the current track, just step in all tracks plus ~4 extra.
;* Otherwise (if DRVTRK=0) do normal bump.
;* Always make sure step bits in $1c00 become 00.
;*
	lda	$1c00
	and	#%00000011
	sta	$4a	;STEPS

	lda	$22	;DRVTRK
	bne	known
	lda	#44
known	and	#$7e
	eor	#$fe	;A=-A-2
	asl		;A=A*2
	ora	$4a	;A=TTTTTTSS
	sec
	sbc	#4
	sta	$4a	;STEPS


;*
;* Setup track
;*
	lda	#1
	sta	$22	;DRVTRK
	sta	ftnum

#if VERIFY
	lda	#10
	sta	fmt_retries
#endif 				; VERIFY

;*
;* return to job loop
;*
	jmp	$f99c	;END


;*
;* Check if we really are on the right track.
;*
format	ldy	#0
	cmp	($32),y		;HDRPNT
	beq	wp
	sta	($32),y		;HDRPNT
	jmp	$f99c	;END

;*
;* Check if write protected.
;*
wp	lda	#16
	bit	$1c00
	bne	wipe
	lsr		;26, 'write protect on'
	jmp	$fddb	;FMTE10


;*
;* start wiping track with $55's
;* this happens automagically once started.
;*
wipe	lda	$1c0c
	and	#$1f
	ora	#$c0
	sta	$1c0c	;set write mode
	lda	#$ff
	sta	$1c03	;port as output
	lda	#$55
	sta	$1c01	;wipe pattern

;*
;* calculate sectorheaders (while still wipeing track)
;*
	ldy	#0		; header buffer pointer
	ldx	#0		; current sector
secthd	lda	$39	;HBID (always $08)
	sta	FMT_HDRBUFFER,y
	iny
	iny
	txa
	sta	FMT_HDRBUFFER,y
	iny
	lda	ftnum
	sta	FMT_HDRBUFFER,y
	iny
	lda	id2
	sta	FMT_HDRBUFFER,y
	iny
	lda	id1
	sta	FMT_HDRBUFFER,y
	iny
	lda	#$0f
	sta	FMT_HDRBUFFER,y
	iny
	sta	FMT_HDRBUFFER,y
	iny

;* calculate header checksum
	lda	FMT_HDRBUFFER-6,y	;SECTOR
	eor	FMT_HDRBUFFER-5,y	;TRACK
	eor	FMT_HDRBUFFER-4,y	;ID2
	eor	FMT_HDRBUFFER-3,y	;ID1
	sta	FMT_HDRBUFFER-7,y	;CHKSUM

	inx
	cpx	$43	;SECTR
	bne	secthd
	stx	fmt_sectortemp

;*
;* GCR code headers (while still wipeing track)
;*
	tya
	pha
	lda	#>FMT_HDRBUFFER
	sta	$31	;BUFPNT+1
	jsr	$fe30	;FBTOG  (always sets BUFPNT to 0)
	pla
	tay
	dey
	jsr	$fde5	;MOVUP
	jsr	$fdf5	;MOVOVR

;*
;* do the actual formatting of a track.
;*

	lda	#0
	sta	$32	;HDRPNT
fsect	jsr	writesector
	dec	fmt_sectortemp
	bne	fsect

	bvc	*
	clv

	jsr	$fe00	;KILL  (go to readmode)

;*
;* Verify the track.
;*
#if VERIFY
	lda	#0
	sta	$32	;HDRPNT
	lda	#200
	sta	fmt_vfyretries

	lda	$43	;SECTR
	sta	fmt_sectortemp
	jsr	verifysect
#endif

;*
;* are we done with the disk?
;*
	inc	ftnum
	lda	ftnum
	cmp	#FMT_TRACKS+1
	bcc	fnext
	lda	#$01		; done
;*
;* set FTNUM to default value.  Set GCRFLAG so that noone will
;* try to convert from GCR to BIN after we have left.
;*
	jmp	$fddb	;FMTE10

fnext	jmp	$f99c	;END

;**************************************************************************
;*
;* writesector
;* must be run from FDC mode!
;*
;******
writesector

;*
;* write leading sync
;*
	jsr	writesync

;*
;* write sector header
;*
	ldy	$32	;HDRPNT
	ldx	#10
	.(
head	bvc	*
	clv
	lda	FMT_HDRBUFFER,y
	sta	$1c01
	iny
	dex
	bne	head
	.)
	sty	$32	;HDRPNT

;*
;* write gap
;*
	ldx	#9
	lda	#$55
	jsr	writegap

;*
;* write sector sync
;*
	jsr	writesync

;*
;* write the actual data block
;*		
	ldy	#4
	.(
wd1	bvc	*
	clv
	lda	emptygcr1,y
	sta	$1c01
	dey
	bpl	wd1

	ldx	#$40
wd2	ldy	#4
wd3	bvc	*
	clv
	lda	emptygcr2,y
	sta	$1c01
	dey
	bpl	wd3
	dex
	bne	wd2
	.)

;*
;* write intersector gap and exit
;*
	ldx	#8
writegap:
	lda	#$55
write	bvc	*
	clv
	sta	$1c01
	dex
	bne	write
	rts
;**************************************************************************
;*
;* writesync
;* writes 5*$ff to the disk.
;* must be run from FDC mode!
;*
;******
writesync:
	ldx	#5
	lda	#$ff
	bne	write

#if VERIFY
	;; failed to verify the sector
verifyfail:
	dec	fmt_vfyretries
	bne	verifysect
	pla			; discard the return address
	pla
	lda	#$06	;24, 'read error'
	dec	fmt_retries
	bne	retry
	jmp	$f99c	;END
retry	jmp	$fddb	;FMTE10

;**************************************************************************
;*
;* verifysect
;* must be run from FDC mode!
;*
;******
verifysect:
;*
;* find sync. 
;*
	jsr	$f556	;SYNC


;*
;* verify header
;*
	ldx	#10
	ldy	$32	;HDRPNT
	.(
vh	bvc	*
	clv
	lda	$1c01
	cmp	FMT_HDRBUFFER,y
	bne	verifyfail
	iny
	dex
	bne	vh
	.)

	sty	$32	;HDRPNT

;*
;* find sync
;*
	jsr	$f556	;SYNC



;*
;* verify the actual data block
;*
	ldy	#4
	.(
vd1	bvc	*
	clv
	lda	$1c01
	cmp	emptygcr1,y
	bne	verifyfail
	dey
	bpl	vd1

	ldx	#$40
vd2	ldy	#4
vd3	bvc	*
	clv
	lda	$1c01
	cmp	emptygcr2,y
	bne	verifyfail
	dey
	bpl	vd3
	dex
	bne	vd2
	.)

	dec	fmt_sectortemp
	bne	verifysect
	rts
#endif				; VERIFY

;**************************************************************************
;*
;* format data
;*
;******

;*
;* written:
;*     1 * emptygcr1
;*   $40 * emptygcr2
;* will result in (when GCR uncoded)
;* $07 followed by 259 $00 bytes.
;*
;* $07 is the data block identifier
;* $00 * 256 is the data bytes
;* $00 is the data block checksum
;* $00 * 2 is the off bytes.
;*
;* the GCR code is stored in reverse byte order
;*
;* emptygcr1:
;*   01010 10111 01010 01010 01010 01010 01010 01010
;*     \     /     \     /     \     /     \     /
;*       $07         $00         $00         $00
;*
;* emptygcr2 (repeat $40 times):
;*   01010 01010 01010 01010 01010 01010 01010 01010
;*     \     /     \     /     \     /     \     /
;*       $00         $00         $00         $00
;*
emptygcr1:
	.byte	$4a,$29,$a5,$d4,$55
emptygcr2:
	.byte	$4a,$29,$a5,$94,$52


;**************************************************************************
;*
;* here is where we enter first.
;* started using a M-E command.
;* Now we are in IP mode.
;*
;******
FormatEntry:

;*
;* set initial formatting parameters
;*
	ldy	#$ff
	lda	#0		; PARAMETER => first ID byte
	ldx	#0		; PARAMETER => second ID byte
	sta	id1
	stx	id2
	sty	ftnum
	iny
	sty	$7f

;*
;* led on, clear all channels
;*
	jsr	$c100	;SETLDA

	jsr	$d307	;CLRCHN

;*
;* do execute job
;*
	ldx	#1
	stx	$08	;HDRS1
	dex
	stx	$09	;HDRS1+1

	lda	#$e0	;JOB => execute
	sta	$01	;in buffer #1 ($0400)
	.(
polljob	lda	$01
	bmi	polljob
	.)

	cmp	#$02
	bcc	mkfs

	lda	#$03	;21, 'read error'
	ldx	#$00
	jmp	$e60a	;ERROR

;*
;* prepare a new BAM
;*
mkfs	jsr	$f005	;CLRBAM
	jsr	$eeb7	;NEWMAP

;*
;* copy disk name
;*
	ldy	#$90
	.(
cpnam	lda	fmt_name - $90,y
	sta	($6d),y
	iny
	cpy	#$a0
	bcc	cpnam
	.)

;*
;* wipe id area with $a0
;*
	tya
	.(
a0	sta	($6d),y
	iny
	cpy	#$ab
	bne	a0
	.)

;*
;* store 1541 format identifier
;*
	lda	#$41
	sta	$0101
	ldy	#$02
	sta	($6d),y
	ldy	#$a6
	sta	($6d),y
	dey
	lda	#"2"
	sta	($6d),y

;*
;* copy id
;*
	lda	id1
	ldy	#$a2
	sta	($6d),y
	lda	id2
	iny
	sta	($6d),y


;*
;* Mark sectors 18,00 and 18,01 as used in BAM
;*
	lda	#18
	sta	$80
	lda	#1
	sta	$81
	jsr	$ef93	;USEDTS
	dec	$81
	jsr	$ef93	;USEDTS

;*
;* write BAM to disk
;*
	jsr	$eeff	;SCRBAM


;*
;* prepare and write out the 18,01 sector
;*
	jsr	$f005	;CLRBAM
	ldy	#1
	lda	#$ff
	sta	($6d),y
	inc	$81
	jsr	$d464	;DRTWRT

;*
;* do init to read back BAM
;*
	jsr	$d042	;DRINIT

;*
;* return to 1541 DOS.
;*
	jmp	$c194

fmt_name		; disk name (16 bytes), filled in by master program

fmt_sectortemp	= fmt_name + 16
#if VERIFY
fmt_retries	= fmt_sectortemp + 1
fmt_vfyretries	= fmt_retries + 1
#endif				; VERIFY
