
#include <qapplication.h>

#include "TVChannel.h"

bool TVChannel::s_TablesFilled = false;
QVector<TVChannel::FrequencyTableEntry> TVChannel::s_Channel2BaseFrequency[TVChannel::FrequencyGrid_MAX];


TVChannel::TVChannel()
{
  Name = QString::null;
  FrequencySystem = FrequencyGrid_INVALID;
  Channel = 0;
  Finetuning = 0;
  ColorSystem = PAL_BG;
}

TVChannel::TVChannel(const TVChannel &org)
{
  Name = org.Name;
  FrequencySystem = org.FrequencySystem;
  Channel = org.Channel;
  Finetuning = org.Finetuning;
  ColorSystem = org.ColorSystem;
}

TVChannel &TVChannel::operator =(const TVChannel &org)
{
  // kkeep ListView parent
  Name = org.Name;
  FrequencySystem = org.FrequencySystem;
  Channel = org.Channel;
  Finetuning = org.Finetuning;
  ColorSystem = org.ColorSystem;
  return *this;
}

// private

/**
  \brief Fill frequency base tables with names and frequencies
*/
void TVChannel::FillTables()
{
   int i, j;
   QString fname;

   if (s_TablesFilled)
     return;

   s_Channel2BaseFrequency[European].resize(109); // Total number of channels for Europe
   s_Channel2BaseFrequency[European].setAutoDelete(true);
   for (i = 0; i < 109; i++)
   {
      s_Channel2BaseFrequency[European].insert(i, new FrequencyTableEntry());
   }

   j = 0;
   for (i = 0; i < 3; i++) {
      s_Channel2BaseFrequency[European][j  ]->Name = fname.sprintf("VHF %d", 2 + i);
      s_Channel2BaseFrequency[European][j++]->Frequency = 48.25 + 7.0 * i;
   }
   for (i = 0; i < 8; i++) {
      s_Channel2BaseFrequency[European][j  ]->Name = fname.sprintf("VHF %d", 5 + i);
      s_Channel2BaseFrequency[European][j++]->Frequency = 175.25 + 7.0 * i;
   }
   /* The following set of frequencies make no sense at all. */
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy A");
   s_Channel2BaseFrequency[European][j++]->Frequency = 53.75;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy B");
   s_Channel2BaseFrequency[European][j++]->Frequency = 62.25;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy C");
   s_Channel2BaseFrequency[European][j++]->Frequency = 82.25;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy D");
   s_Channel2BaseFrequency[European][j++]->Frequency = 175.25;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy E");
   s_Channel2BaseFrequency[European][j++]->Frequency = 183.75;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy F");
   s_Channel2BaseFrequency[European][j++]->Frequency = 192.25;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy G");
   s_Channel2BaseFrequency[European][j++]->Frequency = 201.25;
   s_Channel2BaseFrequency[European][j  ]->Name = tr("Italy H");
   s_Channel2BaseFrequency[European][j++]->Frequency = 210.25;

   /* UHF band */
   for (i = 0; i < 49; i++) {
      s_Channel2BaseFrequency[European][j  ]->Name = fname.sprintf("UHF %d", 21 + i);
      s_Channel2BaseFrequency[European][j++]->Frequency = 471.25 + 8.0 * i;
   }
   /* Cable */
   for (i = 0; i < 10; i++) {
      s_Channel2BaseFrequency[European][j  ]->Name = fname.sprintf("CATV S%d", 1 + i);
      s_Channel2BaseFrequency[European][j++]->Frequency = 105.25 + 7.0 * i;
   }
   for (i = 0; i < 10; i++) {
      s_Channel2BaseFrequency[European][j  ]->Name = fname.sprintf("CATV S%d", 11 + i);
      s_Channel2BaseFrequency[European][j++]->Frequency = 231.25 + 7.0 * i;
   }
   for (i = 0; i < 21; i++) {
      s_Channel2BaseFrequency[European][j  ]->Name = fname.sprintf("CATV S%d", 21 + i);
      s_Channel2BaseFrequency[European][j++]->Frequency = 303.25 + 8.0 * i;
   }

   s_TablesFilled = true;
}



// public

/**
  \brief Calculate frequency for this channel
  \return The frequency in MHz

  See also the function below.

*/

float TVChannel::Frequency() const
{
   return Frequency(FrequencySystem, Channel, Finetuning);
}

/**
  \brief Calculate frequency for channel + finetuning
  \return The frequency in MHz
  \param grid The frequency grid system
  \param channel The channel in this grid
  \param finetuning The finetuning offset, in 0.01 MHz

  Note: the frequency is in MHz, so for the real frequency multiply by 1e6.

*/
float TVChannel::Frequency(FrequencyGrid grid, int channel, int finetuning)
{
   float fr;

   FillTables();

   if (grid < 0 || grid >= FrequencyGrid_MAX)
     return 0.0;

   if (channel < 0 || channel >= s_Channel2BaseFrequency[grid].count())
     return 0.0;

   fr = s_Channel2BaseFrequency[grid][channel]->Frequency + finetuning / 100.0;
   return fr;
}


/**
  \brief Return number of channels within a grid
  \return Number of channels, or 0 if invalid grid
  \see ChannelName

*/
uint TVChannel::GetNumberOfChannels(FrequencyGrid grid)
{
   if (grid < 0 || grid >= FrequencyGrid_MAX)
     return 0;
   FillTables();
   return s_Channel2BaseFrequency[grid].count();
}


/**
  \brief Return name of a channel
  \param grid The frequency grid system
  \param channel The channel in this grid
  \return The name of this channel, like VHF 1, or CATV 23

  Note: the name is not a TV channel name, but a rather systemic name
  given to the channel within the given frequency system, like VHF 3,
  UHF 21, etc.
*/
QString TVChannel::ChannelName(FrequencyGrid grid, int channel)
{
   FillTables();

   if (grid < 0 || grid >= FrequencyGrid_MAX)
     return QString::null;

   if (channel < 0 || channel >= s_Channel2BaseFrequency[grid].count())
     return QString::null;
   return s_Channel2BaseFrequency[grid][channel]->Name;
}

