#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <qaccel.h>
#include <qbuttongroup.h>
#include <qcombobox.h>
#include <qkeycode.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qpixmap.h>
#include <qworkspace.h>

#include "CamStreamApp.h"
#include "CamStreamMainWindow.h"
#include "CamWindow.h"

#include "gui/OpenDevice.h"
#include "VideoCollector.h"

#include "WebCamViewer.h"
#ifdef DEVELOPMENT
#include "WebCamCodecSubband.h"
#endif

#include "icons/canvas.xpm"

#include "tracer.h"

TR_MODULE("CCamStreamMainWindow");

/**
  \class CCamStreamMainWindow
  \brief Name says it all: main window, with menu bar, etc.

  This is the main widget class of the application; It does several things:

  * manages the main menu and the Frames of the various subclasses.
  * handles new and disappearing devices.

  There is only one instance of this object, ever.
*/



/**
  \brief Constructor
*/
CCamStreamMainWindow::CCamStreamMainWindow()
	: QMainWindow(0, "camstreams")
{
   QPopupMenu *file, *window;
   QMenuBar *pMenu;
   QPixmap BackgroundPix((const char **)canvas_xpm);

   // Create main workspace, with a nice background
   m_pWorkspace = new QWorkspace(this); // Just a placeholder for all the windows
   assert(m_pWorkspace != NULL);
   m_pWorkspace->setBackgroundPixmap(BackgroundPix);

   file = new QPopupMenu();
   file->insertItem(tr("&Open viewer..."),   this, SLOT(FileOpenNewViewer()), QAccel::stringToKey(tr("Ctrl+O")));
   file->insertItem(tr("Show &Tracer"),      this, SLOT(ShowTracer()),        QAccel::stringToKey(tr("Shift+Ctrl+D")));
   file->insertItem(tr("E&xit"),             this, SLOT(Quit()),              QAccel::stringToKey(tr("Ctrl+Q")));

   window = new QPopupMenu();
   window->insertItem("&Cascade", m_pWorkspace, SLOT(cascade()));
   window->insertItem("&Tile",    m_pWorkspace, SLOT(tile()));

   pMenu = menuBar();
   pMenu->insertItem("&File", file);
   pMenu->insertItem("&Window", window);

   setCentralWidget(m_pWorkspace);

#if 0
   // Add devices on command line as viewers
   m = g_pCamApp->argc();
   for (i = 1; i < m; i++) {
      const char *dev;

      dev = g_pCamApp->argv()[i];
      fd = open(dev, O_RDONLY);
      if (fd < 0) {
        QMessageBox mb;
        QString message;

        message.sprintf("Failed to open video device: %s" , dev);
        mb.critical(0, "Failure", message);
      }
      else {
        pCam = new CCamWidget(fd, Workspace);
        AddCam(pCam);
      }
   }
#endif
}

CCamStreamMainWindow::~CCamStreamMainWindow()
{
   TR_ENTER();
   CloseAll();
   TR_LEAVE();
}

// private

void CCamStreamMainWindow::Quit()
{
   TR_ENTER();
   CloseAll();
   qApp->quit();
   TR_LEAVE();
}

// private slots

void CCamStreamMainWindow::CloseAll()
{
   QWidgetList wl;
   QWidget *w;

   TR_ENTER();
   wl = m_pWorkspace->windowList();
   for (w = wl.first(); w != 0; w = wl.next())
      delete w;
   TR_LEAVE();
}



/* Menu functions */

/**
  \brief Show dialog to open new viewer frame

  This function will pop-up a small dialog where the user can select
  a (detected) video device, and an viewer to do something useful with
  it. The viewers are now hardcoded, a plug-in structure is planned for
  the future.
 */
void CCamStreamMainWindow::FileOpenNewViewer()
{
   COpenDeviceDlg Dlg(this, "open video device dlg", true);
   QString dname, ser;
   CVideoCollector *pCollector = CVideoCollector::Instance();
   CVideoDevice *pDev = 0;
   CCamWindow *pWindow = 0;
   unsigned int i;

   for (i = 0; i < pCollector->NumberOfVideoDevices(); i++) {
      pDev = pCollector->GetVideoDevice(i);
      dname = pDev->GetIntfName();
      if (dname.isNull())
        dname = pDev->GetNodeName();
      ser = pDev->GetSerialNumber();
      if (!ser.isEmpty())
        dname = dname + "[#" + ser + "]";
      Dlg.m_DeviceSelection->insertItem(dname);
   }

   // Add viewer types to combobox... should be dynamically registerable by
   // 'plugins'
   Dlg.m_ViewerSelection->insertItem(tr("Regular viewer"), 0);
#ifdef DEVELOPMENT
   Dlg.m_ViewerSelection->insertItem(tr("Webcam Recorder (development)"), 1);
   Dlg.m_ViewerSelection->insertItem(tr("Subband codec (experimental)"), 2);
#endif

   if (Dlg.exec() == QDialog::Accepted) {
     pDev = pCollector->GetVideoDevice(Dlg.m_DeviceSelection->currentItem());
     if (pDev == NULL)
       return;

     switch(Dlg.m_ViewerSelection->currentItem())
     {
       case 0:
         pWindow = new CWebCamViewer(pDev, m_pWorkspace);
         break;
#ifdef DEVELOPMENT
       case 2:
         pWindow = new CWebCamCodecSubband(pDev, m_pWorkspace);
         break;
#endif
       default:
         QMessageBox::warning(this, tr("Unknown viewer"),
                            tr("The selection from the viewer dropdown box is unknown."),
                            QMessageBox::Ok, QMessageBox::NoButton);
         return;
         break;
     }
     if (pWindow == NULL) {
       QMessageBox::warning(this, tr("Error creating viewer"),
                    tr("The viewer could not be created. You might be running low on memory."),
                    QMessageBox::Ok, QMessageBox::NoButton);
     }
     else {
       pWindow->show();
     }
   }
}


void CCamStreamMainWindow::ShowTracer()
{
  TR_SHOW();
}


// protected

/**
  \brief Overloaded function from QWidget

  Called when the main window is closed; we properly shut down every widget

*/
void CCamStreamMainWindow::closeEvent(QCloseEvent *e)
{
  TR_ENTER();
  CloseAll();
  qApp->quit();
  TR_LEAVE();
}

