/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

/**
  \class CRingBuffer

  \brief A thread-safe circular buffer with multiple readers and multiple writers.

  A ringbuffer is a FIFO buffer: First In, First Out. It gets its name
  because if data is to be written beyond the end of the internal buffer,
  it wraps around to the beginning (provided that there is enough space
  left and we don't overwrite the tail). Thus the buffer is used 'circular',
  in a ring.

  What makes this class special is that it supports multiple writers and
  readers, \b and is thread-safe. So, multiple threads can write to the
  buffer (appending at the front), and multiple threads can read from the
  tail. (Please note that the readers and writers themselves are not
  thread safe! Use separate readers and writers in different threads).

  However, the functionality is not symmetrical: there is only a single head
  pointer, so every writer always appends to the buffer, not overwriting
  anything. But each reader has its own tail pointer and all readers will
  receive the same data from the buffer. Effectively, the data is copied
  to the various reader threads.

  The readers don't have a head pointer; just a tail and the length; the
  length is increased every time data is added by a writer. The tail and
  length are updated at read operations.

  Note: you must use at least Qt 2.3.0, because there was a bug in the
  QMutex class.
*/

#ifndef RINGBUFFER_DEBUG
#undef  RINBGUFFER_DEBUG
//#define RINGBUFFER_DEBUG
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <stdlib.h>

#include "MemoryAudioSample.h"
#include "RingBuffer.h"

#define RB_EVENT		0x52420000
#define RB_EVENT_NEW_DATA	(RB_EVENT + 1)
/**
  @param buffer_space The desired buffer length, in bytes

  Constructor; allocates space.
*/

CRingBuffer::CRingBuffer(unsigned int buffer_space)
{
   BufferSpace = buffer_space;
   Buffer = NULL;
   BufferHead = 0;
   MaxLength = 0;
   BytesWritten = 0;
   Overflows = 0;

   if (BufferSpace >= 0)
     Buffer = malloc(BufferSpace);
}


CRingBuffer::~CRingBuffer()
{
   free(Buffer);
   Buffer = NULL;
}


// private

/**
  @param s A new buffer writer

  Adds a new CRingBufferWriter to the pool of writers. A writer has the
  right to add data to the circular buffer. A writer may be attached to
  a ring only once.
*/

void CRingBuffer::AttachWriter(CRingBufferWriter *s)
{
#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBuffer::AttachWriter(%p)", s);
#endif
   Lists.lock();
   Writers.append(s);
   Lists.unlock();
   emit WriterAttached();
}

void CRingBuffer::DetachWriter(CRingBufferWriter *s)
{
#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBuffer::DetachWriter(%p)", s);
#endif
   Lists.lock();
   if (!Writers.remove(s))
     qWarning("CRingBuffer::DetachWriter(%p): failed to find item in list.", s);
   Lists.unlock();
   emit WriterDetached();
}


/**
  \brief Append data to buffer-front
  \param data Pointer to data
  \param len Number of byte to put into buffer
  \param must_fit When there's not enough space in the buffer, do not attempt to write data into it
  \return Number of bytes put in buffer; may be smaller than \ref len
*/
int CRingBuffer::AddToBuffer(void *data, int len, bool must_fit)
{
   int plus;
   CRingBufferReader *reader;

   /* Although the lock in this function is rather long, and includes a
      (lenghty) memcpy() operation, there's not much we can do about it.
      There's all sorts of problems with trying to keep the memcpy() outside
      the lock, mainly consistency (a second writer-thread could increase
      the reader buffer lengths, and thus the reader-'heads', to a point
      beyond where the mempcy() of the first writer thread is, should that
      thread have run out of its timeslice).

      So, effectively all writes are serialized, but it doesn't matter
      because usually there are more readers than writers, and the total
      time of all the memcpy()s combined is needed anyway.
    */

   Head.lock();
   if (len + MaxLength > BufferSpace) { // Dang, not enough space left
//qWarning("CRingBuffer::AddToBuffer: buffer overflow (len = %d, space = %d)", len, BufferSpace - MaxLength);
     if (must_fit)
       len = 0;  // Don't bail out yet; fall through to the reader update part below
     else
       len = BufferSpace - MaxLength;
     Overflows++;
   }

   if (len > 0) { // We have space left.
     /* copy data, split if necessary */
     if (BufferHead + len > BufferSpace) {
       /* hrmpf. Copy data in 2 strokes */
       plus = BufferSpace - BufferHead;
       memcpy((char *)Buffer + BufferHead, data, plus);
       memcpy(Buffer, (char *)data + plus, len - plus);
     }
     else // 'ANSI C++ forbids using pointer of type 'void *' in arithmetic'... How inconvenient.
       memcpy((char *)Buffer + BufferHead, data, len);

     /* advance head */
     BufferHead = (BufferHead + len) % BufferSpace;
     BytesWritten += len;
   }

   /* Data is now in the buffer; adjust lengths of our readers, finding
      MaxLength at the same time.
      NB: we always process our readers here, even if our buffer does get full;
      this prevents dead-lock. If we had returned above when len == 0 we would
      never get here, and notice that data got read by our readers. Second,
      any Readers waiting for data will get notified.
    */

   Lists.lock(); // prevent list manipulation while we're busy
   MaxLength = 0;
   reader = Readers.first();
   while (reader != NULL) {
      reader->Lock.lock();
      reader->MyBufferLength += len;
      if (reader->MyBufferLength > MaxLength)
        MaxLength = reader->MyBufferLength;
      if (reader->LowWaterMark > 0 && reader->MyBufferLength > reader->LowWaterMark)
        reader->DataReady.wakeAll(); // In case our thread is waiting for data
      reader->Lock.unlock();
// FIXME: post event when low water mark is *crossed*
//      QThread::postEvent(reader, new QEvent((QEvent::Type)RB_EVENT_NEW_DATA));
      reader = Readers.next();
   }
   Lists.unlock();
   Head.unlock();

   return len;
}

/**
  \brief Return number of bytes available in buffer
*/

int CRingBuffer::SpaceLeft()
{
   int n;
   Head.lock();
   n = BufferSpace - MaxLength;
   Head.unlock();
   return n;
}

/**
  \brief Return number of bytes used in buffer

  Updates internal counters.
*/
int CRingBuffer::SpaceUsed()
{
   CRingBufferReader *reader;
   int n;

   Head.lock();
   Lists.lock(); // prevent list manipulation while we're busy
   MaxLength = 0;
   reader = Readers.first();
   while (reader != NULL) {
      reader->Lock.lock();
      if (reader->MyBufferLength > MaxLength)
        MaxLength = reader->MyBufferLength;
      reader->Lock.unlock();
      reader = Readers.next();
   }
   Lists.unlock();

   n = MaxLength;
   Head.unlock();
   return n;
}

/**
  Adds a reader to our pool; also initializes the Tail pointer.
*/
void CRingBuffer::AttachReader(CRingBufferReader *r)
{
#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBuffer::AttachReader(%p)", r);
#endif
   /* We have to be careful we get a valid BufferHead pointer. If we are
      attaching while we we are in the middle of AddToBuffer, all kinds of
      weird situations might happen.
    */
   Head.lock(); // make sure BufferHead doesn't change
   r->BufferTail = BufferHead;
   Lists.lock();
   Readers.append(r);
   connect(this, SIGNAL(BufferFlushed()), r, SIGNAL(BufferFlushed()));
   Lists.unlock();
   Head.unlock();
   emit ReaderAttached();
}

void CRingBuffer::DetachReader(CRingBufferReader *r)
{
#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBuffer::DetachReader(%p)", r);
#endif
   Lists.lock();
   if (!Readers.remove(r))
     qWarning("CRingBuffer::DetachReader(%p): failed to find item in list.", r);
   Lists.unlock();
   emit ReaderDetached();
}


/** \brief Flush contents of buffer
*/
void CRingBuffer::Flush()
{
   CRingBufferReader *reader = 0;

   Head.lock(); // make sure BufferHead doesn't change
   Lists.lock();
   MaxLength = 0;
   reader = Readers.first();
   while (reader != NULL) {
      reader->Lock.lock();
      reader->BufferTail = BufferHead; // reset poiinters
      reader->MyBufferLength = 0;
      reader->Lock.unlock();
      reader = Readers.next();
   }
   Lists.unlock();
   Head.unlock();

   emit BufferFlushed();
}

// public



unsigned int CRingBuffer::GetBufferLength()
{
   return BufferSpace;
}




/****************************************************************************/

/**
  \brief Constructor for writer; attaches to ring automatically

  There is no default constructor.
*/
CRingBufferWriter::CRingBufferWriter(CRingBuffer *ring)
{
qDebug(">> CRingBufferWriter::CRingBufferWriter(CRingBuffer *)");
   assert(ring != NULL);
   pRing = ring;

   pRing->AttachWriter(this);
qDebug("<< CRingBufferWriter::CRingBufferWriter(CRingBuffer *)");
}

/**
  \brief Destructor; detaches from ring automatically
*/
CRingBufferWriter::~CRingBufferWriter()
{
   pRing->DetachWriter(this);
}


/**
  \brief Put data in buffer
  \param data A pointer to the data
  \param len Number of bytes to write
  \param must_fit Data must fit in buffer
  \return Number of bytes written

  This function tries to put new data in the circular buffer. It returns
  the number of bytes that actually got written, which may be < len. If
  must_fit is TRUE then there must be enough space available in the buffer
  to write the whole data block. When there's not enough room, 0 will
  be returned.

  This function has an interesting side effect: when there are no readers,
  the buffer is never filled!
*/
int CRingBufferWriter::WriteToBuffer(void *data, int len, bool must_fit) const
{
#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBufferWriter::WriteToBuffer(%d) this = %p", len, this);
#endif
   return pRing->AddToBuffer(data, len, must_fit);
}

int CRingBufferWriter::SpaceLeft() const
{
   return pRing->SpaceLeft();
}

int CRingBufferWriter::SpaceUsed() const
{
   return pRing->SpaceUsed();
}

void CRingBufferWriter::Flush() const
{
   pRing->Flush();
}


/*****************/

/**
  \brief Contructor

  Constructs a \ref CRingBuffer reader, and adds it to the pool of readers.
  Initially, our 'buffer' will be empty; we will only see data that is
  written to the ringbuffer after we are attached.
*/

CRingBufferReader::CRingBufferReader(CRingBuffer *ring)
{
qDebug(">> CRingBufferReader::CRingBufferReader(CRingBuffer *)");
   pRing = ring;
   MyBufferLength = 0;
   LowWaterMark = HighWaterMark = 0;
   pRing->AttachReader(this);
qDebug("<< CRingBufferReader::CRingBufferReader(CRingBuffer *)");
}


/**
  \brief Destructor

  The destructor will automatically detach this object from the ringbuffer.
*/
CRingBufferReader::~CRingBufferReader()
{
   pRing->DetachReader(this);
}

// private


// public

int CRingBufferReader::SpaceUsed() const
{
   return MyBufferLength;
}

/**
  \brief Get data from tail of buffer
  \param data Pointer where to store data
  \param len Number of requested bytes
  \return Number of bytes put in data pointer; may be less than \e len

  This function reads data from the tail of the buffer. It updates our
  tailpointer (CRingBuffer will check the tailpointers during a write,
  to see how much data is still in use).
*/
int CRingBufferReader::ReadFromBufferTail(void *data, unsigned int len)
{
   unsigned int buflen;

#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBufferReader::ReadFromBufferTail(%d)", len);
#endif
   buflen = MyBufferLength;
   if (len > buflen)
     len = buflen;
   if (len > 0) {
     // Copy data in 1 or 2 strokes
     if (BufferTail + len > pRing->BufferSpace) {
       buflen = pRing->BufferSpace - BufferTail;
       memcpy(data, (char *)pRing->Buffer + BufferTail, buflen);
       memcpy((char *)data + buflen, pRing->Buffer, len - buflen);
     }
     else
       memcpy(data, (char *)pRing->Buffer + BufferTail, len);
   }
   Lock.lock(); // keep this lock minimal
   MyBufferLength -= len;
   BufferTail = (BufferTail + len) % pRing->BufferSpace;
   Lock.unlock();
   return len;
}



/**
  \brief Get data from head of buffer
  \param data Pointer where to store data
  \param len Number of requested bytes
  \param time In case buffer is empty, wait for data a maximum of \b time milliseconds
  \param clear If TRUE, will reset our lengthpointer (pretend we read all)
  \return Number of bytes put in data pointer; may be less than \e len

  This functions allows you to peek at the data at the head of the buffer.
  It will not update the lengthpointer, unless you set \e clear to TRUE;
  then it will reset our reader-buffer (this does not affect other readers).
*/
int CRingBufferReader::ReadFromBufferHead(void *data, unsigned int len, bool clear, unsigned long time)
{
   unsigned int buflen, head;

#ifdef RINGBUFFER_DEBUG
   qDebug("CRingBufferReader::ReadFromBufferHead(%d, %c, %lu)", len, clear ? 'T' : 'F', time);
#endif
   // If we have a LowWaterMark and we don't have enough data, wait a bit.
   if (LowWaterMark > 0 && MyBufferLength < LowWaterMark) {
//qDebug("ReadFromBufferHead: Waiting for data for %lu ms...", time);
     if (!DataReady.wait(time))
       qDebug("CRingBufferReader::ReadFromBufferHead() Timed out after %lu ms.", time);
   }
   Lock.lock();
   buflen = MyBufferLength;
   if (len > buflen)
     len = buflen;
   head = (BufferTail + buflen - len) % pRing->BufferSpace;
   Lock.unlock();

   if (len > 0) {
     // Copy data in 1 or 2 strokes.
     if (head + len > pRing->BufferSpace) {
       buflen = pRing->BufferSpace - head; // restant
       memcpy(data, (char *)pRing->Buffer + head, buflen);
       memcpy((char *)data + buflen, pRing->Buffer, len - buflen);
     }
     else
       memcpy(data, (char *)pRing->Buffer + head, len);
   }
   if (clear) {
     Lock.lock();
     MyBufferLength = 0;
     BufferTail = pRing->BufferHead;
     Lock.unlock();
   }
   return len;
}


bool CRingBufferReader::event(QEvent *e)
{
   if (e->type() == RB_EVENT_NEW_DATA) {
qDebug("CRingBufferReader::event(dataArrived)");
     emit DataArrived(0);
     return true;
   }
   return false;
}

/**
  \brief Set reader low water mark
  \param mark The new water mark; -1 disables the mark.

  A 'low water mark' is used in a buffer to indicate a minimum fill
  before action is/should be taken; it is often used in combination
  with a high water mark. It prevents things like the 'silly buffer syndrome'
  or nefarious polling on the buffer.

  Basicly, the low and high water marks specify the \e preferred minimum and
  maximum fill of the buffer; you expect the buffer to have at least low mark,
  but not more than high mark bytes of data in it; however, the buffer
  may still drain completely or fill up. But usually action is taken when
  the low or high water mark is crossed.

*/

void CRingBufferReader::SetLowWaterMark(unsigned int mark)
{
   if (mark > pRing->BufferSpace)
     return;
   LowWaterMark = mark;
}

unsigned int CRingBufferReader::GetLowWaterMark() const
{
   return LowWaterMark;
}

void CRingBufferReader::SetHighWaterMark(unsigned int mark)
{
   if (mark > pRing->BufferSpace)
     return;
   HighWaterMark = mark;
}

unsigned int CRingBufferReader::GetHighWaterMark() const
{
   return HighWaterMark;
}

// signals

/**
   \function void CRingBufferReader::DataArrived(int bytes)
   \param bytes Number of newly arrived bytes in buffer


   void BufferFlushed();

*/
