package buoy.widget;

import buoy.event.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

/**
 * A BComboBox is a Widget that displays a popup menu with a list of objects for the user to select. 
 * Typically the objects are Strings, but other types of objects can be used as well.  Optionally,
 * the combo box may be made editable, so the user may type a value directly rather than selecting
 * one from the list.  This allows arbitrary values to be entered, not just standard ones in the list.
 * <p>
 * BComboBox provides methods for adding and removing objects in the list.  Alternatively, you can set
 * a ComboBoxModel to provide more complex behaviors.
 * <p>
 * In addition to the event types generated by all Widgets, BComboBoxes generate the following event types:
 * <ul>
 * <li>{@link buoy.event.ValueChangedEvent ValueChangedEvent}</li>
 * </ul>
 *
 * @author Peter Eastman
 */

public class BComboBox extends Widget
{
  private int suppressEvents;

  /**
   * Create a new uneditable BComboBox containing no objects.
   */
  
  public BComboBox()
  {
    component = createComponent();
    getComponent().addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent ev)
      {
        if (suppressEvents == 0)
          dispatchEvent(new ValueChangedEvent(BComboBox.this));
      }
    });
  }

  /**
   * Create a new uneditable BComboBox containing the objects in an array.
   */
  
  public BComboBox(Object contents[])
  {
    this();
    setContents(contents);
  }

  /**
   * Create a new uneditable BComboBox containing the objects in a Collection.  The objects will be
   * added in the order they are returned by the Collection's Iterator.
   */
  
  public BComboBox(Collection contents)
  {
    this();
    setContents(contents);
  }
  
  /**
   * Create a new uneditable BComboBox whose contents are determined by a ComboBoxModel.
   */
  
  public BComboBox(ComboBoxModel model)
  {
    this();
    getComponent().setModel(model);
  }
  
  /**
   * Create the JComboBox which serves as this Widget's Component.  This method is protected so that
   * subclasses can override it.
   */
  
  protected JComboBox createComponent()
  {
    return new JComboBox() {
      public void reshape(int x, int y, int width, int height)
      {
        super.reshape(x, y, width, height);
        validate();
      }
    };
  }

  public JComboBox getComponent()
  {
    return (JComboBox) component;
  }

  /**
   * Get whether this combo box is editable, allowing the user to type in values other than those in
   * the popup list.
   */
  
  public boolean isEditable()
  {
    return getComponent().isEditable();
  }
  
  /**
   * Set whether this combo box is editable, allowing the user to type in values other than those in
   * the popup list.
   */
  
  public void setEditable(boolean editable)
  {
    getComponent().setEditable(editable);
  }
  
  /**
   * Set the contents of the list to the objects in an array.  This completely replaces the contents of the
   * list, removing any objects that were previously in it.
   *
   * @param o     the objects to put in the combo box's list
   */
  
  public void setContents(Object o[])
  {
    try
    {
      suppressEvents++;
      getComponent().setModel(new DefaultComboBoxModel(o));
    }
    finally
    {
      suppressEvents--;
    }
  }
  
  /**
   * Set the contents of the list to the objects in a Collection.  This completely replaces the contents of the
   * list, removing any objects that were previously in it.  The objects will be added in the order they are
   * returned by the Collection's Iterator.
   *
   * @param c     the objects to put in the combo box's list
   */
  
  public void setContents(Collection c)
  {
    setContents(c.toArray());
  }
  
  /**
   * Get the ComboBoxModel which controls the contents of this BComboBox.
   */
  
  public ComboBoxModel getModel()
  {
    return getComponent().getModel();
  }
  
  /**
   * Set the ComboBoxModel which controls the contents of this BComboBox.
   */
  
  public void setModel(ComboBoxModel model)
  {
    try
    {
      suppressEvents++;
      getComponent().setModel(model);
    }
    finally
    {
      suppressEvents--;
    }
  }
  
  /**
   * Add an object to the end of the list.
   *
   * @param o      the object to add
   */
  
  public void add(Object o)
  {
    try
    {
      suppressEvents++;
      getComponent().addItem(o);
    }
    finally
    {
      suppressEvents--;
    }
  }
  
  /**
   * Add an object at specified position in the middle of the list.
   *
   * @param index  the position at which to add the object
   * @param o      the object to add
   */
  
  public void add(int index, Object o)
  {
    try
    {
      suppressEvents++;
      getComponent().insertItemAt(o, index);
    }
    finally
    {
      suppressEvents--;
    }
  }
  
  /**
   * Replace the object at a specified position in the list with a new one.
   *
   * @param index  the position at which to set the object
   * @param o      the new object to add
   */
  
  public void replace(int index, Object o)
  {
    remove(index);
    add(index, o);
  }
  
  /**
   * Remove an object from the list.
   *
   * @param index  the position from which to remove the object
   */
  
  public void remove(int index)
  {
    try
    {
      suppressEvents++;
      getComponent().removeItemAt(index);
    }
    finally
    {
      suppressEvents--;
    }
  }
  
  /**
   * Remove all objects from the list.
   */
  
  public void removeAll()
  {
    try
    {
      suppressEvents++;
      getComponent().removeAllItems();
    }
    finally
    {
      suppressEvents--;
    }
  }

  /**
   * Get the number of items in the combo box's list.
   */
  
  public int getItemCount()
  {
    return getComponent().getItemCount();
  }
  
  /**
   * Get the item at a specific position in the list.
   */
  
  public Object getItem(int index)
  {
    return getComponent().getItemAt(index);
  }

  /**
   * Get the preferred number of rows which should be visible in the popup menu without using
   * a scrollbar.
   */
  
  public int getPreferredVisibleRows()
  {
    return getComponent().getMaximumRowCount();
  }
  
  /**
   * Get the preferred number of rows which should be visible in the popup menu without using
   * a scrollbar.
   */
  
  public void setPreferredVisibleRows(int rows)
  {
    getComponent().setMaximumRowCount(rows);
  }

  /**
   * Get the index of the selected item.  If this is an editable combo box and the current value
   * is not in the list, this returns -1.
   */
  
  public int getSelectedIndex()
  {
    return getComponent().getSelectedIndex();
  }

  /**
   * Select the item with the specified index.
   */
  
  public void setSelectedIndex(int index)
  {
    try
    {
      suppressEvents++;
      getComponent().setSelectedIndex(index);
    }
    finally
    {
      suppressEvents--;
    }
  }

  /**
   * Get the current value which has been selected from the list or, if this is an editable combo
   * box, typed into the box.
   */
  
  public Object getSelectedValue()
  {
    return getComponent().getSelectedItem();
  }

  /**
   * Set the value shown in the combo box.  If this combo box is not editable and the specified value
   * is not a member of the list, this will have no effect.
   */
  
  public void setSelectedValue(Object value)
  {
    try
    {
      suppressEvents++;
      getComponent().setSelectedItem(value);
    }
    finally
    {
      suppressEvents--;
    }
  }
}
