#!/usr/bin/runhugs

This program converts the Build-Essential (bess) list into a form
suitable for use in a Debian Depends control field.

> module Main (main) where

Due to problems with the Emacs Haskell major mode (see Bug#46115), the
string constants that include a certain magic word are imported from
another file.

> import Constants

> import Maybe
> import System

The list we are parsing is quite free-form.  Everything up to a
certain line (beginConstant) and everything after another certain line
(endConstant) are ignored, as is everything that is indented.  The
function bessDeComment filters out these comments; deComment weeds out
indented lines and stripSurroundings removes the head and the tail.

> bessDeComment :: String -> String
> bessDeComment = unlines . deComment . stripSurroundings . lines
>     where deComment :: [String] -> [String]
>           deComment = filter (not . isComment)
>               where isComment :: String -> Bool
>                     isComment [] = True
>                     isComment (c:_) = c == ' ' || c == '\t'
>           stripSurroundings :: [String] -> [String]
>           stripSurroundings = stripTail . stripHead
>           stripHead :: [String] -> [String]
>           stripHead = tail . dropWhile (/= beginConstant)
>           stripTail = takeWhile (/= endConstant)

The list contains entries that are meant for only some Debian
architectures.  filterByArchitecture assumes de-commented input and
outputs the input without those parts that are not meant for the
current architecture (which is given as the first argument to this
function).

> filterByArchitecture :: String -> String -> String
> filterByArchitecture arch = listToAlts . catMaybes .
>                             filterAltsByArch arch . altsToList
>     where listToAlts [] = ""
>           listToAlts s = join " | " s
>           altsToList = split '|'

Here we continue filtering out unwanted architectures.
filterAltsByArch assumes that the input is in the format used for the
Build-Depends control file field, with no commas or vertical lines.
This function is given a list of package-version-architecture clauses,
and it returns a list.  An element of the return value is either
Nothing, if the clause was not for this arch, or the clause without
the arch spec, lifted to Just, if it was for this arch.

> filterAltsByArch :: String -> [String] -> [Maybe String]
> filterAltsByArch arch ss = map (checkArch arch . parseAlt) ss

parseAlt takes a string of the form "package (>> version) [i386 m68k]"
and parses it into a triplet ("package", ">> version", ["i386",
"m68k"]).

> parseAlt :: String -> (String, String, [String])
> parseAlt s = let (pkg, rest)    = getPkg  . dropWSP $ s
>                  (vers, rest')  = getVer  . dropWSP $ rest
>                  (arch, rest'') = getArch . dropWSP $ rest'
>                  in (pkg, vers, arch)

dropWSP is what many people call "left strip" or "lstrip": it removes
all whitespace from the beginning of the string, so that the first
character of the result is not whitespace.

> dropWSP :: String -> String
> dropWSP = dropWhile (`elem` " \t\n")

getPkg parses "package ..." into ("package", " ..."), returning a pair
whose left element is the package name and the right element is what
remains of the string after the parse.

> getPkg :: String -> (String, String)
> getPkg = span (\c -> c /= ' ' && c /= '\t')

getVer parses "(>> version) ..." into (">> version", " ...").  If
there is no parenthesis at the beginning of the argument, the first
element of the return pair is empty and the second is the argument in
full.

> getVer :: String -> (String, String)
> getVer ('(':s) = let (f, r) = span (/= ')') s
>                      in (f, tail r)
> getVer s@(_:_) = ("", s)
> getVer "" = ("", "")

getArch parses an arch spec "[i386 !m68k] ..." into a pair (["i386",
"!m68k"], " ...") whose first argument is a list of the archs
mentioned in the spec.  The rest of the string is again returned in
the second element of the pair.

> getArch :: String -> ([String], String)
> getArch ('[':ss) = let (f, r) = span (/= ']') ss
>                        rv = (filter (not . onlySpace) . split ' ' $ f, tail r)
>                        in rv
>               where onlySpace [] = True
>                     onlySpace (' ':xs)  = onlySpace xs
>                     onlySpace ('\t':xs) = onlySpace xs
>                     onlySpace (_:_) = False
> getArch s@(_:_) = ([], s)
> getArch "" = ([], "")

checkArch takes two arguments: the first is a string, containing the
name of the current architecture; the second is a triplet produced by
parseAlt.  The result is Nothing, if the pair is not meant for the
current architecture.  Otherwise the result is Just string, where
string is a string representation of the triplet without the
architecture spec.  The logic that determines this is contained in
four intermediate results:

   - isInList is true iff current architecture is in the list without
     a bang

   - isInListWithBang is true iff current architecture is in the list
     with a bang

   - hasbang is true iff there is a bang in the version spec

   - test is what determines whether the return value is Just or Nothing.

> checkArch :: String -> (String, String, [String]) -> Maybe String
> checkArch _ t@(_,_,[]) = Just (stringify t)
> checkArch arch t@(_,_,a)
>     = let isInList = arch `elem` a
>           isInListWithBang = ('!' : arch) `elem` a
>           hasbang = hasBang a
>           test = (not isInList && not hasbang) || isInListWithBang
>           in if test
>              then Nothing
>                   else Just (stringify t)

stringify converts the triplet ("A", "B", ["C", "D"]) into the string
"A (B)", thereby converting the relationship spec into a string form.

> stringify :: (String, String, [String]) -> String
> stringify (p, "", _) = p
> stringify (p, v, _) = p ++ " (" ++ v ++ ")"

hasBang returns true if the argument list contains a string 
that starts with a bang.

> hasBang :: [String] -> Bool
> hasBang [] = False
> hasBang (('!':_):_) = True
> hasBang ((_:_):ss)  = hasBang ss

checkArgs tests whether the command line argument list given is in the
correct format.  If not, we bail out.

> checkArgs :: [String] -> IO ()
> checkArgs [] = fail "Too few arguments"
> checkArgs [x] = return () -- ok
> checkArgs (_:_:_) = fail "Too many arguments"

Here's the main program.  We parse arguments, read in the input from
stdin and then print out the result, which is in the normal Depends
field format.

> main = do args <- getArgs
>           checkArgs args
>           cont <- getContents
>           putStrLn . (join ", ") . filter (/= "") .
>                    map (filterByArchitecture (head args)) .
>                    lines . bessDeComment $ cont



split and join are utility functions that really should be in some
library.  split is a generalization of the standard filter "lines",
allowing one to specify what is used as the line separator.  join is
its conceptual inverse.

> join :: String -> [String] -> String
> join _ [] = []
> join _ [s] = s
> join is (s:ss@(_:_)) = s ++ join' is ss
>     where join' is = foldl (\a b -> a ++ is ++ b) []

> split  :: Char -> String -> [String]
> split _ "" =  []
> split c s  =  let (l, s') = break (== c) s
>                   in  l : case s' of
>                           []      -> []
>                           (_:s'') -> split c s''





