
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *
 * Define a structure that will represent any given IO of an operator.
 * Will require a name, buffer memory, flags (IO, Active, next, etc),
 *
 * These definitions declare the control structures for operator an IO control.
 * We need a separate set of structures that define the IO required by each
 * operator, which will be used by the GUI to understand what parameters need
	(*operator)->destroy = destroy;
 * to be represented to apply parameter changes to the operator.
 */

#ifndef _BRISTOL_H
#define _BRISTOL_H

//#define BRISTOL_DBG

#include "bristolaudio.h"

#include "bristoldebug.h"
#include "bristolmidiapi.h"

extern void * bristolmalloc();
extern void * bristolmalloc0();

#if (ADLIB == 2)
#define bAMD "hw:0,0"
#define bAAD "plughw:0,0"
#else
#define bAMD "0"
#define bAAD "0"
#endif ADLIB
#define bOMD "/dev/midi"
#define bOAD "/dev/audio"

#define BUFSZE 1024

#define BRISTOL_IO_COUNT 8
#define BRISTOL_PARAM_COUNT 8

#define BRISTOL_BUFSIZE BUFSZE

#define BRISTOL_ENGINE "bristolengine"

/*
 * The following are channels and flags. They are applied to bristol system
 * messages, and to the baudio.midiflags
 */
#define BRISTOL_SYSTEM 127
#define BRISTOL_HELLO			0x00002000
#define BRISTOL_COMMASK			0x00001f00
#define BRISTOL_PARAMMASK		0x000000ff
#define BRISTOL_REQSTART		0x00000100
#define BRISTOL_REQSTOP			0x00000200
#define BRISTOL_DEBUGON			0x00000300
#define BRISTOL_DEBUGOF			0x00000400
#define BRISTOL_VOICES			0x00000500
#define BRISTOL_INIT_ALGO		0x00000600
#define BRISTOL_MIDICHANNEL		0x00000700
#define BRISTOL_ALL_NOTES_OFF	0x00000800
#define BRISTOL_LOWKEY			0x00000900
#define BRISTOL_HIGHKEY			0x00000a00
#define BRISTOL_TRANSPOSE		0x00000b00
#define BRISTOL_HOLD			0x00000c00
#define BRISTOL_EVENT_KEYON		0x00000d00 // NOT SENT AS BRISTOL_SYSTEM
#define BRISTOL_EVENT_KEYOFF	0x00000e00 // NOT SENT AS BRISTOL_SYSTEM
#define BRISTOL_EVENT_PITCH		0x00000f00 // NOT SENT AS BRISTOL_SYSTEM
//#define BRISTOL_FREE			0x00000f00
#define BRISTOL_EXIT_ALGO		0x00001000

#define SYSTEM_CHANNEL			0x07f // used for initialisation requests.

#define BRISTOL_MINI			0x0000
#define BRISTOL_HAMMOND			0x0001
#define BRISTOL_PROPHET			0x0002
#define BRISTOL_DX				0x0003
#define BRISTOL_JUNO			0x0004
#define BRISTOL_EXPLORER		0x0005
#define BRISTOL_HAMMONDB3		0x0006
#define BRISTOL_VOX				0x0007
#define BRISTOL_RHODES			0x0008
#define BRISTOL_PROPHET10		0x0009
#define BRISTOL_MIXER			0x000a // THIS SHOULD BE MOVED TO END!!!
#define BRISTOL_SAMPLER			0x000b
#define BRISTOL_BRISTOL			0x000c
#define BRISTOL_DDD				0x000d
#define BRISTOL_PRO52			0x000e

/*
 * Audio interface types
 */
#define BRISTOL_AUDIOMASK		0xf0000000
#define BRISTOL_ALSA			0x00000000
#define BRISTOL_OSS				0x80000000
#define BRISTOL_SLAB			0x40000000
#define BRISTOL_JACK			0xc0000000
#define BRISTOL_LADSP			0x20000000

/*
 * MIDI interface types, may interact with audio drivers.
 */
#define BRISTOL_MIDIMASK		0x0f000000
#define BRISTOL_MIDI_ALSA		0x00000000
#define BRISTOL_MIDI_OSS		0x08000000
#define BRISTOL_MIDI_SLAB		0x04000000
#define BRISTOL_MIDI_JACK		0x0c000000
#define BRISTOL_MIDI_LADSP		0x02000000
#define BRISTOL_MIDI_SEQ		0x0a000000

#define BRISTOL_EXIT -1
#define BRISTOL_OK 0
#define BRISTOL_WAIT 1

/*
 * Global size management.
 */
#define BRISTOL_VPO 12.0
#define BRISTOL_VPO_DIV 10.5833333 /* 127 / this gives us our 12 per octave */

/*
 * Flags for IO management
 */
#define BRISTOL_DONE 4 /* Is this channel ready for use */
#define BRISTOL_GAIN 8 /* Do we have an active gain control on this channel */

/*
 * Parameter description structures.
 */
typedef struct BristolParamSpec {
	int int_val;
	float float_val;
	void *mem;
	float config_val;
	int midi_adj_algo;
	int midi_param;
	float config_amount; 
	float midi_amount; 
} bristolParamSpec;

/*
 * Modifier description structures.
 *
 * Define some modifier identifier flags:
 */
#define BRISTOL_PMOD_KEY 0 /* Apply a key (poly) index modifier */
#define BRISTOL_PMOD_PRESS 1 /* Apply a polypressure (poly) modifier */
/*
 * The rest are channel mods, and apply to all voices on bristolSound
 */
#define BRISTOL_CMOD_CONT_0 0
#define BRISTOL_CMOD_CONT_1 1
#define BRISTOL_CMOD_CONT_2 2
/* ... up to 32 controllers */
#define BRISTOL_CMOD_CONT_30 30
#define BRISTOL_CMOD_CONT_31 31
#define BRISTOL_CMOD_PITCHWHEEL 32
#define BRISTOL_CMOD_CHANPRESS 33
/*
 * And some control flags to define what is applied where.
 */
#define BRISTOL_CHANMOD 1 /* Apply a channel mod: chanpress, controller, etc */
#define BRISTOL_POLYMOD 2 /* Apply a polyphonic modifier: key or polypress */
#define BRISTOL_CHANMOD_INV 4 /* Invert value (ie, max - value) */
#define BRISTOL_POLYMOD_INV 8
#define BRISTOL_CHANMOD_CORRECT 0x10 /* Apply max / 2 - value */
#define BRISTOL_POLYMOD_CORRECT x020
#define BRISTOL_CHANMOD_UNIQUE 0x40 /* Apply uniquely, or adjust configured */
#define BRISTOL_POLYMOD_UNIQUE 0x80
typedef union BristolModSpec {
	unsigned int flags; /* type of modifiers to apply */
	int channelMod; /* index into CHANMOD table */
	int polyMod; /* index into PolyMod table */
	float cmGain;
	float pmGain;
} bristolModSpec;

typedef struct BristolOPParams {
	bristolParamSpec param[BRISTOL_PARAM_COUNT];
} bristolOPParams;

/*
 * Parameter management structures.
 */
typedef struct BristolParam {
	int type;
	int index;
	bristolParamSpec value;
	bristolModSpec modifiers;
} bristolParam;

/*
 * Voice structures
 */
#define BRISTOL_KEYON 1
#define BRISTOL_KEYOFF 2
#define BRISTOL_KEYDONE 4
#define BRISTOL_KEYNEW 8
#define BRISTOL_KEYREON 0x010
#define BRISTOL_KEYOFFING 0x020
/*
 * Actually configured into baudio->mixflags
 */
#define BRISTOL_MULTITRIG		0x00800000
typedef struct BristolVoice {
	struct BristolVoice *next;
	struct BristolVoice *last;
	struct BAudio *baudio;
	int index;
	unsigned int flags;
	char ***locals;
	/*
	 * We need an event structure for each possible poly event. These will be
	 * fead to the voice operators for mods, etc. This should be views as the
	 * PolyMod table, ie, there are only two polyphonic modifiers, the key id
	 * and any polypressure generated.
	 */
	keyMsg key;
	keyMsg keyoff;
	float velocity;
	pressureMsg pressure;
	float press;
	/*
	 * For polyponic portamento
	 */
	unsigned char lastkey;
	float dFreq; /* Desired step frequency index */
	float cFreq; /* Current step frequency index */
	float cFreqstep;
	float oFreq; /* Used to return pitchbend to original value */
	float chanpressure; /* Need a copy here */
} bristolVoice;

/*
 * IO Channel structures
 */
#define BRISTOL_AC 1
#define BRISTOL_DC 2
#define BRISTOL_INPUT 4
#define BRISTOL_OUTPUT 8
typedef struct BristolIO {
	char *IOname;     /* Name of this IO function */
	struct BristolOP *owner; /* Pointer to the operator that owns this IO */
	int index;        /* Index of this IO */
	unsigned int flags;        /* diverse bits */
	struct BristolIO *last; /* Previous operator on list */
	struct BristolIO *next; /* Next operator on list */
	float *bufmem;    /* buffer memory for this IO */
	int samplecnt;    /* number of samples in buffer */
	int samplerate;   /* on this IO */
	bristolModSpec modifiers; /* Midi modifiers to streamed input */
} bristolIO;

typedef int (*bristolAlgo)();

/*
 * These are used for templating, not for operational control. We could change
 * the naming, since it is rather close to bristolOPParams, which pass the run-
 * time parameters to the operating code.
 */
#define BRISTOL_INT 1
#define BRISTOL_FLOAT 2
#define BRISTOL_ENUM 3
#define BRISTOL_TOGGLE 4

#define BRISTOL_ROTARY 1
#define BRISTOL_SLIDER 2
#define BRISTOL_BUTTON 4
#define BRISTOL_HIDE 8
typedef struct BristolOPParam {
	char *pname;
	char *description;
	int type; /* need method of defining enumerations "Square, sine" etc, FFS */
	int low;
	int high;
	unsigned int flags;
} bristolOPParam;

typedef struct BristolOPIO {
	char *ioname;
	char *description;
	int samplerate;
	int samplecount;
	unsigned int flags; /* AC/DC, Hide op, others */
	float *buf;
} bristolOPIO;

typedef struct BristolOPSpec {
	char *opname;
	char *description;
	int pcount;
	bristolOPParam param[BRISTOL_PARAM_COUNT];
	int iocount;
	bristolOPIO io[BRISTOL_IO_COUNT];
	int localsize; /* size of table require for runtime locals */
} bristolOPSpec;

#define BRISTOL_SOUND_START 1
#define BRISTOL_SOUND_END 2
typedef struct BristolSound {
	char *name;
	int index; /* into operator list */
	int (*operate)(struct BristolOP *, bristolVoice *,
		bristolOPParams *, void *);
	//unsigned char *param; /* Param structure for that op. */
	bristolOPParams *param; /* Param structure for that op. */
	int next;
	unsigned int flags;
} bristolSound;

/*
 * Bristol Operator control structure.
 */
typedef struct BristolOP {
	int	index;    /* Index of this OP */
	unsigned int flags;    /* diverse bits */
	struct BristolOP *next; /* Next operator on list */
	struct BristolOP *last; /* Previous operator on list */
	/*
	 * These are used for any information that is managed internally
	 */
	bristolOPSpec *specs; /* any parameter specs for this OP */
	int size; /* Size of specs memory */
	/*
	 * A number of routines called to manage the operator
	 */
	bristolAlgo init;
	bristolAlgo destroy;
	bristolAlgo reset;
	int (*param)(struct BristolOP *, bristolOPParams *, unsigned char, float);
	int (*operate)(struct BristolOP *, bristolVoice *, bristolOPParams *,
		void *);
} bristolOP;

extern bristolOP *bristolOPinit();
extern bristolIO *bristolIOinit();

/*
 * This are mixflags: the system reserves the space 0xff000000, and a poly
 * algorithm can use the rest.
 */
#define BRISTOL_MUST_PRE	0x01000000
#define BRISTOL_HAVE_OPED	0x02000000
#define BRISTOL_HOLDDOWN	0x04000000
#define BRISTOL_REMOVE		0x08000000
#define A_440				0x10000000
#define MASTER_ONOFF		0x20000000
#define BRISTOL_STEREO		0x40000000
#define BRISTOL_KEYHOLD		0x80000000

/*
 * Audio globals structure.
 */
typedef struct BAudio {
	struct BAudio *next;
	struct BAudio *last;
	int soundCount;
	bristolSound **sound; /* operator instance sequences */
	bristolSound **effect;
	int (*param)(struct BAudio *, u_char, u_char, float); /* param change */
	bristolAlgo preops; /* Pre polyphonic (ie, monophonic) voicing routine */
	bristolAlgo operate; /* Polyphonic voice mixing routine */
	bristolAlgo postops; /* Post polyphonic voicing routine: FX, etc. */
	bristolAlgo destroy; /* Voice destruction routine */
	bristolVoice *firstVoice;
	int debuglevel;
	/*
	 * This should become a generic midi modifier table.
	 */
	float contcontroller[MIDI_CONTROLLER_COUNT];
	chanPressMsg chanPress;
	float chanpressure;
	int sid;
	int controlid;
	int midichannel;
	char ***locals; /* Unique locals per voice, a basic necessity. */
	void ***params; /* Unique params per voice, for midi poly support, etc. */
	char ***FXlocals;
	unsigned int mixflags;
	unsigned int midiflags;
	/*
	 * These need to remain to the baudio structure
	 */
	float extgain;
	float glide;
	float gtune;
	float note_diff;
	float gain_diff;
	int transpose;
	float pitchwheel;
	int midi_pitch;
	float midi_pitchfreq;
	int cvoices;
	int lvoices;
	float *leftbuf;
	float *rightbuf;
	float ctab[DEF_TAB_SIZE];
	float *mixlocals;
	int voicecount;
	char lowkey;
	char highkey;
} Baudio;

typedef struct AudioMain {
	Baudio *audiolist;
	bristolVoice *playlist;
	bristolVoice *playlast;
	bristolOP **palette; /* operator templates */
	bristolOP **effects; /* operator templates */
	int voiceCount;
	int opCount;
	int debuglevel;
	unsigned int flags;
	int iosize;
	int preload;
	int samplecount;
	int segmentsize;
	int samplerate;
	int atReq;
	int atStatus;
	int mtReq;
	int mtStatus;
	unsigned int midiflags;
	int s440holder; // not used?
	char *audiodev; // both of these really need to become multiheaded?
	char *mididev;
	int port; // for tcp connecctions
} audioMain;

extern int cleanup();

extern Baudio *findBristolAudio(Baudio *, int, int);
extern Baudio *findBristolAudioByChan(Baudio *, int);
extern bufmerge(float *, float, float *, float, int);

#ifndef NULL
#define NULL 0
#endif

#endif /* _BRISTOL_H */

