#ifndef _NOTE_H_
#define _NOTE_H_

#include "event.h"
#include "table.h"
#include "expression.h"
#include "bow.h"

class Ornament;
class Vector;
class Lyrics;
class RiemannFunction;
class Tuplet;

enum {
  NS_STACCATO, NS_LEGATO, NS_ACCDOWN, NS_ACCUP, NS_ACCENT1, NS_ACCENT2, NS_ACCENT3, NS_ACCENT4,
  NS_ACCENT5, NS_ACCENT6, NS_ACCENT7, NS_ACCENT8, NS_ACCENT9, NS_ACCENT10, NS_ACCENT11, NS_TRILLER,
  NS_TRILL1, NS_TRILL2, NS_INF, NS_BOWUP, NS_BOWDOWN };

enum {
  AS_LABEL1, AS_LABEL2, AS_LABEL3, AS_CASE1, AS_CASE2, AS_PEDAL1, AS_PEDAL2, AS_NOTEDOT,
  AS_NOTE3, AS_NOTE6, AS_NOTEXX, AS_8VA, AS_15VA, AS_TRILLX, AS_TRILLX2, AS_DECRESCENDO,
  AS_CRESCENDO, AS_BRACKETUP, AS_BRACKETDOWN, AS_ARPEGGIO, AS_TEXT,
  DYN_PPP, DYN_PP, DYN_P, DYN_MP, DYN_MF, DYN_F,
  DYN_FF, DYN_FFF, DYN_SFZ, DYN_SF, DYN_SFF, DYN_FP };

/**
 * The note event contains a pitch, velocity, and the midi channel. As well as some visual information like a vector of ornaments, the enharmonic
 * shift (which of course contains measurable harmonical information in theory, but for midi music - as well as generally in the well tempered world -
 * it is a visual information only) and a Tuplet object, used to specify the properties of tuplets.
 *
 * Remember always to make use of the AddElement operation, when adding a new note n to a part p, where sonG is the globally defined song instance:
 * <pre>
 *
 * #include <song.h>
 *
 * extern Song * sonG;
 *
 * ...
 * 
 * Part * p = ...
 * Note * n = new Note(pitch, velocity, duration, position, enhshft);
 * sonG->doo(new AddElement(n, p));
 * </pre>
 *
 * Bows and Ornaments can be added to and removed from a note. So far the following ornaments exist:
 * <pre>
 * NS_STACCATO - staccato
 * NS_LEGATO   - legato
 * NS_ACCDOWN  - accent pointing down
 * NS_ACCUP    - accent pointing up
 * NS_ACCENT1  - different accents
 * NS_ACCENT2  - ...
 * NS_ACCENT3
 * NS_ACCENT4
 * NS_ACCENT5
 * NS_ACCENT6
 * NS_ACCENT7
 * NS_ACCENT8
 * NS_ACCENT9
 * NS_ACCENT10
 * NS_ACCENT11
 * NS_TRILLER  - triller
 * NS_TRILL1   - triller
 * NS_TRILL2   - triller
 * NS_INF      - ...
 * NS_BOWUP    - a bow up
 * NS_BOWDOWN  - a bow down
 * </pre>
 **/
class Note : public Event
{
 private:
  short int _pitch;
  short int _vel;
  short int _prop;
  short int _chan;
  Tuplet *  _tuplet;
  Vector *  _ornament;
  char *    _cPitch;

 public:
  /**
   * Default constructor
  **/
  Note();

  /**
   * Constructor used by copy()
  **/
  Note(const Note&);

  /**
   * Constructor to define a new note
  **/
  Note(int pitch, int vel, long len, Position pos, int enh, int tup=0, int c=-1);

  /**
   * Constructor to define a new note
  **/
  Note(char* pitch, int vel, long len, Position pos, int enh, int tup=0, int c=-1);

  /**
   * This destructor gets rid of the ornament vector
  **/
  ~Note();

  /**
   * Returns the pitch
  **/
  int pitch() const { return _pitch; }

  /**
   * Returns the velocity
  **/
  int vel() const { return _vel; }

  /**
   * Returns the enharmonic shift
  **/
  int enh() const { return (_prop & MASK_ENH)-2; }

  /**
   * Returns the channel. -1 invokes the default (part defined) channel
  **/
  int chan() const { return _chan; }

  /**
   * Returns the ornament-vector
  **/
  Vector * ornament() const { return _ornament; }

  /**
   * The stem can be STEM_UP, STEM_DOWN or STEM_AUTO
  **/
  int stem() const { return (_prop & MASK_STEM) >> SHIFT_STEM; }

  /**
   * Returns true if this is not to be displayed as part of a group of notes
  **/
  bool nogroup() const { return ((_prop & MASK_NOGROUP) >> SHIFT_NOGROUP == 1); }

  /**
   * The display is the value (in ticks) to displayed in the score presentation. In most cases
   * this equals the the duration. Only in case of tuplets, the width has to be shortened
   * (the total width of three eighths triplets is the same as the duration of a regular quarter)
  
  **/
  long display(int res = 1) const;

  /**
   * Returns the tuplet of this note (may be 0 !)
  **/
  Tuplet * tuplet() const;

  /**
   * The base of a tuplet (3 for triplets, 7 for septlets, etc)
   **/
  int tupletBase() const;

  /**
   * The total tuplet duration
  **/
  int tupletDuration() const;

  /**
   * Tuplet with base and duration
  **/
  void tuplet(int base, int duration);

  /**
   * Sets the pitch
  **/
  void setPitch(int f);

  /**
   * Sets the velocity
  **/
  void setVel(int v);

  /**
   * Sets the enharmonic shift
  **/
  void setEnh(int e);

  /**
   * Sets the MIDI channel
  **/
  void setChan(int c);

  /**
   * Sets the stem (STEM_UP, STEM_DOWN or STEM_AUTO)
  **/
  void setStem(int);

  /**
   * Lets this note not to be displayed as a part of a group of notes
  **/
  void setNogroup(bool);

  /**
   * Lets this note be displayed as a tuplet (base 3 = triplet)
  **/
  void setTuplet(Tuplet * tp);

  /**
   * Adds an ornament to the ornament vector
  **/
  void add(Ornament*);

  /**
   * Removes an ornament from the ornament vector.
   * The ornament is not deleted here, deletion is done in AddOrnament's destructor
   **/
  void remove(Ornament*);

  /**
   * Returns an ornament of the ornament vector, and creates it, if necessary
  **/
  Expression * setExpression(int);

  /**
   * Returns an ornament of the ornament vector, and creates it, if necessary
  **/
  Bow * setBow(int,int,int);

  /**
   * Returns the note's lyrics. No new lyrics are created if they don't exist yet.
  **/
  Lyrics * lyrics();

  /**
   * Returns the note's riemann function. No new function is created if it doesn't exist yet.
  **/
  RiemannFunction * function();


  /**
   * Extracts the expresseions from the note into a vector and returns this vector
  **/
  Vector * removeExp();

  /**
   * Extracts the bow(s) from the note into a vector and returns this vector
  **/
  Vector * removeBow();

  /**
   * Returns the pitch in a nice character presentation
  **/
  char * cPitch();

  /**
   * Implementation of the print method
  **/
  virtual ostream & print(int,ostream&) const;

  /**
   * Implementation of the flush method
  **/
  virtual void flush(const char*) const;

  /**
   * Implementation of the copy method
  **/
  virtual Element * copy() const;

  /**
   * Implementation of the load method
  **/
  static Element * load(char*,ifstream*&,Element*);

private:
  static const int MASK_ENH = 7;
  static const int MASK_STEM = 24;
  static const int MASK_NOGROUP = 32;

  static const int SHIFT_STEM = 3;
  static const int SHIFT_NOGROUP = 5;

};


#endif
