#include "config.h"
#if HAVE_GPT_PARTITION

/*
 * Support for parsing GPT partition tables, as used by EFI firmware on
 * ia64.  May be needed for newer ia32 boxes one day.  GPT doesn't have
 * partition types in the same sense as MSDOS-style tables, and has to
 * parse the partition contents to determine a type.  This means that
 * if you use GPT tables you have to use parted to create the filesystem
 * (or swap space), otherwise libfdisk wont be able to report the 
 * partition types properly to dbootstrap.
 *
 * This code is mostly lifted from parted 1.4.24 source, by
 * Richard Hirst <rhirst@linuxcare.com>
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <asm/types.h>
#include <sys/utsname.h>
#include <sys/ioctl.h>
#include <uuid/uuid.h>
#include "byteorder.h"

#include "fdisk.h"

extern int get_kernel_revision(void);	/* in partbl_msdos.c, move to fdisk.c ? */

/* from linux/fs.h */
#define BLKGETSIZE _IO(0x12,96)		/* return device size */
#define BLKGETLASTSECT  _IO(0x12,108)	/* get last sector of block device */

struct blkdev_ioctl_param {
	unsigned int block;
	size_t content_length;
	char * block_contents;
};


typedef struct {
	__u8 boot_ind;		/* 0x80 - active */
	__u8 head;		/* starting head */
	__u8 sector;		/* starting sector */
	__u8 cyl;		/* starting cylinder */
	__u8 sys_ind;		/* What partition type */
	__u8 end_head;		/* end head */
	__u8 end_sector;	/* end sector */
	__u8 end_cyl;		/* end cylinder */
	__u32 start_sect;	/* starting sector counting from 0 */
	__u32 nr_sects;		/* nr of sectors in partition */
} MsdosPartitionEntry_t;

#define GPT_PRIMARY_HEADER_LBA	1
#define GPT_BLOCK_SIZE		512
#define MSDOS_LABEL_MAGIC	0xAA55
#define GPT_HEADER_SIGNATURE	0x5452415020494645

typedef struct {
        uint32_t time_low;
        uint16_t time_mid;
        uint16_t time_hi_and_version;
        uint8_t  clock_seq_hi_and_reserved;
        uint8_t  clock_seq_low;
        uint8_t  node[6];
} __attribute__ ((packed)) efi_guid_t;

efi_guid_t partition_swap_guid =
	((efi_guid_t) { 0x0657fd6d, 0xa4ab, 0x43c4, 0x84, 0xe5, { 0x09, 0x33, 0xc8, 0x4b, 0x4f, 0x4f }});

typedef struct _GuidPartitionTableHeader_t {
	uint64_t Signature;
	uint32_t Revision;
	uint32_t HeaderSize;
	uint32_t HeaderCRC32;
	uint32_t Reserved1;
	uint64_t MyLBA;
	uint64_t AlternateLBA;
	uint64_t FirstUsableLBA;
	uint64_t LastUsableLBA;
	efi_guid_t DiskGUID;
	uint64_t PartitionEntryLBA;
	uint32_t NumberOfPartitionEntries;
	uint32_t SizeOfPartitionEntry;
	uint32_t PartitionEntryArrayCRC32;
	uint8_t Reserved2[GPT_BLOCK_SIZE - 92];
} __attribute__ ((packed)) GuidPartitionTableHeader_t;

typedef struct _GuidPartitionEntryAttributes_t {
	uint64_t RequiredToFunction:1;
	uint64_t Reserved:47;
        uint64_t GuidSpecific:16;
} __attribute__ ((packed)) GuidPartitionEntryAttributes_t;

typedef struct _GuidPartitionEntry_t {
	efi_guid_t PartitionTypeGuid;
	efi_guid_t UniquePartitionGuid;
	uint64_t StartingLBA;
	uint64_t EndingLBA;
	GuidPartitionEntryAttributes_t Attributes;
	uint16_t PartitionName[36];
} __attribute__ ((packed)) GuidPartitionEntry_t;


/* Identify linux swap partitions */

static int
is_swap(int fd, uint64_t lba)
{
	int pagesize = getpagesize();
	uint8_t *p = malloc(pagesize);
	int i, res = 0;

	for (i = 0; i < pagesize/512; i++) {
		if (!sread(fd, lba + i, p + i * 512)) {
			free(p);
			return 0;
		}
	}
	if (!strncmp(p + pagesize - 10, "SWAP-SPACE", 10))
		res = 1;
	else if (!strncmp(p + pagesize - 10, "SWAPSPACE2", 10))
		res = 1;
	free(p);

	return res;
}

/* Identify linux ext2/ext3 filesystems */

#define EXT2_SUPER_MAGIC(sb)		(__le16_to_cpu((sb)->s_magic))
#define EXT2_SUPER_FEATURE_COMPAT(sb)	(__le32_to_cpu((sb)->s_feature_compat))
#define EXT2_SUPER_MAGIC_CONST		0xEF53
#define EXT3_FEATURE_COMPAT_HAS_JOURNAL	0x0004

struct ext2_super_block
{
	__u32	s_inodes_count;		/* Inodes count */
	__u32	s_blocks_count;		/* Blocks count */
	__u32	s_r_blocks_count;	/* Reserved blocks count */
	__u32	s_free_blocks_count;	/* Free blocks count */
	__u32	s_free_inodes_count;	/* Free inodes count */
	__u32	s_first_data_block;	/* First Data Block */
	__u32	s_log_block_size;	/* Block size */
	__s32	s_log_frag_size;	/* Fragment size */
	__u32	s_blocks_per_group;	/* # Blocks per group */
	__u32	s_frags_per_group;	/* # Fragments per group */
	__u32	s_inodes_per_group;	/* # Inodes per group */
	__u32	s_mtime;		/* Mount time */
	__u32	s_wtime;		/* Write time */
	__u16	s_mnt_count;		/* Mount count */
	__s16	s_max_mnt_count;	/* Maximal mount count */
	__u16	s_magic;		/* Magic signature */
	__u16	s_state;		/* File system state */
	__u16	s_errors;		/* Behaviour when detecting errors */
	__u16	s_minor_rev_level;	/* minor revision level */
	__u32	s_lastcheck;		/* time of last check */
	__u32	s_checkinterval;	/* max. time between checks */
	__u32	s_creator_os;		/* OS */
	__u32	s_rev_level;		/* Revision level */
	__u16	s_def_resuid;		/* Default uid for reserved blocks */
	__u16	s_def_resgid;		/* Default gid for reserved blocks */
	/*
	 * These fields are for EXT2_DYNAMIC_REV superblocks only.
	 *
	 * Note: the difference between the compatible feature set and
	 * the incompatible feature set is that if there is a bit set
	 * in the incompatible feature set that the kernel doesn't
	 * know about, it should refuse to mount the filesystem.
	 * 
	 * e2fsck's requirements are more strict; if it doesn't know
	 * about a feature in either the compatible or incompatible
	 * feature set, it must abort and not try to meddle with
	 * things it doesn't understand...
	 */
	__u32	s_first_ino;		/* First non-reserved inode */
	__u16	s_inode_size;		/* size of inode structure */
	__u16	s_block_group_nr;	/* block group # of this superblock */
	__u32	s_feature_compat;	/* compatible feature set */
	__u32	s_feature_incompat;	/* incompatible feature set */
	__u32	s_feature_ro_compat;	/* readonly-compatible feature set */
	__u8	s_uuid[16];		/* 128-bit uuid for volume */
	char	s_volume_name[16];	/* volume name */
	char	s_last_mounted[64];	/* directory where last mounted */
	__u32	s_algorithm_usage_bitmap;  /* For compression */
	/*
	 * Performance hints.  Directory preallocation should only
	 * happen if the EXT2_COMPAT_PREALLOC flag is on.
	 */
	__u8	s_prealloc_blocks;	/* Nr of blocks to try to preallocate*/
	__u8	s_prealloc_dir_blocks;	/* Nr to preallocate for dirs */
	__u16	s_padding1;
	/*
	 * Journaling support valid if EXT2_FEATURE_COMPAT_HAS_JOURNAL set.
	 */
	__u8	s_journal_uuid[16];	/* uuid of journal superblock */
	__u32	s_journal_inum;		/* inode number of journal file */
	__u32	s_journal_dev;		/* device number of journal file */
	__u32	s_last_orphan;		/* start of list of inodes to delete */

	__u32	s_reserved[197];	/* Padding to the end of the block */
};

static int is_ext2(int fd, uint64_t lba)
{
	struct ext2_super_block *sb;
	uint8_t buf[1024];

	sb = (struct ext2_super_block *)buf;
	if (!sread(fd, lba+2, buf) || !sread(fd, lba+3, buf+512) ||
			EXT2_SUPER_MAGIC(sb) != EXT2_SUPER_MAGIC_CONST ||
			(EXT2_SUPER_FEATURE_COMPAT(sb) & EXT3_FEATURE_COMPAT_HAS_JOURNAL))
		return 0;

	return 1;
}

static int is_ext3(int fd, uint64_t lba)
{
	struct ext2_super_block *sb;
	uint8_t buf[1024];

	sb = (struct ext2_super_block *)buf;
	if (!sread(fd, lba+2, buf) || !sread(fd, lba+3, buf+512) ||
			EXT2_SUPER_MAGIC(sb) != EXT2_SUPER_MAGIC_CONST ||
			!(EXT2_SUPER_FEATURE_COMPAT(sb) & EXT3_FEATURE_COMPAT_HAS_JOURNAL))
		return 0;

	return 1;
}

/* identify xfs partitions */

typedef __uint32_t	xfs_agblock_t;	/* blockno in alloc. group */
typedef	__uint32_t	xfs_extlen_t;	/* extent length in blocks */
typedef	__uint32_t	xfs_agnumber_t;	/* allocation group number */
typedef __int32_t	xfs_extnum_t;	/* # of extents in a file */
typedef __int16_t	xfs_aextnum_t;	/* # extents in an attribute fork */
typedef	__int64_t	xfs_fsize_t;	/* bytes in a file */
typedef __uint64_t	xfs_ufsize_t;	/* unsigned bytes in a file */
typedef	__int32_t	xfs_suminfo_t;	/* type of bitmap summary info */
typedef	__int32_t	xfs_rtword_t;	/* word type for bitmap manipulations */
typedef	__int64_t	xfs_lsn_t;	/* log sequence number */
typedef	__int32_t	xfs_tid_t;	/* transaction identifier */
typedef	__uint32_t	xfs_dablk_t;	/* dir/attr block number (in file) */
typedef	__uint32_t	xfs_dahash_t;	/* dir/attr hash value */
typedef __uint16_t	xfs_prid_t;	/* prid_t truncated to 16bits in XFS */
typedef __uint64_t	xfs_dfsbno_t;	/* blockno in filesystem (agno|agbno) */
typedef __uint64_t	xfs_drfsbno_t;	/* blockno in filesystem (raw) */
typedef	__uint64_t	xfs_drtbno_t;	/* extent (block) in realtime area */
typedef	__uint64_t	xfs_dfiloff_t;	/* block number in a file */
typedef	__uint64_t	xfs_dfilblks_t;	/* number of blocks in a file */
typedef __uint64_t	xfs_ino_t;

#define XFS_SB_MAGIC	0x58465342	/* 'XFSB' */

typedef struct xfs_sb
{
	__uint32_t	sb_magicnum;	/* magic number == XFS_SB_MAGIC */
	__uint32_t	sb_blocksize;	/* logical block size, bytes */
	xfs_drfsbno_t	sb_dblocks;	/* number of data blocks */
	xfs_drfsbno_t	sb_rblocks;	/* number of realtime blocks */
	xfs_drtbno_t	sb_rextents;	/* number of realtime extents */
	uuid_t		sb_uuid;	/* file system unique id */
	xfs_dfsbno_t	sb_logstart;	/* starting block of log if internal */
	xfs_ino_t	sb_rootino;	/* root inode number */
	xfs_ino_t	sb_rbmino;	/* bitmap inode for realtime extents */
	xfs_ino_t	sb_rsumino;	/* summary inode for rt bitmap */
	xfs_agblock_t	sb_rextsize;	/* realtime extent size, blocks */
	xfs_agblock_t	sb_agblocks;	/* size of an allocation group */
	xfs_agnumber_t	sb_agcount;	/* number of allocation groups */
	xfs_extlen_t	sb_rbmblocks;	/* number of rt bitmap blocks */
	xfs_extlen_t	sb_logblocks;	/* number of log blocks */
	__uint16_t	sb_versionnum;	/* header version == XFS_SB_VERSION */
	__uint16_t	sb_sectsize;	/* volume sector size, bytes */
	__uint16_t	sb_inodesize;	/* inode size, bytes */
	__uint16_t	sb_inopblock;	/* inodes per block */
	char		sb_fname[12];	/* file system name */
	__uint8_t	sb_blocklog;	/* log2 of sb_blocksize */
	__uint8_t	sb_sectlog;	/* log2 of sb_sectsize */
	__uint8_t	sb_inodelog;	/* log2 of sb_inodesize */
	__uint8_t	sb_inopblog;	/* log2 of sb_inopblock */
	__uint8_t	sb_agblklog;	/* log2 of sb_agblocks (rounded up) */
	__uint8_t	sb_rextslog;	/* log2 of sb_rextents */
	__uint8_t	sb_inprogress;	/* mkfs is in progress, don't mount */
	__uint8_t	sb_imax_pct;	/* max % of fs for inode space */
					/* statistics */
	/*
	 * These fields must remain contiguous.  If you really
	 * want to change their layout, make sure you fix the
	 * code in xfs_trans_apply_sb_deltas().
	 */
	__uint64_t	sb_icount;	/* allocated inodes */
	__uint64_t	sb_ifree;	/* free inodes */
	__uint64_t	sb_fdblocks;	/* free data blocks */
	__uint64_t	sb_frextents;	/* free realtime extents */
	/*
	 * End contiguous fields.
	 */
	xfs_ino_t	sb_uquotino;	/* user quota inode */
	xfs_ino_t	sb_gquotino;	/* group quota inode */
	__uint16_t	sb_qflags;	/* quota flags */
	__uint8_t	sb_flags;	/* misc. flags */
	__uint8_t	sb_shared_vn;	/* shared version number */
	xfs_extlen_t	sb_inoalignmt;	/* inode chunk alignment, fsblocks */
	__uint32_t	sb_unit;	/* stripe or raid unit */
	__uint32_t	sb_width;	/* stripe or raid width */	
	__uint8_t	sb_dirblklog;	/* log2 of dir block size (fsbs) */
        __uint8_t       sb_dummy[7];    /* padding */
} xfs_sb_t;

static int
is_xfs(int fd, uint64_t lba)
{
	union {
		struct xfs_sb	sb;
		char		bytes [512];
	} buf;

	return sread(fd, lba, &buf) && 
		(__le32_to_cpu(buf.sb.sb_magicnum) == XFS_SB_MAGIC ||
		 __be32_to_cpu(buf.sb.sb_magicnum) == XFS_SB_MAGIC);
}

/* Identify FAT filesystems */

struct FatBootSector_s {
	__u8	boot_jump[3];	/* 00: Boot strap short or near jump */
	__u8	system_id[8];	/* 03: system name */
	__u16	sector_size;	/* 0b: bytes per logical sector */
	__u8	cluster_size;	/* 0d: sectors/cluster */
	__u16	reserved;	/* 0e: reserved sectors */
	__u8	fats;		/* 10: number of FATs */
	__u16	dir_entries;	/* 11: number of root directory entries */
	__u16	sectors;	/* 13: if 0, total_sect supersedes */
	__u8	media;		/* 15: media code */
	__u16	fat_length;	/* 16: sectors/FAT for FAT12/16 */
	__u16	secs_track;	/* 18: sectors per track */
	__u16	heads;		/* 1a: number of heads */
	__u32	hidden;		/* 1c: hidden sectors (partition start) */
	__u32	sector_count;	/* 20: no. of sectors (if sectors == 0) */

	union {
		/* FAT16 fields */
		struct {
			__u8	drive_num;	/* 24: */
			__u8	empty_1;	/* 25: */
			__u8	ext_signature;	/* 26: always 0x29 */
			__u32	serial_number;	/* 27: */
			__u8	volume_name [11];       /* 2b: */
			__u8	fat_name [8];	/* 36: */

			__u8	boot_code[448];	/* 3f: Boot code (or message) */
		} __attribute__ ((packed)) fat16;

		/* FAT32 fields */
		struct {
			__u32	fat_length;	/* 24: size of FAT in sectors */
			__u16	flags;		/* 28: bit8: fat mirroring, low4: active fat */
			__u16	version;        /* 2a: minor * 256 + major */
			__u32	root_dir_cluster;	/* 2c: */
			__u16	info_sector;    /* 30: */
			__u16	backup_sector;	/* 32: */

			__u8	empty_1 [12];	/* 34: */

			__u16	drive_num;	/* 40: */
			__u8	ext_signature;	/* 42: always 0x29 */
			__u32	serial_number;	/* 43: */
			__u8	volume_name [11];	/* 47: */
			__u8	fat_name [8];	/* 52: */

			__u8	boot_code[420];	/* 5a: Boot code (or message) */
		} __attribute ((packed)) fat32;
	} u;

	__u16	boot_sign;	/* 1fe: always 0xAA55 */
} __attribute__ ((packed));

typedef struct FatBootSector_s FatBootSector_t;

struct FatDirEntry_s {
	__u8		name[8];
	__u8		extension[3];
	__u8		attributes;
	__u8		is_upper_case_name;
	__u8		creation_time_low;      /* milliseconds */
	__u16		creation_time_high;
	__u16		creation_date;
	__u16		access_date;
	__u16		first_cluster_high;     /* for FAT32 */
	__u16		time;
	__u16		date;
	__u16		first_cluster;
	__u32		length;
} __attribute__ ((packed));

typedef struct FatDirEntry_s FatDirEntry_t;

#define MAX_FAT12_CLUSTERS	4086

/* returns partition type or zero */

static int
is_fatx(int fd, uint64_t lba)
{
	uint8_t buf[512];
	FatBootSector_t * bs = (FatBootSector_t *)buf;
	int first_cluster_sector;
	int logical_sector_size;
	int cluster_count;
	int cluster_sectors;
	int sector_count;
	int fat_offset;

	if (!sread(fd, lba, buf))
		return 0;
	if (__le16_to_cpu(bs->boot_sign) != 0xAA55)
		return 0;
	if (!bs->system_id[0])
		return 0;
	if (!bs->sector_size || __le16_to_cpu(bs->sector_size) % 512)
		return 0;
	if (!bs->reserved)
		return 0;
	if (bs->fats < 1 || bs->fats > 4)
		return 0;
	if (!__le16_to_cpu(bs->dir_entries))
		return PTYPE_WFAT32;

	logical_sector_size = __le16_to_cpu(bs->sector_size)/512;
	if (bs->sectors)
		sector_count = __le16_to_cpu(bs->sectors) * logical_sector_size;
	else
		sector_count = __le32_to_cpu(bs->sector_count) * logical_sector_size;
	cluster_sectors = __le16_to_cpu(bs->cluster_size) * logical_sector_size;
	fat_offset = __le16_to_cpu(bs->reserved) * logical_sector_size;

	first_cluster_sector = fat_offset +
		2 * __le16_to_cpu(bs->fat_length) +
		__le16_to_cpu(bs->dir_entries) / (512 / sizeof (FatDirEntry_t));
	cluster_count = (sector_count - first_cluster_sector) / cluster_sectors;

	if (cluster_count > MAX_FAT12_CLUSTERS)
		return PTYPE_DOS16L;
	else
		return PTYPE_DOS12;
}


/************************************************************
 * efi_crc32()
 * Requires:
 *  - a buffer of length len
 * Modifies: nothing
 * Returns:
 *  EFI-style CRC32 value for buf
 *  
 * This function uses the crc32 function by Gary S. Brown,
 * but seeds the function with ~0, and xor's with ~0 at the end.
 ************************************************************/

/* 
 * Dec 5, 2000 Matt Domsch <Matt_Domsch@dell.com>
 * - Copied crc32.c from the linux/drivers/net/cipe directory.
 * - Now pass seed as an arg
 * - changed unsigned long to uint32_t, added #include<stdint.h>
 * - changed len to be an unsigned long
 * - changed crc32val to be a register
 * - License remains unchanged!  It's still GPL-compatable!
 */

  /* ============================================================= */
  /*  COPYRIGHT (C) 1986 Gary S. Brown.  You may use this program, or       */
  /*  code or tables extracted from it, as desired without restriction.     */
  /*                                                                        */
  /*  First, the polynomial itself and its table of feedback terms.  The    */
  /*  polynomial is                                                         */
  /*  X^32+X^26+X^23+X^22+X^16+X^12+X^11+X^10+X^8+X^7+X^5+X^4+X^2+X^1+X^0   */
  /*                                                                        */
  /*  Note that we take it "backwards" and put the highest-order term in    */
  /*  the lowest-order bit.  The X^32 term is "implied"; the LSB is the     */
  /*  X^31 term, etc.  The X^0 term (usually shown as "+1") results in      */
  /*  the MSB being 1.                                                      */
  /*                                                                        */
  /*  Note that the usual hardware shift register implementation, which     */
  /*  is what we're using (we're merely optimizing it by doing eight-bit    */
  /*  chunks at a time) shifts bits into the lowest-order term.  In our     */
  /*  implementation, that means shifting towards the right.  Why do we     */
  /*  do it this way?  Because the calculated CRC must be transmitted in    */
  /*  order from highest-order term to lowest-order term.  UARTs transmit   */
  /*  characters in order from LSB to MSB.  By storing the CRC this way,    */
  /*  we hand it to the UART in the order low-byte to high-byte; the UART   */
  /*  sends each low-bit to hight-bit; and the result is transmission bit   */
  /*  by bit from highest- to lowest-order term without requiring any bit   */
  /*  shuffling on our part.  Reception works similarly.                    */
  /*                                                                        */
  /*  The feedback terms table consists of 256, 32-bit entries.  Notes:     */
  /*                                                                        */
  /*      The table can be generated at runtime if desired; code to do so   */
  /*      is shown later.  It might not be obvious, but the feedback        */
  /*      terms simply represent the results of eight shift/xor opera-      */
  /*      tions for all combinations of data and CRC register values.       */
  /*                                                                        */
  /*      The values must be right-shifted by eight bits by the "updcrc"    */
  /*      logic; the shift must be unsigned (bring in zeroes).  On some     */
  /*      hardware you could probably optimize the shift in assembler by    */
  /*      using byte-swap instructions.                                     */
  /*      polynomial $edb88320                                              */
  /*                                                                        */
  /*  --------------------------------------------------------------------  */

static uint32_t crc32_tab[] = {
      0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
      0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
      0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
      0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
      0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
      0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
      0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
      0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
      0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
      0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
      0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
      0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
      0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
      0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
      0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
      0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
      0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
      0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
      0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
      0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
      0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
      0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
      0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
      0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
      0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
      0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
      0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
      0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
      0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
      0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
      0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
      0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
      0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
      0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
      0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
      0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
      0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
      0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
      0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
      0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
      0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
      0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
      0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
      0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
      0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
      0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
      0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
      0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
      0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
      0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
      0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
      0x2d02ef8dL
   };

/* Return a 32-bit CRC of the contents of the buffer. */

uint32_t
__efi_crc32(const void *buf, unsigned long len, uint32_t seed)
{
	unsigned long i;
	register uint32_t crc32val;
	const unsigned char *s = buf;

	crc32val = seed;
	for (i = 0;  i < len;  i ++) {
		crc32val = crc32_tab[(crc32val ^ s[i]) & 0xff] ^
					(crc32val >> 8);
	}
	return crc32val;
}


static inline uint32_t
efi_crc32(const void *buf, unsigned long len)
{
	return (__efi_crc32(buf, len, ~0L) ^ ~0L);
}

#if 0
static int
efi_guidcmp(efi_guid_t *value, efi_guid_t *expected)
{
	/* This is bad, some versions of parted write the guid shifted right
	 * one byte.  Compiler bug, probably.
	 */
	uint8_t *val = (uint8_t *)value;
	uint8_t *exp = (uint8_t *)expected;

	if (!memcmp(val, exp, 16))
		return 0;
	else
		return memcmp(val+1, exp, 15);
}
#endif

static GuidPartitionEntry_t *
gpt_read_part_entries(int fd, GuidPartitionTableHeader_t * gpt)
{
	GuidPartitionEntry_t *ptes;
	unsigned int i, sectors;

	sectors = (gpt->NumberOfPartitionEntries * gpt->SizeOfPartitionEntry + 511)/512;
	ptes = (GuidPartitionEntry_t *) malloc(sectors * 512);
	if (ptes == NULL)
		return NULL;

	for (i = 0; i < sectors; i++) {
		if (!sread(fd, gpt->PartitionEntryLBA + i, (char *)ptes + i * 512)) {
			free(ptes);
			return NULL;
		}
	}

#if (__BYTE_ORDER != __LITTLE_ENDIAN)
	/* Fixup endianness */
	for (i=0; i<gpt->NumberOfPartitionEntries; i++) {
		gpt_le_guid_to_cpu(&ptes[i].PartitionTypeGuid);
		gpt_le_guid_to_cpu(&ptes[i].UniquePartitionGuid);
		ptes[i].StartingLBA = PED_LE64_TO_CPU(ptes[i].StartingLBA);
		ptes[i].EndingLBA   = PED_LE64_TO_CPU(ptes[i].EndingLBA);
		gpt_le_part_attributes_to_cpu(&ptes[i].Attributes);
		for (j=0; j<(72/sizeof(efi_char16_t)); j++) {
			ptes[i].PartitionName[j] = (efi_char16_t)(PED_LE16_TO_CPU((uint16_t)(ptes[i].PartitionName[j])));
		}
	}
#endif

	return ptes;
}

#if (__BYTE_ORDER == __LITTLE_ENDIAN)
#define le_to_cpu_gpt(x)
#else
static void le_to_cpu_gpt(GuidPartitionTableHeader_t *gpt)
{
	/* Fixup endianness */
	gpt->Signature                = PED_LE64_TO_CPU(gpt->Signature);
	gpt->Revision                 = PED_LE32_TO_CPU(gpt->Revision);
	gpt->HeaderSize               = PED_LE32_TO_CPU(gpt->HeaderSize);
	gpt->HeaderCRC32              = PED_LE32_TO_CPU(gpt->HeaderCRC32);
	gpt->Reserved1                = PED_LE32_TO_CPU(gpt->Reserved1);
	gpt->MyLBA                    = PED_LE64_TO_CPU(gpt->MyLBA);
	gpt->AlternateLBA             = PED_LE64_TO_CPU(gpt->AlternateLBA);
	gpt->FirstUsableLBA           = PED_LE64_TO_CPU(gpt->FirstUsableLBA);
	gpt->LastUsableLBA            = PED_LE64_TO_CPU(gpt->LastUsableLBA);
	gpt->PartitionEntryLBA        = PED_LE64_TO_CPU(gpt->PartitionEntryLBA);
	gpt->NumberOfPartitionEntries = PED_LE32_TO_CPU(gpt->NumberOfPartitionEntries);
	gpt->SizeOfPartitionEntry     = PED_LE32_TO_CPU(gpt->SizeOfPartitionEntry);
	gpt->PartitionEntryArrayCRC32 = PED_LE32_TO_CPU(gpt->PartitionEntryArrayCRC32);
	//gpt_le_guid_to_cpu(&gpt->DiskGUID);
}
#endif

static GuidPartitionEntry_t *
gpt_is_valid(int fd, GuidPartitionTableHeader_t *gpt, uint64_t lba)
{
	uint32_t crc, origcrc;
	GuidPartitionEntry_t *ptes;

	/* Check the GUID Partition Table Signature */
	if (gpt->Signature != GPT_HEADER_SIGNATURE)
		return NULL;

	/* Check the GUID Partition Table Header CRC */
	origcrc = gpt->HeaderCRC32;
	gpt->HeaderCRC32 = 0;
	crc = efi_crc32(gpt, gpt->HeaderSize);
	if (crc != origcrc) {
		gpt->HeaderCRC32 = origcrc;
		return NULL;
	}
	gpt->HeaderCRC32 = origcrc;

	/* Check that the MyLBA entry points to the LBA
	   that contains the GPT we read */
	if (gpt->MyLBA != lba)
		return NULL;

	if (!(ptes = gpt_read_part_entries(fd, gpt)))
		return NULL;

	/* Check the GUID Partition Entry Array CRC */
	crc = efi_crc32(ptes, gpt->NumberOfPartitionEntries *
			   gpt->SizeOfPartitionEntry);
	if (crc != gpt->PartitionEntryArrayCRC32) {
		free(ptes);
		return NULL;
	}

	/* We're done, all's well */
	return ptes;
}

/*
 * Return:
 *
 * 0    if this is not a GPT
 * -1   for serious error on disk
 * 1    if GPT partition found
 */

int parse_gpt_partition(char *device, int fd)
{
	unsigned char mbr[512];
	MsdosPartitionEntry_t *p;
	GuidPartitionTableHeader_t pgpt, agpt;
	GuidPartitionEntry_t *ppte, *apte, zeropte;
	int i, nextminor = 1;
	unsigned long devsectors;

	memset(&zeropte, 0, sizeof(zeropte));
	if (!sread( fd, 0, mbr ))
		return -1;
	if (*(unsigned short *) (0x1fe + mbr) != __cpu_to_le16(MSDOS_LABEL_MAGIC))
		return 0;
	p = (MsdosPartitionEntry_t *) (0x1be + mbr);
	if (p->sys_ind != PTYPE_GPT)
		return 0;
	if (!sread(fd, GPT_PRIMARY_HEADER_LBA, &pgpt))
		return -1;
	le_to_cpu_gpt(&pgpt);
	ppte = gpt_is_valid(fd, &pgpt, 1);
	if (ppte == NULL)
		return 0;
	/* Mess about to handle last sector on drives with odd number
	 * of sectors.  You currently need a kernel patch to support the
	 * ioctl.  Apparently wont be needed in 2.5.
	 */
	if (ioctl(fd, BLKGETSIZE, &devsectors)) {
		/* can't find device size */
		free(ppte);
		return 0;
	}
	else if ((devsectors & 1) && (pgpt.AlternateLBA == devsectors-1)) {
		struct blkdev_ioctl_param ioctl_param;

		ioctl_param.block = 0; /* read the last sector */
		ioctl_param.content_length = 512;
		ioctl_param.block_contents = (void *)&agpt;

		if (ioctl(fd, BLKGETLASTSECT, &ioctl_param) == -1) {
			free(ppte);
			return 0;
		}
	}
	else if (!sread(fd, pgpt.AlternateLBA, &agpt)) {
		free(ppte);
		return 0;
	}
	le_to_cpu_gpt(&agpt);
	apte = gpt_is_valid(fd, &agpt, pgpt.AlternateLBA);
	if (apte == NULL) {
		free(ppte);
		free(apte);
		return 0;
	}
	free(apte);

	for (i = 0; i < pgpt.NumberOfPartitionEntries; i++) {
		GuidPartitionEntry_t *pte = (GuidPartitionEntry_t *)((char *)ppte + i * pgpt.SizeOfPartitionEntry);
		if (memcmp(&zeropte, pte, sizeof(zeropte))) {
			int type, fattype;

			/* Try to identify partition type from contents; if that
			 * fails, check for SwapGUID, otherwise declare it unknown.
			 * dbootstrap wont offer it for swap unless we report it as
			 * swap.  Maybe we should insist people use parted to
			 * create swap and filesystems...
			 */
			if (is_swap(fd, pte->StartingLBA))
				type = PTYPE_LSWAP;
			else if (is_ext2(fd, pte->StartingLBA))
				type = PTYPE_LINUX;
			else if (is_ext3(fd, pte->StartingLBA))
				type = PTYPE_LINUX;
			else if (is_xfs(fd, pte->StartingLBA))
				type = PTYPE_LINUX;
			else if ((fattype = is_fatx(fd, pte->StartingLBA)))
				type = fattype;
#if 0
			else if (is_reiser(fd, pte->StartingLBA))
				type = PTYPE_LINUX;
			else if (!efi_guidcmp(&(pte->PartitionTypeGuid), &partition_swap_guid))
				type = PTYPE_LSWAP;
#endif
			else
				type = PTYPE_UNKNOWN;

			fdisk_add_partition(device, nextminor++, type, (pte->EndingLBA - pte->StartingLBA + 1)/2);
		}
	}
	free(ppte);

	return 1;
}
#endif

