/* Network configuration during Debian installation
 * 
 * 1. Get a list of all available network interfaces from /proc/net/dev
 * 2. Let the user chose thich one to configure (either manually or using
 *    DHCP/BOOTP): IP address, netmask, broadcast address, default gateway
 * 3. Ask for the hostname, domain name and DNS servers (use predefined
 *    values we might have received from a DHCP server)
 * 4. Write the network configuration to /etc/network/interfaces (non-PCMCIA
 *    devices, at least loopback) or /etc/pcmcia/network.opts (PCMCIA devices)
 * 5. Activate the configuration
 */

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <resolv.h>
#include <stdlib.h>
#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

/*
 * Strings and other constants
 */
#define NC_ETC_DIR         "/etc"
#define NC_NETWORKS_FILE   "/etc/networks"
#define NC_RESOLV_FILE     "/etc/resolv.conf"
#define NC_ETC_NETWORK_DIR "/etc/network"
#define NC_INTERFACES_FILE "/etc/network/interfaces"
#define NC_HOSTS_FILE      "/etc/hosts"
/* 
  Use 2.2.x because dhclient forces -q (quiet) and I like logging.
  Using dhclient would make adding 2.0.x support just a matter of 
  including the file in scripts/rootdisk/SMALL_BASE_LIST_all.
  Also, right now dhclient is not in SMALL_BASE_LIST_all either 
  since it doesn't do anything and it takes (a little) space.
*/
/* #define NC_DHCLIENT_FILE   "/sbin/dhclient" */
#define NC_DHCLIENT_FILE   "/sbin/dhclient-2.2.x"

#define MAX_INTERFACES (16)
#define IP4_ADDR_SZ (4)
#define IP4_ADDR_LN (3 * IP4_ADDR_SZ + 4)
typedef struct ip4_addr
{
  int i[IP4_ADDR_SZ];
} ip4_addr_t;


/*
 * Global variables for this module
 */
char *host = NULL;
char *domain = NULL;
char *transceiver = NULL;
char *netinterface = NULL;
char *nameserver = NULL;
ip4_addr_t ipaddr = {{192, 168, 1, 1}}; 
ip4_addr_t netmask = {{255, 255, 255, 0}};
ip4_addr_t network = {{192, 168, 1, 0}};
ip4_addr_t broadcast = {{192, 168, 1, 255}};
ip4_addr_t gateway = {{0, 0, 0, 0}};
ip4_addr_t peer = {{0, 0, 0, 0}};
int has_pointopoint = 0;
int has_gw = 0;
int has_pcmcia = 0;
int use_dhcp = 0;
struct stat nc_statbuf;

/*
 * Function prototypes
 */
extern int check_dir(const char *);
extern int activate_pcmcia();
int atoIP4(char *, ip4_addr_t *);
char *IP4toa(ip4_addr_t);
char *IP4tostr(char *, ip4_addr_t);
int ask_for_IP4(ip4_addr_t *, char *, char *, char *);
char *get_ifdsc(const char *);
int check_ip4_nameserver(char *);
int get_host();
int get_netinterface();
int get_ifconfig();
int get_domain();
int get_dnsconfig();
int get_pcmcia_transceiver();
int write_common_network();
int write_pcmcia_network();
int write_static_network();
int write_dhcp_network();
int activate_static_network();
int configure_static_network();


/*
 * Checks a string with an IPv4-address for validity and converts it into an
 * ip4_addr_t type;
 */
int atoIP4(char* addr, ip4_addr_t* iaddr)
{
  char* end;
  char* tmp_a;
  char* current = strdup(addr);
  char* next = NULL;
  int ix = 0;
  int tmp;

  tmp_a = current;
  end = current + strlen(current) + 1;
  while(next !=  end)
    {
      next = strchr(current, '.');
      if(next == NULL)
	next = end;
      else
	{
	  *next = '\0';
	  next++;
	}
      
      if(ix == IP4_ADDR_SZ)
	{
	  free(tmp_a);
	  return 255;
	}
      else
	{
	  tmp = atoi(current);
	  if ((tmp < 0) || (tmp > 255))
	    {
	      free(tmp_a);
	      return 255;
	    }
	  iaddr->i[ix++] = tmp;
	  current = next;
	}
    }
  free(tmp_a);
  if(ix != IP4_ADDR_SZ)
    return 255;
  return 0;
}

char* IP4toa(ip4_addr_t iaddr)
{
  int length = 3 * IP4_ADDR_SZ + 4;
  char* addr = calloc(length, sizeof(char));
  snprintf(addr, length, "%d.%d.%d.%d",
	   iaddr.i[0],
	   iaddr.i[1],
	   iaddr.i[2],
	   iaddr.i[3]);
  return addr;
}

char* IP4tostr(char *s, ip4_addr_t iaddr)
{
  snprintf(s, 3 * IP4_ADDR_SZ + 4, "%d.%d.%d.%d",
	   iaddr.i[0],
	   iaddr.i[1],
	   iaddr.i[2],
	   iaddr.i[3]);
  return s;
}

/* Ask for a generic IPv4 address */
int ask_for_IP4(ip4_addr_t *ip, char *title, char *mesg, char *err)
{
  char *def;
  char *addr = IP4toa(*ip);
  for (;;) {
    def = addr;
    addr = inputBox(title, mesg, def);
    free(def);
    if ((addr == NULL) || (strlen(addr) == 0)) {
      return 255;
    }
    if (atoIP4(addr, ip) != 0) {
      problemBox(err, _("Problem"));
    }
    else {
      break;
    }
  }
  free(addr);
  return 0;
}



/*
 * Get a description for an interface (i.e. "Ethernet" for ethX).
 */
char
*get_ifdsc (const char *ifp)
{
  int i;
  struct if_alist_struct {
    char *interface;
    char *description;
  } interface_alist[] = {
    /*
     * A _("string") is an element of a char *[], and the linker will
     * know where to find the elements.  This works with `pointerize';
     * perhaps that's different with `gettext'... though if it expands
     * to a function call, this initializer should still work fine.
     * Also see the 1.66 version of choose_cdrom(), which uses the
     * similar technique.  If it works there, it will work here.
     */
    {"eth",    _("Ethernet or Fast Ethernet")},
    {"pcmcia", _("PC-Card (PCMCIA) Ethernet or Token Ring")},
    {"tr",     _("Token Ring")},
    {"arc",    _("Arcnet")},
    {"slip",   _("Serial-line IP")},
    {"plip",   _("Parallel-line IP")},
    {NULL,     NULL}};

  for (i = 0; interface_alist[i].interface != NULL; i++)
    if (! strncmp(ifp, interface_alist[i].interface, 
                  strlen (interface_alist[i].interface)))
      return strdup (interface_alist[i].description);
  return NULL;
}



/* Check for valid IPv4 addesses in nameservers' string */
int check_ip4_nameserver(char *ns) {
  ip4_addr_t ip;
  char msg[100];
  char *buf = strdup(ns);
  char *cur = buf;
  char *tmp, *end;
  end = buf + strlen(ns);
  do
    {
    if ((tmp = strstr(cur, " ")) != NULL)
      *tmp = '\0';
    if (atoIP4(cur, &ip) != 0)
      {
      snprintf(msg, 100, _("%s isn't a valid IP address!"), cur);
      problemBox(msg, _("Problem"));
      free(buf);
      return -1;
      }
    cur = (tmp == NULL) ? end : tmp + 1;  
    }
  while (cur < end);
  free(buf);
  return 0;
}



/*
 * Get the hostname of this system
 */
int get_host()
{
  char *def = NULL;
  static const char *valid_hostname_chars =
	  "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-";
  int len;

  for (;;) {
    if(host)
      def = host;
    else
      def = strdup("debian");

    snprintf(prtbuf, PRTBUFSIZE,
             _("Every Debian system has a name - even if it is not on a network. This name is called its \"hostname\". You should now specify a hostname for your new system.\n\nIf your system is going to be on a network, you should talk to the administrator of the network before you choose a hostname. If not, you may choose any name you like.\n\nThe hostname must not contain dots or underscores, and you must not append the domain name here.\n\nIf you can't think of a hostname for your new system, you may press <ENTER> to use the default hostname of \"%s\".\nPlease enter your Hostname"), def);

    host = inputBox(prtbuf, _("Choose the Hostname"), def);
    free(def);
    def = NULL;

    if (host == NULL)
      return 255;

    if ((len = strlen(host)) == 0) {
      free(host);
      host = NULL;
      return 255;
    }

    /* Check the hostname for RFC 1123 compliance.  */
    if (!isalnum(host[0])) {
      problemBox(_("The hostname must begin with an alphanumeric character."),
		 _("Problem"));
    } else if (host[strlen(host) - 1] == '-') {
      problemBox(_("The hostname must not end in a minus sign."),
		 _("Problem"));
    } else if (host[strspn(host, valid_hostname_chars)] != '\0') {
      problemBox(_("The hostname may only contain alphanumeric characters and the minus sign."),
		 _("Problem"));
    } else if (!(2 <= len && len <= 63)) {
      problemBox(_("The hostname must contain between 2 and 63 characters."),
		 _("Problem"));
    } else {
      break;
    }
  }
  return 0;
}

/*
 * Get all available interfaces from the kernel and ask the user which one
 * he wants to configure
 */
int get_netinterface()
{
  int i = 0, ret, ifcnt = is_network_up(1); /* the total number of interfaces */
  struct d_choices *opt = calloc(ifcnt, sizeof(struct d_choices));
  char *inter;

  getif_start();
  while ((inter = getif(1)) != NULL) {
    opt[i].tag = strdup(inter);
    opt[i++].string = get_ifdsc(inter);
  }
  getif_end();

  if (ifcnt==0) {
    if(netinterface) {
      free(netinterface);
      netinterface=NULL;
    }
    INFOMSG("no network devices found");
    problemBox(_("No network interfaces were found.   That means that the installation system was unable to find a network device.  If you do have a network card, then it is possible that the module for it hasn't been selected yet.  Go back to the 'Configure Device Driver Modules' menu and load the required kernel modules first.\n\nIf your only network connection is PPP, don't worry -- you will be asked if you want PPP later in the process."),
               _("Problem"));
    ret = 0;
  } else if (ifcnt == 1) {
    /* only one interface found, so use it */
    netinterface = strdup(opt[0].tag);
    ret = 0;
  } else {
    ret = menuBox(_("Choose the type of your primary network interface that you will need for installing Debian (via NFS or HTTP)"),
		_("Choose Network Interface"), opt, ifcnt, 1);
    if (ret >= 0) {
      netinterface = strdup(opt[ret].tag);
      ret = 0;
    } else {
      if(netinterface) {
	free(netinterface);
        netinterface=NULL;
      }
      ret = 255;
    }
  }

  /* check if this is a Point-To-Point interface */
  has_pointopoint = 0;
  snprintf(prtbuf, PRTBUFSIZE, "ifconfig %s | grep -q POINTOPOINT", netinterface);
  if (execlog(prtbuf, LOG_INFO) == 0) has_pointopoint = 1;

  has_pcmcia = 0;
  if (netinterface && NAME_ISDIR("/proc/bus/pccard", &nc_statbuf)) {
    snprintf(prtbuf, PRTBUFSIZE, _("The interface %s (%s) seems to be a PCMCIA card. Is this correct?"), 
             netinterface, get_ifdsc(netinterface));
    if (bootargs.isquiet || yesNoBox(prtbuf, _("Confirmation")) == DLG_YES) {
      has_pcmcia = 1;
    }
  }

  for (i=0; i<ifcnt; i++) {
    free(opt[i].tag);
    free(opt[i].string);
  }
  free(opt);
  return ret;
}



/*
 * Get IPv4 address, netmask and default gateway
 */
int get_ifconfig() {
  int ret, ix;
#if #cpu (s390)
  /* consider netsetup information gathered during first installation step*/
  char * env_gateway = getenv("ip_gateway");
#endif

  /* Get the IP address */
  snprintf(prtbuf, PRTBUFSIZE,
           _("What is the IP address of system %s?"), host);
  ret = ask_for_IP4(&ipaddr, prtbuf,
                    _("Choose the IP Address"),
                    _("You entered an invalid IP address.  Ask your network administrator for the proper value."));
  if (ret == 255) return ret;

  if(!has_pointopoint) {
    /* Get the netmask */
    ret = ask_for_IP4(&netmask, _("What is your netmask?"),
                    _("Choose the Network Mask"),
		    _("You entered an invalid netmask.  Ask your network administrator for the proper value."));
    if (ret == 255) return ret;
  }
  else {
    /* netmask is 255.255.255.255 */
    for (ix = 0; ix < IP4_ADDR_SZ; ix++) {
      netmask.i[ix] = 255;
    }
  }

  /*
   * Generate the network, broadcast and default gateway address
   * Default gateway address is network with last number set to "1", 
   * or "2" if "1" is the local address.
   */
  if (!has_gw) {
    for (ix = 0; ix < IP4_ADDR_SZ; ix++) {
      gateway.i[ix] = network.i[ix] = ipaddr.i[ix] & netmask.i[ix];
      broadcast.i[ix] = (~netmask.i[ix] & 255) | ipaddr.i[ix];
    }
    gateway.i[IP4_ADDR_SZ - 1] |=
      (ipaddr.i[IP4_ADDR_SZ - 1] == (network.i[IP4_ADDR_SZ - 1] | 1)) ? 2 : 1;
  }

#if #cpu (s390)
  /* use gateway from initial netsetup */
  if(env_gateway)
    sscanf(env_gateway, "%d.%d.%d.%d", &gateway.i[0], &gateway.i[1], &gateway.i[2], &gateway.i[3]);
#endif

  /* Get the gateway */
  has_gw = 1;
  if(!has_pointopoint) {
    ret = ask_for_IP4(&gateway, _("A gateway system is one that connects your network to other networks such as your company wide-area net or the Internet.\n\nIf there is no gateway in your subnet just leave this field blank."),
                    _("What is your IP gateway address?"),
                    _("You entered an invalid gateway address.  Ask your network administrator for the proper value."));
    if (ret == 255) has_gw = 0;
  }
  else {
    /* don't overwrite peer; important at least for s390 */
    if(peer.i[0] == 0) {
      for (ix = 0; ix < IP4_ADDR_SZ; ix++) {
        peer.i[ix] = gateway.i[ix];
      }
    } 
    /* Get the peer */
    ret = ask_for_IP4(&peer, _("You are configuring a point-to-point network. The peer address is also used as gateway address."),
                    _("What is your IP peer address?"),
                    _("You entered an invalid peer address.  Ask your network administrator for the proper value."));
    if (ret == 255) return ret;
    for (ix = 0; ix < IP4_ADDR_SZ; ix++) {
      gateway.i[ix] = peer.i[ix];
    }
  }

  return 0;
}



/*
 * Get the domain name of this system
 */
int get_domain() {
  char *def;
  
  for(;;) {
    if (domain) {
      def = strdup(domain);
      free(domain);
      domain = NULL;
    } else {
      def = strdup("");
    }
    domain = inputBox(_("As with individual systems, every network has a name. This name is called the domain name.\n\nPlease enter your domain name or leave this field empty if you don't have a domain."),
                      _("Choose the Domain name"), def);
    free(def);
    if (!domain) return 255;

    if (strchr(domain, ' ') || strchr(domain, '_')) {
      problemBox(_("Domain names cannot contain spaces or underscores"),
		 _("Problem"));
    } else {
      if (strlen(domain)==0) {
        free(domain);
        domain = NULL;
      }
      break;
    }
  }
  return 0;
}



/*
 * Get the IPv4 addresses of up to 3 name servers
 */
int get_dnsconfig() {
  char *def;
  
  for(;;) {
    if (nameserver) {
      def = strdup(nameserver);
      free(nameserver);
      nameserver = NULL;
    } else {
      if (has_gw) {
        def = IP4toa(gateway);
      } else {
        def = strdup("");
      }
    }

    nameserver = inputBox(_("Please enter the IP addresses (not hostnames) of up to 3 name servers, separated by spaces. Do not use commas. The servers will be queried in the order in which you enter them.\n\nIf you don't want to use any name servers just leave this field blank."),
                          _("Choose the DNS Server Addresses"), def);
    free(def);

    if (!nameserver) return 255;
    if (strlen(nameserver) == 0) {
      free(nameserver);
      nameserver = NULL;
      break;
    }
    
    if (check_ip4_nameserver(nameserver) == 0) {
      break;
    }
  }
  return 0;
}



#ifdef PCMCIA
/*
 * Get the PCMCIA transceiver type
 */
int get_pcmcia_transceiver() {
  for (;;) {
    transceiver = inputBox(_("Some PCMCIA Cards can detect the transceiver type although this might take some time. Others cannot detect it at all. Possible values for this parameter are \"auto\", \"10baseT\", \"10base2\", \"aui\", and \"100baseT\", or a blank for cards that don't support transceiver selection.\nPlease specify what transceiver type you would like to use. If you aren't sure, just use the default value of \"auto\"."),
			     _("PCMCIA Transceiver"), transceiver);
    if (transceiver) {
      if (transceiver[0] == '\0') {
        break;
      }
      if ((strcmp(transceiver, "auto") == 0 ) ||
          (strcmp(transceiver, "10baseT") == 0 ) ||
          (strcmp(transceiver, "10base2") == 0 ) ||
          (strcmp(transceiver, "aui") == 0 ) ||
          (strcmp(transceiver, "100baseT") == 0 )) {
         break;
      }
    }
  }
  return 0;
}
#endif /* PCMCIA */

static int write_hostname() {
    FILE *p_file;
    const char *path;

    check_target_path();
    path = target_path(NC_ETC_DIR);
    if (check_dir(path) == -1) {
	mkdir(path, 0755);
    }

    if ((p_file = fopen(target_path(NC_HOSTNAME_FILE), "w"))) {
	fprintf(p_file, "%s\n", host);
	fclose(p_file);
    } else
	return 255;

    /* Just do this for now */
    if ((p_file = fopen(target_path(NC_HOSTS_FILE), "w"))) {
	fprintf(p_file, "127.0.0.1\t%s\n", host);
	fclose(p_file);
    } else
	return 255;

    return 0;
}

/*
 * Write the network configuration files that are common to a static,
 * DHCP and PCMCIA configuration.
 */
int write_common_network() {
  FILE *p_file;
  const char* path;
  int ret = 0;

  /* Create /target/etc and /target/etc/network if they don't exist */
  check_target_path();
  path = target_path(NC_ETC_DIR);
  if (check_dir(path) == -1) {
    mkdir(path, 0755);
  }
  path = target_path(NC_ETC_NETWORK_DIR);
  if (check_dir(path) == -1) {
    mkdir(path, 0755);
  }

  write_hostname();
  
  if (!netinterface || use_dhcp) {
    if ((p_file = fopen(target_path(NC_HOSTS_FILE), "w"))) {
      /* write_static_network will overwrite this file so we can just assume
       * that we use DHCP or have no netinterface at all
       */
      fprintf(p_file, "127.0.0.1\t%s\tlocalhost\n", host);
      fclose(p_file);
    } else {
      ret = 255;
    }
  }

  if ((p_file = fopen(target_path(NC_INTERFACES_FILE), "w"))) {
    fprintf(p_file, "# /etc/network/interfaces -- configuration file for ifup(8), ifdown(8)\n\n");
    fprintf(p_file, "# The loopback interface\n");
    fprintf(p_file, "auto lo\n");
    fprintf(p_file, "iface lo inet loopback\n");
    fclose(p_file);
  } else {
    ret = 255;
  }

  return ret;
}



/*
 * Write the static network configuration
 * NOTE: This function must not be called with netinterface == NULL
 */
int write_static_network() {
  FILE *p_file;
  char *tmp, *ns, *aux;
  int ret;

  ret = write_common_network();
  if (ret == 255) return ret;

  if ((p_file = fopen(target_path(NC_HOSTS_FILE), "w"))) {
    fprintf(p_file, "127.0.0.1\tlocalhost\n");
    if (domain) {
      fprintf(p_file, "%s\t%s.%s\t%s\n", IP4tostr(prtbuf, ipaddr),
                                         host, domain, host);
    } else {
      fprintf(p_file, "%s\t%s\n", IP4tostr(prtbuf, ipaddr), host);
    }
    fclose(p_file);
  } else {
    ret = 255;
  }

  if ((p_file = fopen(target_path(NC_NETWORKS_FILE), "w"))) {
    fprintf(p_file, "localnet %s\n", IP4tostr(prtbuf, network));
    fclose(p_file);
  } else {
    ret = 255;
  }

  if ((p_file = fopen(target_path(NC_RESOLV_FILE), "w"))) {
    if (domain) fprintf(p_file, "search %s\n", domain);
    if (nameserver) {
      aux = ns = strdup(nameserver);
      while ((tmp = strchr(ns, ' ')) != NULL) {
        *tmp = '\0';
        tmp++;
        fprintf(p_file, "nameserver %s\n", ns);
        ns = tmp;
      }
      fprintf(p_file, "nameserver %s\n", ns);
      free(aux);
    }
    fclose(p_file);
    /* if this is a live CD filesystem, symlink to /tmp/resolv.conf too */
    if (is_live_cd() == 2) {
	unlink("/tmp/resolv.conf");
	symlink(target_path(NC_RESOLV_FILE), "/tmp/resolv.conf");
    }
  } else {
    ret = 255;
  }

  if ((p_file = fopen(target_path(NC_INTERFACES_FILE), "a"))) {
    fprintf(p_file, "\n# The first network card - this entry was created during the Debian installation\n");
    fprintf(p_file, "# (network, broadcast and gateway are optional)\n");
    if (!has_pcmcia)
      fprintf(p_file, "auto %s\n", netinterface);
    fprintf(p_file, "iface %s inet static\n", netinterface);
    fprintf(p_file, "\taddress %s\n", IP4tostr(prtbuf, ipaddr));
    fprintf(p_file, "\tnetmask %s\n", IP4tostr(prtbuf, netmask));
    if (!has_pointopoint) {
      fprintf(p_file, "\tnetwork %s\n", IP4tostr(prtbuf, network));
      fprintf(p_file, "\tbroadcast %s\n", IP4tostr(prtbuf, broadcast));
    } else {
      fprintf(p_file, "\tpointopoint %s\n", IP4tostr(prtbuf, peer));
    }
    if (has_gw) {
      fprintf(p_file, "\tgateway %s\n", IP4tostr(prtbuf, gateway));
    }
    fclose(p_file);
  } else {
    ret = 255;
  }

  return ret;
}



/*
 * Write the network configuration for a DHCP setup
 */
int write_dhcp_network() {
  FILE *p_file;
  int ret;

  ret = write_common_network();
  if (ret == 255) return ret;

  /* Copy /etc/resolv.conf (created by dhcp) to the target system */
  vexeclog (LOG_DEBUG, "cp %s %s", NC_RESOLV_FILE, target_path(NC_RESOLV_FILE));
  p_file = fopen(target_path(NC_INTERFACES_FILE), "a");
  fprintf(p_file, "\n# The first network card - this entry was created during the Debian installation\n");
  if (!has_pcmcia)
    fprintf(p_file, "auto %s\n", netinterface);
  fprintf(p_file, "iface %s inet dhcp\n", netinterface);
  fclose(p_file);
  return 0;
}



/*
 * Activates the static network configuration (we don't add the network
 * route to the interfaces, this is not required for kernel 2.2)
 */
int activate_static_network()
{
  int ret = 0;
  if (NAME_ISEXE("/sbin/ifconfig", &nc_statbuf)) {
#ifndef _TESTING_
    unlink(NC_RESOLV_FILE);
    symlink(target_path(NC_RESOLV_FILE), NC_RESOLV_FILE);
#endif

    /*
     * Always activate the loopback interface and don't
     * activate PCMCIA (this is done in activate_pcmcia)
     */

    ret = execlog("/sbin/ifconfig lo 127.0.0.1", LOG_INFO);
    if (!use_dhcp && !has_pcmcia && (ret==0) && netinterface) {
      char cmdline[MAXLINE];

      /* Down the interface to clear old gateways.  */
      snprintf(cmdline, MAXLINE, "/sbin/ifconfig %s down", netinterface);
      execlog(cmdline, LOG_INFO);
      
      /* Kill any dhcp client daemon that may have been running if the
	 user previously chose DHCP */
      if (NAME_ISEXE("/sbin/pump", &nc_statbuf))      
	execlog("/sbin/pump -k", LOG_DEBUG);      
      else if (NAME_ISEXE(NC_DHCLIENT_FILE, &nc_statbuf)) {
	char *name = strrchr(NC_DHCLIENT_FILE, '/');
	if (name) {
	  snprintf(cmdline, MAXLINE, "kill `pidof %s`", name+1);
	  execlog(cmdline, LOG_DEBUG);
	}
      }

      snprintf(cmdline, MAXLINE-5, "/sbin/ifconfig %s %s netmask ",
               netinterface, IP4tostr(prtbuf, ipaddr));
      strncat(cmdline, IP4tostr(prtbuf, netmask), MAXLINE-strlen(cmdline)-5);
      strncat(cmdline, " broadcast ", MAXLINE-strlen(cmdline)-5);
      strncat(cmdline, IP4tostr(prtbuf, broadcast), MAXLINE-strlen(cmdline)-5);
      if (has_pointopoint) {
       strncat(cmdline, " pointopoint ", MAXLINE-strlen(cmdline)-5);
       strncat(cmdline, IP4tostr(prtbuf, peer), MAXLINE-strlen(cmdline)-5);
      }
      ret = execlog(cmdline, LOG_INFO);
      if (has_gw && (ret==0)) {
        snprintf(cmdline, MAXLINE-5, "/sbin/route add default gw %s metric 1",
                 IP4tostr(prtbuf, gateway));
        ret = execlog(cmdline, LOG_INFO);
      }
    }
  }
  return ret;
}



/*
 * Configuration for static (i.e. not DHCP) network setups. This
 * function is called by configure_network() and calls the other
 * get_XXX() functions
 */
int configure_static_network() {
  int ret;
  transceiver = strdup("auto");

  if (!netinterface) {
    write_common_network();
  } else {
    ret = get_ifconfig();
    if (ret == 255) return ret;

    ret = get_domain();
    if (ret == 255) return ret;

    ret = get_dnsconfig();
    if (ret == 255) return ret;
#ifdef PCMCIA
    if (has_pcmcia) {
      ret = get_pcmcia_transceiver();
      if (ret == 255) return ret;
    }
#endif

    write_static_network();
#ifdef PCMCIA
    if (has_pcmcia && activate_pcmcia()) {
        problemBox(_("As soon as you configure the PCMCIA support, your network configuration will be automatically activated."),
                   _("PCMCIA Networking"));
    }
#endif
  }

  ret = activate_static_network();

  return ret;
}

int configure_hostname() {
    int ret;
#if #cpu (s390)
    /* consider netsetup information gathered during first installation step*/
    char * env_host = getenv("ip_host");

    if(env_host && !host) {
      host = malloc(strlen(env_host)+1);
      strcpy(host, env_host);
      }
#endif

    ret = get_host();
    if(ret == 255) return ret;
    ret = write_common_network();
    return ret;
}

/*
 * Main configuration routine: First get the network interface to
 * configure, then ask about DHCP/BOOTP.
 */
int configure_network() {
  int ret;
  struct stat hf;
#if #cpu (s390)
  /* consider netsetup information gathered during first installation step*/
  char * env_netinterface = getenv("network_interface");
  char * env_ipaddr = getenv("ip_addr");
  char * env_netmask = getenv("ip_netmask");
  char * env_network = getenv("ip_network");
  char * env_broadcast = getenv("ip_broadcast");
  char * env_peer = getenv("ip_peer");
  char * env_gateway = getenv("ip_gateway");
  char * env_nameserver = getenv("ip_dns");
  char * env_host = getenv("ip_host");
  char * env_domain = getenv("ip_search");

  if(env_host && !host) {
    host = malloc(strlen(env_host)+1);
    strcpy(host, env_host);
  }
#endif

  use_dhcp = 0;

  /* TODO: we need to see if there is already a network configuration, and
   * the interfaces are just not up. Give the user a choice to simply
   * activate it. This will require us to read in the config files and
   * setup out vars from there, however. */

  if (!NAME_ISREG(target_path(NC_HOSTNAME_FILE),&hf) || !host) {
    ret = get_host();
    if (ret == 255) return ret;
  }

  ret = get_netinterface();
  if (ret == 255) return ret;

#if #cpu (s390)
  if(env_netinterface && !strcmp(netinterface, env_netinterface)) {
    if(env_domain && !domain) {
      domain = malloc(strlen(env_domain)+1);
      strcpy(domain, env_domain);
    }
    if(env_nameserver && !nameserver) {
      nameserver = malloc(strlen(env_nameserver)+1);
      strcpy(nameserver, env_nameserver);
    }
    if(env_ipaddr)
      sscanf(env_ipaddr, "%d.%d.%d.%d", &ipaddr.i[0], &ipaddr.i[1], &ipaddr.i[2], &ipaddr.i[3]);
    if(env_netmask)
      sscanf(env_netmask, "%d.%d.%d.%d", &netmask.i[0], &netmask.i[1], &netmask.i[2], &netmask.i[3]);
    if(env_network)
      sscanf(env_network, "%d.%d.%d.%d", &network.i[0], &network.i[1], &network.i[2], &network.i[3]);
    if(env_broadcast)
      sscanf(env_broadcast, "%d.%d.%d.%d", &broadcast.i[0], &broadcast.i[1], &broadcast.i[2], &broadcast.i[3]);
    if(env_gateway)
      sscanf(env_gateway, "%d.%d.%d.%d", &gateway.i[0], &gateway.i[1], &gateway.i[2], &gateway.i[3]);
    if(env_peer)
      sscanf(env_peer, "%d.%d.%d.%d", &peer.i[0], &peer.i[1], &peer.i[2], &peer.i[3]);
  }
#endif

  /* DHCP is only supported on Ethernet */
  if (netinterface &&
      (strncmp(netinterface, "eth", 3) == 0) &&
#if #cpu (s390)
      /* if !static during initial setup for s390 */ 
      (!env_netinterface || strcmp(netinterface, env_netinterface) || !env_ipaddr) &&
#endif
      (NAME_ISEXE("/sbin/pump", &nc_statbuf) || 
       NAME_ISEXE( NC_DHCLIENT_FILE, &nc_statbuf ))) {
    snprintf(prtbuf, PRTBUFSIZE, 
	     _("Configuration for %s (%s)\n\nDo you want to use DHCP or BOOTP to automatically configure this interface? You'll need a DHCP or BOOTP server in the local network for this to work."), 
	     netinterface, get_ifdsc(netinterface));
    if (yesNoBox(prtbuf, _("Automatic Network Configuration")) == DLG_NO) {
      ret = configure_static_network();
    } else {
      pleaseWaitBox(_("Trying to get network configuration using DHCP/BOOTP...\n\nThis may take a few minutes, be patient."));
      /* Kill pump first, in case it is already running - currently disabled
       * because of problems with PCMCIA cards
       */
#if 0
      execlog("/sbin/pump -k", LOG_DEBUG);
#endif
    /* Look for dhclient first, if we find it we'll use it. */
      if (NAME_ISEXE( NC_DHCLIENT_FILE, &nc_statbuf ))  
      {
      /* Setup host info for dhclient (I suppose we need to do this?) */
	FILE * fp = fopen( "/etc/dhclient.conf", "w" );
	if ( NULL != fp )
	{
	    fprintf( fp, "interface \"%s\" {\n send host-name \"%s\";\n }\n", 
		netinterface, host );
	    fclose( fp );
	}

      	snprintf( prtbuf, MAXLINE-5, "%s -e -lf /tmp/dhcp.leases %s", 
		NC_DHCLIENT_FILE, netinterface );
      }

    /* No dhclient, so default to pump. */
      else 
      	snprintf(prtbuf, MAXLINE-5, "/sbin/pump -i %s -h %s", 
		netinterface, host);
      if (execlog(prtbuf, LOG_INFO) || !is_interface_up(netinterface)) {
	boxPopWindow();
        snprintf(prtbuf, PRTBUFSIZE, _("Automatic configuration using DHCP/BOOTP failed for %s (%s).\n\nDo you want to manually configure this interface?"), netinterface, get_ifdsc(netinterface));
        if (yesNoBox(prtbuf, _("Automatic Network Configuration")) == DLG_YES) {
          ret = configure_static_network();
        } else {
	  return DLG_CANCEL;
        }
      } else {
	boxPopWindow();
        use_dhcp = 1;
        if (!bootargs.isquiet) {
          problemBox(_("The network has been successfully configured using DHCP/BOOTP."),
                     _("Information"));
        }
        ret = write_dhcp_network();
        ret = activate_static_network();     /* For loopback interface */
      }
    }
  } else {
    ret = configure_static_network();
  }
  res_init();
  return ret;
}



#ifdef _TESTING_
/*
 * To test, compile using: make netconfig_test
 */
int main()
{
  LOAD_TRMFILE("test.trm");
  get_kver();
  if (check_dir("targetdir_test") == -1)
    mkdir("targetdir_test", 0755);

  boxInit();

  configure_network();

  boxFinished();
  return 0;
}
#endif
