#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/klog.h>
#ifndef _TESTING_
# include <sys/reboot.h>
#endif
#ifdef _TESTING_
# define RB_AUTOBOOT 1
void reboot(int blah){}
#endif
#include <sys/types.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <ctype.h>
#include <asm/types.h>
#include <linux/serial.h>
#include <syslog.h>

#ifdef USE_LANGUAGE_CHOOSER
#include <locale.h>
#include <wchar.h>
#endif

#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

char *kver;
char *append_opts;
char *Arch2;
char kernel_image_path[PATH_MAX + 1];
char drivers_path[PATH_MAX + 1];
char *base_cd;
char *kernel_cd;
int wants_cd;

/* Defaults to no serial console present */
int serialConsole = -1;

#if #cpu(alpha)
int srm_boot = 1;
char milo_binary_path[PATH_MAX + 1];
#endif

#ifdef USE_LANGUAGE_CHOOSER
const struct language_item *lang = NULL;
#endif

struct Arg {
  const char *name;
  int isflag;
  void *value;
};

int parsefile(const char *filename, struct Arg *options, 
	      int hasinterspace, int toeol) {
  int fd, bytes, lineend=0;
  char *start= prtbuf, *end;
  struct Arg *a;

  if ((fd = open(filename, O_RDONLY, 0)) < 0) {
    ERRMSG(_("Cannot open %s: %s\n"), filename, strerror(errno) );
    return 1;
  }
  while ( (bytes = read(fd, prtbuf, PRTBUFSIZE - 1)) != 0) {
    prtbuf[bytes]= '\0';

    /* parse line */
    while (! lineend) {
      /* skip space at the beginning of the line */
      while (isspace(*start)) start++;
      if (!(*start)) break;

      /* take the first word on the string as the option name */
      end = start;
      while (*end && !isspace(*end)) end++;
      if (!*end) lineend = 1;
      *end = '\0';
      a = options;
      while ( a->name != NULL ) {
	if (a->isflag) {
	  if (! strcmp(start, a->name)) {
	    /* the option is valid. It is a flag, switch it on */
	    *((int *)a->value) = 1;
	  }
	} else {
	  int namelen=strlen(a->name);
	  if (!strncmp(start, a->name, namelen)) {
	    /* the option is valid. It expects a value */
	    if (hasinterspace) {
	      if (lineend) {
		/* If the file syntax is "name value" (hasinterspace==1) 
		 * and lineend == 1, we have found a name with no value. */
		break;
	      }
	      /* skip whitespace after the option name */
	      namelen = 0;
	      start = end + 1;
	      /* MDP: Skip a colon since PPC uses a different format */
	      while (isspace(*start) || *start == ':') start++;
	      if (!(*start)) {
		/* We have reached the end of the line, that means we have
		 * found a name with no value. */
		break;
	      }
	      end = start;
	      if (toeol)  /* include embedded spaces in value */
		while (*end >= ' ') end++;
	      else
		while (*end && !isspace(*end)) end++;
	      *end = '\0';
	      /* Skip the rest of the line */
	      lineend = 1;
	    }
	    *((char **)a->value)=strdup(start+namelen);
	  }
	}
	a++;
      }
      start = end + 1;
    }
  }
  close(fd);
  return 0;
}

#if #cpu(alpha) || #cpu(sparc) || #cpu(mips)
static int
parse_cpuinfo(const char *key, char *val, int len)
{
    FILE *cpuinfo;
    char format[256];
    int found = 0;

    snprintf(format, sizeof(format), "%s : %%%d[A-Za-z /0-9]", key, len-1);

#if defined(_EXETESTING_) || defined(_TESTING_)
    if ((cpuinfo = fopen ("testcpuinfo", "r")) != NULL)
#else
    if ((cpuinfo = fopen ("/proc/cpuinfo", "r")) != NULL)
#endif
      {
	while (!feof(cpuinfo)) {
	    if (fscanf (cpuinfo, format, val) == 1) {
	      found = 1;
	      break;
	    }
	    /* Grab the rest of the line */
	    fgets (val, len, cpuinfo);
	}
	fclose(cpuinfo);
      }
    return found;
}
#endif

/* Get arguments from boot commmand line */
int
readcmdline (void)
{
  int  status;

  struct Arg bootoptions[] = {
    { "root=",     0, &bootargs.root     },
    { "disksize=", 0, &bootargs.disksize },
    { "bootkbd=",  0, &bootargs.kbd      },
    { "mouse=",    0, &bootargs.mouse    },
    { "flavor=",   0, &bootargs.flavor   },
    { "console=",  0, &bootargs.console  },
    { "mono",      1, &bootargs.ismono   },
    { "debug",     1, &bootargs.isdebug  },
    { "verbose",   1, &bootargs.isverbose}, /* chatty mode */
    { "quiet",     1, &bootargs.isquiet  }, /* quiet mode */
/*    { "cdrom",     1, &bootargs.cdrom    }, // cdrom install */
    { NULL,        0, NULL               }
  };

  memset ((void *) &bootargs, 0, sizeof(struct BootArgs));
#if defined(_EXETESTING_) || defined(_TESTING_)
  status = parsefile("testcmdline", bootoptions, 0, 0);
#else
  status = parsefile("/proc/cmdline", bootoptions, 0, 0);
#endif

  return status;
}

#if #cpu (m68k)
static char *get_m68k_model( void )
{
  char *model = NULL;
  int status;
  struct Arg archoptions[] = {
    { "Model:", 0, &model },
    { NULL, 0, NULL}
  };
  status = parsefile("/proc/hardware", archoptions, 1, 0);
  if(status) {
    fprintf( stderr, _("m68k machine type unknown!\n") );
    return NULL;
  }
  if (!model || !*model) {
     fprintf( stderr, _("Unable to determine m68k machine type!\n") );
     return NULL;
  }

  if (strcmp(model, "Motorola") == 0)
  {
    /* reparse to get MVME model number as well */
    parsefile("/proc/hardware", archoptions, 1, 1);
  }
 
  return model;
}
#endif

#if #cpu (powerpc)
char * Arch3=NULL;
char * Arch4=NULL;
static char *get_powerpc_model( void )
{
  char *model = NULL;
  char *pmacgeneration = NULL;
  int status;
  struct Arg archoptions[] = {
    { "machine", 0, &model },
    { "pmac-generation", 0, &pmacgeneration },
    { NULL, 0, NULL}
  };
  status = parsefile("/proc/cpuinfo", archoptions, 2, 0);
  if (status) {
    fprintf( stderr, _("PowerPC machine type unknown!\n") );
    return NULL;
  }
  if (!model || !*model)
    fprintf( stderr, _("Unable to determine PowerPC machine type!\n") );
  if ((strncasecmp(model, "Power", 5) == 0)
     || (strncasecmp(model, "iMac",  4) == 0)) {
    /* Hack: if we're running a 2.4 kernel, then use the new-powermac
       subarch */
    if (!strncmp("2.4", kver, 3))
      model = "NewPowerMac";
    else
      model = "PowerMac";
/* 
    FIXME
    Okay, parsefile should be probably be fixed to do right the first time, 
    but this works now. (parsefile doesn't seem to read to the end of 
    the file.)
*/
    archoptions[0].name = "pmac-generation";
    archoptions[0].value = &pmacgeneration;
    status = parsefile("/proc/cpuinfo", archoptions, 2, 0);
    if ( status )
      ERRMSG( "Unable to determine PowerMac generation!" );
    Arch3 = pmacgeneration;	
    
    archoptions[0].name = "motherboard";
    archoptions[0].value = &Arch4;
    status = parsefile("/proc/cpuinfo", archoptions, 2, 0);
    if ( status )
      ERRMSG( "Unable to determine PowerMac motherboard!" );
    
  }
  if (strcmp(model,"Amiga") == 0)
    model = "apus";

  return model;
}
#endif

#if #cpu(alpha)
static int
is_in_cpuinfo(const char *key, const char *str)
{
  char buf[256];

  if (parse_cpuinfo(key, buf, sizeof(buf))) {
    if (strncmp(buf, str, strlen(str)) == 0)
      return 1;
  }
  return 0;
}

int get_alpha_boot_method(void)
{
  if (is_in_cpuinfo("system serial number", "MILO"))
    srm_boot = 0;
  else
    srm_boot = 1;
  return srm_boot;
}

/* These are ones that can't be resolved simply by downcasing */
static struct alpha_model {
  const char *systype, *model;
} alpha_models[] = {
  {"Nautilus", "nautilus"},
  {"Jensen", "jensen"},
  {"AlphaBook1", "book1"},
  {"EB64+", "eb64p"},
  {"EB66+", "eb66p"},
  {"Platform2000", "p2k"},
};
#define ALPHA_MODEL_COUNT (sizeof(alpha_models) / sizeof(alpha_models[0]))

char* get_alpha_model(void)
{
  char buf[256];
  if (parse_cpuinfo("system type", buf, sizeof(buf))) {
    int i;
    char *c;
    for (i = 0; i < ALPHA_MODEL_COUNT; i++) {
      if (strncmp(alpha_models[i].systype, buf, strlen (alpha_models[i].systype)) == 0)
	return strdup(alpha_models[i].model);
    }
    /* else, take the one we got and downcase it */
    for (c = buf; *c; c++)
      *c = tolower(*c);
    return strdup(buf);
  }
  return strdup(""); /* generic */
}
#endif

#if #cpu(sparc)
/* FIXME: what do we do about sun4e?! */
static char* get_sparc_model(void)
{
  char buf[16];

  if (parse_cpuinfo("type", buf, sizeof(buf))) {
    if (! strncmp(buf, "sun4", 4)) {
      if (buf[4] == 'c' || buf[4] == 'd' || buf[4] == 'm')
	return strdup("sun4cdm");
    }
    return strdup(buf);
  }
  return NULL;
}
#endif

#if #cpu(arm)
char *Arch3;

int strongarm_is_broken;

static char* get_arm_model(void)
{
  char *model = NULL;
  char *proc = NULL;
  int status;
  struct Arg archoptions1[] = {
    { "Hardware", 0, &model },
    { NULL, 0, NULL}
  };
  struct Arg archoptions2[] = {
    { "Processor", 0, &proc },
    { NULL, 0, NULL}
  };
  status = parsefile("/proc/cpuinfo", archoptions1, 2, 1);
  status |= parsefile("/proc/cpuinfo", archoptions2, 2, 1);
  if (status || !model || !*model) {
    fprintf( stderr, _("ARM machine type unknown!\n") );
    model = "";
  }
  Arch3 = model;
  if (strncasecmp(model, "acorn-", 6) == 0)
    model = "riscpc";
  else if (strcasecmp(model, "chalice-cats") == 0
      || strcasecmp(model, "rebel-netwinder") == 0
      || strcasecmp(model, "ebsa285") == 0)
    model = "netwinder";
  else if (strcasecmp(model, "lart") == 0)
    model = "lart";
  else if (strcasecmp(model, "shark") == 0)
    model = "shark";
  if (proc && strstr(proc, "StrongARM-110 ")) {
    char *rev_str = strstr(proc, " rev ");
    if (rev_str) {
      int rev_nr = atoi(rev_str + 5);
      if (rev_nr < 3)
	strongarm_is_broken = 1;
    }
  }
  return model;
}
#endif

#if #cpu(mips)
static char* get_mips_model(void)
{

  /*
   * DECstation 5000/20,25,33 is KN02-CA (R3k), should work with r3k-kn02
   * DECstation 5000/50 is ?? (R4k), should work with r4k-kn02 
   * DECstation 5000/120,125,133 is KN02 (R3k)
   * DECstation 5000/150 is KN04 (R4k)
   * DECstation 5000/240 is ??, should work with r3k-kn02 
   * DECstation 5000/260 is KN05 (R4k4), should work with r4k-kn04
   */

  char sys_type[80]; /* 32 bytes is too short for some types of DECstation */
  char cpu_type[80];

  memset (sys_type, 0, sizeof (sys_type));
  memset (cpu_type, 0, sizeof (cpu_type));
  if (parse_cpuinfo("system type", sys_type, sizeof(sys_type))) {
    /* safe enough since all of them use arc firmware */
    if (! strncmp(sys_type, "SGI",3)) {
      return strdup("r4k-ip22");
    } else if ((! strncmp (sys_type, "Digital DECstation 5000/1xx", 27)) || 
           (! strncmp (sys_type, "Digital Personal DECstation 5000/xx", 35)) ||
           (! strncmp (sys_type, "Digital DECstation 5000/2x0", 27))) 
    {
      if (parse_cpuinfo("cpu model", cpu_type, sizeof(cpu_type))) {
        if ((! strncmp(cpu_type, "R4000",5)) ||
           (! strncmp(cpu_type, "R4400",5)))
        {
          return strdup("r4k-kn04");
        } else {
          return strdup("r3k-kn02");
        }
      }
    }
  }
  return strdup("unknown");
}
#endif

#if #cpu(hppa)
static char* get_hppa_model(void)
{
  struct utsname buf;

  if (uname(&buf) < 0)
    return "";
  else if (!strcmp(buf.machine, "parisc64"))
    return "64";
  else if (!strcmp(buf.machine, "parisc"))
    return "32";
  else
    return "";
}
#endif

void get_subarch_name(void)
{
#if #cpu(m68k)
  Arch2 = get_m68k_model();
#elif #cpu(powerpc)
  Arch2 = get_powerpc_model();
#elif #cpu(alpha)
  Arch2    = get_alpha_model();
#elif #cpu(sparc)
  Arch2 = get_sparc_model();
#elif #cpu(arm)
  Arch2 = get_arm_model();
#elif #cpu(mips)
  Arch2 = get_mips_model();
#elif #cpu(hppa)
  Arch2 = get_hppa_model();
#else
  Arch2 = strdup(""); /* prevent errors */
#endif
}

void get_kver(void)
{
  FILE *f;
  char osrelease[64];
  if ((f = fopen("/proc/sys/kernel/osrelease", "r"))) {
    fscanf(f, "%63s", osrelease);
    fclose(f);
    kver = strdup(osrelease);
  } else {
    /* panic! */
    kver = strdup("2.2.19"); /* default for now */
  }
}

/* try to eject root floppy */
#if #cpu(sparc)
void
try_to_eject_root_floppy(void)
{
  if (bootargs.root && 0==strncmp(bootargs.root,"/dev/fd",7)) {
      /* extract floppy device name */
      eject_floppy(bootargs.root);
  }
}
#endif

/* try to eject root floppy */
#if #cpu(powerpc)
void
try_to_eject_root_floppy(void)
{
 /* Eject that pesky root floppy for PowerMacs */
  if ((strstr(Arch2, "PowerMac")) && (!strcmp(Arch3, "OldWorld")))
      eject_floppy("/dev/fd0");
}
#endif

#ifdef SCSI_FLOPPY
/* scan /proc/scsi/scsi for TEAC FC-1 floppy drive
 * and create /dev/sfd[0-7] symlinks if found
 */
void
find_scsi_floppy()
{
  FILE *fp;
  char buf[100];
  char dev[8];
  int  i = -1;
  int  n = 0;
 
  if ((fp = fopen("/proc/scsi/scsi", "r")) != NULL) {
    while (fgets(buf, sizeof(buf), fp)) {
      if ((strncmp(buf, "Host:", 5) == 0) && strstr(buf, "Channel:"))
        i++;
      else {
        if (strstr(buf, "Vendor:")
        &&  strstr(buf, "TEAC"   )
        &&  strstr(buf, "Model:" )
        &&  strstr(buf, "FC-1"   )) {
          sprintf(dev, "sd%c", 'a' + i);
          sprintf(buf, "/dev/sfd%d", n++);
          unlink(buf);
          symlink(dev, buf);
        }
      }
    }
    fclose(fp);
  }
}
#endif

/*
 * check for supported filesystem in kernel
 */
int
supported_filesystem(const char* fsname)
{
  char buf[12];
  int found=0;
  FILE *f = fopen("/proc/filesystems", "r");
  if (f) {
    while (fscanf(f, "%11s", buf) > 0) {
      if (!strcmp(buf, fsname)) {
	found=1;
	break;
      }
    }
    fclose(f);
  }
  return found;
}

void
setup_image_names(void)
{
  /* Keep track of lengths - start with trailing zero */
  int kplen = 1, dplen = 1;
  kernel_image_path[0] = 0;
  drivers_path[0] = 0;
 
  /* Subarchitecture */
  if (Arch2) {
    int len = strlen(Arch2);
    kplen += len + 1;
    dplen += len + 1;
    if (kplen > sizeof(kernel_image_path)
	|| dplen > sizeof(drivers_path)) {
      ERRMSG("kernel_image_path or drivers_path too long (%d %d), punting",
	     kplen, dplen);
      kernel_image_path[0] = 0;
      drivers_path[0] = 0;
      goto append_names;
    }
      
    if (len > 0) {
      /* default */
      const char *arch_path = Arch2;

#if #cpu(powerpc)
      if (!strcmp(Arch2, "Amiga"))
	 arch_path = "apus";
      else if (!strcmp(Arch2, "CHRP"))
	 arch_path = "chrp";
      else if (!strcmp(Arch2, "PowerMac"))
	  arch_path = "powermac";
      else if (!strcmp(Arch2, "NewPowerMac")) {
	  arch_path = "new-powermac";
	  /* UGLY UGLY UGLY */
	  bootargs.disksize = "2.88";
      } else if (!strcmp(Arch2, "PReP"))
	 arch_path =  "prep";

#elif #cpu(m68k)
      if (!strncmp(Arch2, "BVME", 4))
	 arch_path = "bvme6000";
      else if (!strncmp(Arch2, "Motorola", 8)) {
      	 if (!strncmp(Arch2 + 8, " MVME147", 8))
      	   arch_path = "mvme147";
      	 else
	   arch_path = "mvme16x";
      }
      else if (!strcmp(Arch2, "Atari"))
	 arch_path = "atari";
      else if (!strcmp(Arch2, "Amiga"))
      	 arch_path = "amiga";
      else if (!strcmp(Arch2, "Macintosh"))
      	 arch_path = "mac";

#elif #cpu(alpha)
      /* The rescue disk is generic for most Alphas, but MILO is not.
         So we have to fix up here :P */
      if (   !(!strcmp(Arch2, "nautilus") && !srm_boot)
	  && !(!strcmp(Arch2, "jensen")))
	arch_path = NULL;
      /* Furthermore, choose_medium() is really broken, so we have to
         cook up a MILO path here while we're at it */
      sprintf(milo_binary_path, "MILO/%s", Arch2);

#elif #cpu(arm)
      if (!strcmp(Arch2, "netwinder"))
	/* UGLY UGLY UGLY */
	bootargs.disksize = "2.88";

#endif

      if (arch_path != NULL) {
	  strcat(kernel_image_path, arch_path);
	  strcat(kernel_image_path, "/");

	  strcat(drivers_path, arch_path);
	  strcat(drivers_path, "/");
      }
    }
  }

  /*
   * Some architectures have several sizes of diskette images.
   */
  if (bootargs.disksize) {
    int len = strlen(bootargs.disksize);
    kplen += len + 8;
    if (kplen > sizeof(kernel_image_path)
	|| dplen > sizeof(drivers_path)) {
      ERRMSG("kernel_image_path or drivers_path too long (%d %d), punting",
	     kplen, dplen);
      kernel_image_path[0] = 0;
      drivers_path[0] = 0;
      goto append_names;
    }
      
    if (len > 0) {
      char part[len + 9];
      sprintf(part, "images-%s/", bootargs.disksize);
      strcat(kernel_image_path, part);
    }
  }

  /*
   * `flavor' is something like "compact", "safe", or "standard".  For
   *   some flavors, `kver' (aka `uname -r`) will be the same as the
   *   flavor, but for others it won't be. (eg: The "safe" flavor on
   *   i386 is the standard kernel with special options given to the
   *   boot loader installer to make it work with old BIOSes.  Its
   *   modules path and module set is the same as for the vanilla
   *   "2.2.xx" flavor kernel, but its flavor is different because it
   *   uses a different rescue disk image.
   */
  if (bootargs.flavor) {
    int len = strlen(bootargs.flavor);
    kplen += len + 1;
    dplen += len + 1;
    if (kplen > sizeof(kernel_image_path)
	|| dplen > sizeof(drivers_path)) {
      ERRMSG("kernel_image_path or drivers_path too long (%d %d), punting",
	     kplen, dplen);
      kernel_image_path[0] = 0;
      drivers_path[0] = 0;
      goto append_names;
    }

    if (len > 0) {
      /* special case ... the safe flavor doesn't have its own kernel
	 and drivers, so just use the standard ones */
      if (strcmp(bootargs.flavor, "safe") != 0) {
	strcat(kernel_image_path, bootargs.flavor);
	strcat(kernel_image_path, "/");
	strcat(drivers_path, bootargs.flavor);
	strcat(drivers_path, "/");
      }
    }
  }

 append_names:
  strcat(kernel_image_path, KERDISKFILE);
  DEBUGMSG("kernel_image_path is '%s'", kernel_image_path);
  strcat(drivers_path, DRVTGZFILE);
  DEBUGMSG("drivers_path is '%s'", drivers_path);
}

#ifndef _TESTING_
/* This simply checks that /proc/version exists, which should give us a
 * good indicator that /proc is in fact mounted - BenC
 */
static void check_proc (void) {
    struct stat statbuf;
    if (!NAME_ISREG("/proc/version", &statbuf)) {
	fprintf(stderr, "E: /proc does not appear to be mounted (%s)\n",
		strerror(errno));
    	exit(1);
    }
}

int main (void) {
    char *term;
    struct serial_struct ser;

    /* SIGCHLD must be DEFAULT and not IGNORE for popen/pclose to work */
    signal(SIGCHLD, SIG_DFL);
    /* make sure that /proc is mounted */
    check_proc();
    /* Downgrade kernel verbosity */
    klogctl(8, NULL, 4);

    /* ensure sane umask */
    umask(022);

    INFOMSG("dbootstrap starting; built " BUILDTIME " for " ARCHNAME);

    readcmdline();

    /* Default disksize, in case disksize= is not set in bootloader config. */
    if (bootargs.disksize == NULL) {
      bootargs.disksize = "1.44";
    }

    if ( bootargs.isverbose && bootargs.isquiet ) {
      ERRMSG("both verbose and quiet set, going with verbose");
      bootargs.isquiet = 0;
    }

    if (bootargs.isdebug)
      install_segv_handler();

    /* get kernel version */
    get_kver();

    /* get subarchitecture name */
    get_subarch_name();

    /* initialize boot parameters */
    append_opts = strdup ("");

    /* Alpha is weird, we act totally differently based on what kind
       of firmware we were booted from */
#if #cpu(alpha)
    get_alpha_boot_method();
#endif

    /* prompt first on install */
    base_cd=NULL;
    kernel_cd=NULL;
    wants_cd=0;

    /* setup rescue & drivers disks images filenames
     * (should be done before the first call to stderrToTTY) */
    setup_image_names();

    /* Are we on a serial console? */
    if (ioctl(0, TIOCGSERIAL, &ser) < 0)
      /* Not on a serial console */
      serialConsole = -1;
    else {
      /* Serial console on ttySx */
      serialConsole = ser.line;
      INFOMSG("serial console found on line %d", serialConsole);
    }
    /* Set term type and initialize it. */
    term = getenv("TERM");
    if (!term || (term && ! strncmp(term, "linux", 5))) {
      setenv("TERM", "linux", 1);
      setMono();
    }
#ifdef USE_LANGUAGE_CHOOSER
    slurp_language_pack ();

    setlocale (LC_CTYPE, "");   /* Language chooser relies on this call! */
#endif
    boxInit();

#ifdef USE_LANGUAGE_CHOOSER
    {
        char *msgcat = (char *)malloc (PATH_MAX);
	char *locale = getenv("LC_CTYPE");
	const struct language_definition *langs = available_languages ();

	if (locale && strstr(locale, "utf-8")) {
	  /* UTF-8 mode, choose a language */
	  do {
	    lang = boxChooseLanguageVariant (langs);
	  } while (lang == NULL);
	  snprintf (msgcat, PATH_MAX, "/etc/messages.%s", lang->msgcat);
	  msgcat[PATH_MAX - 1] = '\0';
	  DEBUGMSG ("loading message catalog %s", msgcat);
	  
	  if (LOAD_TRMFILE (msgcat) == 0)         /* Failed to load localized strings? */
	  {
	    if (LOAD_TRMFILE (TRMBACKUP) == 0)  /* Failed to load English strings as well? */
            {
	      problemBoxEn ("An error occured while loading application messages.", "Problem");
	      
	      reboot (RB_AUTOBOOT);
	      
	      /* when not root and debugging */
	      boxFinished();
	      exit(1); 
	    }
	    else
	    {
	      char message[255];

	      snprintf (message, sizeof (message), "An error occured while loading localized application messages from '%s'. English messages will be used instead.", msgcat);
	      message[254] = '\0'; /* Just in case */
	      
	      problemBoxEn (message, "Problem");
	    }
	  }
	} else {
	  /* Try to force the language to English */
	  while (langs->name != NULL && strcmp(langs->name, "English"))
	    langs++;
	  if (langs->name == NULL) {
	    problemBoxEn ("UTF-8 not available and no default language present -- can't continue", "Problem");
	    reboot (RB_AUTOBOOT);
	  }
	  lang = langs->list->items[0].p;

	  snprintf (msgcat, PATH_MAX, "/etc/messages.%s", lang->msgcat);
	  msgcat[PATH_MAX - 1] = '\0';
	  DEBUGMSG ("i18n mode, loading message catalog %s", msgcat);

	  /* The English messages might be /etc/messages.en, or
	     /etc/messages.trm in a single locale system */
	  if (LOAD_TRMFILE (msgcat) == 0
	      && LOAD_TRMFILE ("/etc/messages.trm") == 0)
	  {
	    problemBoxEn ("An error occurred while loading application messages", "Problem");
	    reboot(RB_AUTOBOOT);
	    /* when not root and debugging */
	    boxFinished();
	    exit(1); 
	  }
	}

        free (msgcat);
    }
#else
    /* We can use problemBox only AFTER boxInit */
    /* So it's important that no (NO!!!) function before this call (LOAD_TRMFILE) uses gettext function */
    if (LOAD_TRMFILE(TRMFILE) == 0 && LOAD_TRMFILE(TRMBACKUP) == 0) {
	char message[255];
	snprintf(message, sizeof(message), 
		 "An error occured while loading application messages from '%s'.", TRMFILE);
    message[254] = '\0'; /* Just in case */
	problemBoxEn (message, "Problem");
	reboot(RB_AUTOBOOT);
	/* when not root and debugging */
	boxFinished();
	exit(1); 
    }
#endif

#if #cpu (m68k)
    if ((strncmp(Arch2, "BVME",     4) == 0)
    ||  (strncmp(Arch2, "Motorola", 8) == 0)) Arch2 = "VME";
    else
    if ( (strcmp(Arch2,"Atari")) && (strcmp(Arch2,"Amiga")) &&
	 (strcmp(Arch2,"Macintosh"))) {
      problemBox(_("Your m68k architecture is not supported yet."), _("Problem"));
      reboot(RB_AUTOBOOT);
    }

    /* stop kernel complaining to console about scsi floppy with no disk */
    if (strcmp(Arch2, "VME") == 0)
      klogctl(6, NULL, 0);
#elif #cpu (powerpc)
    if ( !strstr(Arch2,"PowerMac") &&
	 strcmp(Arch2,"CHRP") && strcmp(Arch2,"PReP")
	 && strcasecmp(Arch2,"apus")) {
      problemBox(_("Your PowerPC architecture is not supported yet."), _("Problem"));
      reboot(RB_AUTOBOOT);
    }
#endif

#ifdef SCSI_FLOPPY
    find_scsi_floppy();
#endif

#if #cpu(sparc) || #cpu(powerpc)
    try_to_eject_root_floppy();
#endif

    Boot = Root = NULL;
    noSwap = 0;
    notCreatedBootFloppy = 1;
    notInstalledLILO = 1;
    Archive_Dir=NULL;

    InstallationRootDevice=block_device("/");
    if (!InstallationRootDevice) {
      /* something's borked, cannot determine where / comes from */
      /* leave time to see any error message */
      sleep(8);
      problemBox(_("I had trouble checking the choice of root device. I don't know what to use as the root fs."),_("Problem"));
#if #cpu(alpha)
      exit(1);
#else
      reboot(RB_AUTOBOOT);
      boxFinished();
      exit(1); /* when not root and debugging */
#endif
    }
   
#ifdef USE_LANGUAGE_CHOOSER
    release_notes (lang->msgcat);
#else
    release_notes (NULL);
#endif

#if #cpu(arm)
    if (strongarm_is_broken)
      wideMessageBox(_(
"This machine seems to be using an old StrongARM CPU.  SA-110\n"
"processors prior to revision S suffered from bugs that will prevent\n"
"Linux from running reliably.\n\n"
"You can go ahead and install anyway, but you may experience problems\n"
"later.\n\n"
"For more information, refer to http://www.armlinux.org/strongarm.html\n"), 
		     _("Warning"));
#endif

    /* See how much memory is fitted.  If it's less than the
       recommended amount, say so. */
    {
      char *smem = NULL;
      int status;
      struct Arg args[] = { 
	{ "MemTotal", 0, &smem }, 
	{ NULL, 0, NULL } 
      };
      status = parsefile("/proc/meminfo", args, 2, 1);
      if (status == 0 && smem != NULL) {
	unsigned long mem = atol(smem), min_mem;
	mem /= 1024;
	/* This information comes from documentation/default.ent */
#if defined __i386__
	min_mem = 12;
#elif defined __arm__ || defined __s390__ || defined __alpha__ || defined __mips__ || defined __powerpc__
	min_mem = 16;
#else
	min_mem = 5;
#endif
	if (mem < min_mem) {
	  snprintf(prtbuf, PRTBUFSIZE - 1, _(
"Your computer seems to have only %ldMB of RAM.  For this\n"
"version of Debian, a minimum of %ldMB is recommended.\n"), 
		   mem, min_mem);
	  wideMessageBox(prtbuf, _("Warning: Low Memory"));
	}
      }
    }

    is_root_a_floppy ();

    main_menu ();

    boxFinished();

#if #cpu (m68k)
    /* restart kernel logging to console */
    if (strcmp(Arch2,"VME") == 0)
      klogctl(7,NULL,0);
#endif

    closelog();

    return 0;
}
#endif /* _TESTING_ */
