/***************************************************************************
                          dlgsettings.cpp  -  description
                             -------------------
    begin                : Sat Nov 16 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include "dlgsettings.h"
#include "dlgbrowsers.h"

#include "browserlist.h"
#include "bknode.h"
#include "bridgecfg.h"

#include <QTreeWidget>
#include <QMessageBox>

extern BROWSERLIST BrowserList[];

#include <vector>
#include <algorithm>
#include <stdlib.h>
#include <time.h>

using namespace std;

/***************************************************************************
 *                                                                         *
 *   The BrowserCfgItem class (derived from QListViewItem) represents an   *
 *   item in the list of configured browsers.  All it really does is hold  *
 *   an iterator into the Browsers list, part of the BridgeCfg class.      *
 *                                                                         *
 ***************************************************************************/
 
class BrowserCfgItem : public QTreeWidgetItem
{
public:
	BrowserCfgItem(QTreeWidget *parent) : QTreeWidgetItem(parent) { };

	std::vector<BrowserCfg>::iterator it;
};

/***************************************************************************
 *                                                                         *
 *   DlgSettings::DlgSettings()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg                                               *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class constructor.  The first parameter is a constant reference to *
 *      a copy of the BridgeCfg class.  This function makes a copy of the  *
 *      instance of the BridgeCfg class and uses this copy to allows the   *
 *      user to edit their configuration.  If the user hits Ok then the    *
 *      DlgMain class copies this class's copy back into its copy.         *
 *                                                                         *
 ***************************************************************************/

DlgSettings::DlgSettings(const BridgeCfg &cfg, QWidget* parent)
	: QDialog( parent ), m_cfg(cfg)
{
	QStringList headers;

	setupUi(this);

	headers << "Browser" << "Description";
	listBrowsers->setHeaderLabels(headers);
	RefreshList();
	RefreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::~DlgSettings()                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      Class destructor                                                   *
 *                                                                         *
 ***************************************************************************/

DlgSettings::~DlgSettings()
{
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::RefreshButtons()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called to enable to disable the buttons in the    *
 *      dialog box depending on the dialog's state.                        *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::RefreshButtons(void)
{
	QList<QTreeWidgetItem *> items = listBrowsers->selectedItems();

	buttonDelete->setEnabled(items.count() > 0);
	labelLocation->setEnabled(items.count() > 0);
	buttonBrowse->setEnabled(items.count() > 0);
	buttonDelete->setEnabled(items.count() > 0);
	checkReadonly->setEnabled(items.count() > 0);
	labelDesc->setEnabled(items.count() > 0);
	editDesc->setEnabled(items.count() > 0);
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::RefreshList()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function is called to refresh the list of configured browsers.*
 *                                                                         *
 ***************************************************************************/

void DlgSettings::RefreshList(void)
{
	std::vector<BrowserCfg>::iterator it;

	listBrowsers->clear();

	for(it=m_cfg.browsers.begin(); it!=m_cfg.browsers.end(); ++it)
	{
		BrowserCfgItem *item = new BrowserCfgItem(listBrowsers);
		item->setText(0, it->browser());
		item->setText(1, it->Descrip());
		item->it = it;
	}
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::on_buttonAdd_clicked()                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Add (browser) button.  It presents the user *
 *      with a list of BookmarkBridge's supported browsers (using the      *
 *      DlgBrowsers class).  If the user didn't hit Cancel then the        *
 *      browser they selected is added to the list.                        *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::on_buttonAdd_clicked(void)
{
	DlgBrowsers dlg(this);

	if(dlg.exec() != QDialog::Accepted)
		return;

	BrowserCfg cfg;
	QListWidgetItem *item = dlg.listBrowsers->currentItem();
	if(item == NULL)
		return;
	cfg.setBrowser(item->text());
	cfg.setDescrip(item->text()+" Bookmarks");
	cfg.setBrowserNum(dlg.browser);
	m_cfg.browsers.push_back(cfg);
	RefreshList();
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::on_listBrowsers_itemSelectionChanged()                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks somewhere within the browser list.  It updates (or *
 *      clears) the edit fields representing that browser.                 *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::on_listBrowsers_itemSelectionChanged(void)
{
	RefreshButtons();

	BrowserCfgItem *item;

	QList<QTreeWidgetItem *> items = listBrowsers->selectedItems();
	if(items.count() != 1)
		item = NULL;
	else
		item = static_cast<BrowserCfgItem *>(items[0]);

	editDesc->setText(item==NULL?"":item->it->Descrip());
	editLocation->setText(item==NULL?"":item->it->Location());
	checkReadonly->setChecked(item==NULL?false:item->it->readOnly());
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::on_buttonDelete_clicked()                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks the Delete (browser) button.  It confirms this     *
 *      action with the user and removes the browser from the              *
 *      configuration.                                                     *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::on_buttonDelete_clicked(void)
{
	// make sure a browser is actually selected

	QList<QTreeWidgetItem *> items = listBrowsers->selectedItems();
	if(items.count() != 1)
		return;
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(items[0]);
	if(item == NULL)
		return;

	// confirm this action with the user
	
	int ret = QMessageBox::warning(this, "Delete Browser",
		"Are you sure you want to delete this browser?\n"
		"If you delete this browser from your settings,\n"
		"any sources or destinations referencing this\n"
		"browser will also be deleted.\n"
		"Warning! This action cannot be undone.",
		QMessageBox::Yes, QMessageBox::No);
	if(ret != QMessageBox::Yes)
		return;

	// before we actually delete the browser, see if it is listed
	// as a source or destination.  if so, remove it from those lists too

	vector<BRWSNUM>::iterator oit;
	oit = std::find(m_cfg.sources.begin(), m_cfg.sources.end(), item->it->ordinal());
	if(oit != m_cfg.sources.end())
		m_cfg.sources.erase(oit);
	oit = std::find(m_cfg.destinations.begin(), m_cfg.destinations.end(), item->it->ordinal());
	if(oit != m_cfg.destinations.end())
		m_cfg.destinations.erase(oit);

	// remove it from the list
	
	m_cfg.browsers.erase(item->it);
	RefreshList();
	RefreshButtons();
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::on_editDesc_textChanged()                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &                                                    *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user changes the text in the Description edit field.           *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::on_editDesc_textChanged(const QString &text)
{
	// make sure a browser is actually selected

	QList<QTreeWidgetItem *> items = listBrowsers->selectedItems();
	if(items.count() != 1)
		return;

	// change the browser configuration to reflect the description change

	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(items[0]);
	
	item->it->setDescrip(text);
	item->setText(1, text);
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::on_buttonBrowse_clicked()                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the Browse button to select the browser's       *
 *      bookmark database location.                                        *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::on_buttonBrowse_clicked(void)
{
	// make sure a browser is actually selected

	QList<QTreeWidgetItem *> items = listBrowsers->selectedItems();
	if(items.count() != 1)
		return;
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(items[0]);
	if(item == NULL)
		return;

	BrowserBk *browserbk = BrowserList[item->it->browserNum()-1].Factory();
	if(browserbk == NULL)
		return;

	// tell the browser's class instance to allow the user to browse
	// for bookmarks.
	
	QString bookmarks = item->it->Location();
	for(;;)
	{
		// if the user hit Cancel, forget it
		
		if(browserbk->BrowseForBookmarks(m_cfg, bookmarks, this))
		{
			delete browserbk;
			return;
		}

		// If the bookmarks selected by the user are read-only,
		// confirm this with the user
		
		if(browserbk->AreBookmarksReadonly(bookmarks))
		{
			int res = QMessageBox::warning(this, "Bookmarks Are Read-only",
				"The bookmarks you selected are currently read-only.\nDo you "
				"wish to never save changes to this browser's bookmarks?",
			QMessageBox::Yes, QMessageBox::No, QMessageBox::Retry);
			if(res == QMessageBox::Retry)
				continue;
			item->it->setReadOnly(res == QMessageBox::Yes);
			checkReadonly->setChecked(res == QMessageBox::Yes);
		}

		// make sure the bookmark database location selected by
		// the user is actually valid.
		
		if(browserbk->AreBookmarksValid(bookmarks) == false)
		{
			int res = QMessageBox::warning(this, "Bookmarks Are Not Valid",
				"The location you specified does not appear to be a valid\n"
				"location for bookmarks.  Are you sure?",
				QMessageBox::Yes, QMessageBox::No, QMessageBox::Retry);
			if(res == QMessageBox::Retry)
				continue;
			else if(res == QMessageBox::No)
			{
				delete browserbk;
				return;
			}
		}

		// okay, the user selected a bookmark database location
		
		break;
	}

	// we're done with the bookmark class instance
	
	delete browserbk;

	// save the new location selected by the user
	
	item->it->setLocation(bookmarks);
	editLocation->setText(bookmarks);
}

/***************************************************************************
 *                                                                         *
 *   DlgSettings::on_checkReadonly_clicked()                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function (actually a slot) is called by the Qt framework when *
 *      the user clicks on the read-only checkbox.                         *
 *                                                                         *
 ***************************************************************************/

void DlgSettings::on_checkReadonly_clicked(void)
{
	// make sure a browser is actually selected

	QList<QTreeWidgetItem *> items = listBrowsers->selectedItems();
	if(items.count() != 1)
		return;
	
	BrowserCfgItem *item = static_cast<BrowserCfgItem *>(items[0]);
	if(item == NULL)
		return;

	// if the user is making the browser read-only, make sure it
	// is not currently listed as a destination

	if(checkReadonly->isChecked())
	{
		vector<BRWSNUM>::iterator oit =
			std::find(m_cfg.destinations.begin(), m_cfg.destinations.end(),
			item->it->ordinal());
		if(oit != m_cfg.destinations.end())
		{
			int res = QMessageBox::warning(this, "Notice",
				"This browser is currently listed as a destination.  Would\n"
				"you like to remove it from the destination list?  If you\n"
				"say No, the browser will not be made read-only.",
			QMessageBox::Yes, QMessageBox::No);
			if(res == QMessageBox::No)
			{
				checkReadonly->setChecked(false);
				return;
			}
			m_cfg.destinations.erase(oit);
		}
	}

	// save the user's choice
	
	item->it->setReadOnly(checkReadonly->isChecked());
}
