/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * Copyright (C) 1999  Rob Crittenden (rcrit@greyoak.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#endif
#include <errno.h>
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include "packet.h"
#include "tag.h"
#include "message.h"
#include "eventlog.h"
#include "command.h"
#include "account.h"
#include "connection.h"
#include "channel.h"
#include "game.h"
#include "queue.h"
#include "tick.h"
#include "file.h"
#include "prefs.h"
#include "util.h"
#include "bnethash.h"
#include "bnethashconv.h"
#include "bn_type.h"
#include "field_sizes.h"
#include "ladder.h"
#include "adbanner.h"
#include "list.h"
#include "bnettime.h"
#include "bnetd.h"


static t_game_type bngtype_to_gtype(unsigned short bngtype, char const * clienttag);
static unsigned short gtype_to_bngtype(t_game_type gtype);
static t_game_result bngresult_to_gresult(unsigned int bngresult);


static t_game_type bngtype_to_gtype(unsigned short bngtype, char const * clienttag)
{
    if (!clienttag)
    {
	eventlog(eventlog_level_error,"bngtype_to_gtype","got NULL clienttag");
	return game_type_none;
    }
    if (strcmp(clienttag,CLIENTTAG_DIABLORTL)==0 ||
	strcmp(clienttag,CLIENTTAG_DIABLOSHR)==0)
    {
	switch (bngtype)
	{
	case CLIENT_GAMELISTREQ_ALL:
	    return game_type_all;
	case CLIENT_GAMETYPE_DIABLO1:
	case CLIENT_GAMETYPE_DIABLO2:
	case CLIENT_GAMETYPE_DIABLO3:
	    return game_type_diablo;
	default:
	    eventlog(eventlog_level_error,"bngtype_to_gtype","bad Diablo bnet game type \"%s\" %hu",clienttag,bngtype);
	    return game_type_none;
	}
    }
    else if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0 ||
	     strcmp(clienttag,CLIENTTAG_BROODWARS)==0 ||
	     strcmp(clienttag,CLIENTTAG_SHAREWARE)==0)
    {
	switch (bngtype)
	{
	case CLIENT_GAMELISTREQ_ALL:
	    return game_type_all;
	case CLIENT_GAMELISTREQ_MELEE:
	    return game_type_melee;
	case CLIENT_GAMELISTREQ_FFA:
	    return game_type_ffa;
	case CLIENT_GAMELISTREQ_ONEONONE:
	    return game_type_oneonone;
	case CLIENT_GAMELISTREQ_CTF:
	    return game_type_ctf;
	case CLIENT_GAMELISTREQ_GREED:
	    return game_type_greed;
	case CLIENT_GAMELISTREQ_SLAUGHTER:
	    return game_type_slaughter;
	case CLIENT_GAMELISTREQ_SDEATH:
	    return game_type_sdeath;
	case CLIENT_GAMELISTREQ_LADDER:
	    return game_type_ladder;
	case CLIENT_GAMELISTREQ_MAPSET:
	    return game_type_mapset;
	case CLIENT_GAMELISTREQ_TEAMMELEE:
	    return game_type_teammelee;
	case CLIENT_GAMELISTREQ_TEAMFFA:
	    return game_type_teamffa;
	case CLIENT_GAMELISTREQ_TEAMCTF:
	    return game_type_teamctf;
	case CLIENT_GAMELISTREQ_PGL:
	    return game_type_pgl;
	default:
	    eventlog(eventlog_level_error,"bngtype_to_gtype","bad Starcraft bnet game type \"%s\" %hu",clienttag,bngtype);
	    return game_type_none;
	}
    }
    else
    {
	eventlog(eventlog_level_error,"bngtype_to_gtype","unknown game clienttag \"%s\" %hu",clienttag,bngtype);
	return game_type_none;
    }
}


static unsigned short gtype_to_bngtype(t_game_type gtype)
{
    switch (gtype)
    {
    case game_type_all:
	return CLIENT_GAMELISTREQ_ALL;
    case game_type_melee:
	return CLIENT_GAMELISTREQ_MELEE;
    case game_type_ffa:
	return CLIENT_GAMELISTREQ_FFA;
    case game_type_oneonone:
        return CLIENT_GAMELISTREQ_ONEONONE;
    case game_type_ctf:
        return CLIENT_GAMELISTREQ_CTF;
    case game_type_greed:
        return CLIENT_GAMELISTREQ_GREED;
    case game_type_slaughter:
        return CLIENT_GAMELISTREQ_SLAUGHTER;
    case game_type_sdeath:
	return CLIENT_GAMELISTREQ_SDEATH;
    case game_type_ladder:
	return CLIENT_GAMELISTREQ_LADDER;
    case game_type_mapset:
	return CLIENT_GAMELISTREQ_MAPSET;
    case game_type_teammelee:
	return CLIENT_GAMELISTREQ_TEAMMELEE;
    case game_type_teamffa:
	return CLIENT_GAMELISTREQ_TEAMFFA;
    case game_type_teamctf:
	return CLIENT_GAMELISTREQ_TEAMCTF;
    case game_type_pgl:
	return CLIENT_GAMELISTREQ_PGL;
    case game_type_diablo:
	return CLIENT_GAMELISTREQ_DIABLO;
    case game_type_none:
    default:
        eventlog(eventlog_level_error,"gtype_to_bngtype","bad game type %u",(unsigned int)gtype);
	return 0xffff;
    }
}


static t_game_result bngresult_to_gresult(unsigned int bngresult)
{
    switch (bngresult)
    {
    case CLIENT_GAME_REPORT_RESULT_PLAYING:
	return game_result_playing;
	
    case CLIENT_GAME_REPORT_RESULT_WIN:
	return game_result_win;
	
    case CLIENT_GAME_REPORT_RESULT_LOSS:
	return game_result_loss;
	
    case CLIENT_GAME_REPORT_RESULT_DRAW:
	return game_result_draw;
	
    case CLIENT_GAME_REPORT_RESULT_DISCONNECT:
	return game_result_disconnect;
	
    default:
        eventlog(eventlog_level_error,"bngresult_to_gresult","bad game result %u",bngresult);
	return game_result_disconnect; /* bad packet? */
    }
}


extern int handle_packet(t_connection * c, t_packet const * const packet)
{
    t_packet *    rpacket=NULL;
    
    switch (conn_get_class(c))
    {
    case conn_class_normal:
	switch (conn_get_state(c))
	{
	case conn_state_connected:
	    switch (packet_get_type(packet))
	    {
	    case CLIENT_UNKNOWN_1B:
		if (packet_get_size(packet)<sizeof(t_client_unknown_1b))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad UNKNOWN_1B packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_unknown_1b),packet_get_size(packet));
		    break;
		}
		
		eventlog(eventlog_level_debug,"handle_packet","[%d] UNKNOWN_1B unknown1=0x%04hx",conn_get_socket(c),bn_short_get(packet->u.client_unknown_1b.unknown1));
		eventlog(eventlog_level_debug,"handle_packet","[%d] UNKNOWN_1B unknown1=0x%04hx",conn_get_socket(c),bn_short_get(packet->u.client_unknown_1b.port));
		eventlog(eventlog_level_debug,"handle_packet","[%d] UNKNOWN_1B unknown2=0x%08x",conn_get_socket(c),bn_int_get(packet->u.client_unknown_1b.ip));
		eventlog(eventlog_level_debug,"handle_packet","[%d] UNKNOWN_1B unknown3=0x%08x",conn_get_socket(c),bn_int_get(packet->u.client_unknown_1b.unknown2));
		eventlog(eventlog_level_debug,"handle_packet","[%d] UNKNOWN_1B unknown4=0x%08x",conn_get_socket(c),bn_int_get(packet->u.client_unknown_1b.unknown3));
		break;
		
	    case CLIENT_COMPINFO1:
		if (packet_get_size(packet)<sizeof(t_client_compinfo1))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad COMPINFO1 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_compinfo1),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * host;
		    char const * user;
		    
		    if (!(host = packet_get_str_const(packet,sizeof(t_client_compinfo1),128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad COMPINFO1 packet (missing or too long host)",conn_get_socket(c));
			break;
		    }
		    if (!(user = packet_get_str_const(packet,sizeof(t_client_compinfo1)+strlen(host)+1,128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad COMPINFO1 packet (missing or too long user)",conn_get_socket(c));
			break;
		    }
		    
                    conn_set_host(c,host);
                    conn_set_user(c,user);
                }
		
		conn_set_protflag(c,1);
		
		break;
		
	    case CLIENT_COMPINFO2:
		if (packet_get_size(packet)<sizeof(t_client_compinfo2))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad COMPINFO2 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_compinfo2),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * host;
		    char const * user;
		    
		    if (!(host = packet_get_str_const(packet,sizeof(t_client_compinfo2),128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad COMPINFO2 packet (missing or too long host)",conn_get_socket(c));
			break;
		    }
		    if (!(user = packet_get_str_const(packet,sizeof(t_client_compinfo2)+strlen(host)+1,128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad COMPINFO2 packet (missing or too long user)",conn_get_socket(c));
			break;
		    }
		    
                    conn_set_host(c,host);
                    conn_set_user(c,user);
                }
		
		conn_set_protflag(c,2);
		
		break;
		
	    case CLIENT_COUNTRYINFO:
		if (packet_get_size(packet)<sizeof(t_client_countryinfo))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad COUNTRYINFO packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_countryinfo),packet_get_size(packet));
		    break;
		}
		break;
		
	    case CLIENT_UNKNOWN_2B: /* FIXME: what is this? */
		if (packet_get_size(packet)<sizeof(t_client_unknown_2b))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad UNKNOWN_2B packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_unknown_2b),packet_get_size(packet));
		    break;
		}
		break;
		
	    case CLIENT_PROGIDENT:
		if (packet_get_size(packet)<sizeof(t_client_progident))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad PROGIDENT packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_progident),packet_get_size(packet));
		    break;
		}
		
		eventlog(eventlog_level_error,"handle_packet","[%d] CLIENT_PROGIDENT archtag=0x%08x clienttag=0x%08x versionid=0x%08x unknown1=0x%08x",
			 conn_get_socket(c),
			 bn_int_get(packet->u.client_progident.archtag),
			 bn_int_get(packet->u.client_progident.clienttag),
			 bn_int_get(packet->u.client_progident.versionid),
			 bn_int_get(packet->u.client_progident.unknown1));
		
		if (bn_int_tag_eq(packet->u.client_progident.clienttag,CLIENTTAG_STARCRAFT)==0)
		    conn_set_clienttag(c,CLIENTTAG_STARCRAFT);
		else if (bn_int_tag_eq(packet->u.client_progident.clienttag,CLIENTTAG_BROODWARS)==0)
		    conn_set_clienttag(c,CLIENTTAG_BROODWARS);
		else if (bn_int_tag_eq(packet->u.client_progident.clienttag,CLIENTTAG_SHAREWARE)==0)
		    conn_set_clienttag(c,CLIENTTAG_SHAREWARE);
		else if (bn_int_tag_eq(packet->u.client_progident.clienttag,CLIENTTAG_DIABLORTL)==0)
		    conn_set_clienttag(c,CLIENTTAG_DIABLORTL);
		else if (bn_int_tag_eq(packet->u.client_progident.clienttag,CLIENTTAG_WARCIIBNE)==0)
		    conn_set_clienttag(c,CLIENTTAG_WARCIIBNE);
		else
		    eventlog(eventlog_level_error,"handle_packet","[%d] unknown client program type 0x%08x, don't expect this to work",conn_get_socket(c),bn_int_get(packet->u.client_progident.clienttag));
        	
                if ((rpacket = packet_create(packet_class_normal)))
		{
		    packet_set_size(rpacket,sizeof(t_server_compreply));
		    packet_set_type(rpacket,SERVER_COMPREPLY); /* same values as sent to us in COMPINFO[12] */
		    bn_int_set(&rpacket->u.server_compreply.reg_version,SERVER_COMPREPLY_REG_VERSION);
		    bn_int_set(&rpacket->u.server_compreply.reg_auth,SERVER_COMPREPLY_REG_AUTH);
		    bn_int_set(&rpacket->u.server_compreply.client_id,SERVER_COMPREPLY_CLIENT_ID);
		    bn_int_set(&rpacket->u.server_compreply.client_token,SERVER_COMPREPLY_CLIENT_TOKEN);
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		
                switch (conn_get_protflag(c))
		{
		case 1:
                    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_sessionkey1));
			packet_set_type(rpacket,SERVER_SESSIONKEY1);
			bn_int_set(&rpacket->u.server_sessionkey1.sessionkey,conn_get_sessionkey(c));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		    break;
		case 2:
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_sessionkey2));
			packet_set_type(rpacket,SERVER_SESSIONKEY2);
			bn_int_set(&rpacket->u.server_sessionkey2.unknown1,SERVER_SESSIONKEY2_UNKNOWN1);
			bn_int_set(&rpacket->u.server_sessionkey2.sessionkey,conn_get_sessionkey(c));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		    break;
		default:
		    eventlog(eventlog_level_error,"handle_packet","got PROGIDENT before determining protocol version");
		}
		
		if (prefs_get_allow_autoupdate()) /* then cause the client to download the mpq file and request authorization */
		{
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_authreq));
			packet_set_type(rpacket,SERVER_AUTHREQ);
			file_to_mod_time(prefs_get_mpqfile(),&rpacket->u.server_authreq.timestamp);
			packet_append_string(rpacket,prefs_get_mpqfile());
			packet_append_string(rpacket,SERVER_AUTHREQ_EQN); /* FIXME: randomize this when we figure out how to calculate the result */
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		}
		else
		{
		    /* skip over SERVER_AUTHREQ and CLIENT_AUTHREQ */
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_authreply));
			packet_set_type(rpacket,SERVER_AUTHREPLY);
			bn_int_set(&rpacket->u.server_authreply.message,SERVER_AUTHREPLY_MESSAGE_OK);
			packet_append_string(rpacket,"");
			packet_append_string(rpacket,""); /* FIXME: what's the second strings for? */
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
                }
		
		break;
		
	    case CLIENT_CLOSEGAME: /* FIXME: Why do we get this here? */
		break;
		
	    case CLIENT_CREATEACCTREQ:
		if (packet_get_size(packet)<sizeof(t_client_createacctreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad CREATEACCTREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_createacctreq),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const * username;
		    t_account *  temp;
		    t_hash       newpasshash1;
		    
		    if (!(username = packet_get_str_const(packet,sizeof(t_client_createacctreq),USER_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CREATEACCTREQ (missing or too long username)",conn_get_socket(c));
			break;
		    }
		    
		    eventlog(eventlog_level_info,"handle_packet","[%d] new account requested for \"%s\"",conn_get_socket(c),username);
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_createacctreply));
		    packet_set_type(rpacket,SERVER_CREATEACCTREPLY);
		    
		    if (prefs_get_allow_new_accounts()==0)
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] account not created (disabled)",conn_get_socket(c));
			bn_int_set(&rpacket->u.server_createacctreply.result,SERVER_CREATEACCTREPLY_RESULT_NO);
		    }
		    else
		    {
		        bnhash_to_hash(packet->u.client_createacctreq.password_hash1,&newpasshash1);
		        if (!(temp = account_create(username,hash_get_str((t_hash const *)&newpasshash1)))) /* avoid warning */
		        {
			    eventlog(eventlog_level_info,"handle_packet","[%d] account not created (failed)",conn_get_socket(c));
			    bn_int_set(&rpacket->u.server_createacctreply.result,SERVER_CREATEACCTREPLY_RESULT_NO);
		        }
		        else if (!accountlist_add_account(temp))
		        {
			    account_destroy(temp);
			    eventlog(eventlog_level_info,"handle_packet","[%d] account not inserted",conn_get_socket(c));
			    bn_int_set(&rpacket->u.server_createacctreply.result,SERVER_CREATEACCTREPLY_RESULT_NO);
		        }
		        else
		        {
			    eventlog(eventlog_level_info,"handle_packet","[%d] account created",conn_get_socket(c));
			    bn_int_set(&rpacket->u.server_createacctreply.result,SERVER_CREATEACCTREPLY_RESULT_OK);
		        }
		    }
		    bn_int_set(&rpacket->u.server_createacctreply.unknown1,SERVER_CREATEACCTREPLY_UNKNOWN1);
		    bn_int_set(&rpacket->u.server_createacctreply.unknown2,SERVER_CREATEACCTREPLY_UNKNOWN2);
		    bn_int_set(&rpacket->u.server_createacctreply.unknown3,SERVER_CREATEACCTREPLY_UNKNOWN3);
		    bn_int_set(&rpacket->u.server_createacctreply.unknown4,SERVER_CREATEACCTREPLY_UNKNOWN4);
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
	        break;
	        
	    case CLIENT_CHANGEPASSREQ:
		if (packet_get_size(packet)<sizeof(t_client_changepassreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad CHANGEPASSREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_changepassreq),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const * username;
		    t_account *  account;
		    
		    if (!(username = packet_get_str_const(packet,sizeof(t_client_changepassreq),USER_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CHANGEPASSREQ (missing or too long username)",conn_get_socket(c));
			break;
		    }
		    
		    eventlog(eventlog_level_info,"handle_packet","[%d] password change requested for \"%s\"",conn_get_socket(c),username);
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_changepassack));
		    packet_set_type(rpacket,SERVER_CHANGEPASSACK);
		    
		    /* fail if logged in or no account */
		    if (connlist_find_connection(username) || !(account = accountlist_find_account(username)))
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] password change for \"%s\" refused (bad account)",conn_get_socket(c),username);
			bn_int_set(&rpacket->u.server_changepassack.message,SERVER_CHANGEPASSACK_MESSAGE_FAIL);
		    }
		    else if (account_get_auth_changepass(account)==0) /* default to true */
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] password change for \"%s\" refused (no change access)",conn_get_socket(c),username);
			bn_int_set(&rpacket->u.server_changepassack.message,SERVER_CHANGEPASSACK_MESSAGE_FAIL);
		    }
		    else
		    {
			struct
			{
			    bn_int   ticks;
			    bn_int   sessionkey;
			    bn_int   passhash1[5];
			}            temp;
			char const * oldstrhash1;
			t_hash       oldpasshash1;
			t_hash       oldpasshash2;
			t_hash       trypasshash2;
			t_hash       newpasshash1;
			char const * tname;
			
		        if (conn_get_sessionkey(c)!=bn_int_get(packet->u.client_changepassreq.sessionkey))
			    eventlog(eventlog_level_error,"handle_packet","[%d] password change: got session key 0x%08x expected 0x%08x, using it anyway",conn_get_socket(c),bn_int_get(packet->u.client_changepassreq.sessionkey),conn_get_sessionkey(c));
			
			if ((oldstrhash1 = account_get_pass(account)))
			{
			    bn_int_set(&temp.ticks,bn_int_get(packet->u.client_changepassreq.ticks));
			    bn_int_set(&temp.sessionkey,bn_int_get(packet->u.client_changepassreq.sessionkey));
			    if (hash_set_str(&oldpasshash1,oldstrhash1)<0)
			    {
				account_unget_pass(oldstrhash1);
				bnhash_to_hash(packet->u.client_changepassreq.newpassword_hash1,&newpasshash1);
				account_set_pass(account,hash_get_str((t_hash const *)&newpasshash1)); /* avoid warning */
				eventlog(eventlog_level_info,"handle_packet","[%d] password change for \"%s\" successful (bad previous password)",conn_get_socket(c),(tname = account_get_name(account)));
				account_unget_name(tname);
				bn_int_set(&rpacket->u.server_changepassack.message,SERVER_CHANGEPASSACK_MESSAGE_SUCCESS);
			    }
			    else
			    {
				account_unget_pass(oldstrhash1);
				hash_to_bnhash((t_hash const *)&oldpasshash1,temp.passhash1); /* avoid warning */
				bnet_hash(&oldpasshash2,sizeof(temp),&temp); /* do the double hash */
				bnhash_to_hash(packet->u.client_changepassreq.oldpassword_hash2,&trypasshash2);
				
				if (memcmp(trypasshash2,oldpasshash2,BNETHASH_LEN)==0)
				{
				    bnhash_to_hash(packet->u.client_changepassreq.newpassword_hash1,&newpasshash1);
				    account_set_pass(account,hash_get_str((t_hash const *)&newpasshash1)); /* avoid warning */
				    eventlog(eventlog_level_info,"handle_packet","[%d] password change for \"%s\" successful (previous password)",conn_get_socket(c),(tname = account_get_name(account)));
				    account_unget_name(tname);
				    bn_int_set(&rpacket->u.server_changepassack.message,SERVER_CHANGEPASSACK_MESSAGE_SUCCESS);
				}
				else
				{
				    eventlog(eventlog_level_info,"handle_packet","[%d] password change for \"%s\" refused (wrong password)",conn_get_socket(c),(tname = account_get_name(account)));
				    account_unget_name(tname);
				    bn_int_set(&rpacket->u.server_changepassack.message,SERVER_CHANGEPASSACK_MESSAGE_FAIL);
				}
			    }
			}
			else
			{
			    bnhash_to_hash(packet->u.client_changepassreq.newpassword_hash1,&newpasshash1);
			    account_set_pass(account,hash_get_str((t_hash const *)&newpasshash1)); /* avoid warning */
			    eventlog(eventlog_level_info,"handle_packet","[%d] password change for \"%s\" successful (no previous password)",conn_get_socket(c),(tname = account_get_name(account)));
			    account_unget_name(tname);
			    bn_int_set(&rpacket->u.server_changepassack.message,SERVER_CHANGEPASSACK_MESSAGE_SUCCESS);
			}
		    }
		    
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
	        break;
		
	    case CLIENT_ECHOREPLY:
		if (packet_get_size(packet)<sizeof(t_client_echoreply))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad ECHOREPLY packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_echoreply),packet_get_size(packet));
		    break;
		}
		
		{
		    unsigned int now;
		    unsigned int then;
		    
		    now = get_ticks();
		    then = bn_int_get(packet->u.client_echoreply.ticks);
		    if (!now || !then || now<then)
		        eventlog(eventlog_level_warn,"handle_packet","[%d] bad timing in echo reply: now=%u then=%u",conn_get_socket(c),now,then);
		    else
			conn_set_latency(c,now-then);
		}
		break;
		
	    case CLIENT_AUTHREQ:
		if (packet_get_size(packet)<sizeof(t_client_authreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad AUTHREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_authreq),packet_get_size(packet));
		    break;
		}
		
		eventlog(eventlog_level_error,"handle_packet","[%d] CLIENT_AUTHREQ archtag=0x%08x clienttag=0x%08x versionid=0x%08x v=%02x.%02x.%02x.%02x answer=0x%08x",
			 conn_get_socket(c),
			 bn_int_get(packet->u.client_authreq.archtag),
			 bn_int_get(packet->u.client_authreq.clienttag),
			 bn_int_get(packet->u.client_authreq.versionid),
			 (unsigned int)bn_int_get(packet->u.client_authreq.v4),
			 (unsigned int)bn_int_get(packet->u.client_authreq.v3),
			 (unsigned int)bn_int_get(packet->u.client_authreq.v2),
			 (unsigned int)bn_int_get(packet->u.client_authreq.v1),
			 bn_int_get(packet->u.client_authreq.answer));
		
		{
                    int  version;
                    int  server_authreply=SERVER_AUTHREPLY_MESSAGE_OK;
		    char tempver[16];
		    
		    {
			char const * exename;
			
	 		if (!(exename = packet_get_str_const(packet,sizeof(t_client_authreq),128)))
	 		{
			    eventlog(eventlog_level_error,"handle_packet","[%d] got bad AUTHREQ (missing or too long exename)",conn_get_socket(c));
			    break;
	 		}
			conn_set_clientexe(c,exename);
	 	    }
		    
		    sprintf(tempver,"%u.%u.%u.%u",
			    (unsigned int)bn_byte_get(packet->u.client_authreq.v1),
			    (unsigned int)bn_byte_get(packet->u.client_authreq.v2),
			    (unsigned int)bn_byte_get(packet->u.client_authreq.v3),
			    (unsigned int)bn_byte_get(packet->u.client_authreq.v4));
		    conn_set_clientver(c,tempver);
		    version = 0; /* FIXME: how to calculate version # */
		    
                    eventlog(eventlog_level_info,"handle_packet","[%d] clienttag=\"%s\" version=%u.%u.%u.%u versionid=0x%08x answer=0x%08x",
			     conn_get_socket(c),
			     conn_get_clienttag(c),
			     (unsigned int)bn_byte_get(packet->u.client_authreq.v1),
			     (unsigned int)bn_byte_get(packet->u.client_authreq.v2),
			     (unsigned int)bn_byte_get(packet->u.client_authreq.v3),
			     (unsigned int)bn_byte_get(packet->u.client_authreq.v4),
			     bn_int_get(packet->u.client_authreq.versionid),
			     bn_int_get(packet->u.client_authreq.answer));
		    
		    /* only check the version upon login, and only for Starcraft for now */
                    if (strcmp(conn_get_clienttag(c),CLIENTTAG_STARCRAFT)==0 ||
                        strcmp(conn_get_clienttag(c),CLIENTTAG_BROODWARS)==0)
			if (version!=prefs_get_mpqversion())
			    server_authreply = SERVER_AUTHREPLY_MESSAGE_UPDATE;
		    
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_authreply));
			packet_set_type(rpacket,SERVER_AUTHREPLY);
			bn_int_set(&rpacket->u.server_authreply.message,server_authreply);
			if (server_authreply==SERVER_AUTHREPLY_MESSAGE_UPDATE)
			{
			    eventlog(eventlog_level_info,"handle_packet","[%d] an upgrade from %d to %d is available \"%s\"",conn_get_socket(c),version,prefs_get_mpqversion(),prefs_get_mpqfile());
			    packet_append_string(rpacket,prefs_get_mpqfile());
			}
			else
			    packet_append_string(rpacket,"");
			packet_append_string(rpacket,""); /* FIXME: what's the second strings for? */
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
                }
		break;
		
	    case CLIENT_ICONREQ:
		if (packet_get_size(packet)<sizeof(t_client_iconreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad ICONREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_iconreq),packet_get_size(packet));
		    break;
		}
		
		if ((rpacket = packet_create(packet_class_normal)))
		{
		    packet_set_size(rpacket,sizeof(t_server_iconreply));
		    packet_set_type(rpacket,SERVER_ICONREPLY);
		    file_to_mod_time(prefs_get_iconfile(),&rpacket->u.server_iconreply.timestamp);
		    packet_append_string(rpacket,prefs_get_iconfile());
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_CDKEY:
		if (packet_get_size(packet)<sizeof(t_client_cdkey))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad CDKEY packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_cdkey),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * cdkey;
		    char const * owner;
		    
		    if (!(cdkey = packet_get_str_const(packet,sizeof(t_client_cdkey),128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CDKEY packet (missing or too long cdkey)",conn_get_socket(c));
			break;
		    }
		    if (!(owner = packet_get_str_const(packet,sizeof(t_client_cdkey)+strlen(cdkey)+1,128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CDKEY packet (missing or too long owner)",conn_get_socket(c));
			break;
		    }
		    
            	    conn_set_cdkey(c,cdkey);
            	    conn_set_owner(c,owner);
                    
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_cdkeyreply));
			packet_set_type(rpacket,SERVER_CDKEYREPLY);
			bn_int_set(&rpacket->u.server_cdkeyreply.message,SERVER_CDKEYREPLY_MESSAGE_OK);
			packet_append_string(rpacket,owner);
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		}
		break;
		
	    case CLIENT_CDKEY2:
		if (packet_get_size(packet)<sizeof(t_client_cdkey2))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad CDKEY2 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_cdkey2),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * owner;
		    
		    if (!(owner = packet_get_str_const(packet,sizeof(t_client_cdkey2),128)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CDKEY2 packet (missing or too long owner)",conn_get_socket(c));
			break;
		    }
		    
            	    conn_set_owner(c,owner);
                    
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_cdkeyreply2));
			packet_set_type(rpacket,SERVER_CDKEYREPLY2);
			bn_int_set(&rpacket->u.server_cdkeyreply2.message,SERVER_CDKEYREPLY2_MESSAGE_OK);
			packet_append_string(rpacket,owner);
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		}
		break;
		
	    case CLIENT_UNKNOWN_14: /* FIXME: What is this for? */
		if (packet_get_size(packet)<sizeof(t_client_unknown_14))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad UNKNOWN_14 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_unknown_14),packet_get_size(packet));
		    break;
		}
		break;
		
	    case CLIENT_TOSREQ:
		if (packet_get_size(packet)<sizeof(t_client_tosreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad TOSREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_tosreq),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * tosfile;
		    
		    if (!(tosfile = packet_get_str_const(packet,sizeof(t_client_tosreq),128)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad TOSREQ packet (missing or too long tosfile)",conn_get_socket(c));
			break;
		    }
		    eventlog(eventlog_level_info,"handle_packet","[%d] TOS requested: \"%s\"",conn_get_socket(c),tosfile);
		}
		
		if ((rpacket = packet_create(packet_class_normal)))
		{
		    packet_set_size(rpacket,sizeof(t_server_tosreply));
		    packet_set_type(rpacket,SERVER_TOSREPLY);
		    bn_int_set(&rpacket->u.server_tosreply.unknown1,SERVER_TOSREPLY_UNKNOWN1);
		    bn_int_set(&rpacket->u.server_tosreply.unknown2,SERVER_TOSREPLY_UNKNOWN2);
		    file_to_mod_time(prefs_get_tosfile(),&rpacket->u.server_tosreply.timestamp);
		    packet_append_string(rpacket,prefs_get_tosfile());
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_STATSREQ:
		if (packet_get_size(packet)<sizeof(t_client_statsreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_statsreq),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * name;
		    char const * key;
		    unsigned int name_count;
		    unsigned int key_count;
		    unsigned int i,j;
		    unsigned int name_off;
		    unsigned int keys_off;
		    unsigned int key_off;
		    t_account *  account;
		    char const * tval;
		    char const * tname;
		    
		    name_count = bn_int_get(packet->u.client_statsreq.name_count);
		    key_count = bn_int_get(packet->u.client_statsreq.key_count);
		    
		    for (i=0,name_off=sizeof(t_client_statsreq);
			 i<name_count && (name = packet_get_str_const(packet,name_off,128));
			 i++,name_off+=strlen(name)+1);
		    if (i<name_count)
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSREQ packet (only %u names of %u)",conn_get_socket(c),i,name_count);
			break;
		    }
		    keys_off = name_off;
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_statsreply));
		    packet_set_type(rpacket,SERVER_STATSREPLY);
		    bn_int_set(&rpacket->u.server_statsreply.name_count,name_count);
		    bn_int_set(&rpacket->u.server_statsreply.key_count,key_count);
		    bn_int_set(&rpacket->u.server_statsreply.unknown1,bn_int_get(packet->u.client_statsreq.unknown1));
		    
		    for (i=0,name_off=sizeof(t_client_statsreq);
			 i<name_count && (name = packet_get_str_const(packet,name_off,128));
			 i++,name_off+=strlen(name)+1)
		    {
			account = accountlist_find_account(name);
		        for (j=0,key_off=keys_off;
			     j<key_count && (key = packet_get_str_const(packet,key_off,512));
			     j++,key_off+=strlen(key)+1)
			    if (account && (tval = account_get_strattr(account,key)))
			    {
				packet_append_string(rpacket,tval);
				account_unget_strattr(tval);
			    }
			    else
			    {
				packet_append_string(rpacket,""); /* FIXME: what should really happen here? */
				if (account && key[0]!='\0')
				{
				    eventlog(eventlog_level_debug,"handle_packet","[%d] no entry \"%s\" in account \"%s\"",conn_get_socket(c),key,(tname = account_get_name(account)));
				    account_unget_name(tname);
				}
			    }
		    }
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
      		}
		
		break;
		
	    case CLIENT_LOGINREQ:
		if (packet_get_size(packet)<sizeof(t_client_loginreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad LOGINREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_loginreq),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const * username;
		    t_account *  account;
		    
		    if (!(username = packet_get_str_const(packet,sizeof(t_client_loginreq),USER_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad LOGINREQ (missing or too long username)",conn_get_socket(c));
			break;
		    }
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_loginreply));
		    packet_set_type(rpacket,SERVER_LOGINREPLY);
		    
		    /* already logged in */
		    if (connlist_find_connection(username) &&
			prefs_get_kick_old_login()==0)
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] login for \"%s\" refused (already logged in)",conn_get_socket(c),username);
			bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_FAIL);
		    }
		    else
			/* fail if no account */
			if (!(account = accountlist_find_account(username)))
			{
			    eventlog(eventlog_level_info,"handle_packet","[%d] login for \"%s\" refused (bad account)",conn_get_socket(c),username);
			    bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_FAIL);
			}
			else if (account_get_auth_normallogin(account)==0) /* default to true */
			{
			    eventlog(eventlog_level_info,"handle_packet","[%d] login for \"%s\" refused (no normal access)",conn_get_socket(c),username);
			    bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_FAIL);
			}
			else
			{
			    struct
			    {
				bn_int   ticks;
				bn_int   sessionkey;
				bn_int   passhash1[5];
			    }            temp;
			    char const * oldstrhash1;
			    t_hash       oldpasshash1;
			    t_hash       oldpasshash2;
			    t_hash       trypasshash2;
			    char const * tname;
			    
			    if (conn_get_sessionkey(c)!=bn_int_get(packet->u.client_changepassreq.sessionkey))
				eventlog(eventlog_level_error,"handle_packet","[%d] login: got session key 0x%08x expected 0x%08x, using it anyway",conn_get_socket(c),bn_int_get(packet->u.client_changepassreq.sessionkey),conn_get_sessionkey(c));
			    
			    if ((oldstrhash1 = account_get_pass(account)))
			    {
				bn_int_set(&temp.ticks,bn_int_get(packet->u.client_loginreq.ticks));
				bn_int_set(&temp.sessionkey,bn_int_get(packet->u.client_loginreq.sessionkey));
				if (hash_set_str(&oldpasshash1,oldstrhash1)<0)
				{
				    account_unget_pass(oldstrhash1);
				    eventlog(eventlog_level_info,"handle_packet","[%d] login for \"%s\" refused (corrupted passhash1?)",conn_get_socket(c),(tname = account_get_name(account)));
				    account_unget_name(tname);
				    bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_FAIL);
				}
				else
				{
				    account_unget_pass(oldstrhash1);
				    hash_to_bnhash((t_hash const *)&oldpasshash1,temp.passhash1); /* avoid warning */
				    
				    bnet_hash(&oldpasshash2,sizeof(temp),&temp); /* do the double hash */
				    bnhash_to_hash(packet->u.client_loginreq.password_hash2,&trypasshash2);
				    
				    if (memcmp(trypasshash2,oldpasshash2,BNETHASH_LEN)==0)
				    {
					conn_set_account(c,account);
					eventlog(eventlog_level_info,"handle_packet","[%d] \"%s\" logged in (correct password)",conn_get_socket(c),(tname = conn_get_username(c)));
					conn_unget_username(tname);
					bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_SUCCESS);
				    }
				    else
				    {
					eventlog(eventlog_level_info,"handle_packet","[%d] login for \"%s\" refused (wrong password)",conn_get_socket(c),(tname = account_get_name(account)));
					account_unget_name(tname);
					bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_FAIL);
				    }
				}
			    }
			    else
			    {
				conn_set_account(c,account);
				eventlog(eventlog_level_info,"handle_packet","[%d] \"%s\" logged in (no password)",conn_get_socket(c),(tname = account_get_name(account)));
				account_unget_name(tname);
				bn_int_set(&rpacket->u.server_loginreply.message,SERVER_LOGINREPLY_MESSAGE_SUCCESS);
			    }
			}
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
	        break;
		
	    default:
		eventlog(eventlog_level_error,"handle_packet","[%d] unknown (unlogged in) packet type 0x%04hx, len %hu",conn_get_socket(c),packet_get_type(packet),packet_get_size(packet));
	    }
	    break;
	    
	case conn_state_loggedin:
	    
	    switch (packet_get_type(packet))
	    {
	    case CLIENT_ECHOREPLY:
		if (packet_get_size(packet)<sizeof(t_client_echoreply))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad ECHOREPLY packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_echoreply),packet_get_size(packet));
		    break;
		}
		
	        {
		    unsigned int now;
		    unsigned int then;
		    
		    now = get_ticks();
		    then = bn_int_get(packet->u.client_echoreply.ticks);
		    if (!now || !then || now<then)
			eventlog(eventlog_level_warn,"handle_packet","[%d] bad timing in echo reply: now=%u then=%u",conn_get_socket(c),now,then);
		    else
			conn_set_latency(c,now-then);
		}
		break;
		
	    case CLIENT_PINGREQ:
		if (packet_get_size(packet)<sizeof(t_client_pingreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad PINGREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_pingreq),packet_get_size(packet));
		    break;
		}
		
		if ((rpacket = packet_create(packet_class_normal)))
		{
		    packet_set_size(rpacket,sizeof(t_server_pingreply));
		    packet_set_type(rpacket,SERVER_PINGREPLY);
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_ADREQ:
		if (packet_get_size(packet)<sizeof(t_client_adreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad ADREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_adreq),packet_get_size(packet));
		    break;
		}
		{
		    t_adbanner * ad;
		    
		    if (!(ad = adbanner_pick(c,bn_int_get(packet->u.client_adreq.prev_adid))))
			break;
		    
/*		    eventlog(eventlog_level_debug,"handle_packet","[%d] picking ad file=\"%s\" id=0x%06x tag=%u",conn_get_socket(c),adbanner_get_filename(ad),adbanner_get_id(ad),adbanner_get_extensiontag(ad));*/
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			packet_set_size(rpacket,sizeof(t_server_adreply));
			packet_set_type(rpacket,SERVER_ADREPLY);
			bn_int_set(&rpacket->u.server_adreply.adid,adbanner_get_id(ad));
			bn_int_set(&rpacket->u.server_adreply.extensiontag,adbanner_get_extensiontag(ad));
			file_to_mod_time(adbanner_get_filename(ad),&rpacket->u.server_adreply.timestamp);
			packet_append_string(rpacket,adbanner_get_filename(ad));
			packet_append_string(rpacket,adbanner_get_link(ad));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		}
		break;
		
	    case CLIENT_ADACK:
		if (packet_get_size(packet)<sizeof(t_client_adack))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad ADACK packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_adack),packet_get_size(packet));
		    break;
		}
	        {
		    char const * tname;
		    
		    eventlog(eventlog_level_info,"handle_packet","[%d] ad acknowledgement for adid 0x%04hx from \"%s\"",conn_get_socket(c),bn_int_get(packet->u.client_adack.adid),(tname = conn_get_username(c)));
		    conn_unget_username(tname);
		}
		break;
		
	    case CLIENT_ADCLICK:
		if (packet_get_size(packet)<sizeof(t_client_adclick))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad ADCLICK packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_adclick),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * tname;
		    
		    eventlog(eventlog_level_info,"handle_packet","[%d] ad click for adid 0x%04hx from \"%s\"",
			     conn_get_socket(c),bn_int_get(packet->u.client_adclick.adid),(tname = conn_get_username(c)));
		    conn_unget_username(tname);
		}
		break;
		
	    case CLIENT_STATSREQ:
		if (packet_get_size(packet)<sizeof(t_client_statsreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_statsreq),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * name;
		    char const * key;
		    unsigned int name_count;
		    unsigned int key_count;
		    unsigned int i,j;
		    unsigned int name_off;
		    unsigned int keys_off;
		    unsigned int key_off;
		    t_account *  account;
		    char const * tval;
		    char const * tname;
		    
		    name_count = bn_int_get(packet->u.client_statsreq.name_count);
		    key_count = bn_int_get(packet->u.client_statsreq.key_count);
		    
		    for (i=0,name_off=sizeof(t_client_statsreq);
			 i<name_count && (name = packet_get_str_const(packet,name_off,128));
			 i++,name_off+=strlen(name)+1);
		    if (i<name_count)
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSREQ packet (only %u names of %u)",conn_get_socket(c),i,name_count);
			break;
		    }
		    keys_off = name_off;
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_statsreply));
		    packet_set_type(rpacket,SERVER_STATSREPLY);
		    bn_int_set(&rpacket->u.server_statsreply.name_count,name_count);
		    bn_int_set(&rpacket->u.server_statsreply.key_count,key_count);
		    bn_int_set(&rpacket->u.server_statsreply.unknown1,bn_int_get(packet->u.client_statsreq.unknown1));
		    
		    for (i=0,name_off=sizeof(t_client_statsreq);
			 i<name_count && (name = packet_get_str_const(packet,name_off,128));
			 i++,name_off+=strlen(name)+1)
		    {
			if (name[0]=='\0')
			    account = conn_get_account(c);
			else
			    account = accountlist_find_account(name);
			eventlog(eventlog_level_debug,"handle_packet","[%d] sending account info for account \"%s\" (\"%s\")",conn_get_socket(c),account_get_name(account),name);
		        for (j=0,key_off=keys_off;
			     j<key_count && (key = packet_get_str_const(packet,key_off,512));
			     j++,key_off+=strlen(key)+1)
			    if (account && (tval = account_get_strattr(account,key)))
			    {
				packet_append_string(rpacket,tval);
				account_unget_strattr(tval);
			    }
			    else
			    {
				packet_append_string(rpacket,""); /* FIXME: what should really happen here? */
				if (account && key[0]!='\0')
				{
				    eventlog(eventlog_level_debug,"handle_packet","[%d] no entry \"%s\" in account \"%s\"",conn_get_socket(c),key,(tname = account_get_name(account)));
				    account_unget_name(tname);
				}
			    }
		    }
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_STATSUPDATE:
		if (packet_get_size(packet)<sizeof(t_client_statsupdate))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSUPDATE packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_statsupdate),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * name;
		    char const * key;
		    char const * val;
		    unsigned int name_count;
		    unsigned int key_count;
		    unsigned int i,j;
		    unsigned int name_off;
		    unsigned int keys_off;
		    unsigned int key_off;
		    unsigned int vals_off;
		    unsigned int val_off;
		    t_account *  account;
		    
		    name_count = bn_int_get(packet->u.client_statsupdate.name_count);
		    key_count = bn_int_get(packet->u.client_statsupdate.key_count);
		    
		    if (name_count!=1)
			eventlog(eventlog_level_warn,"handle_packet","[%d] got suspicious STATSUPDATE packet (name_count=%u)",conn_get_socket(c),name_count);
		    
		    for (i=0,name_off=sizeof(t_client_statsupdate);
			 i<name_count && (name = packet_get_str_const(packet,name_off,128));
			 i++,name_off+=strlen(name)+1);
		    if (i<name_count)
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSUPDATE packet (only %u names of %u)",conn_get_socket(c),i,name_count);
			break;
		    }
		    keys_off = name_off;
		    
		    for (i=0,key_off=keys_off;
			 i<key_count && (key = packet_get_str_const(packet,key_off,1024));
			 i++,key_off+=strlen(key)+1);
		    if (i<key_count)
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad STATSUPDATE packet (only %u keys of %u)",conn_get_socket(c),i,key_count);
			break;
		    }
		    vals_off = key_off;
		    
		    if ((account = conn_get_account(c)))
		    {
			char const * tname;
			
			if (account_get_auth_changeprofile(account)==0) /* default to true */
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] stats update for \"%s\" refused (no change profile access)",conn_get_socket(c),(tname = conn_get_username(c)));
			    conn_unget_username(tname);
			    break;
			}
			eventlog(eventlog_level_info,"handle_packet","[%d] updating player profile for \"%s\"",conn_get_socket(c),(tname = conn_get_username(c)));
			conn_unget_username(tname);
			    
			for (i=0,name_off=sizeof(t_client_statsupdate);
			     i<name_count && (name = packet_get_str_const(packet,name_off,128));
			     i++,name_off+=strlen(name)+1)
			    for (j=0,key_off=keys_off,val_off=vals_off;
				 j<key_count && (key = packet_get_str_const(packet,key_off,1024)) && (val = packet_get_str_const(packet,val_off,1024));
				 j++,key_off+=strlen(key)+1,val_off+=strlen(val)+1)
				if (strlen(key)<9 || strncasecmp(key,"profile\\",8)!=0)
				    eventlog(eventlog_level_error,"handle_packet","[%d] got STATSUPDATE with suspicious key \"%s\" value \"%s\"",conn_get_socket(c),key,val);
				else
				    account_set_strattr(account,key,val);
		    }
		}
		break;
		
	    case CLIENT_PLAYERINFOREQ:
		if (packet_get_size(packet)<sizeof(t_client_playerinforeq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad PLAYERINFOREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_playerinforeq),packet_get_size(packet));
		    break;
		}
		
		{
		    char const * username;
		    char const * info;
		    t_account *  account;
		    
		    if (!(username = packet_get_str_const(packet,sizeof(t_client_playerinforeq),USER_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad PLAYERINFOREQ (missing or too long username)",conn_get_socket(c));
			break;
		    }
		    if (!(info = packet_get_str_const(packet,sizeof(t_client_playerinforeq)+strlen(username)+1,1024)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad PLAYERINFOREQ (missing or too long info)",conn_get_socket(c));
			break;
		    }
		    
		    if (info[0]!='\0')
			conn_set_playerinfo(c,info);
		    
		    account = accountlist_find_account(username);
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_playerinforeply));
		    packet_set_type(rpacket,SERVER_PLAYERINFOREPLY);
		    
		    if (account)
		    {
			char const * tname;
			
			packet_append_string(rpacket,(tname = account_get_name(account)));
			packet_append_string(rpacket,conn_get_playerinfo(c,conn_get_clienttag(c)));
			packet_append_string(rpacket,tname);
			account_unget_name(tname);
		    }
		    else
		    {
			packet_append_string(rpacket,"");
			packet_append_string(rpacket,"");
			packet_append_string(rpacket,"");
		    }
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_PROGIDENT2:
		if (packet_get_size(packet)<sizeof(t_client_progident2))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad PROGIDENT2 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_progident2),packet_get_size(packet));
		    break;
		}
		
		if (bn_int_tag_eq(packet->u.client_progident2.clienttag,CLIENTTAG_STARCRAFT)==0)
		    conn_set_clienttag(c,CLIENTTAG_STARCRAFT);
		else if (bn_int_tag_eq(packet->u.client_progident2.clienttag,CLIENTTAG_BROODWARS)==0)
		    conn_set_clienttag(c,CLIENTTAG_BROODWARS);
		else if (bn_int_tag_eq(packet->u.client_progident2.clienttag,CLIENTTAG_SHAREWARE)==0)
		    conn_set_clienttag(c,CLIENTTAG_SHAREWARE);
		else if (bn_int_tag_eq(packet->u.client_progident2.clienttag,CLIENTTAG_DIABLORTL)==0)
		    conn_set_clienttag(c,CLIENTTAG_DIABLORTL);
		else if (bn_int_tag_eq(packet->u.client_progident2.clienttag,CLIENTTAG_WARCIIBNE)==0)
		    conn_set_clienttag(c,CLIENTTAG_WARCIIBNE);
		else
		    eventlog(eventlog_level_error,"handle_packet","[%d] unknown client program type 0x%08x, don't expect this to work",conn_get_socket(c),bn_int_get(packet->u.client_progident2.clienttag));
		
		if ((rpacket = packet_create(packet_class_normal)))
		{
		    packet_set_size(rpacket,sizeof(t_server_channellist));
		    packet_set_type(rpacket,SERVER_CHANNELLIST);
		    {
			t_channel * ch;
			
			for (ch=channellist_get_first(); ch; ch=channellist_get_next())
			    if ((!prefs_get_hide_temp_channels() || channel_get_permanent(ch)) &&
				(!channel_get_clienttag(ch) || strcmp(channel_get_clienttag(ch),conn_get_clienttag(c))==0))
				packet_append_string(rpacket,channel_get_name(ch));
		    }
		    packet_append_string(rpacket,"");
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_JOINCHANNEL:
		if (packet_get_size(packet)<sizeof(t_client_joinchannel))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad JOINCHANNEL packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_joinchannel),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const *      cname;
		    t_channel const * chan;
		    int               found=1;
		    
		    if (!(cname = packet_get_str_const(packet,sizeof(t_client_joinchannel),CHANNEL_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad JOINCHANNEL (missing or too long cname)",conn_get_socket(c));
			break;
		    }

		    switch (bn_int_get(packet->u.client_joinchannel.channelflag))
		    {
		    case CLIENT_JOINCHANNEL_NORMAL:
			eventlog(eventlog_level_info,"handle_packet","[%d] CLIENT_JOINCHANNEL_NORMAL channel \"%s\"",conn_get_socket(c),cname);
			
			/* Does that channel exist? */
			if ((chan = channellist_find_channel(cname))!=NULL)
			    break; /* just join it */
			else if (prefs_get_ask_new_channel())
			{
			    found=0;
			    eventlog(eventlog_level_info,"handle_packet","[%d] didn't find channel \"%s\" to join",conn_get_socket(c),cname);
			    message_send(c,MT_CHANNELDOESNOTEXIST,c,cname);
			}
			break;
		    case CLIENT_JOINCHANNEL_GENERIC:
			eventlog(eventlog_level_info,"handle_packet","[%d] CLIENT_JOINCHANNEL_GENERIC channel \"%s\"",conn_get_socket(c),cname);
			/* don't have to do anything here */
			break;
		    case CLIENT_JOINCHANNEL_CREATE:
			eventlog(eventlog_level_info,"handle_packet","[%d] CLIENT_JOINCHANNEL_CREATE channel \"%s\"",conn_get_socket(c),cname);
			/* don't have to do anything here */
			break;
		    }
		    if (found && conn_set_channel(c,cname)<0)
			conn_set_channel(c,CHANNEL_NAME_BANNED); /* should not fail */
		}
		break;
		
	    case CLIENT_MESSAGE:
		if (packet_get_size(packet)<sizeof(t_client_message))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad MESSAGE packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_message),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const * text;
		    
		    if (!(text = packet_get_str_const(packet,sizeof(t_client_message),MAX_MESSAGE_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad MESSAGE (missing or too long text)",conn_get_socket(c));
			break;
		    }
		    
		    if (text[0]=='/')
		    {
			handle_command(c,text);
			break;
		    }
		    
		    if (conn_get_channel(c))
			channel_message_send(conn_get_channel(c),MT_MESSAGE,c,text);
		}
		break;
		
	    case CLIENT_GAMELISTREQ:
		if (packet_get_size(packet)<sizeof(t_client_gamelistreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad GAMELISTREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_gamelistreq),packet_get_size(packet));
		    break;
		}
		
		{
		    char const *                gamename;
		    unsigned short              bngtype;
		    t_game_type                 gtype;
		    t_game *                    game;
		    t_server_gamelistreply_game glgame;
		    
		    if (!(gamename = packet_get_str_const(packet,sizeof(t_client_gamelistreq),GAME_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad GAMELISTREQ (missing or too long gamename)",conn_get_socket(c));
			break;
		    }
		    
		    bngtype = bn_short_get(packet->u.client_gamelistreq.gametype);
		    gtype = bngtype_to_gtype(bngtype,conn_get_clienttag(c));
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_gamelistreply));
		    packet_set_type(rpacket,SERVER_GAMELISTREPLY);
		    
		    /* specific game requested? */
		    if (gamename[0]!='\0')
		    {
			eventlog(eventlog_level_debug,"handle_packet","[%d] GAMELISTREPLY looking for specific game tag=\"%s\" bngtype=0x%08x gtype=%d name=\"%s\"",conn_get_socket(c),conn_get_clienttag(c),bngtype,(int)gtype,gamename);
			if ((game = gamelist_find_game(gamename,gtype)))
			{
			    bn_int_set(&glgame.unknown7,SERVER_GAMELISTREPLY_GAME_UNKNOWN7);
			    bn_short_set(&glgame.gametype,gtype_to_bngtype(game_get_type(game)));
			    bn_short_set(&glgame.unknown1,SERVER_GAMELISTREPLY_GAME_UNKNOWN1);
			    bn_short_set(&glgame.unknown3,SERVER_GAMELISTREPLY_GAME_UNKNOWN3);
			    bn_short_nset(&glgame.port,game_get_port(game));
			    bn_int_nset(&glgame.game_ip,game_get_addr(game));
			    bn_int_set(&glgame.unknown4,SERVER_GAMELISTREPLY_GAME_UNKNOWN4);
			    bn_int_set(&glgame.unknown5,SERVER_GAMELISTREPLY_GAME_UNKNOWN5);
			    switch (game_get_status(game))
			    {
			    case game_status_started:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_STARTED);
				break;
			    case game_status_full:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_FULL);
				break;
			    case game_status_open:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_OPEN);
				break;
			    case game_status_done:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_DONE);
				break;
			    default:
				eventlog(eventlog_level_warn,"handle_packet","[%d] game \"%s\" has bad status %d",conn_get_socket(c),game_get_name(game),game_get_status(game));
				bn_int_set(&glgame.status,0);
			    }
			    bn_int_set(&glgame.unknown6,SERVER_GAMELISTREPLY_GAME_UNKNOWN6);
			    
			    packet_append_data(rpacket,&glgame,sizeof(glgame));
			    packet_append_string(rpacket,game_get_name(game));
			    packet_append_string(rpacket,game_get_pass(game));
			    packet_append_string(rpacket,game_get_info(game));
			    bn_int_set(&rpacket->u.server_gamelistreply.gamecount,1);
			    eventlog(eventlog_level_debug,"handle_packet","[%d] GAMELISTREPLY found it",conn_get_socket(c));
			}
			else
			{
			    bn_int_set(&rpacket->u.server_gamelistreply.gamecount,0);
			    eventlog(eventlog_level_debug,"handle_packet","[%d] GAMELISTREPLY doesn't seem to exist",conn_get_socket(c));
			}
		    }
		    else /* list all public games of this type */
		    {
			unsigned int           counter=0;
			unsigned int           tcount;
			t_list const * const * save;
			
			if (gtype==game_type_all)
			    eventlog(eventlog_level_debug,"handle_packet","GAMELISTREPLY looking for public games tag=\"%s\" bngtype=0x%08x gtype=all",conn_get_clienttag(c),bngtype);
			else
			    eventlog(eventlog_level_debug,"handle_packet","GAMELISTREPLY looking for public games tag=\"%s\" bngtype=0x%08x gtype=%d",conn_get_clienttag(c),bngtype,(int)gtype);
			
			for (game=gamelist_get_first(&save),tcount=0; game; game=gamelist_get_next(&save),tcount++)
			{
			    eventlog(eventlog_level_debug,"handle_packet","[%d] considering listing game=\"%s\", pass=\"%s\" clienttag=\"%s\" gtype=%d",conn_get_socket(c),game_get_name(game),game_get_pass(game),game_get_clienttag(game),(int)game_get_type(game));
			    
			    if (prefs_get_hide_pass_games() && strcmp(game_get_pass(game),"")!=0)
			    {
				eventlog(eventlog_level_debug,"handle_packet","[%d] not listing because game is private",conn_get_socket(c));
				continue;
			    }
			    if (strcmp(game_get_clienttag(game),conn_get_clienttag(c))!=0)
			    {
				eventlog(eventlog_level_debug,"handle_packet","[%d] not listing because game is for a different client",conn_get_socket(c));
				continue;
			    }
			    if (gtype!=game_type_all && game_get_type(game)!=gtype)
			    {
				eventlog(eventlog_level_debug,"handle_packet","[%d] not listing because game is wrong type",conn_get_socket(c));
				continue;
			    }
				
			    bn_int_set(&glgame.unknown7,SERVER_GAMELISTREPLY_GAME_UNKNOWN7);
			    bn_short_set(&glgame.gametype,gtype_to_bngtype(game_get_type(game)));
			    bn_short_set(&glgame.unknown1,SERVER_GAMELISTREPLY_GAME_UNKNOWN1);
			    bn_short_set(&glgame.unknown3,SERVER_GAMELISTREPLY_GAME_UNKNOWN3);
			    bn_short_nset(&glgame.port,game_get_port(game));
			    bn_int_nset(&glgame.game_ip,game_get_addr(game));
			    bn_int_set(&glgame.unknown4,SERVER_GAMELISTREPLY_GAME_UNKNOWN4);
			    bn_int_set(&glgame.unknown5,SERVER_GAMELISTREPLY_GAME_UNKNOWN5);
			    switch (game_get_status(game))
			    {
			    case game_status_started:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_STARTED);
				break;
			    case game_status_full:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_FULL);
				break;
			    case game_status_open:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_OPEN);
				break;
			    case game_status_done:
				bn_int_set(&glgame.status,SERVER_GAMELISTREPLY_GAME_STATUS_DONE);
				break;
			    default:
				eventlog(eventlog_level_warn,"handle_packet","[%d] game \"%s\" has bad status=%d",conn_get_socket(c),game_get_name(game),(int)game_get_status(game));
				bn_int_set(&glgame.status,0);
			    }
			    bn_int_set(&glgame.unknown6,SERVER_GAMELISTREPLY_GAME_UNKNOWN6);
			    
			    if (packet_get_size(rpacket)+
				sizeof(glgame)+
				strlen(game_get_name(game))+1+
				strlen(game_get_pass(game))+1+
				strlen(game_get_info(game))+1>MAX_PACKET_SIZE)
			    {
				eventlog(eventlog_level_debug,"handle_packet","[%d] out of room for games",conn_get_socket(c));
				break; /* no more room */
			    }
			    
			    packet_append_data(rpacket,&glgame,sizeof(glgame));
			    packet_append_string(rpacket,game_get_name(game));
			    packet_append_string(rpacket,game_get_pass(game));
			    packet_append_string(rpacket,game_get_info(game));
			    counter++;
			}
			
			bn_int_set(&rpacket->u.server_gamelistreply.gamecount,counter);
			eventlog(eventlog_level_debug,"handle_packet","[%d] GAMELISTREPLY sent %u of %u games",conn_get_socket(c),counter,tcount);
		    }
		    
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_JOIN_GAME:
	        {
		    char const * gamename;
		    char const * gamepass;
		    char const * tname;
		    t_game *     game;
		    t_game_type  gtype;
		    
		    if (!(gamename = packet_get_str_const(packet,sizeof(t_client_join_game),GAME_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CLIENT_JOIN_GAME (missing or too long gamename)",conn_get_socket(c));
			break;
		    }
		    if (!(gamepass = packet_get_str_const(packet,sizeof(t_client_join_game)+strlen(gamename)+1,GAME_PASS_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad CLIENT_JOIN_GAME packet (missing or too long gamepass)",conn_get_socket(c));
			break;
		    }
		    
		    eventlog(eventlog_level_debug,"handle_packet","[%d] trying to join game \"%s\" pass=\"%s\"",conn_get_socket(c),gamename,gamepass);
		    if (!(game = gamelist_find_game(gamename,game_type_all)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] unable to find game \"%s\" for user to join",conn_get_socket(c),gamename);
			break;
		    }
		    gtype = game_get_type(game);
		    gamename = game_get_name(game);
		    
		    if ((gtype==game_type_ladder && account_get_auth_joinladdergame(conn_get_account(c))==0) || /* default to true */
			(gtype!=game_type_ladder && account_get_auth_joinnormalgame(conn_get_account(c))==0)) /* default to true */
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] game join for \"%s\" to \"%s\" refused (no authority)",conn_get_socket(c),(tname = conn_get_username(c)),gamename);
			conn_unget_username(tname);
			/* If the user is not in a game, then map authorization
			   will fail and keep them from playing. */
			break;
		    }
		    
		    if (conn_set_game(c,gamename,gamepass,"",gtype,GAME_VERSION_UNKNOWN)<0)
		        eventlog(eventlog_level_info,"handle_packet","[%d] \"%s\" joined game \"%s\", but could not be recorded on server",conn_get_socket(c),(tname = conn_get_username(c)),gamename);
		    else
		        eventlog(eventlog_level_info,"handle_packet","[%d] \"%s\" joined game \"%s\"",conn_get_socket(c),(tname = conn_get_username(c)),gamename);
		    conn_unget_username(tname);
		    break;
		}
		break;
		
	    case CLIENT_STARTGAME1:
		if (packet_get_size(packet)<sizeof(t_client_startgame1))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME1 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_startgame1),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const *   gamename;
		    char const *   gamepass;
		    char const *   gameinfo;
		    unsigned short bngtype;
		    unsigned int   status;
		    t_game *       currgame;
		    
		    if (!(gamename = packet_get_str_const(packet,sizeof(t_client_startgame1),GAME_NAME_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME1 packet (missing or too long gamename)",conn_get_socket(c));
			break;
		    }
		    if (!(gamepass = packet_get_str_const(packet,sizeof(t_client_startgame1)+strlen(gamename)+1,GAME_PASS_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME1 packet (missing or too long gamepass)",conn_get_socket(c));
			break;
		    }
		    if (!(gameinfo = packet_get_str_const(packet,sizeof(t_client_startgame1)+strlen(gamename)+1+strlen(gamepass)+1,GAME_INFO_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME1 packet (missing or too long gameinfo)",conn_get_socket(c));
			break;
		    }
		    
		    bngtype = bn_short_get(packet->u.client_startgame1.gametype);
		    eventlog(eventlog_level_debug,"handle_packet","[%d] got startgame1 status for \"%s\" is 0x%08x",conn_get_socket(c),gamename,bn_int_get(packet->u.client_startgame1.status));
		    status = bn_int_get(packet->u.client_startgame1.status)&CLIENT_STARTGAME1_STATUSMASK;
		    
		    if ((currgame = conn_get_game(c)))
		    {
			switch (status)
			{
			case CLIENT_STARTGAME1_STATUS_STARTED:
			    game_set_status(currgame,game_status_started);
			    break;
			case CLIENT_STARTGAME1_STATUS_FULL:
			    game_set_status(currgame,game_status_full);
			    break;
			case CLIENT_STARTGAME1_STATUS_OPEN:
			    game_set_status(currgame,game_status_open);
			    break;
			case CLIENT_STARTGAME1_STATUS_DONE:
			    game_set_status(currgame,game_status_done);
			    eventlog(eventlog_level_info,"handle_packet","[%d] game \"%s\" is finished",conn_get_socket(c),gamename);
			    break;
			}
		    }
		    else
			if (status!=CLIENT_STARTGAME1_STATUS_DONE)
			{
			    t_game_type gtype;
			    
			    gtype = bngtype_to_gtype(bngtype,conn_get_clienttag(c));
			    if ((gtype==game_type_ladder && account_get_auth_createladdergame(conn_get_account(c))==0) || /* default to true */
				(gtype!=game_type_ladder && account_get_auth_createnormalgame(conn_get_account(c))==0)) /* default to true */
			    {
				char const * tname;
				
				eventlog(eventlog_level_info,"handle_packet","[%d] game start for \"%s\" refused (no authority)",conn_get_socket(c),(tname = conn_get_username(c)));
				conn_unget_username(tname);
			    }
			    else
				conn_set_game(c,gamename,gamepass,gameinfo,gtype,GAME_VERSION_1);
			    
			    if ((rpacket = packet_create(packet_class_normal)))
			    {
				packet_set_size(rpacket,sizeof(t_server_startgame1_ack));
				packet_set_type(rpacket,SERVER_STARTGAME1_ACK);
				
				if (conn_get_game(c))
				    bn_int_set(&rpacket->u.server_startgame1_ack.reply,SERVER_STARTGAME1_ACK_OK);
				else
				    bn_int_set(&rpacket->u.server_startgame1_ack.reply,SERVER_STARTGAME1_ACK_NO);
				
				queue_push_packet(conn_get_out_queue(c),rpacket);
				packet_del_ref(rpacket);
			    }
			}
			else
			    eventlog(eventlog_level_info,"handle_packet","[%d] client tried to set game status to DONE",conn_get_socket(c));
		}
		break;
		
	    case CLIENT_STARTGAME3:
		if (packet_get_size(packet)<sizeof(t_client_startgame3))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME3 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_startgame3),packet_get_size(packet));
		    break;
		}
		
		{
		    char const *   gamename;
		    char const *   gamepass;
		    char const *   gameinfo;
		    unsigned short bngtype;
		    unsigned int   status;
		    t_game *       currgame;
		    
		    if (!(gamename = packet_get_str_const(packet,sizeof(t_client_startgame3),GAME_NAME_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME3 packet (missing or too long gamename)",conn_get_socket(c));
			break;
		    }
		    if (!(gamepass = packet_get_str_const(packet,sizeof(t_client_startgame3)+strlen(gamename)+1,GAME_PASS_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME3 packet (missing or too long gamepass)",conn_get_socket(c));
			break;
		    }
		    if (!(gameinfo = packet_get_str_const(packet,sizeof(t_client_startgame3)+strlen(gamename)+1+strlen(gamepass)+1,GAME_INFO_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME3 packet (missing or too long gameinfo)",conn_get_socket(c));
			break;
		    }
		    
		    bngtype = bn_short_get(packet->u.client_startgame3.gametype);
		    eventlog(eventlog_level_debug,"handle_packet","[%d] got startgame3 status for \"%s\" is 0x%08x",conn_get_socket(c),gamename,bn_int_get(packet->u.client_startgame3.status));
		    status = bn_int_get(packet->u.client_startgame3.status)&CLIENT_STARTGAME3_STATUSMASK;
		    
		    if ((currgame = conn_get_game(c)))
		    {
			switch (status)
			{
			case CLIENT_STARTGAME3_STATUS_STARTED:
			    game_set_status(currgame,game_status_started);
			    break;
			case CLIENT_STARTGAME3_STATUS_FULL:
			    game_set_status(currgame,game_status_full);
			    break;
			case CLIENT_STARTGAME3_STATUS_OPEN1:
			case CLIENT_STARTGAME3_STATUS_OPEN:
			    game_set_status(currgame,game_status_open);
			    break;
			case CLIENT_STARTGAME3_STATUS_DONE:
			    game_set_status(currgame,game_status_done);
			    eventlog(eventlog_level_info,"handle_packet","[%d] game \"%s\" is finished",conn_get_socket(c),gamename);
			    break;
			}
		    }
		    else
			if (status!=CLIENT_STARTGAME3_STATUS_DONE)
			{
			    t_game_type gtype;
			    
			    gtype = bngtype_to_gtype(bngtype,conn_get_clienttag(c));
			    if ((gtype==game_type_ladder && account_get_auth_createladdergame(conn_get_account(c))==0) ||
				(gtype!=game_type_ladder && account_get_auth_createnormalgame(conn_get_account(c))==0))
			    {
				char const * tname;
				
				eventlog(eventlog_level_info,"handle_packet","[%d] game start for \"%s\" refused (no authority)",conn_get_socket(c),(tname = conn_get_username(c)));
				conn_unget_username(tname);
			    }
			    else
				conn_set_game(c,gamename,gamepass,gameinfo,gtype,GAME_VERSION_3);
			    
			    if ((rpacket = packet_create(packet_class_normal)))
			    {
				packet_set_size(rpacket,sizeof(t_server_startgame3_ack));
				packet_set_type(rpacket,SERVER_STARTGAME3_ACK);
			        
				if (conn_get_game(c))
				    bn_int_set(&rpacket->u.server_startgame3_ack.reply,SERVER_STARTGAME3_ACK_OK);
				else
				    bn_int_set(&rpacket->u.server_startgame3_ack.reply,SERVER_STARTGAME3_ACK_NO);
			        
				queue_push_packet(conn_get_out_queue(c),rpacket);
				packet_del_ref(rpacket);
			    }
			}
			else
			    eventlog(eventlog_level_info,"handle_packet","[%d] client tried to set game status to DONE",conn_get_socket(c));
		}
		break;
		
	    case CLIENT_STARTGAME4:
		if (packet_get_size(packet)<sizeof(t_client_startgame4))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME4 packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_startgame4),packet_get_size(packet));
		    break;
		}
		
		{
		    char const *   gamename;
		    char const *   gamepass;
		    char const *   gameinfo;
		    unsigned short bngtype;
		    unsigned int   status;
		    t_game *       currgame;
		    
		    if (!(gamename = packet_get_str_const(packet,sizeof(t_client_startgame4),GAME_NAME_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME4 packet (missing or too long gamename)",conn_get_socket(c));
			break;
		    }
		    if (!(gamepass = packet_get_str_const(packet,sizeof(t_client_startgame4)+strlen(gamename)+1,GAME_PASS_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME4 packet (missing or too long gamepass)",conn_get_socket(c));
			break;
		    }
		    if (!(gameinfo = packet_get_str_const(packet,sizeof(t_client_startgame4)+strlen(gamename)+1+strlen(gamepass)+1,GAME_INFO_LEN)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] got bad STARTGAME4 packet (missing or too long gameinfo)",conn_get_socket(c));
			break;
		    }
		    
		    bngtype = bn_short_get(packet->u.client_startgame4.gametype);
		    eventlog(eventlog_level_debug,"handle_packet","[%d] got startgame4 status for \"%s\" is 0x%08x",conn_get_socket(c),gamename,bn_int_get(packet->u.client_startgame4.status));
		    status = bn_int_get(packet->u.client_startgame4.status)&CLIENT_STARTGAME4_STATUSMASK;
		    
		    if ((currgame = conn_get_game(c)))
		    {
			switch (status)
			{
			case CLIENT_STARTGAME4_STATUS_STARTED:
			    game_set_status(currgame,game_status_started);
			    break;
			case CLIENT_STARTGAME4_STATUS_FULL1:
			case CLIENT_STARTGAME4_STATUS_FULL2:
			    game_set_status(currgame,game_status_full);
			    break;
			case CLIENT_STARTGAME4_STATUS_INIT:
			case CLIENT_STARTGAME4_STATUS_OPEN1:
			case CLIENT_STARTGAME4_STATUS_OPEN2:
			case CLIENT_STARTGAME4_STATUS_OPEN3:
			    game_set_status(currgame,game_status_open);
			    break;
			case CLIENT_STARTGAME4_STATUS_DONE1:
			case CLIENT_STARTGAME4_STATUS_DONE2:
			    game_set_status(currgame,game_status_done);
			    eventlog(eventlog_level_info,"handle_packet","[%d] game \"%s\" is finished",conn_get_socket(c),gamename);
			    break;
			}
		    }
		    else
			if (status!=CLIENT_STARTGAME4_STATUS_DONE1 &&
			    status!=CLIENT_STARTGAME4_STATUS_DONE2)
			{
			    			    t_game_type gtype;
			    
			    gtype = bngtype_to_gtype(bngtype,conn_get_clienttag(c));
			    if ((gtype==game_type_ladder && account_get_auth_createladdergame(conn_get_account(c))==0) ||
				(gtype!=game_type_ladder && account_get_auth_createnormalgame(conn_get_account(c))==0))
			    {
				char const * tname;
				
				eventlog(eventlog_level_info,"handle_packet","[%d] game start for \"%s\" refused (no authority)",conn_get_socket(c),(tname = conn_get_username(c)));
				conn_unget_username(tname);
			    }
			    else
				conn_set_game(c,gamename,gamepass,gameinfo,gtype,GAME_VERSION_4);
			    
			    if ((rpacket = packet_create(packet_class_normal)))
			    {
				packet_set_size(rpacket,sizeof(t_server_startgame4_ack));
				packet_set_type(rpacket,SERVER_STARTGAME4_ACK);
				
				if (conn_get_game(c))
				    bn_int_set(&rpacket->u.server_startgame4_ack.reply,SERVER_STARTGAME4_ACK_OK);
				else
				    bn_int_set(&rpacket->u.server_startgame4_ack.reply,SERVER_STARTGAME4_ACK_NO);
				queue_push_packet(conn_get_out_queue(c),rpacket);
				packet_del_ref(rpacket);
			    }
			}
			else
			    eventlog(eventlog_level_info,"handle_packet","[%d] client tried to set game status to DONE",conn_get_socket(c));
		}
		break;
		
	    case CLIENT_CLOSEGAME:
		eventlog(eventlog_level_info,"handle_packet","[%d] client closing game",conn_get_socket(c));
		conn_set_game(c,NULL,NULL,NULL,game_type_none,0);
		break;
		
	    case CLIENT_GAME_REPORT:
		if (packet_get_size(packet)<sizeof(t_client_game_report))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad GAME_REPORT packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_game_report),packet_get_size(packet));
		    break;
		}
		
	        {
		    t_account *                         my_account;
		    t_account *                         other_account;
		    t_game_result                       my_result=game_result_none;
		    t_game *                            game;
		    unsigned int                        player_count;
		    unsigned int                        i;
		    t_client_game_report_result const * result_data;
		    unsigned int                        result_off;
		    t_game_result                       result;
		    char const *                        player;
		    unsigned int                        player_off;
		    char const *                        tname;
		    
		    player_count = bn_int_get(packet->u.client_gamerep.count);
		    
		    if (!(game = conn_get_game(c)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got GAME_REPORT when not in a game for user \"%s\"",conn_get_socket(c),(tname = conn_get_username(c)));
			conn_unget_username(tname);
			break;
		    }
		    
		    eventlog(eventlog_level_info,"handle_packet","[%d] CLIENT_GAME_REPORT: %s (%u players)",conn_get_socket(c),(tname = conn_get_username(c)),player_count);
		    my_account = conn_get_account(c);
		    
		    for (i=0,result_off=sizeof(t_client_game_report),player_off=sizeof(t_client_game_report)+player_count*sizeof(t_client_game_report_result);
			 i<player_count;
			 i++,result_off+=sizeof(t_client_game_report_result),player_off+=strlen(player)+1)
		    {
			if (!(result_data = packet_get_data_const(packet,result_off,sizeof(t_client_game_report_result))))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] got corrupt GAME_REPORT packet (missing results %u-%u)",conn_get_socket(c),i+1,player_count);
			    break;
			}
			if (!(player = packet_get_str_const(packet,player_off,USER_NAME_LEN)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] got corrupt GAME_REPORT packet (missing players %u-%u)",conn_get_socket(c),i+1,player_count);
			    break;
			}
			
			result = bngresult_to_gresult(bn_int_get(result_data->result));
			eventlog(eventlog_level_debug,"handle_packet","[%d] got player %d (\"%s\") result 0x%08x",conn_get_socket(c),i,player,result);
			
			if (player[0]=='\0') /* empty slots have empty player name */
			    continue;
			
		        if (!(other_account = accountlist_find_account(player)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] got GAME_REPORT with unknown player \"%s\"",conn_get_socket(c),player);
			    break;
			}
			
			if (my_account==other_account)
			    my_result = result;
			else
			    if (game_check_result(game,other_account,result)<0)
				break;
		    }
		    
		    conn_unget_username(tname);
		    
		    if (i==player_count) /* if everything checked out... */
		    {
			char const * head;
			char const * body;
			
			if (!(head = packet_get_str_const(packet,player_off,2048)))
			    eventlog(eventlog_level_debug,"handle_packet","[%d] got GAME_REPORT with missing or too long report head",conn_get_socket(c));
			else
			{
			    player_off += strlen(head)+1;
			    if (!(body = packet_get_str_const(packet,player_off,8192)))
				eventlog(eventlog_level_debug,"handle_packet","[%d] got GAME_REPORT with missing or too long report body",conn_get_socket(c));
			    else
				game_set_result(game,my_account,my_result,head,body); /* finally we can report the info */
			}
		    }
		    
		    eventlog(eventlog_level_debug,"handle_packet","[%d] finished parsing result... now leaving game",conn_get_socket(c));
		    conn_set_game(c,NULL,NULL,NULL,game_type_none,0);
		}
		break;
		
	    case CLIENT_LEAVECHANNEL:
		/* If this user in a channel, notify everyone that the user has left */
		if (conn_get_channel(c))
		    conn_set_channel(c,NULL);
		break;
		
	    case CLIENT_LADDERREQ:
		if (packet_get_size(packet)<sizeof(t_client_ladderreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad LADDERREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_ladderreq),packet_get_size(packet));
		    break;
		}
		
	        {
		    t_ladder_entry entry;
		    int            i;
		    unsigned int   type;
		    unsigned int   start;
		    unsigned int   count;
		    t_account *    account;
		    char const *   clienttag;
		    char const *   tname;
		    char const *   timestr;
		    t_bnettime     bt;
		    
		    clienttag = conn_get_clienttag(c);
		    
		    type = bn_int_get(packet->u.client_ladderreq.type);
		    start = bn_int_get(packet->u.client_ladderreq.startplace);
		    count = bn_int_get(packet->u.client_ladderreq.count);
		    /* eventlog(eventlog_level_debug,"handle_packet","got LADDERREQ type=%u start=%u count=%u",type,start,count); */
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_ladderreply));
		    packet_set_type(rpacket,SERVER_LADDERREPLY);
		    
		    bn_int_tag_set(&rpacket->u.server_ladderreply.clienttag,clienttag);
		    bn_int_set(&rpacket->u.server_ladderreply.unknown1,SERVER_LADDERREPLY_UNKNOWN1);
		    bn_int_set(&rpacket->u.server_ladderreply.type,type);
		    bn_int_set(&rpacket->u.server_ladderreply.startplace,start);
		    bn_int_set(&rpacket->u.server_ladderreply.count,count);
		    
		    for (i=start; i<start+count; i++)
		    {
			switch (type)
			{
			case CLIENT_LADDERREQ_TYPE_HIGHESTRATED:
			    if (!(account = ladder_get_account_by_rank(i+1,ladder_sort_highestrated,ladder_time_active,clienttag)))
				account = ladder_get_account_by_rank(i+1,ladder_sort_highestrated,ladder_time_current,clienttag);
			    break;
			case CLIENT_LADDERREQ_TYPE_MOSTWINS:
			    if (!(account = ladder_get_account_by_rank(i+1,ladder_sort_mostwins,ladder_time_active,clienttag)))
				account = ladder_get_account_by_rank(i+1,ladder_sort_mostwins,ladder_time_current,clienttag);
			    break;
			case CLIENT_LADDERREQ_TYPE_MOSTGAMES:
			    if (!(account = ladder_get_account_by_rank(i+1,ladder_sort_mostgames,ladder_time_active,clienttag)))
				account = ladder_get_account_by_rank(i+1,ladder_sort_mostgames,ladder_time_current,clienttag);
			    break;
			default:
			    account = NULL;
			    eventlog(eventlog_level_error,"handle_packet","[%d] got unknown value for ladderreq.type=%u",conn_get_socket(c),type);
			}
			
			if (account)
			{
			    bn_int_set(&entry.active.wins,account_get_ladder_active_wins(account,clienttag));
			    bn_int_set(&entry.active.loss,account_get_ladder_active_losses(account,clienttag));
			    bn_int_set(&entry.active.disconnect,account_get_ladder_active_disconnects(account,clienttag));
			    bn_int_set(&entry.active.rating,account_get_ladder_active_rating(account,clienttag));
			    bn_int_set(&entry.active.unknown,10); /* FIXME: rank,draws,?! */
			    if (!(timestr = account_get_ladder_active_last_time(account,clienttag)))
				timestr = BNETD_LADDER_DEFAULT_TIME;
			    bnettime_set_str(&bt,timestr);
			    bnettime_to_bn_long(bt,&entry.lastgame_active);
			    
			    bn_int_set(&entry.current.wins,account_get_ladder_wins(account,clienttag));
			    bn_int_set(&entry.current.loss,account_get_ladder_losses(account,clienttag));
			    bn_int_set(&entry.current.disconnect,account_get_ladder_disconnects(account,clienttag));
			    bn_int_set(&entry.current.rating,account_get_ladder_rating(account,clienttag));
			    bn_int_set(&entry.current.unknown,5); /* FIXME: rank,draws,?! */
			    if (!(timestr = account_get_ladder_last_time(account,clienttag)))
				timestr = BNETD_LADDER_DEFAULT_TIME;
			    bnettime_set_str(&bt,timestr);
			    bnettime_to_bn_long(bt,&entry.lastgame_current);
			}
			else
			{
			    bn_int_set(&entry.active.wins,0);
			    bn_int_set(&entry.active.loss,0);
			    bn_int_set(&entry.active.disconnect,0);
			    bn_int_set(&entry.active.rating,0);
			    bn_int_set(&entry.active.unknown,0);
			    bn_long_set_a_b(&entry.lastgame_active,0,0);
			    
			    bn_int_set(&entry.current.wins,0);
			    bn_int_set(&entry.current.loss,0);
			    bn_int_set(&entry.current.disconnect,0);
			    bn_int_set(&entry.current.rating,0);
			    bn_int_set(&entry.current.unknown,0);
			    bn_long_set_a_b(&entry.lastgame_current,0,0);
			}
			
			bn_int_set(&entry.ttest[0],0); /* FIXME: what is ttest? */
			bn_int_set(&entry.ttest[1],0);
			bn_int_set(&entry.ttest[2],0);
			bn_int_set(&entry.ttest[3],0);
			bn_int_set(&entry.ttest[4],0);
			bn_int_set(&entry.ttest[5],0);
			
			packet_append_data(rpacket,&entry,sizeof(entry));
			
			if (account)
			{
			    packet_append_string(rpacket,(tname = account_get_name(account)));
			    account_unget_name(tname);
			}
			else
			    packet_append_string(rpacket," "); /* use a space so the client won't show the user's own account when double-clicked on */
		    }
		    
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_LADDERSEARCHREQ:
		if (packet_get_size(packet)<sizeof(t_client_laddersearchreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad LADDERSEARCHREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_laddersearchreq),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const * playername;
		    t_account *  account;
		    unsigned int rank; /* starts at zero */
		    
		    if (!(playername = packet_get_str_const(packet,sizeof(t_client_laddersearchreq),USER_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad LADDERSEARCHREQ packet (missing or too long playername)",conn_get_socket(c));
			break;
		    }
		    
		    if (!(account = accountlist_find_account(playername)))
			rank = SERVER_LADDERSEARCHREPLY_RANK_NONE;
		    else
		    {
			switch (bn_int_get(packet->u.client_laddersearchreq.type))
			{
			case CLIENT_LADDERSEARCHREQ_TYPE_HIGHESTRATED:
			    if ((rank = ladder_get_rank_by_account(account,ladder_sort_highestrated,ladder_time_active,conn_get_clienttag(c)))==0)
			    {
				rank = ladder_get_rank_by_account(account,ladder_sort_highestrated,ladder_time_current,conn_get_clienttag(c));
				if (ladder_get_account_by_rank(rank,ladder_sort_highestrated,ladder_time_active,conn_get_clienttag(c)))
				    rank = 0;
			    }
			    break;
			case CLIENT_LADDERSEARCHREQ_TYPE_MOSTWINS:
			    if ((rank = ladder_get_rank_by_account(account,ladder_sort_mostwins,ladder_time_active,conn_get_clienttag(c)))==0)
			    {
			        rank = ladder_get_rank_by_account(account,ladder_sort_mostwins,ladder_time_current,conn_get_clienttag(c));
				if (ladder_get_account_by_rank(rank,ladder_sort_mostwins,ladder_time_active,conn_get_clienttag(c)))
				    rank = 0;
			    }
			    break;
			case CLIENT_LADDERSEARCHREQ_TYPE_MOSTGAMES:
			    if ((rank = ladder_get_rank_by_account(account,ladder_sort_mostgames,ladder_time_active,conn_get_clienttag(c)))==0)
			    {
			        rank = ladder_get_rank_by_account(account,ladder_sort_mostgames,ladder_time_current,conn_get_clienttag(c));
				if (ladder_get_account_by_rank(rank,ladder_sort_mostgames,ladder_time_active,conn_get_clienttag(c)))
				    rank = 0;
			    }
			    break;
			default:
			    rank = 0;
			    eventlog(eventlog_level_error,"handle_packet","[%d] got unknown ladder search type %u",conn_get_socket(c),bn_int_get(packet->u.client_laddersearchreq.type));
			}
			
			if (rank==0)
			    rank = SERVER_LADDERSEARCHREPLY_RANK_NONE;
			else
			    rank--;
		    }
		    
		    if (!(rpacket = packet_create(packet_class_normal)))
			break;
		    packet_set_size(rpacket,sizeof(t_server_laddersearchreply));
		    packet_set_type(rpacket,SERVER_LADDERSEARCHREPLY);
		    bn_int_set(&rpacket->u.server_laddersearchreply.rank,rank);
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case CLIENT_MAPAUTHREQ:
		if (packet_get_size(packet)<sizeof(t_client_mapauthreq))
		{
		    eventlog(eventlog_level_error,"handle_packet","[%d] got bad MAPAUTHREQ packet (expected %u bytes, got %u)",conn_get_socket(c),sizeof(t_client_mapauthreq),packet_get_size(packet));
		    break;
		}
		
	        {
		    char const * mapname;
		    
		    if (!(mapname = packet_get_str_const(packet,sizeof(t_client_mapauthreq),MAP_NAME_LEN)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad MAPAUTHREQ packet (missing or too long mapname)",conn_get_socket(c));
			break;
		    }
		    
		    eventlog(eventlog_level_info,"handle_packet","[%d] map auth requested for map \"%s\" gametype \"%s\"",conn_get_socket(c),mapname,game_type_get_str(game_get_type(conn_get_game(c))));
		    
		    if ((rpacket = packet_create(packet_class_normal)))
		    {
			t_game *     game;
			unsigned int val;
			
			if (!(game = conn_get_game(c)))
			{
			    val = SERVER_MAPAUTHREPLY_NO;
			    eventlog(eventlog_level_debug,"handle_packet","[%d] map ladder authorization denied (not in a game)",conn_get_socket(c));
			}
			else
			{
			    game_set_status(game,game_status_started);
			    
			    if (game_get_type(game)==game_type_ladder)
			    {
				val = SERVER_MAPAUTHREPLY_LADDER_OK;
				eventlog(eventlog_level_debug,"handle_packet","[%d] giving map ladder authorization (in a ladder game)",conn_get_socket(c));
			    }
			    else if (ladder_check_map(mapname,conn_get_clienttag(c)))
			    {
				val = SERVER_MAPAUTHREPLY_LADDER_OK;
				eventlog(eventlog_level_debug,"handle_packet","[%d] giving map ladder authorization (is a ladder map)",conn_get_socket(c));
			    }
			    else
			    {
				val = SERVER_MAPAUTHREPLY_OK;
				eventlog(eventlog_level_debug,"handle_packet","[%d] giving map normal authorization",conn_get_socket(c));
			    }
			}
			
			packet_set_size(rpacket,sizeof(t_server_mapauthreply));
			packet_set_type(rpacket,SERVER_MAPAUTHREPLY);
			bn_int_set(&rpacket->u.server_mapauthreply.response,val);
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		}
		break;
		
	    default:
		eventlog(eventlog_level_error,"handle_packet","[%d] unknown (logged in) packet type 0x%04hx, len %hu",conn_get_socket(c),packet_get_type(packet),packet_get_size(packet));
	    }
	    break;
	    
	default:
	    eventlog(eventlog_level_error,"handle_packet","[%d] invalid login state %d",conn_get_socket(c),conn_get_state(c));
	}
	break;
	
    case conn_class_file:
	switch (conn_get_state(c))
	{
	case conn_state_connected:
	    switch (packet_get_type(packet))
	    {
	    case CLIENT_FILE_REQ:
	        {
		    char const * rawname;
		    
		    if (!(rawname = packet_get_str_const(packet,sizeof(t_client_file_req),2048)))
		    {
			eventlog(eventlog_level_error,"handle_packet","[%d] got bad FILE_REQ (missing or too long filename)",conn_get_socket(c));

			return -1;
		    }
		    
		    if (strcmp(rawname,prefs_get_mpqfile())==0)
		    {
			if (file_send(c,rawname,0,0,0)<0)
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] client requested \"%s\", but none is avaliable",conn_get_socket(c),rawname);
			    return -1; /* not much else we can do... */
			}
		    }
		    else
			file_send(c,rawname,
				  bn_int_get(packet->u.client_file_req.adid),
				  bn_int_get(packet->u.client_file_req.extensiontag),
				  1);
		}
		break;
		
	    default:
		eventlog(eventlog_level_error,"handle_packet","[%d] unknown file packet type 0x%04hx, len %hu",conn_get_socket(c),packet_get_type(packet),packet_get_size(packet));
		
		break;
	    }
	    break;
	    
	default:
	    eventlog(eventlog_level_error,"handle_packet","[%d] unknown file connection state %d",conn_get_socket(c),(int)conn_get_state(c));
	}
	
	break;
	
    case conn_class_bot:
        {
	    char const * const linestr=packet_get_str_const(packet,0,MAX_MESSAGE_LEN);
	    
	    if (packet_get_size(packet)<2) /* empty line */
		break;
	    if (!linestr)
	    {
		eventlog(eventlog_level_warn,"handle_packet","[%d] line too long",conn_get_socket(c));
		break;
	    }
	    
	    switch (conn_get_state(c))
	    {
	    case conn_state_connected:
		conn_add_flags(c,MF_PLUG);
		conn_set_clienttag(c,CLIENTTAG_BNCHATBOT);
		
		{
		    char const * temp=linestr;
		    
		    if (temp[0]=='\004') /* FIXME: no echo, ignore for now (we always do no echo) */
			temp = &temp[1];
		    
		    if (temp[0]=='\0') /* empty line */
		    {
			conn_set_state(c,conn_state_bot_username); /* don't look for ^D or reset tag and flags */
			break;
		    }
		    
		    conn_set_state(c,conn_state_bot_password);
		    
		    if (conn_set_botuser(c,temp)<0)
			eventlog(eventlog_level_error,"handle_packet","[%d] could not set username to \"%s\"",conn_get_socket(c),temp);
		    
		    {
			char const * const msg="\r\nPassword: ";
			
			if (!(rpacket = packet_create(packet_class_raw)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
			    break;
			}
#if 0 /* don't echo */
			packet_append_data(rpacket,conn_get_botuser(c),strlen(conn_get_botuser(c)));
#endif
			packet_append_data(rpacket,msg,strlen(msg));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		}
		break;
		
	    case conn_state_bot_username:
		conn_set_state(c,conn_state_bot_password);
		
		if (conn_set_botuser(c,linestr)<0)
		    eventlog(eventlog_level_error,"handle_packet","[%d] could not set username to \"%s\"",conn_get_socket(c),linestr);
		
		{
		    char const * const temp="\r\nPassword: ";
	    	
		    if (!(rpacket = packet_create(packet_class_raw)))
		    {
		        eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
			break;
		    }
#if 0 /* don't echo */
		    packet_append_data(rpacket,linestr,strlen(linestr));
#endif
		    packet_append_data(rpacket,temp,strlen(temp));
		    queue_push_packet(conn_get_out_queue(c),rpacket);
		    packet_del_ref(rpacket);
		}
		break;
		
	    case conn_state_bot_password:
	        {
		    char const * const tempa="\r\nLogin failed.\r\n\r\nUsername: ";
		    char const * const tempb="\r\nAccount has no bot access.\r\n\r\nUsername: ";
		    char const * const botuser=conn_get_botuser(c);
		    t_account *        account;
		    char const *       oldstrhash1;
		    t_hash             trypasshash1;
		    t_hash             oldpasshash1;
		    char const *       tname;
		    
		    if (!botuser) /* error earlier in login */
		    {
			/* no log message... */
			conn_set_state(c,conn_state_bot_username);
			
			if (!(rpacket = packet_create(packet_class_raw)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
			    break;
			}
			
			packet_append_data(rpacket,tempa,strlen(tempa));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
			break;
		    }
		    if (connlist_find_connection(botuser))
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] bot login for \"%s\" refused (already logged in)",conn_get_socket(c),botuser);
			conn_set_state(c,conn_state_bot_username);
			
			if (!(rpacket = packet_create(packet_class_raw)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
			    break;
			}
			
			packet_append_data(rpacket,tempa,strlen(tempa));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
			break;
		    }
		    if (!(account = accountlist_find_account(botuser)))
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] bot login for \"%s\" refused (bad account)",conn_get_socket(c),botuser);
			conn_set_state(c,conn_state_bot_username);
			
			if (!(rpacket = packet_create(packet_class_raw)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
			    break;
			}
			
			packet_append_data(rpacket,tempa,strlen(tempa));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
			break;
		    }
		    if ((oldstrhash1 = account_get_pass(account)))
		    {
			if (hash_set_str(&oldpasshash1,oldstrhash1)<0)
			{
			    account_unget_pass(oldstrhash1);
			    eventlog(eventlog_level_info,"handle_packet","[%d] bot login for \"%s\" refused (corrupted passhash1?)",conn_get_socket(c),(tname = account_get_name(account)));
			    account_unget_name(tname);
			    conn_set_state(c,conn_state_bot_username);
			    
			    if (!(rpacket = packet_create(packet_class_raw)))
			    {
				eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
				break;
			    }
			    
			    packet_append_data(rpacket,tempa,strlen(tempa));
			    queue_push_packet(conn_get_out_queue(c),rpacket);
			    packet_del_ref(rpacket);
			    break;
			}
			account_unget_pass(oldstrhash1);
			if (bnet_hash(&trypasshash1,strlen(linestr),linestr)<0)
			{
			    eventlog(eventlog_level_info,"handle_packet","[%d] bot login for \"%s\" refused (unable to hash password)",conn_get_socket(c),(tname = account_get_name(account)));
			    account_unget_name(tname);
			    conn_set_state(c,conn_state_bot_username);
			    
			    if (!(rpacket = packet_create(packet_class_raw)))
			    {
				eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
				break;
			    }
			    
			    packet_append_data(rpacket,tempa,strlen(tempa));
			    queue_push_packet(conn_get_out_queue(c),rpacket);
			    packet_del_ref(rpacket);
			    break;
			}
			if (memcmp(trypasshash1,oldpasshash1,BNETHASH_LEN)!=0)
			{
			    eventlog(eventlog_level_info,"handle_packet","[%d] bot login for \"%s\" refused (wrong password)",conn_get_socket(c),(tname = account_get_name(account)));
			    account_unget_name(tname);
			    conn_set_state(c,conn_state_bot_username);
			    
			    if (!(rpacket = packet_create(packet_class_raw)))
			    {
				eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
				break;
			    }
			    
			    packet_append_data(rpacket,tempa,strlen(tempa));
			    queue_push_packet(conn_get_out_queue(c),rpacket);
			    packet_del_ref(rpacket);
			    break;
			}



		    if (account_get_auth_botlogin(account)!=1) /* default to false */
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] bot login for \"%s\" refused (no bot access)",conn_get_socket(c),(tname = account_get_name(account)));
			account_unget_name(tname);
			conn_set_state(c,conn_state_bot_username);
			
			if (!(rpacket = packet_create(packet_class_raw)))
			{
			    eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
			    break;
			}
			
			packet_append_data(rpacket,tempb,strlen(tempb));
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
			break;
		    }


			eventlog(eventlog_level_info,"handle_packet","[%d] \"%s\" bot logged in (correct password)",conn_get_socket(c),(tname = account_get_name(account)));
			account_unget_name(tname);
		    }
		    else
		    {
			eventlog(eventlog_level_info,"handle_packet","[%d] \"%s\" bot logged in (no password)",conn_get_socket(c),(tname = account_get_name(account)));
			account_unget_name(tname);
		    }
		    
		    if (!(rpacket = packet_create(packet_class_raw))) /* if we got this far, let them log in even if this fails */
			eventlog(eventlog_level_error,"handle_packet","[%d] could not create rpacket",conn_get_socket(c));
		    else
		    {
			packet_append_data(rpacket,"\r\n",2);
			message_bot_format(rpacket,0,NULL,(tname = account_get_name(account)));
			account_unget_name(tname);
			queue_push_packet(conn_get_out_queue(c),rpacket);
			packet_del_ref(rpacket);
		    }
		    
		    conn_set_account(c,account);
		    
		    if (conn_set_channel(c,CHANNEL_NAME_CHAT)<0)
			conn_set_channel(c,CHANNEL_NAME_BANNED); /* should not fail */
		}
		break;
		
	    case conn_state_loggedin:
	 	if (linestr[0]=='/')
	 	{
		    handle_command(c,linestr);
		    break;
	 	}
		
		if (conn_get_channel(c))
	 	    channel_message_send(conn_get_channel(c),MT_MESSAGE,c,linestr);
		break;
		
	    default:
		eventlog(eventlog_level_error,"handle_packet","[%d] unknown bot connection state %d",conn_get_socket(c),(int)conn_get_state(c));
	    }
	}
        break;
	
    default:
	eventlog(eventlog_level_error,"handle_packet","[%d] unknown connection type %d",conn_get_socket(c),(int)conn_get_class(c));
    }
    
    return 0;
}
