//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/shared_ptr.hpp>

// Musicbrainz
#include <musicbrainz/musicbrainz.h>
#include "mbxml.hh"

#include "main.hh"
#include "paths.hh"
#include "stock.hh"
#include "debug.hh"
#include "util.hh"
#include "util_file.hh"

#include "uri++.hh"
#include "ui_toolbox.hh"
#include "network.hh"

#include "x_library.hh"
#include "x_play.hh"
#include "x_vfs.hh"
#include "x_mcsbind.hh"

#ifdef HAVE_HAL
#  include "hal.hh"
#endif //HAVE_HAL

#include "ui-part-playlist.hh"

namespace
{

#define PLAYLIST_ACTION_EXPORT          "playlist-action-export"
#define PLAYLIST_ACTION_REMOVE          "playlist-action-remove"
#define PLAYLIST_ACTION_REMOVE_ALL      "playlist-action-remove-all"
#define PLAYLIST_ACTION_ADD_FILES       "playlist-action-add-files"
#define PLAYLIST_ACTION_ADD_FILES_FC    "playlist-action-add-files-fc"
#define PLAYLIST_ACTION_OPEN_FILES      "playlist-action-open-files"

#define PLAYLIST_ACTION_SHOW_FSTREE "playlist-action-show-fstree"

  const char *ui_menu_playlist =
  "<ui>"
  ""
  "<menubar name='MenuBarMain'>"
  "   <menu action='MenuUiPartPlaylist'>"
  "     <menuitem action='" PLAYLIST_ACTION_SHOW_FSTREE "'/>"
  "         <separator name='playlist-sep-6'/>"
  "     <menuitem action='" PLAYLIST_ACTION_ADD_FILES "'/>"
  "     <menuitem action='" PLAYLIST_ACTION_OPEN_FILES "'/>"
  "         <separator name='playlist-sep-7'/>"
  "     <menuitem action='" PLAYLIST_ACTION_ADD_FILES_FC "'/>"
  "         <separator name='playlist-sep-8'/>"
  "     <menuitem action='" PLAYLIST_ACTION_REMOVE "'/>"
  "     <menuitem action='" PLAYLIST_ACTION_REMOVE_ALL "'/>"
  "         <separator name='playlist-sep-9'/>"
  "     <menuitem action='" PLAYLIST_ACTION_EXPORT "'/>"
  "   </menu>"
  "</menubar>"
  ""
  "</ui>";

  const char *ui_popup_playlist =
  "<ui>"
  ""
  "<menubar name='popup-tracklist'>"
  "   <menu action='dummy' name='menu-tracklist'>"
  "     <menuitem action='" PLAYLIST_ACTION_ADD_FILES_FC "'/>"
  "         <separator name='playlist-sep-3'/>"
  "     <menuitem action='" PLAYLIST_ACTION_ADD_FILES "'/>"
  "     <menuitem action='" PLAYLIST_ACTION_OPEN_FILES "'/>"
  "         <separator name='playlist-sep-4'/>"
  "     <menuitem action='" PLAYLIST_ACTION_REMOVE "'/>"
  "     <menuitem action='" PLAYLIST_ACTION_REMOVE_ALL "'/>"
  "         <separator name='playlist-sep-5'/>"
  "     <menuitem action='" PLAYLIST_ACTION_EXPORT "'/>"
  "   </menu>"
  "</menubar>"
  ""
  "</ui>";



  Gtk::Widget*
  get_popup (Glib::RefPtr<Gtk::UIManager> ui_manager,
             Glib::ustring                path) 
  {
      Gtk::Widget *menu_item;
      menu_item = ui_manager->get_widget (path);
      if ( menu_item )
          return dynamic_cast<Gtk::MenuItem*>(menu_item)->get_submenu ();
      else
          return 0;
  }
}

namespace Bmp
{
  namespace UiPart
  {
    void
    Playlist::append_track (Glib::ustring const& uri)
    {
      using namespace Gtk;

      try {
          Library::Track t;          
          library->get (uri, t);
          TreeModel::iterator m_iter = playlist_store->append ();
          if (t.artist) (*m_iter)[track.artist] = (t.artist.get());
          if (t.album) (*m_iter)[track.album] = (t.album.get());
          if (t.title) (*m_iter)[track.title] = (t.title.get());
          if (t.tracknumber) (*m_iter)[track.tracknumber] = (t.tracknumber.get());
          if (t.genre) (*m_iter)[track.genre] = (t.genre.get());
          if (t.date) (*m_iter)[track.date] = t.date.get();
          if (t.duration) (*m_iter)[track.duration] = t.duration.get();
          if (t.asin) (*m_iter)[track.asin] = t.asin.get();
          (*m_iter)[track.location] = t.location.get();
          (*m_iter)[track.guid] = m_guid++; 
        }
      catch (Bmp::Library::Exception& cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Library Error for URI %s: %s", uri.c_str(), cxe.what());
        }
#ifdef HAVE_HAL
      catch (Bmp::Library::HAL::Exception& cxe) 
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "HAL Error: %s", cxe.what());
        }
#endif //HAVE_HAL
      catch (...) {}
    }

    void
    Playlist::tree_selection_changed ()
    {
      bool selected (tree->get_selection()->count_selected_rows());
      if (selected)
        {
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->set_sensitive (true);
          m_actions->get_action (PLAYLIST_ACTION_OPEN_FILES)->set_sensitive (true);
        }
      else
        {
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->set_sensitive (false);
          m_actions->get_action (PLAYLIST_ACTION_OPEN_FILES)->set_sensitive (false);
        }
    }

    void  
    Playlist::check_nextprev_caps ()
    {
      using namespace Gtk;
      using namespace Glib;

      bool next = false;
      bool prev = false;

      if (m_playing_row && m_playing_row.get().is_valid())
      {
        TreeModel::Children const& children (playlist_store->children());
        TreeModel::Path path (m_playing_row.get().get_path());      

        unsigned int position = path.get_indices().data()[0];
        unsigned int size = children.size(); 

        if (position > 0) prev = true;
        if (position+1 != size) next = true;
      }

      if (!next)
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);

      if (!prev)
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);

      s_caps_.emit (m_caps);
    }

    void
    Playlist::add_fc ()
    {
      using namespace Gtk;
      using namespace Glib;

      Playlist::PlaylistFCDialog dialog;
      int response = dialog.run ();
      dialog.hide ();

      if (response != GTK_RESPONSE_OK)
        return;
      
      typedef Glib::SListHandle<Glib::ustring> URISLH;
      URISLH uris = dialog.get_uris ();

      for (URISLH::const_iterator u = uris.begin() ; u != uris.end() ; ++u) 
        {
          if (!vfs->has_container (*u))
            {
              append_track (*u);
            }
          else
            {
              VFS::Handle h (*u);
              VUri list;

              try {
                if (Glib::file_test (Glib::filename_from_uri (*u), Glib::FILE_TEST_IS_DIR))
                    vfs->read (h, list, VFS::CONTAINER); 
                else
                    vfs->read (h, list, VFS::ProcessingFlags (VFS::TRANSPORT | VFS::CONTAINER)); 

                for (VUri::const_iterator u = list.begin(); u != list.end() ; ++u)
                  {
                    Bmp::URI uri (*u);
                    Bmp::URI::Protocol p (uri.get_protocol());

                    switch (p)
                    {
                      case URI::PROTOCOL_UNKNOWN:
                      case URI::PROTOCOL_CDDA:
                      case URI::PROTOCOL_LASTFM:
                      case URI::PROTOCOL_FTP:
                      case URI::PROTOCOL_QUERY:
                      case URI::PROTOCOL_TRACK:
                              break;

                      case URI::PROTOCOL_FILE:
                              append_track (*u);
                              break;

                      case URI::PROTOCOL_HTTP:
                      case URI::PROTOCOL_MMS:
                      case URI::PROTOCOL_MMSU:
                      case URI::PROTOCOL_MMST:
                              std::string basepath (Glib::path_get_basename (uri.path));
                              TreeModel::iterator m_iter = playlist_store->append ();
                              (*m_iter)[track.artist]   = uri.hostname; 
                              (*m_iter)[track.album]    = ""; 
                              (*m_iter)[track.title]    = ""; 
                              (*m_iter)[track.tracknumber] = 0; 
                              (*m_iter)[track.genre]    = ""; 
                              (*m_iter)[track.date]     = 0; 
                              (*m_iter)[track.duration] = 0; 
                              (*m_iter)[track.location] = (*u); 
                              (*m_iter)[track.guid] = m_guid++; 
                              break;
                    }
                  }
                }
              catch (...) {} //FIXME: Catch conversion error
            }
        } 
      check_nextprev_caps ();
    }

    void
    Playlist::add (bool play)
    {
      using namespace Gtk;
      using namespace Glib;

      int n_rows = tree->get_selection()->count_selected_rows();
      if (!n_rows)
        return;

      VUri uris;
      tree->get_selected_uris (uris);
      
      TreePath path;
      if (play)
          path = TreePath (1, playlist_store->children().size());
     
      for (VUri::const_iterator u = uris.begin() ; u != uris.end() ; ++u) 
        {
          if (!vfs->has_container (*u))
            {
              append_track (*u);
            }
          else
            {
              VFS::Handle h (*u);
              VUri list;
              vfs->read (h, list, VFS::CONTAINER); 
              for (VUri::const_iterator u = list.begin(); u != list.end() ; ++u)
                {
                  append_track (*u);
                }
            }
        } 

      if (play)
        {
          m_playing_row = TreeModel::RowReference (playlist_store, path);
          s_playback_request_.emit();
          send_title ();
        }
    
      check_nextprev_caps ();
    }

    void
    Playlist::clear_selected ()
    {
      using namespace Gtk;
      using namespace Glib;

      boost::optional<Gtk::TreeModel::Path> p;
      if (m_playing_row)
        {
          p = m_playing_row.get().get_path();
        }

      PathList paths = playlist_view->get_selection()->get_selected_rows ();
      ReferenceList references; 
      for (PathList::const_iterator path  = paths.begin (),
                                    end   = paths.end ();
                                    path != end; ++path)
        {
          references.push_back (TreeModel::RowReference (playlist_store, *path));
          if ((p) && (p.get() == (*path)))
            {
              m_playing_row.reset ();
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
              m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
              s_caps_.emit (m_caps);
              playlist_view->queue_draw ();
            }
        }

      for (ReferenceList::const_iterator rref  = references.begin (),
                                         end   = references.end ();
                                         rref != end; ++rref)
        {
          TreeModel::iterator m_iter = playlist_store->get_iter (rref->get_path()); 
          playlist_store->erase (m_iter);
        }
      check_nextprev_caps ();
    }

    void
    Playlist::clear ()
    {
      using namespace Gtk;

      playlist_store->clear ();
      if (m_playing_row) m_playing_row.reset ();

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      s_caps_.emit (m_caps);
    }
 
    guint 
    Playlist::add_ui ()
    {
      return m_ui_manager->add_ui_from_string (ui_menu_playlist);
    };

    Playlist::Playlist (Glib::RefPtr<Gnome::Glade::Xml> const& xml, Glib::RefPtr<Gtk::UIManager> ui_manager)

         :  PlaybackSource (PlaybackSource::CAN_SEEK, PlaybackSource::F_HANDLE_STREAMINFO), Base (xml, ui_manager), 
            m_playing (Gdk::Pixbuf::create_from_file (BMP_IMAGE_DIR_MAIN G_DIR_SEPARATOR_S "playing.png")),
            m_init    (true),
            m_guid    (0)
    {
        using namespace std;
        using namespace Gtk;
        using namespace Glib;

        export_dialog = ExportDialog::create ();

        m_actions = Gtk::ActionGroup::create ("ActionsPlaylist"); 

        m_actions->add ( Gtk::Action::create ("dummy", "dummy"));
        m_actions->add ( Gtk::Action::create ("MenuUiPartPlaylist", _("_Playlist")));
        m_actions->add ( Gtk::Action::create (PLAYLIST_ACTION_EXPORT,
                                  Gtk::StockID (GTK_STOCK_SAVE_AS),
                                  _("_Export Playlist")),
                                  sigc::mem_fun (this, &Bmp::UiPart::Playlist::export_playlist));

        m_actions->add ( Gtk::Action::create (PLAYLIST_ACTION_REMOVE,
                                  Gtk::StockID (GTK_STOCK_REMOVE),
                                  _("Remove _Selected Tracks")),
                                  sigc::mem_fun (this, &Bmp::UiPart::Playlist::clear_selected));

        m_actions->add ( Gtk::Action::create (PLAYLIST_ACTION_REMOVE_ALL,
                                  Gtk::StockID (GTK_STOCK_REMOVE),
                                  _("Remove _All Tracks")),
                                  sigc::mem_fun (this, &Bmp::UiPart::Playlist::clear));

        m_actions->add ( Gtk::Action::create (PLAYLIST_ACTION_ADD_FILES,
                                  Gtk::StockID (GTK_STOCK_ADD),
                                  _("_Add Files")),
                                  sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Playlist::add), false));

        m_actions->add ( Gtk::Action::create (PLAYLIST_ACTION_ADD_FILES_FC,
                                  Gtk::StockID (GTK_STOCK_ADD),
                                  _("_Add Files (Using File Chooser)")),
                                  sigc::mem_fun (this, &Bmp::UiPart::Playlist::add_fc));

        m_actions->add ( Gtk::Action::create (PLAYLIST_ACTION_OPEN_FILES,
                                  Gtk::StockID (GTK_STOCK_ADD),
                                  _("Add Files and _Play")) ,
                                  sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Playlist::add), true));

        m_actions->add ( Gtk::ToggleAction::create (PLAYLIST_ACTION_SHOW_FSTREE, 
                          _("Show Filesystem Tree")),
                          sigc::mem_fun (this, &Bmp::UiPart::Playlist::on_show_tree_toggled));

        m_action_show_fstree = Glib::RefPtr<Gtk::ToggleAction>::cast_static
            (m_actions->get_action (PLAYLIST_ACTION_SHOW_FSTREE));

        m_action_show_fstree->set_active (true);

        m_ref_xml->get_widget ("playlist-notebook", playlist_notebook);
        m_ref_xml->get_widget ("playlist-alignment-single", playlist_alignment_single);
        m_ref_xml->get_widget ("playlist-hpaned", playlist_hpaned);

        m_ui_manager->insert_action_group (m_actions);
        m_ui_manager->add_ui_from_string (ui_popup_playlist);

        m_actions->get_action (PLAYLIST_ACTION_REMOVE)->set_sensitive (false);
        m_actions->get_action (PLAYLIST_ACTION_EXPORT)->set_sensitive (false);
        m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->set_sensitive (false);
        m_actions->get_action (PLAYLIST_ACTION_OPEN_FILES)->set_sensitive (false);

        m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->connect_proxy
                                ((*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_playlist_add")))));

        m_actions->get_action (PLAYLIST_ACTION_REMOVE_ALL)->connect_proxy
                                ((*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_playlist_clear")))));

        const char *headers[]=
        {
             N_("Title"), 
             N_("Track"), 
             N_("Artist"), 
             N_("Album"), 
             N_("Genre"), 
             N_("Date")
        };

        // --- Filesystem Tree
        m_ref_xml->get_widget_derived ("playlist-treeview-dirtree", tree);
        tree->get_selection()->signal_changed().connect
          (sigc::mem_fun (this, &Bmp::UiPart::Playlist::tree_selection_changed));

        mcs_bind->bind_filechooser (dynamic_cast<Gtk::FileChooser *>
                                    (m_ref_xml->get_widget ("playlist-rootpath-fcb")),
                                    "playlist", "rootpath");

        mcs->subscribe ("Playlist",
                        "playlist",
                        "rootpath",
          (sigc::mem_fun (this, &Bmp::UiPart::Playlist::mcs_playlist_dirtree_rootpath_changed)));

        tree->on_path_activated().connect 
          (sigc::bind (sigc::mem_fun (this, &Bmp::UiPart::Playlist::add), false));

        // --- Playlist View+Store 
        m_ref_xml->get_widget ("playlist-treeview", playlist_view);

        Gtk::TreeViewColumn *column = 0;

        cell_playing = Gtk::manage (new Gtk::CellRendererPixbuf());
        cell_playing->property_xalign() = 0.5; 
        playlist_view->append_column ("", *cell_playing);
        column = playlist_view->get_column (0);
        column->set_resizable (false);
        column->set_expand (false);
        column->property_min_width() = 30; 
        column->property_max_width() = 30; 
        column->set_cell_data_func (*cell_playing,
          sigc::mem_fun(this, &Bmp::UiPart::Playlist::cell_data_func_playing));

        Gtk::Image * image = Gtk::manage ( new Gtk::Image() );
        image->set (Glib::build_filename (BMP_IMAGE_DIR, "blue-speaker.png")); 
        column->set_widget (*image); 
        image->show();

        for (unsigned int n = 0; n < G_N_ELEMENTS(headers); ++n)
        {
          Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
          playlist_view->append_column (_(headers[n]), *cell);
          column = playlist_view->get_column (n+1);
          column->set_sort_column (n);
          column->add_attribute (*cell, "text", n);
          column->set_resizable (true);
          column->set_expand (true);
          column->property_min_width() = 50; 
          column->signal_clicked().connect
            (sigc::mem_fun (this, &Bmp::UiPart::Playlist::check_nextprev_caps));
        }

        Gtk::CellRendererText *cell = Gtk::manage ( new Gtk::CellRendererText() );
        playlist_view->append_column (_("Length"), *cell);
        column = playlist_view->get_column (7);
        column->set_resizable (false);
        column->set_expand (false);
        column->set_cell_data_func (*cell,
          sigc::mem_fun(this, &Bmp::UiPart::Playlist::cell_data_func_numbers));

        playlist_store = Gtk::ListStore::create (track);
        playlist_store->set_default_sort_func
          (sigc::mem_fun (*this, &Bmp::UiPart::Playlist::default_sort_func));

        string filename (build_filename (BMP_PATH_USER_DIR, BMP_DEFAULT_PLAYLIST_NAME));
        if (file_test (filename, FILE_TEST_EXISTS))
          {
            ustring uri (filename_to_uri (filename));
            VUri list;
            VFS::Handle handle (uri);
            try {
              vfs->read (handle, list, VFS::ProcessingFlags (VFS::TRANSPORT | VFS::CONTAINER));
              for (VUri::const_iterator u = list.begin(), e = list.end(); u != e; ++u) 
                {
                  append_track (*u);
                }
              }
            catch (Bmp::VFS::Exception& cxe) {}
          }

        playlist_view->set_model (playlist_store);
        playlist_view->get_selection()->set_mode (Gtk::SELECTION_MULTIPLE);
        playlist_view->get_selection()->signal_changed().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Playlist::list_selection_changed));
        playlist_view->signal_event().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Playlist::popup_menu));
        playlist_view->signal_row_activated().connect
                (sigc::mem_fun (this, &Bmp::UiPart::Playlist::activate_default));

#if GTK_CHECK_VERSION(2, 10, 0)
        gtk_tree_view_set_rubber_banding (GTK_TREE_VIEW (playlist_view->gobj()), TRUE);
        gtk_tree_view_set_grid_lines (GTK_TREE_VIEW (playlist_view->gobj()), GTK_TREE_VIEW_GRID_LINES_VERTICAL);
#endif

        m_init = false;
    }

    void
    Playlist::on_show_tree_toggled ()
    {
      if (m_init) return;

      bool active (m_action_show_fstree->get_active());
  
      GtkWidget * w = GTK_WIDGET (m_ref_xml->get_widget ("scrolledwindow-playlist")->gobj());
      g_object_ref (G_OBJECT (w));

      if (active)
        {
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES_FC)->disconnect_proxy
                                ((*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_playlist_add")))));
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->connect_proxy
                                ((*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_playlist_add")))));

          playlist_alignment_single->remove ();
          playlist_notebook->set_current_page (0);    
          playlist_hpaned->pack1 (*(m_ref_xml->get_widget ("scrolledwindow-playlist"))); 
          tree_selection_changed ();
        }
      else
        {
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->disconnect_proxy
                                ((*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_playlist_add")))));
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES_FC)->connect_proxy
                                ((*(dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("b_playlist_add")))));

          playlist_hpaned->remove (*(m_ref_xml->get_widget ("scrolledwindow-playlist")));
          playlist_notebook->set_current_page (1);    
          playlist_alignment_single->add (*(m_ref_xml->get_widget ("scrolledwindow-playlist"))); 
          m_actions->get_action (PLAYLIST_ACTION_ADD_FILES)->set_sensitive (false);
          m_actions->get_action (PLAYLIST_ACTION_OPEN_FILES)->set_sensitive (false);
        }
    }

    void
    Playlist::mcs_playlist_dirtree_rootpath_changed (MCS_CB_DEFAULT_SIGNATURE)
    {
      tree->set_root (boost::get<std::string>(value));
    }

    Playlist::~Playlist ()
    {
      using namespace Gtk;
      using namespace Glib;

      delete export_dialog;

      VUri list;
      TreeModel::Children const& nodes = playlist_store->children();

      for (TreeModel::Children::const_iterator n = nodes.begin(), e = nodes.end(); n != e; ++n)
        {
          try{
            Bmp::URI uri ((*n)[track.location]);
            if (uri.get_protocol() == URI::PROTOCOL_FILE)
              list.push_back ((*n)[track.location]);
            }
          catch (...) {}
        }

      ustring location = filename_to_uri (build_filename (BMP_PATH_USER_DIR, BMP_DEFAULT_PLAYLIST_NAME));
      try
        {
          vfs->write (list, location, "xspf");
        }
      catch (Bmp::VFS::Exception& cxe)
        {
        	g_warning ("Couldn't save playlist to %s", location.c_str ());
        }
    }

    void
    Playlist::activate_default (Gtk::TreeModel::Path const& path, Gtk::TreeViewColumn* column)
    {
      dynamic_cast<Gtk::Window *>(m_ref_xml->get_widget ("main-ui"))->activate_default ();
    }

    void
    Playlist::send_title ()
    {
      using namespace Gtk;
      using namespace Glib;

      TreeModel::iterator const& m_iter = playlist_store->get_iter (m_playing_row.get().get_path());

      boost::optional<ustring> eartist, ealbum, etitle;
      Glib::ustring artist, album, title;

      eartist = Glib::Markup::escape_text ((*m_iter)[track.artist]);
      ealbum  = Glib::Markup::escape_text ((*m_iter)[track.album]);
      etitle  = Glib::Markup::escape_text ((*m_iter)[track.title]);

      title.append ("<b>").append(etitle.get()).append("</b>");

      if (ealbum.get().length())
        {
          album.append ("<i>").append(ealbum.get()).append("</i>"); 
        }

      if (((unsigned int)(*m_iter)[track.date]) > 0)
        {
          static boost::format format_uint ("%u");
          unsigned int date = (*m_iter)[track.date];
          album.append(", ").append((format_uint % date).str());
        }

      char *text = g_strdup_printf ("%s (%s%s%s)", (etitle  ? title.c_str() : ""),
                                                   (eartist ? eartist.get().c_str()  : ""),
                                                   ((!eartist || (!ealbum))) ? "" : ": ",
                                                   (ealbum  ? album.c_str() : ""));

      ustring _title (text);
      g_free (text);

      SimpleTrackInfo sti;

      sti.artist    = (*m_iter)[track.artist]; 
      sti.title     = (*m_iter)[track.title]; 
      sti.album     = (*m_iter)[track.album]; 
      sti.duration  = (*m_iter)[track.duration]; 
  
      Glib::ustring asin ((*m_iter)[track.asin]);
      if (asin.size()) sti.asin = asin;

      s_track_info_.emit (_title, sti);
    }

    int
    Playlist::default_sort_func (Gtk::TreeModel::iterator const& iter_a, Gtk::TreeModel::iterator const& iter_b)
    {
      guint64 a = ((*iter_a)[track.guid]);
      guint64 b = ((*iter_b)[track.guid]);

      int cmp = 0;
    
      if (a < b)
        cmp -= 1;

      if (a > b)
        cmp += 1;

      return cmp;
    }

    // Bmp::PlaybackSource
    void
    Playlist::cell_data_func_numbers (Gtk::CellRenderer* cell, Gtk::TreeModel::iterator const& m_iter)
    {
      static boost::format format_duration ("%d:%02d");
      Gtk::CellRendererText *cell_text = dynamic_cast<Gtk::CellRendererText *>(cell);
      unsigned int duration = (*m_iter)[track.duration];
      unsigned int minutes = duration / 60;
      unsigned int seconds = duration % 60;
      cell_text->property_text() = (format_duration % minutes % seconds).str(); 
    }

    void
    Playlist::cell_data_func_playing (Gtk::CellRenderer* cell_, Gtk::TreeModel::iterator const& m_iter)
    {
      using namespace Gtk;

      CellRendererPixbuf *cell = dynamic_cast<CellRendererPixbuf *>(cell_);
      if (m_playing_row)
        {
          Gtk::TreeModel::Path path1 (playlist_store->get_path (m_iter)); 
          Gtk::TreeModel::Path path2 (m_playing_row.get().get_path());
          if (path1 == path2)
            {
              cell->property_pixbuf() = m_playing; 
              return;
            } 
        }
      cell->property_pixbuf() = Glib::RefPtr<Gdk::Pixbuf>(0);
    }

    Glib::ustring
    Playlist::get_uri ()
    {
      using namespace Gtk;
      TreeModel::iterator m_iter = playlist_store->get_iter (m_playing_row.get().get_path());
      return (*m_iter)[track.location];
    }

    GHashTable*
    Playlist::get_metadata ()
    {
      return library->get_hashtable (::play->property_stream().get_value());
    }

    bool
    Playlist::go_next ()
    {
      using namespace Gtk;
      using namespace Glib;
 
      TreeModel::Path path_old (m_playing_row.get().get_path());

      unsigned int size = playlist_view->get_model()->children().size(); 
      unsigned int position = path_old.get_indices().data()[0];

      if ((position + 1) == size) return false;

      TreeModel::Path path_new (1, position+1); 
      m_playing_row = TreeModel::RowReference (playlist_store, path_new);

      playlist_view->get_model()->row_changed (path_old, playlist_store->get_iter (path_old));
      playlist_view->get_model()->row_changed (path_new, playlist_store->get_iter (path_new));

      position += 1; 
      if (position < size)
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);

      s_caps_.emit (m_caps);
      send_title ();
      return true;
    }

    bool
    Playlist::go_prev ()
    {
      using namespace Gtk;
      using namespace Glib;
 
      TreeModel::Path path_old (m_playing_row.get().get_path());
      unsigned int position = path_old.get_indices().data()[0];
      if (position == 0) return false;

      TreeModel::Path path_new (1, position-1); 
      m_playing_row = TreeModel::RowReference (playlist_store, path_new);

      playlist_view->get_model()->row_changed (path_old, playlist_store->get_iter (path_old));
      playlist_view->get_model()->row_changed (path_new, playlist_store->get_iter (path_new));

      position -= 1; 
      unsigned int size = playlist_view->get_model()->children().size(); 
      if (position < (size-1))
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);

      s_caps_.emit (m_caps);
      send_title ();
      return true;
    }

    void
    Playlist::stop ()
    {
      if (m_playing_row)
        {
          Gtk::TreeModel::Path path (m_playing_row.get().get_path());
          Gtk::TreeModel::iterator m_iter = playlist_store->get_iter (path); 
          m_playing_row.reset ();
          playlist_view->get_model()->row_changed (path, m_iter);
        }

      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps & ~PlaybackSource::CAN_PROVIDE_METADATA);

      s_caps_.emit (m_caps);
    }

    void
    Playlist::play_requested ()
    {
      using namespace Gtk;
      using namespace Glib;

      TreeModel::Path const& path = m_playing_row.get().get_path();
      TreeModel::iterator const& m_iter = playlist_store->get_iter (path); 
      playlist_view->get_model()->row_changed (path, m_iter); 

      unsigned int position = path.get_indices().data()[0];
      unsigned int size     = playlist_view->get_model()->children().size()-1; 

      if (position < size)
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);

      s_caps_.emit (m_caps);
      send_title ();
    }

    void
    Playlist::play ()
    {
      using namespace Gtk;
      using namespace Glib;

      PathList paths = playlist_view->get_selection()->get_selected_rows ();
      TreeModel::iterator m_iter = playlist_store->get_iter (paths[0]); 

      m_playing_row = TreeModel::RowReference (playlist_store, paths[0]);
      playlist_view->get_model()->row_changed (paths[0], m_iter); 

      unsigned int position = paths[0].get_indices().data()[0];
      unsigned int size     = playlist_view->get_model()->children().size()-1; 

      if (position < size)
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_NEXT);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_NEXT);

      if (position > 0) 
        m_caps = Caps (m_caps |  PlaybackSource::CAN_GO_PREV);
      else
        m_caps = Caps (m_caps & ~PlaybackSource::CAN_GO_PREV);

      m_caps = Caps (m_caps | PlaybackSource::CAN_PAUSE);
      m_caps = Caps (m_caps | PlaybackSource::CAN_PROVIDE_METADATA);

      try
        {
        Bmp::URI u ((*m_iter)[track.location]);
        Bmp::URI::Protocol p (u.get_protocol());
        if ((p == Bmp::URI::PROTOCOL_HTTP)  ||
            (p == Bmp::URI::PROTOCOL_MMS)   ||
            (p == Bmp::URI::PROTOCOL_MMSU)  ||
            (p == Bmp::URI::PROTOCOL_MMST))
          {
            m_flags = Flags (m_flags |  PlaybackSource::F_HANDLE_STREAMINFO);
          }
        else
          {
            m_flags = Flags (m_flags & ~PlaybackSource::F_HANDLE_STREAMINFO);
          }
        }
      catch (...)
        {
          m_flags = Flags (m_flags & ~PlaybackSource::F_HANDLE_STREAMINFO);
        }

      s_flags_.emit (m_flags);
      s_caps_.emit (m_caps);
      send_title ();
    }

    void
    Playlist::restore_context ()
    {
    }

    void
    Playlist::add_uris (VUri& list, bool playback)
    {
      using namespace Gtk;

      TreeModel::Path path;

      if (playback)
        {
          path = TreeModel::Path (1, playlist_store->children().size());
        }

      for (VUri::const_iterator u = list.begin(), e = list.end(); u !=e ; ++u)
        {
          append_track (*u);
        }

      if (playback)
        {
          m_playing_row.reset ();
          m_playing_row = TreeModel::RowReference (playlist_store, path);
        }
    }

    // Related to PlaybackSource but not directly part of the class
    void
    Playlist::list_selection_changed ()
    {
      bool selected = (playlist_view->get_selection ()->count_selected_rows() > 0);
      m_actions->get_action (PLAYLIST_ACTION_REMOVE)->set_sensitive (selected);

      if (playlist_view->get_selection ()->count_selected_rows() == 1)
          m_caps = Caps (m_caps |  PlaybackSource::CAN_PLAY);
      else
          m_caps = Caps (m_caps & ~PlaybackSource::CAN_PLAY);
      s_caps_.emit (m_caps);
    }

    bool
    Playlist::popup_menu (GdkEvent *ev)
    {
      using namespace Gtk;

      GdkEventButton *event = 0;
      if (ev->type == GDK_BUTTON_PRESS)
        {
          event = reinterpret_cast<GdkEventButton *>(ev);
          if ((event->button == 3) && (event->type == GDK_BUTTON_PRESS))
            {
              Menu *menu = dynamic_cast<Menu *>(get_popup (m_ui_manager, "/popup-tracklist/menu-tracklist"));
              m_actions->get_action (PLAYLIST_ACTION_REMOVE_ALL)->set_sensitive (!playlist_store->children().empty());
              m_actions->get_action (PLAYLIST_ACTION_EXPORT)->set_sensitive (!playlist_store->children().empty());
              menu->popup (event->button, event->time);
              return true;
            }
        } 
      return false;
    }

    void
    Playlist::export_playlist ()
    {
      using namespace Gtk;

      int response = export_dialog->run (); 
      export_dialog->hide ();

      if (response == GTK_RESPONSE_CANCEL) return;

      VUri list;
  
      if (export_dialog->all_files) 
        {
          TreeNodeChildren const& nodes = playlist_store->children(); 
          for (TreeNodeChildren::const_iterator n = nodes.begin(), e = nodes.end(); n != e; ++n) 
            {
              list.push_back ((*n)[track.location]);
            }
        }
      else
        {
          PathList paths = playlist_view->get_selection()->get_selected_rows ();
          for (PathList::const_iterator path  = paths.begin (),
                                        end   = paths.end ();
                                        path != end; ++path)
            {
              list.push_back (playlist_store->get_iter (*path)->operator[](track.location));
            }
        }

        try {
          vfs->write (list, Glib::filename_to_uri (export_dialog->entry_filename->get_text()), export_dialog->container);
          }
        catch (Bmp::VFS::Exception& cxe)
          {
            //FIXME: We need to enhance VFS to check for file permissions, etc
          }
    }

    ////////////// Playlist FileChooser

    Playlist::PlaylistFCDialog::~PlaylistFCDialog ()
    {
    }
  
    void
    Playlist::PlaylistFCDialog::on_hide ()
    {
      mcs->key_set<std::string>("bmp", "file-chooser-path", Glib::filename_from_uri(get_current_folder_uri()));
    }

    bool
    Playlist::PlaylistFCDialog::audio_files_filter (Gtk::FileFilter::Info const& info)
    {
      return (::play->is_audio_file (info.uri));
    }
 
    Playlist::PlaylistFCDialog::PlaylistFCDialog () : Gtk::FileChooserDialog (_("Select Files to Add - BMP"),
                                                                              Gtk::FILE_CHOOSER_ACTION_OPEN)
    {
        struct {
          const char *name;
          const char *glob;
        } filter_list[] = {
          { "FLAC", "*.[fF][lL][aA][cC]" },
          { "M4A" , "*.[mM]4[aA]" },
          { "MP3" , "*.[mM][pP]3" },
          { "MPC" , "*.[mM][pP][cC]" },
          { "OGG" , "*.[oO][gG][gG]" },
          { "WMA" , "*.[wW][mM][aA]" }
        };

        // Add "all" filter
        Gtk::FileFilter all;
        all.set_name (_("All Files"));
        all.add_pattern ("*");
        add_filter (all);

        // Add audiofiles filter
        Gtk::FileFilter audios;
        audios.set_name (_("Audio Files"));
        audios.add_custom (Gtk::FILE_FILTER_URI,
          sigc::mem_fun (this, &Bmp::UiPart::Playlist::PlaylistFCDialog::audio_files_filter));
        add_filter (audios);

        // Add container plugin filters
        Bmp::VFS::ExportDataList containers;
        vfs->get_containers (containers);

        for (Bmp::VFS::ExportDataList::const_iterator i = containers.begin () ; i != containers.end() ; ++i)
        {
          std::string glob = Util::create_glob (i->extension);
          std::string glob_suffix = std::string ("*.") + glob;

          Gtk::FileFilter filter_container;
          filter_container.set_name (i->description);
          filter_container.add_pattern (glob_suffix);
          add_filter (filter_container);
        }

        // Add glob filefilters
        for (unsigned int n = 0; n < G_N_ELEMENTS(filter_list); ++n)
        {
          Gtk::FileFilter filter;
          filter.set_name (filter_list[n].name);
          filter.add_pattern (filter_list[n].glob);
          add_filter (filter);
        }

        // Set current folder to the filechooser
        set_current_folder (mcs->key_get<std::string>("bmp", "file-chooser-path"));
        add_button (Gtk::StockID (GTK_STOCK_CANCEL), GTK_RESPONSE_CANCEL);
        add_button (Gtk::StockID (GTK_STOCK_OPEN), GTK_RESPONSE_OK);
        set_select_multiple ();
        set_default_size (750, 550);
    }
  } // namespace UiPart
} // namespace Bmp 
