//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glibmm.h>
#include <glibmm/markup.h>
#include <iostream>

#include "debug.hh"
#include "library.hh"
#include "uri++.hh"

#include "x_vfs.hh"

#include "mbxml.hh"

namespace Bmp
{
  namespace MusicBrainzXML
  {
    void
    mb_track_to_update_track (MBTrack const& track, Bmp::Library::UTrack & u)
    {
      u.artist              = track.artist_name;
      u.title               = track.track_title;
      u.mb_track_id         = track.track_id;
      u.mb_album_id         = track.release_id;
      u.mb_artist_id        = track.artist_id;
      u.mb_artist_sort_name = track.artist_sort_name;
      u.tracknumber         = track.tracknumber;
    }

    void
    mb_release_to_update_track (MBRelease const& release, Bmp::Library::UTrack & u)
    {
      u.asin                       = release.asin;
      u.album                      = release.release_title;
      u.mb_album_artist            = release.artist_name;
      u.mb_album_artist_sort_name  = release.artist_sort_name;
      u.mb_album_artist_id         = release.artist_id;
      u.mb_release_date            = release.date;
      u.mb_album_id                = release.release_id;
    }

    void
    mb_tracks_by_puid (Glib::ustring const& puid,
                       MBTrackV& tracks)
    {
      using namespace Glib;

      if (puid.empty() || !puid.size()) return;

      ustring uri;
      uri .append ("http://musicbrainz.org/ws/1/track/?puid=")
          .append (puid)
          .append ("&type=xml&inc=releases");

      Bmp::URI u (uri, true);
      Bmp::VFS::Handle handle ((ustring(u)));
      try
        {
          vfs->read (handle, Bmp::VFS::TRANSPORT);
          MBTracksParser parser (tracks);
          Markup::ParseContext context (parser);
          ustring data = reinterpret_cast<const char *>(handle.get_buffer()); 	  
          context.parse (data);
          context.end_parse ();
          parser.check_sanity ();
        }
      catch (Bmp::VFS::Exception& cxe)
        {
          //FIXME: VFS exception
        }
     catch (ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
     catch (MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
    }

    void
    mb_releases_by_id (Glib::ustring const& mb_release_id,
                       MBReleaseV& releases)
    {
      using namespace Glib;

      if (mb_release_id.empty() || !mb_release_id.size()) return;

      ustring uri;
      uri .append ("http://musicbrainz.org/ws/1/release/")
          .append (mb_release_id)
          .append ("?type=xml&inc=artist+tracks+counts+release-events");

      Bmp::URI u (uri, true);
      Bmp::VFS::Handle handle ((ustring(u)));
      try
        {
          vfs->read (handle, Bmp::VFS::TRANSPORT);
          MBReleaseParser parser (releases);
          Markup::ParseContext context (parser);
          ustring data = reinterpret_cast<const char *>(handle.get_buffer()); 	  
          context.parse (data);
          context.end_parse ();
          parser.check_sanity ();
        }
      catch (Bmp::VFS::Exception& cxe)
        {
          //FIXME: VFS exception
        }
     catch (ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
     catch (MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
    }

    void
    mb_query_releases (Glib::ustring const& artist,
                       Glib::ustring const& album,
                       MBReleaseV& releases)
    {
      using namespace Glib;

      if (artist.empty() || !artist.size()) return;
      if (album.empty() || !album.size()) return;

      ustring uri;

      uri .append ("http://musicbrainz.org/ws/1/release/?type=xml&title=")
          .append (album)
          .append ("&artist=")
          .append (artist)
          .append ("&releasetypes=Official&limit=25&inc=release-events");

      try {
          Bmp::URI u (uri, true);
          Bmp::VFS::Handle handle ((ustring(u)));
          vfs->read (handle, Bmp::VFS::TRANSPORT);
          MBReleaseParser parser (releases);
          Glib::Markup::ParseContext context (parser);
          Glib::ustring data = reinterpret_cast<const char *>(handle.get_buffer()); 	  
          context.parse (data);
          context.end_parse ();
          parser.check_sanity ();
        }
      catch (Bmp::URI::Exception& cxe)
        {
          //FIXME: URI exception
        }
      catch (Bmp::VFS::Exception& cxe)
        {
          //FIXME: VFS exception
        }
      catch (Glib::ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
      catch (Glib::MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
    }

    void
    MBReleaseParser::check_sanity  ()
    {
      if (state)
        {
          g_warning (G_STRLOC ": State should be 0, but is %d", state);
        }
    }

    MBReleaseParser::MBReleaseParser (MBReleaseV &releases) 
      : m_releases (releases), state (0), tracknum (0), has_offset (false)
    {
    }

    MBReleaseParser::~MBReleaseParser () 
    {
    }
 
    void
    MBReleaseParser::on_start_element  (Glib::Markup::ParseContext& context,
                                        Glib::ustring const& name,
                                        const AttributeMap& attributes)
	  {
        if (name == "release-list")
        {
          state |= E_RELEASE_LIST;
          return;
        }

        if (name == "release-event-list")
        {
          state |= E_RELEASE_EVENT_LIST;
          return;
        }

        if (name == "event")
        {
          state |= E_EVENT;
          if (attributes.find("date") != attributes.end())
            {
              c_release.date = attributes.find("date")->second;
            }
          return;
        }

        if (name == "release")
        {
          state |= E_RELEASE;
          if (state & E_TRACK)
            {
              if (attributes.find("id") != attributes.end())
                {
                  c_track.release_id = attributes.find ("id")->second;
                }
            }
          else
            {
              c_release = MBRelease();

              if (attributes.find("id") != attributes.end())
                {
                  c_release.release_id = attributes.find ("id")->second;
                }

              if (attributes.find ("ext:score") != attributes.end())
                {
                  c_release.score = g_ascii_strtoull (attributes.find ("ext:score")->second.c_str(), NULL, 10);
                }
            }
          return;
        }

        if (name == "title")
        {
          state |= E_TITLE;
          return;
        }

        if (name == "asin")
        {
          state |= E_ASIN;
          return;
        }

        if (name == "artist")
        {
          state |= E_ARTIST;

          if (state & E_TRACK)
            {
              if (attributes.find("id") != attributes.end())
                {
                  c_track.artist_id = attributes.find ("id")->second;
                }
            }
          else
            {
              if (attributes.find("id") != attributes.end())
                {
                  c_release.artist_id = attributes.find ("id")->second;
                }
            }
          return;
        }

        if (name == "name")
        {
          state |= E_NAME;
          return;
        }

        if (name == "sort-name")
        {
          state |= E_SORT_NAME;
          return;
        }

        if (name == "track-list")
        {
          state |= E_TRACK_LIST;
          if ((state & E_TRACK) && attributes.find ("offset") != attributes.end())
            {
              tracknum = g_ascii_strtoull (attributes.find ("offset")->second.c_str(), NULL, 10) + 1;
              has_offset = true;
            }
          else
            {
              tracknum = 1;
              has_offset = false;
            }

          if (!(state & E_TRACK))
            {
              c_release.tracks = MBTrackV();
              if (attributes.find("count") != attributes.end())
                {
                  c_release.track_list = g_ascii_strtoull(attributes.find ("count")->second.c_str(), NULL, 10);
                }
            }

          return;
        }

        if (name == "track")
        {
          state |= E_TRACK;
          c_track = MBTrack();
          if (attributes.find("id") != attributes.end())
            {
              c_track.track_id = attributes.find ("id")->second;
            }
          return;
        }

        if (name == "duration")
        {
          state |= E_DURATION;
          return;
        }

        if (name == "disc-list")
        {
          state |= E_DISC_LIST;
          if (attributes.find("id") != attributes.end())
            {
              c_release.disc_list = g_ascii_strtoull (attributes.find ("id")->second.c_str(), NULL, 10);
            }
          return;
        }
    }

    void
    MBReleaseParser::on_end_element    (Glib::Markup::ParseContext& context,
                                        Glib::ustring const& name)
	  {
        if (name == "release-list")
        {
          state &= ~ E_RELEASE_LIST;
          return;
        }

        if (name == "release-event-list")
        {
          state &= ~ E_RELEASE_EVENT_LIST;
          return;
        }

        if (name == "event")
        {
          state &= ~ E_EVENT;
          return;
        }

        if (name == "release")
        {
          state &= ~ E_RELEASE;
          if (!(state & E_TRACK))
          {
            m_releases.push_back (c_release);
          }
          return;
        }

        if (name == "title")
        {
          state &= ~ E_TITLE;
          return;
        }

        if (name == "asin")
        {
          state &= ~ E_ASIN;
          return;
        }

        if (name == "artist")
        {
          state &= ~ E_ARTIST;
          return;
        }

        if (name == "name")
        {
          state &= ~ E_NAME;
          return;
        }

        if (name == "sort-name")
        {
          state &= ~ E_SORT_NAME;
          return;
        }

        if (name == "track-list")
        {
          state &= ~ E_TRACK_LIST;
          return;
        }

        if (name == "track")
        {
          state &= ~ E_TRACK;

          if (!c_track.artist_id.size())
            {
                c_track.artist_id
                  = c_release.artist_id;
            }

          if (!c_track.artist_name.size())
            { 
                c_track.artist_name
                  = c_release.artist_name;
            }

          if (!c_track.artist_sort_name)
            {
                c_track.artist_sort_name 
                  = c_release.artist_sort_name;
            }

          if (!c_track.release_id)
            {
                c_track.release_id
                  = c_release.release_id;
            }

          if (!has_offset)
              c_track.tracknumber = tracknum++;
          else
              c_track.tracknumber = tracknum;
  
          c_release.tracks.get().push_back (c_track);

          return;
        }

        if (name == "duration")
        {
          state &= ~ E_DURATION;
          return;
        }

        if (name == "disc-list")
        {
          state &= ~ E_DISC_LIST;
          return;
        }
	  }

    void
    MBReleaseParser::on_text       (Glib::Markup::ParseContext& context,
                                    Glib::ustring const&  text)
    {
        if ((state & E_RELEASE) && !(state & E_TRACK))
        {
          if (state & E_TITLE)
          {
            c_release.release_title = text;
            return;
          }

          if (state & E_ASIN)
          {
            c_release.asin = text;
            return;
          }

          if ((state & E_ARTIST) && (state & E_NAME))
          {
            c_release.artist_name = text;
            return;
          }

          if ((state & E_ARTIST) && (state & E_SORT_NAME))
          {
            c_release.artist_sort_name = text;
            return;
          }
        }

        /// Track
        if ((state & E_RELEASE) && (state & E_TRACK))
        {
          if (state & E_TITLE)
          {
            c_track.track_title = text;
            return;
          }

          if (state & E_DURATION)
          {
            c_track.duration = g_ascii_strtoull (text.c_str(), NULL, 10);
            return;
          }

          if ((state & E_ARTIST) && (state & E_NAME))
          {
            c_track.artist_name = text;
            return;
          }

          if ((state & E_ARTIST) && (state & E_SORT_NAME))
          {
            c_track.artist_sort_name = text;
            return;
          }

        }
    }

    void
    MBReleaseParser::on_passtrough (Glib::Markup::ParseContext& context,
                                    Glib::ustring const&  text) {}

    void
    MBReleaseParser::on_error      (Glib::Markup::ParseContext& context,
                                    Glib::MarkupError const& error) {}

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    MBTracksParser::MBTracksParser (MBTrackV &tracks) : m_tracks (tracks), state (0), tracknum (0), has_offset (false)
    {
    }

    MBTracksParser::~MBTracksParser () 
    {
    }

    void
    MBTracksParser::check_sanity  ()
    {
      if (state)
        {
          g_warning (G_STRLOC ": State should be 0, but is %d", state);
        }
    }

    void
    MBTracksParser::on_start_element  (Glib::Markup::ParseContext& context,
                                      Glib::ustring const& name,
                                      const AttributeMap& attributes)
	  {
        if (name == "release-list")
        {
          state |= E_RELEASE_LIST;
          return;
        }

        if (name == "release")
        {
          state |= E_RELEASE;
          c_track.release_id = attributes.find ("id")->second;
          return;
        }

        if (name == "title")
        {
          state |= E_TITLE;
          return;
        }

        if (name == "artist")
        {
          state |= E_ARTIST;
          c_track.artist_id = attributes.find ("id")->second;
          return;
        }

        if (name == "name")
        {
          state |= E_NAME;
          return;
        }

        if (name == "sort-name")
        {
          state |= E_SORT_NAME;
          return;
        }

        if (name == "track-list")
        {
          state |= E_TRACK_LIST;
          if ((state & E_TRACK) && attributes.find ("offset") != attributes.end())
            {
              tracknum = g_ascii_strtoull (attributes.find ("offset")->second.c_str(), NULL, 10) + 1;
              has_offset = true;
            }
          else
            {
              tracknum = 0;
              has_offset = false;
            }
          return;
        }

        if (name == "track")
        {
          state |= E_TRACK;
          c_track = MBTrack();
          c_track.track_id = attributes.find ("id")->second;
          return;
        }

        if (name == "duration")
        {
          state |= E_DURATION;
          return;
        }
    }

    void
    MBTracksParser::on_end_element    (Glib::Markup::ParseContext& context,
                                        Glib::ustring const& name)
	  {
        if (name == "release-list")
        {
          state &= ~ E_RELEASE_LIST;
          return;
        }

        if (name == "release")
        {
          state &= ~ E_RELEASE;
          return;
        }

        if (name == "title")
        {
          state &= ~ E_TITLE;
          return;
        }

        if (name == "artist")
        {
          state &= ~ E_ARTIST;
          return;
        }

        if (name == "name")
        {
          state &= ~ E_NAME;
          return;
        }

        if (name == "sort-name")
        {
          state &= ~ E_SORT_NAME;
          return;
        }

        if (name == "track-list")
        {
          state &= ~ E_TRACK_LIST;
          return;
        }

        if (name == "track")
        {
          state &= ~ E_TRACK;

          if (has_offset)
            {
              c_track.tracknumber = tracknum;
            }
          m_tracks.push_back (c_track);
          return;
        }

        if (name == "duration")
        {
          state &= ~ E_DURATION;
          return;
        }
	  }

    void
    MBTracksParser::on_text       (Glib::Markup::ParseContext& context,
                                    Glib::ustring const&  text)
    {
        if (state & E_TRACK)
        {
          if ((state & E_TITLE) && !(state & E_RELEASE))
          {
            c_track.track_title = text;
            return;
          }

          if (state & E_DURATION)
          {
            c_track.duration = g_ascii_strtoull (text.c_str(), NULL, 10);
            return;
          }

          if ((state & E_ARTIST) && (state & E_NAME) && !(state & E_RELEASE))
          {
            c_track.artist_name = text;
            return;
          }

          if ((state & E_ARTIST) && (state & E_SORT_NAME) && !(state & E_RELEASE))
          {
            c_track.artist_sort_name = text;
            return;
          }
        }
    }

    void
    MBTracksParser::on_passtrough (Glib::Markup::ParseContext& context,
                                    Glib::ustring const&  text) {}

    void
    MBTracksParser::on_error      (Glib::Markup::ParseContext& context,
                                    Glib::MarkupError const& error) {}


  }
}
