//-*- Mode: C++; indent-tabs-mode: nil; -*-

//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <gtk/gtk.h>
#include <gtkmm.h>
#include <glibmm.h>
#include <glibmm/i18n.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cctype>
#include <iostream>
#include <fstream>
#include <vector>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <strings.h>

#include <boost/format.hpp>
#include <boost/regex.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/regex.hpp>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlreader.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include "neon++/request.hh"
#include "neon++/util.hh"

#include "x_lastfm.hh"
#include "x_library.hh"
#include "x_service_core.hh"

#include <bmp/bmp.hh>

#include "main.hh"
#include "debug.hh"
#include "paths.hh"
#include "network.hh"
#include "stock.hh"
#include "md5.h"
#include "util.hh"
#include "uri++.hh"

#define STATE(e) ((state & e) != 0)
#define SET_STATE(e) ((state |= e))
#define CLEAR_STATE(e) ((state &= ~ e))

namespace
{
  typedef std::map < std::string, std::string > StringMap;

  void
  parse_to_map (StringMap & map, std::string const& buffer)
  {
      using namespace Bmp;

      using boost::algorithm::split;
      using boost::algorithm::split_regex;
      using boost::algorithm::is_any_of;
      using boost::algorithm::find_nth;
      using boost::iterator_range;

      std::vector<std::string> lines;
      split_regex (lines, buffer, boost::regex ("\\\r?\\\n"));

      for (unsigned int n = 0; n < lines.size(); ++n)
        {
          std::string str = lines[n];
          iterator_range<std::string::iterator> match = find_nth (str, "=", 0);
          if (!match) continue;

          std::string str1 (str.begin(), match.begin());
          std::string str2 (match.end(), str.end());
          map[str1] = str2; 
          debug ("lastfm-radio", "%s=%s", str1.c_str(), str2.c_str());
        }
  }

  int
  read_block (void *data, const char *buf, size_t len)
  {
    std::string *str = (std::string *)data;
    str->append (buf, len);
    return 0;
  }
}

#define BMP_LASTFM_USERAGENT      "BMP2"

//-- LastFM Radio 
#define BMP_LASTFMRADIO_VERSION   "1.0.0.1" 
#define BMP_LASTFMRADIO_PLATFORM  "linux"
#define BMP_LASTFMRADIO_HOST      "ws.audioscrobbler.com"
#define BMP_LASTFMRADIO_HS_REQS   "/radio/handshake.php?version=%s&platform=%s&username=%s&passwordmd5=%s"

namespace Bmp
{
  namespace LastFM
  {
    void
    neon_status_check (int result, int status, std::string const& error_message, std::string const& response)
    {
      std::string message;

      debug ("lastfm-radio", "Response: %s", response.c_str());
      debug ("lastfm-radio", "Status: %d", status); 
      debug ("lastfm-radio", "Result: %d", result); 

      if (result != NE_OK)
          message = (error_message.empty() ? (response) : (error_message + " (" + response + ")"));
      else
          message = error_message;

      switch (result)
        {
          case NE_OK:
            break;

         case NE_CONNECT:
            throw LastFM::ConnectionError (message);

          case NE_TIMEOUT:
            throw LastFM::TimeOutError (message);

          case NE_AUTH:
            throw LastFM::AuthenticationError (message);

          default:
            throw LastFM::NetworkError (message);
        } 

      switch (status)
        {
          case 200:
              break;
          case 400:
              throw LastFM::RequestError (message);
          case 401:
              throw LastFM::AuthenticationError (message);
          case 404:
              throw LastFM::ResourceNotFoundError (message);
          default:
              throw LastFM::NetworkError (message);
        }
    }

    Radio::Radio () : m_connected (false)
    {
    }

    void Radio::connect ()
    {
      if (!Network::check_host (BMP_LASTFMRADIO_HOST, 80, true))
        {
          m_connected = false;
          throw LastFMNotConnectedError();
        }

      std::string        path; 
      std::string        response;

      std::string user = mcs->key_get<std::string>("lastfm", "username");
      std::string pass = mcs->key_get<std::string>("lastfm", "password");
      std::string pmd5 = Util::md5_hex ((char*)pass.c_str(), strlen(pass.c_str()));

      m_connected = false;

      static boost::format lastfmradio_hs_format (BMP_LASTFMRADIO_HS_REQS);
      path = (lastfmradio_hs_format % BMP_LASTFMRADIO_VERSION 
                                    % BMP_LASTFMRADIO_PLATFORM % user % pmd5).str();

      Neon::Request request (BMP_LASTFMRADIO_HOST, path);
      request.dispatch ();
      int result = request.dispatch (); 
      int status = request.get_status ()->code;
      std::string error_message = request.get_error (); 
      request >> response;
      neon_status_check (result, status, error_message, response);
      request.clear ();

      StringMap m;
      parse_to_map (m, response); 

      if (m.find( "session" ) != m.end ())
        {
          std::string session = m.find( "session" )->second;
          if (session == std::string("FAILED"))
          {
            m_connected = FALSE;
            throw LastFMBadauthError(_("Please check your username and password!"));
          }
        }

      if (m.find( "session" ) != m.end ())
          m_session.session = m.find ("session")->second;

      if (m.find( "stream_url" ) != m.end ())
          m_session.streamurl = m.find ("stream_url")->second;

      if (m.find( "subscriber" ) != m.end ())
          m_session.subscriber = bool(atoi(m.find ("subscriber")->second.c_str()));

      if (m.find( "framehack" ) != m.end ())
          m_session.framehack = bool(atoi(m.find ("framehack")->second.c_str()));

      if (m.find( "base_url" ) != m.end ())
          m_session.base_url = m.find ("base_url")->second;

      if (m.find( "base_path" ) != m.end ())
          m_session.base_path = m.find ("base_path")->second;

      m_connected = ( !m_session.streamurl.empty() && 
                      !m_session.session.empty() && 
                      !m_session.base_url.empty() && 
                      !m_session.base_path.empty()) ? true : false;

      if (!m_connected)
        throw LastFMNotConnectedError();
    }

    void Radio::session (Session& session) const
    {
      session = m_session;
    }

    void Radio::metadata (Metadata& metadata) const
    {
      if (!m_connected) throw LastFMNotConnectedError();

      std::string        path; 
      std::string        response;

      path = m_session.base_path;
      path += "/np.php?session=" + m_session.session;

      debug ("lastfm-radio", "Now-Playing Path: %s", path.c_str());

      Neon::Request request (m_session.base_url, path);
      request.dispatch ();
      int result = request.dispatch (); 
      int status = request.get_status ()->code;
      std::string error_message = request.get_error (); 
      request >> response;
      neon_status_check (result, status, error_message, response);
      request.clear ();

      StringMap m;
      parse_to_map (m, response); 

      if (m.find("station") != m.end ()) metadata.station = m.find ("station")->second;
      if (m.find("station_url") != m.end ()) metadata.station_url = m.find ("station_url")->second;
      if (m.find("artist") != m.end ()) metadata.artist = m.find ("artist")->second;
      if (m.find("artist_url") != m.end ()) metadata.artist_url = m.find ("artist_url")->second;
      if (m.find("track") != m.end ()) metadata.track = m.find ("track")->second;
      if (m.find("track_url") != m.end ()) metadata.track_url = m.find ("track_url")->second;
      if (m.find("album") != m.end ()) metadata.album = m.find ("album")->second;
      if (m.find("album_url") != m.end ()) metadata.album_url = m.find ("album_url")->second;
      if (m.find("albumcover_large") != m.end ()) metadata.albumcover_large = m.find ("albumcover_large")->second;
      if (m.find("trackduration") != m.end ()) metadata.duration = atoi(m.find ("trackduration")->second.c_str());
      if (m.find("radiomode") != m.end ()) metadata.radiomode = bool(atoi(m.find ("radiomode")->second.c_str()));
      if (m.find("streaming") != m.end ()) metadata.streaming = bool(m.find ("streaming")->second == "true");
      if (m.find("recordtoprofile") != m.end ()) metadata.rtp = bool(atoi(m.find ("recordtoprofile")->second.c_str()));

      debug ("lastfm-radio", "Now-Playing Response: %s", response.c_str());
    }

    void Radio::tune (std::string& url)
    {
      using namespace std;
    
      if (!m_connected)
        throw LastFMNotConnectedError();

      std::string path; 
      std::string response;

      char * url_escaped = ne_path_escape (url.c_str());

      path  = m_session.base_path;
      path  +=  string  ("/adjust.php?") 
            +   string  ("session=") 
            +   m_session.session 
            +   string  ("&url=") 
            +   string  (url_escaped); 
      g_free (url_escaped);

      debug ("lastfm-radio", "Tune-In Path: %s", path.c_str());

      Neon::Request request (m_session.base_url, path);
      request.dispatch ();
      int result = request.dispatch (); 
      int status = request.get_status ()->code;
      std::string error_message = request.get_error (); 
      request >> response;
      neon_status_check (result, status, error_message, response);
      request.clear ();

      debug ("lastfm-radio", "Tune-In Response: %s", response.c_str());

      StringMap m;
      parse_to_map (m, response); 
      if (m["response"] != std::string("OK"))
        {
          throw LastFMStreamTuningError();
        }
    }

    void Radio::command (Command command)
    {
      if (!m_connected) throw LastFMNotConnectedError();

      std::string        path; 
      std::string        response;

      const std::string commands[] =
      {
        "love",
        "skip",
        "ban",
        "rtp",
        "nortp"
      };

      path  =   m_session.base_path;

      path  += std::string ("/control.php?command=") 
            +  commands[command] 
            +  std::string ("&session=") 
            +  m_session.session;

      debug ("lastfm-radio", "Command Path: %s", path.c_str());

      Neon::Request request (m_session.base_url, path);
      request.dispatch ();
      int result = request.dispatch (); 
      int status = request.get_status ()->code;
      std::string error_message = request.get_error (); 
      request >> response;
      neon_status_check (result, status, error_message, response);

      debug ("lastfm-radio", "Command Response: %s", response.c_str());
    }

    Radio::~Radio ()
    {
    }

    bool Radio::connected () const
    {
      return m_connected;
    }

  }
}

//-- Last.FM Submissions

#define BMP_LASTFM_URI_HOST       "post.audioscrobbler.com"

#define RESP_UPTODATE		          "UPTODATE"
#define RESP_UPDATE		            "UPDATE"
#define RESP_FAILED		            "FAILED"
#define RESP_BADUSER		          "BADUSER"

#define BMP_LASTFM_CLIENT_ID      "mpx"
#define BMP_LASTFM_CLIENT_VERSION "1.0"

#define MILLISECOND (1000)

#define N_LQM         "lqm"
#define N_LQM_QUEUE   "lqm-queue"
#define N_LQM_TRACK   "lqm-track"

namespace
{
  Glib::ustring
  strip_ns (Glib::ustring const& name)
  {
    using boost::algorithm::split;
    using boost::algorithm::is_any_of;

    ::Bmp::StrV subs;
    split (subs, name, is_any_of (":"));

    if (subs.size() == 2)
      return Glib::ustring(subs[1]);
    else
      return Glib::ustring(subs[0]);
  }
}

namespace Bmp
{
  namespace LastFM
  {
    void
    LQMQueueParser::on_start_element  (Glib::Markup::ParseContext & context,
                                       Glib::ustring const& name_,
                                       AttributeMap  const& props)
	  {
      Glib::ustring name (strip_ns (name_));

      if (name == N_LQM) 
        {        
          SET_STATE(E_LQM);
          return;
        }

      if (name == N_LQM_QUEUE) 
        {        
          SET_STATE(E_QUEUE);
          return;
        }

      if (name == N_LQM_TRACK) 
        {        
          SET_STATE(E_TRACK);
      
          LQMItem item;
  
          item.mbid    = props.find ("mbid")->second;   
          item.date    = g_ascii_strtoull (props.find ("date")->second.c_str(), NULL, 10);
          item.length  = g_ascii_strtoull (props.find ("length")->second.c_str(), NULL, 10);

          item.artist  = props.find ("artist")->second;
          item.track   = props.find ("track")->second;
          item.album   = props.find ("album")->second;

          m_queue.push (item);

          return;
        }
    }

    void
    LQMQueueParser::on_end_element    (Glib::Markup::ParseContext& context,
                                       Glib::ustring const& name_)
	  {
      Glib::ustring name (strip_ns (name_));

      if (name == N_LQM) 
        {        
          CLEAR_STATE(E_LQM);
          return;
        }

      if (name == N_LQM_QUEUE) 
        {        
          CLEAR_STATE(E_QUEUE);
          return;
        }

      if (name == N_LQM_TRACK) 
        {        
          CLEAR_STATE(E_TRACK);
          return;
        }
    }

    void
    LQMQueueParser::check_sanity ()
    {
      if (state)
        {
          g_warning (G_STRLOC ": State should be 0, but is %d", state);
        }
    }

    void
    load_lqm (std::string const& filename, LQMQueue & queue)
    {
      using namespace Glib;

      if (!file_test (filename, FILE_TEST_EXISTS))
        return;

      try
        {
          std::string data (file_get_contents (filename));
          LQMQueueParser parser (queue);
          Markup::ParseContext context (parser);
          context.parse (data);
          context.end_parse ();
          parser.check_sanity ();
        }
      catch (ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
      catch (MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
      catch (...) {}
    }

    void
    save_lqm (std::string const& filename, LQMQueue const& queue)
    {
        xmlDocPtr   doc = xmlNewDoc   (BAD_CAST "1.0");
        xmlNodePtr  lqm = xmlNewNode  (0, BAD_CAST N_LQM); 

        xmlNsPtr    bmp = xmlNewNs    (lqm, BAD_CAST XML_NS_BMP, BAD_CAST "bmp");

        xmlSetProp (lqm, BAD_CAST "version", BAD_CAST "1.0");
        xmlDocSetRootElement (doc, lqm);

        xmlNodePtr  q = xmlNewChild (lqm, bmp, BAD_CAST N_LQM_QUEUE, 0);
       
        for (unsigned int n = 0 ; n < queue.size(); ++n)
        {
          static boost::format fllu ("%llu");
          static boost::format fu ("%u");

          LQMItem const& i (queue.get_nth (n));
      
          xmlNodePtr n = xmlNewChild (q, bmp, BAD_CAST N_LQM_TRACK, 0); 

          xmlSetProp (n, BAD_CAST "mbid",
                         BAD_CAST i.mbid.c_str()); 

          xmlSetProp (n, BAD_CAST "artist",
                         BAD_CAST i.artist.c_str()); 

          xmlSetProp (n, BAD_CAST "track",
                         BAD_CAST i.track.c_str()); 

          xmlSetProp (n, BAD_CAST "album",
                         BAD_CAST i.album.c_str()); 

          xmlSetProp (n, BAD_CAST "date",
                         BAD_CAST (fllu % i.date).str().c_str()); 

          xmlSetProp (n, BAD_CAST "length",
                         BAD_CAST (fu % i.length).str().c_str()); 
      }

      xmlCreateIntSubset (doc,
                          BAD_CAST "lqm",
                          NULL,
                          BAD_CAST "http://bmpx.beep-media-player.org/dtd/lqm-1.dtd" );

      xmlThrDefIndentTreeOutput (1);
      xmlKeepBlanksDefault (0);
      g_message ("Saving LQM to %s", filename.c_str());
      int ret = xmlSaveFormatFileEnc (filename.c_str(), doc, "utf-8", 1);
      g_message ("Saved %d characters", ret);
      xmlFreeDoc (doc);
    }

    //////////////////////////////////////////////////////////////////////////////////////////

    Scrobbler::Scrobbler()
        : m_interval_reached (false), handshaked (false), enabled (false)
    {
      using namespace Glib;

      for (unsigned int i = 0; i < N_HANDLERS; handler[i++] = 0);

      std::string filename = build_filename (BMP_PATH_USER_DIR, "lastfm.lqm");
      load_lqm (filename, items);
      mcs->subscribe ("LastFM",
                      "lastfm",
                      "queue-enable",
                      sigc::mem_fun (this,
                      &::Bmp::LastFM::Scrobbler::queue_enable));
    }

    Scrobbler::~Scrobbler ()
    {
      using namespace Glib;

      Glib::Mutex::Lock lc (lock_conn_items);
      Glib::Mutex::Lock lg (lock_m_interval_reached);
      Glib::Mutex::Lock lq (lock_queue);

      conn_m_interval_reached.
        disconnect ();
      conn_items.
        disconnect ();
      save_lqm (build_filename (BMP_PATH_USER_DIR, "lastfm.lqm"), items);
    }

    void
    Scrobbler::queue_enable (std::string const&	 domain,
		                       std::string const&	 key,
                           Mcs::KeyVariant const& value) 
    {
      if (boost::get<bool>(value))
        {
          debug ("lastfm", "Connecting handlers.");

          mcs->subscribe ("LastFM",
                          "lastfm",
                          "submit-enable",
          sigc::mem_fun (this, &::Bmp::LastFM::Scrobbler::submit_enable));

          if (mcs->key_get<bool>("lastfm", "submit-enable") && !handshaked)
            run_handshake ();
        }
      else
        {
          debug ("lastfm", "Disconnecting handlers.");

          if (handler[H_LASTFM_ENABLE])
            mcs->unsubscribe ("LastFM", "lastfm", "submit-enable");

          lock_m_interval_reached.lock ();
          if (conn_m_interval_reached.connected ()) conn_m_interval_reached.disconnect ();
          lock_m_interval_reached.unlock ();

          lock_conn_items.lock (); 
          if (conn_items.connected ()) conn_items.disconnect ();
          lock_conn_items.unlock ();

          handshaked = false;
      }
    }

    Scrobbler::HandshakeStatus
    Scrobbler::handshake ()
    {
        static boost::format path ("/?hs=true&p=1.1&c=%s&v=%s&u=%s");

        Glib::ustring		   user;

        std::vector<std::string>  resp, line;	    
        std::string               response;
        Scrobbler::HandshakeStatus  handshake_status = LASTFM_HANDSHAKE_STATE_UNDEFINED;
    
        user = mcs->key_get<std::string>("lastfm", "username");

        Neon::Request request (BMP_LASTFM_URI_HOST,
          (path % BMP_LASTFM_CLIENT_ID % BMP_LASTFM_CLIENT_VERSION % user).str());

        request.dispatch ();
        int result = request.dispatch (); 
        int status = request.get_status ()->code;
        std::string error_message = request.get_error (); 
        request >> response;
        neon_status_check (result, status, error_message, response);

        boost::algorithm::split (resp, response, boost::algorithm::is_any_of("\n"));
        boost::algorithm::split (line, resp[0], boost::algorithm::is_any_of(" "));

        if (!line[0]. compare(RESP_UPTODATE))
          {
            info.uptodate = true;
            info.md5 = resp[1];
            info.uri = resp[2];

            boost::algorithm::split (line, resp[3], boost::algorithm::is_any_of(" "));
            if (line. size() >= 2) info.interval = std::atoi (line[1]. c_str ());
            handshake_status = LASTFM_HANDSHAKE_UPTODATE;
          }
        else
        if (!line[0]. compare(RESP_UPDATE))
          {
            info.uptodate = false;
            info.info = resp[0];
            info.md5  = resp[1];
            info.uri  = resp[2];

            boost::algorithm::split (line, resp[3], boost::algorithm::is_any_of(" "));
            if (line. size() >= 2) info.interval = std::atoi (line[1]. c_str ());
            handshake_status = LASTFM_HANDSHAKE_UPDATE;
        }
      else
      if (!line[0]. compare(RESP_BADUSER))
        {
            boost::algorithm::split (line, resp[1], boost::algorithm::is_any_of(" "));
            if (line. size() >= 2) info.interval = std::atoi (line[1]. c_str ());
            handshake_status = LASTFM_HANDSHAKE_BADUSER;
        }
      else
      if (!line[0]. compare(RESP_FAILED))
        {
            info.info = resp[0];
            boost::algorithm::split (line, resp[1], boost::algorithm::is_any_of(" "));
            if (line. size() >= 2) info.interval = std::atoi (line[1]. c_str ());
            handshake_status = LASTFM_HANDSHAKE_FAILED;
        }

      if ((handshake_status == LASTFM_HANDSHAKE_BADUSER) || (handshake_status == LASTFM_HANDSHAKE_FAILED))
        {
            boost::algorithm::split (line, resp[1], boost::algorithm::is_any_of(" "));
            interval_reached_timeout (atoi(line[1]. c_str()));
        }
      else
        {
            boost::algorithm::split (line, resp[3], boost::algorithm::is_any_of(" "));
            interval_reached_timeout (atoi(line[1]. c_str()));
        }
   
      return handshake_status;
    }
    
    void
    Scrobbler::run_handshake ()
    {
        Scrobbler::HandshakeStatus handshake_result;
      	Glib::ustring user, pass;
    
        if (handshaked) return;
    
        user = mcs->key_get<std::string>("lastfm", "username");
        pass = mcs->key_get<std::string>("lastfm", "password");

        if (user.empty () || pass.empty ())
          {
            signal_submit_error_.emit(_("Authentication Error: Password or username empty"));
            return;
          }
   
        try { 
          handshake_result = handshake ();
          }
        catch (LastFM::ConnectionError& cxe)
          {
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "ConnectionError: %s", cxe.what());
            signal_submit_error_.emit(_("Connection Error: ") + Glib::ustring(cxe.what()));
            return;
          }
        catch (LastFM::TimeOutError& cxe)
          {
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "TimeOutError: %s", cxe.what());
            signal_submit_error_.emit(_("Timeout Error: ") + Glib::ustring(cxe.what()));
            return;
          }
        catch (LastFM::AuthenticationError& cxe)
          {
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "AuthenticationError: %s", cxe.what());
            signal_submit_error_.emit(_("Authentication Error: ") + Glib::ustring(cxe.what()));
            return;
          }
        catch (LastFM::NetworkError& cxe)
          {
            g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "NetworkError: %s", cxe.what());
            signal_submit_error_.emit(_("Network Error: ") + Glib::ustring(cxe.what()));
            return;
          }
    
        switch (handshake_result)
            {
              case LASTFM_HANDSHAKE_UPTODATE:
                  {
                    md5_state_t  md5state;
                    char         md5pword[16];
                    char	      *str;
                    char	      *pass, *aux;

                    g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                            "%s: UPTODATE. MD5:'%s', URL:'%s', INTERVAL:'%d'", G_STRLOC,
                            info.md5.c_str (),
                            info.uri.c_str (),
                            info.interval);

                    pass = g_strdup (mcs->key_get<std::string>("lastfm", "password"). c_str ());

                    md5_init (&md5state);
                    md5_append (&md5state, (unsigned const char *)pass, std::strlen(pass));
                    md5_finish (&md5state, (unsigned char *)md5pword);
                    aux = Util::hexify (md5pword, sizeof(md5pword));

                    str = g_strconcat (aux, info.md5.c_str (), NULL);
                    md5_init (&md5state);
                    md5_append (&md5state, (unsigned const char *)str, std::strlen(str));
                    md5_finish (&md5state, (unsigned char *)md5pword);

                    md5_response = Util::hexify (md5pword, sizeof(md5pword));

                    lock_conn_items.lock (); 
                    if (!conn_items.connected ())
                      {
                        g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                          "%s: Restarting queue processing", G_STRLOC);
                        conn_items = Glib::signal_timeout().connect
                          (sigc::mem_fun (this, &Bmp::LastFM::Scrobbler::process_queue ), 2000);
                      } 
                    lock_conn_items.unlock (); 
              
                    g_free (aux);
                    g_free (str);
                    g_free (pass);
                    break;
                  }

              case LASTFM_HANDSHAKE_UPDATE:
                  {
                    md5_state_t  md5state;
                    char         md5pword[16];
                    char	*str;
                    char	*pass, *aux;

                    g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                            "%s: UPDATE. UPDATE_URL:'%s', MD5:'%s', URL:'%s', INTERVAL:'%d'", G_STRLOC,
                            info.info.c_str (),
                            info.md5.c_str (),
                            info.uri.c_str (),
                            info.interval);

                    pass = g_strdup (mcs->key_get<std::string>("lastfm", "password"). c_str ());

                    md5_init (&md5state);
                    md5_append (&md5state, (unsigned const char *)pass, std::strlen(pass));
                    md5_finish (&md5state, (unsigned char *)md5pword);
                    aux = Util::hexify (md5pword, sizeof(md5pword));

                    str = g_strconcat (aux, info.md5.c_str (), NULL);
                    md5_init (&md5state);
                    md5_append (&md5state, (unsigned const char *)str, std::strlen(str));
                    md5_finish (&md5state, (unsigned char *)md5pword);

                    md5_response = Util::hexify (md5pword, sizeof(md5pword));

                    lock_conn_items.lock (); 
                    if (!conn_items.connected ())
                      {
                        g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                          "%s: Restarting queue processing", G_STRLOC);
                        conn_items = Glib::signal_timeout().connect
                          (sigc::mem_fun (this, &Bmp::LastFM::Scrobbler::process_queue ), 2000);
                      }
                    lock_conn_items.unlock (); 

                    g_free (aux);
                    g_free (str);
                    g_free (pass);
                    break;
                  }

              case LASTFM_HANDSHAKE_BADUSER:
                  {
                    g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                      "%s: BADUSER. INTERVAL:'%d'", G_STRLOC, info.interval);
                    signal_submit_error_.emit(_("Invalid Credentials Provided!"));
                    return;
                  }

              case LASTFM_HANDSHAKE_FAILED:
                  {
                    g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                      "%s: FAILED. REASON:'%s', INTERVAL:'%d'", G_STRLOC, info.info.c_str (), info.interval);
                    signal_submit_error_.emit(_("Handshake Failed: ") + Glib::ustring (info.info));
                    return;
                  }

              case LASTFM_HANDSHAKE_STATE_UNDEFINED:
                  {
                    g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                      "%s: STATE UNDEFINED", G_STRLOC);
                    signal_submit_error_.emit(_("Unknown Error during Handshake"));
                    return;
                  }

              default:
                  {
                    g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_WARNING,
                      "%s: Unknown LastFM handshake status response!", G_STRLOC);
                    return;
                  }
            }
    
        handshaked = true;
    }

    bool
    Scrobbler::process_queue ()
    {
      if (items.empty())
        return false;

      if (!handshaked) 
	      {
	        conn_m_interval_reached.disconnect ();
	        return false;
      	}

      lock_m_interval_reached.lock();
      if (!m_interval_reached)
	      {
	        g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s: Interval not reached yet: cannot send", G_STRLOC);
          lock_m_interval_reached.unlock();
	        return false;
	      }
      lock_m_interval_reached.unlock();

      signal_submit_start_.emit ();
      process_queue_real ();
      signal_submit_end_.emit ();

      signal_queue_size_.emit(items.size());

      return (!items.empty ());
    }

    void
    Scrobbler::process_queue_real ()
    {
      using namespace Glib;

      static boost::format f_queue_head ("u=%s&s=%s&");
      static boost::format f_queue_item ("a[%u]=%s&t[%u]=%s&b[%u]=%s&m[%u]=%s&l[%u]=%u&i[%u]=%s");

      Mutex::Lock L (lock_queue);

      ::Bmp::StrV resp, line; 
      std::string response;
      std::string postdata;
  
      unsigned int send_items = (items.size() >= 10) ? 10 : items.size();

      postdata.append ((f_queue_head  % mcs->key_get<std::string>("lastfm", "username")
                                      % md5_response ).str());

      for (unsigned int n = 0 ; n < send_items ; ++n)
        {
          LQMItem const& item (items.get_nth (n));
          postdata.append ((f_queue_item  % n % Neon::Util::escape (item.artist)
                                          % n % Neon::Util::escape (item.track)
                                          % n % Neon::Util::escape (item.album)
                                          % n % item.mbid
                                          % n % item.length
                                          % n % Neon::Util::escape (Util::get_timestr (item.date, 1))).str());

          if ( n != (send_items-1))
            postdata.append ("&");
        }
                  
      g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s: Submitting Songs with POST data '%s'", G_STRLOC, postdata.c_str ());

      URI uri (info.uri);

      Neon::Request request ( uri.hostname,
                              uri.path,
                              80,
                              Neon::Request::RESPONSE_READ,
                              Neon::Request::METHOD_POST    );

      request.add_request_header (Neon::Request::HEADER_WWWFORM);
      request.set_request_data (postdata.c_str(), postdata.size());
      request.dispatch ();

      int result = request.dispatch (); 
      int status = request.get_status ()->code;
      std::string error_message (request.get_error ());

      request >> response;
      request.clear ();

      try {
        neon_status_check (result, status, error_message, response);
        } 
      catch (LastFM::ConnectionError& cxe)
        {
          signal_submit_error_.emit(_("Connection Error"));
          return;
        }
      catch (LastFM::TimeOutError& cxe)
        {
          signal_submit_error_.emit(_("Connection Timeout"));
          return;
        }
      catch (LastFM::AuthenticationError& cxe)
        {
          signal_submit_error_.emit(_("Authentication Error: ") + Glib::ustring(cxe.what()));
          return;
        }
      catch (LastFM::RequestError& cxe)
        {
          signal_submit_error_.emit(_("Error Posting Request"));
          return;
        }
      catch (LastFM::ResourceNotFoundError& cxe)
        {
          signal_submit_error_.emit(_("Resource Not Found (HTTP 404 Status)"));
          return;
        }
      catch (LastFM::NetworkError& cxe)
        {
          signal_submit_error_.emit(_("Network Error: ") + Glib::ustring(cxe.what()));
          return;
        }

      boost::algorithm::split (resp, response, boost::algorithm::is_any_of("\n"));

      if (resp[0] == "BADAUTH")
	      {
          signal_submit_error_.emit(_("Authentication Error (Please Check Credentials)"));
          return;
	      }
      else
      if (resp[0].substr(0,6) == "FAILED")
	      {
	        g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                  "%s: Submitting Songs failed: %s", G_STRLOC, resp[0].c_str());
	      }
      else // OK assumed
	      {
	        g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                  "%s: Submitting Songs succeeded.", G_STRLOC);

          for (unsigned int n = 0 ; n < send_items ; ++n)
            items.pop ();
          save_lqm (build_filename (BMP_PATH_USER_DIR, "lastfm.lqm"), items);
	      }
      boost::algorithm::split (line, resp[1], boost::algorithm::is_any_of(" "));
      interval_reached_timeout (std::atoi(line[1]. c_str()));
    }

    void
    Scrobbler::interval_reached_timeout (int seconds)
    {
      Glib::Mutex::Lock lock (lock_m_interval_reached);

      if (conn_m_interval_reached.connected ()) return;

      g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                  "%s: INTERVAL in %d seconds", G_STRLOC, seconds);
      conn_m_interval_reached = Glib::signal_timeout().connect
                  (sigc::mem_fun (this, &Bmp::LastFM::Scrobbler::interval_reached ), seconds * MILLISECOND);
    }

    bool
    Scrobbler::interval_reached ()
    {
      Glib::Mutex::Lock lock (lock_m_interval_reached);
      m_interval_reached = true;
      return false;
    }

    void
    Scrobbler::submit_enable (std::string const&     domain,
                            std::string const&     key,
                            const Mcs::KeyVariant&  value)
    {
      if (boost::get<bool>(value))
          run_handshake ();
      else
          handshaked = false;
    }

    void
    Scrobbler::send_song_information (Glib::ustring const& uri)
    {
      using namespace Glib;

      Mutex::Lock lq (lock_queue);
      debug ("lastfm", "Sending song information.");

      Library::Track track;
      try
        {
          debug ("lastfm", "send_song_information: %s", uri.c_str());
          library->get (uri.c_str(), track);
        }
      catch (Bmp::Library::Exception& cxe)
        {
          g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Library Error: %s", cxe.what());
        }

      if (track.duration && track.duration.get() < 30)
        {
          g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                 "%s: Not submitting '%s' to lastfm: duration of less than 30 seconds", G_STRLOC, uri.c_str());
          return;
        }

      if (!track.duration || !track.artist || !track.title)
        {
          g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO,
                 "%s: Not submitting '%s' to lastfm: missing metadata!", G_STRLOC, uri.c_str());
          return;
        }

      g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s: Pushing '%s' onto the queue", G_STRLOC, uri.c_str());

      LQMItem item;
      item.mbid     = track.mb_track_id?  track.mb_track_id.get() : ustring();
      item.artist   = track.artist?       track.artist.get()      : ustring();
      item.track    = track.title?        track.title.get()       : ustring();
      item.album    = track.album?        track.album.get()       : ustring();
      item.date     = time (NULL);
      item.length   = track.duration? track.duration.get () : 0;

      items.push (item);
      signal_queue_size_.emit(items. size());

      lock_conn_items.lock (); 
      if ((!conn_items.connected ()) && mcs->key_get<bool>("lastfm", "submit-enable"))
        {
          g_log (BMP_LASTFM_LOG_DOMAIN, G_LOG_LEVEL_INFO, "%s: Restarting queue processing", G_STRLOC);
          conn_items = signal_timeout().connect
              (sigc::mem_fun (this, &Bmp::LastFM::Scrobbler::process_queue), 2000);
        }
      lock_conn_items.unlock (); 
    }

    void
    Scrobbler::emit_queue_size ()
    {
      signal_queue_size_.emit (items. size());
    }
  }
}

namespace Bmp
{
  namespace LastFM
  {
    void
    get_tags (TagType type,
              TagOrigin origin,
              Tags & tags,
              Glib::ustring const& username,
              Glib::ustring const& id1,
              Glib::ustring const& id2)
    {
      using namespace Glib;

      ustring uri;
      switch (origin)
        {
          case TAG_ORIGIN_USER:
            {
              uri.append( "http://ws.audioscrobbler.com/1.0/user/" ).append( username );
              switch (type)
              {
                  case TAGS_TOPTAGS:
                      uri.append( "/tags.xml" );
                      break;

                  case TAGS_ARTIST:
                      uri.append( "/artisttags.xml" ).append( "?artist=" ).append ( id1 );
                      break;

                  case TAGS_ALBUM:
                      uri.append( "/albumtags.xml" ).append( "?artist=" ).append ( id1 ).append( "&album=" ).append( id2 );
                      break;

                  case TAGS_TRACK:
                      uri.append( "/tracktags.xml" ).append( "?artist=" ).append ( id1 ).append( "&track=" ).append( id2 );
                      break;
              }
              break;
            }

          case TAG_ORIGIN_GLOBAL:
            {
              uri.append( "http://ws.audioscrobbler.com/1.0/tag" );
              switch (type)
              {
                  case TAGS_TOPTAGS:
                      uri.append( "/toptags.xml" );
                      break;

                  default: return;

            }
            break;
          }
      }


      Bmp::URI u (uri, true);
      try
        {
          Neon::Request request (u.hostname, u.path);
          std::string response;
          request >> response;
          request.clear ();
          TagParser parser (tags, type, origin);
          Markup::ParseContext context (parser);
          context.parse (response);
          context.end_parse ();
        }
      catch (Neon::Request::Exception& cxe)
        {
          //FIXME: Neon exception
        }
      catch (Glib::ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
     catch (Glib::MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
    }
  
    void
    TagParser::on_start_element  (Glib::Markup::ParseContext & context,
                                  Glib::ustring const& element_name,
                                  AttributeMap const& attributes)
	  {
      if (element_name == "tag")
      {
        state |= E_TAG; 
        current = Tag();
        if (m_type == TAGS_TOPTAGS && m_origin == TAG_ORIGIN_GLOBAL) 
          {
            current.name = attributes.find( "name" )->second; 
            current.count = g_ascii_strtoull (attributes.find( "count" )->second.c_str(), NULL, 10);
            current.url = attributes.find( "url" )->second; 
          }
      }

      if (element_name == "name")
      {
        state |= E_NAME; 
      }

      if (element_name == "count")
      {
        state |= E_COUNT; 
      }

      if (element_name == "url")
      {
        state |= E_URL; 
      }
    }

    void
    TagParser::on_end_element    (Glib::Markup::ParseContext& context,
                                      Glib::ustring const& element_name)
	  {
      if (element_name == "tag")
      {
        state &= ~E_TAG; 
        tags.push_back (current);
      }

      if (element_name == "name")
      {
        state &= ~E_NAME; 
      }

      if (element_name == "count")
      {
        state &= ~E_COUNT; 
      }

      if (element_name == "url")
      {
        state &= ~E_URL; 
      }
    }

    void
    TagParser::on_text  (Glib::Markup::ParseContext & context,
                         Glib::ustring const& text)
	  {
      if (state & E_TAG)
      {
        if (state & E_NAME)
        {
          current.name = text;
          return;
        }

        if (state & E_COUNT)
        {
          current.count = atoi (text.c_str()); 
          return;
        }

        if (state & E_URL)
        {
          current.url = text; 
          return;
        }
      }
    }
  }
}

namespace Bmp
{
  namespace LastFM
  {
    void
    get_neighbours (Glib::ustring const& username, Friends & r)
	                 
    {
      using namespace Glib;

      ustring uri;
      uri .append ("http://ws.audioscrobbler.com/1.0/user/")
          .append (username)
          .append ("/neighbours.xml");

      Bmp::URI u (uri, true);
      try
        {
          Neon::Request request (u.hostname, u.path);
          std::string response;
          request >> response;
          request.clear ();
          FriendParser parser (r);
          Markup::ParseContext context (parser);
          context.parse (response);
          context.end_parse ();
        }
      catch (Neon::Request::Exception& cxe)
        {
          //FIXME: Neon exception
        }
      catch (Glib::ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
     catch (Glib::MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
    }

    void
    get_friends (Glib::ustring const& username, Friends & r)
	              
    {
      using namespace Glib;

      ustring uri;
      uri .append ("http://ws.audioscrobbler.com/1.0/user/")
          .append (username)
          .append ("/friends.xml");

      Bmp::URI u (uri, true);
      try
        {
          Neon::Request request (u.hostname, u.path);
          std::string response;
          request >> response;
          request.clear ();
          FriendParser parser (r);
          Markup::ParseContext context (parser);
          context.parse (response);
          context.end_parse ();
        }
      catch (Neon::Request::Exception& cxe)
        {
          //FIXME: Neon exception
        }
      catch (Glib::ConvertError& cxe)
        {
          //FIXME: Glib convert error (ustring) 
        }
     catch (Glib::MarkupError& cxe)
        {
          //FIXME: Glib Markup error
        }
    }
  
    void
    FriendParser::on_start_element  (Glib::Markup::ParseContext& context,
                                     Glib::ustring const& element_name,
                                     const AttributeMap& attributes)
	  {
      if (element_name == "user")
      {
        state |= E_USER;
        current = Friend();
        current.username = (*(attributes.find ("username"))).second;
        return;
      }

      if (element_name == "url")
      {
        if (!(state & E_USER)) return;
        state |= E_URL;
        return;
      }

      if (element_name == "image")
      {
        if (!(state & E_USER)) return;
        state |= E_IMAGE;
        return;
      }
    }

    void
    FriendParser::on_end_element    (Glib::Markup::ParseContext& context,
                                      Glib::ustring const& element_name)
	  {
        if (element_name == "user")
        {
          state &= ~ E_USER;
          if (state)
          {
            g_warning (G_STRLOC ": State should be 0, but is %d", state);
            state = 0;
          }
          friends.push_back (current);
          return;
        }

        if (element_name == "url")
        {
          state &= ~ E_URL;
          return;
        }

        if (element_name == "image")
        {
          state &= ~ E_IMAGE;
          return;
        }
    }

    void
    FriendParser::on_text  (Glib::Markup::ParseContext& context,
                            Glib::ustring const& text)
	  {
        if ((state & E_USER) && (state & E_URL))
        {
          current.url = text;
          return;
        }

        if ((state & E_USER) && (state & E_IMAGE))
        {
          current.image = text;
          return;
        }
    }
  }
}

namespace Bmp
{
  namespace LastFM
  {
      RecommendDialog*
      RecommendDialog::create ()
      {
        const std::string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "dialog-lastfm-recommend.glade");
        Glib::RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
        RecommendDialog *i = 0;
        glade_xml->get_widget_derived ("recommend", i);
        return i;
      }

      RecommendDialog::~RecommendDialog () {}
      RecommendDialog::RecommendDialog (BaseObjectType*                       cobject,
                                        const Glib::RefPtr<Gnome::Glade::Xml> &xml)
            : Gtk::Dialog (cobject),
              m_ref_xml     (xml)
      {
        m_ref_xml->get_widget ("e_artist", e_artist);
        m_ref_xml->get_widget ("e_album",  e_album);
        m_ref_xml->get_widget ("e_title",  e_title);

        m_ref_xml->get_widget ("l_artist", l_artist);
        m_ref_xml->get_widget ("l_album",  l_album);
        m_ref_xml->get_widget ("l_title",  l_title);

        m_ref_xml->get_widget ("cbox_recommend",  cbox_recommend);
        m_ref_xml->get_widget ("view_notes",  view_notes);

        m_ref_xml->get_widget ("notebook", notebook);
        m_ref_xml->get_widget ("image", image);
        image->set (Glib::build_filename (BMP_IMAGE_DIR, BMP_THROBBER));

        m_ref_xml->get_widget ("image_lastfm", image_lastfm);
        image_lastfm->set (Gtk::StockID (BMP_STOCK_LASTFM), Gtk::ICON_SIZE_SMALL_TOOLBAR);

        m_ref_xml->get_widget ("treeview", treeview);
        treeview->set_headers_clickable (false);
        treeview->set_headers_visible (false);

        Gtk::CellRendererPixbuf *cell_pixbuf = Gtk::manage (new Gtk::CellRendererPixbuf ());
        cell_pixbuf->property_ypad () = 3;
        cell_pixbuf->property_xpad () = 3;
        treeview->append_column ("Avatar", *cell_pixbuf);
        treeview->get_column (0)->add_attribute (*cell_pixbuf, "pixbuf", 0);
        treeview->get_column (0)->set_resizable (false);

        Gtk::CellRendererText *cell_text = Gtk::manage (new Gtk::CellRendererText ());
        cell_text->property_yalign() = 0.5; 
        treeview->append_column ("Name", *cell_text);
        treeview->get_column (1)->add_attribute (*cell_text, "markup", 1);
        treeview->get_column (1)->set_resizable (false);

        notebook->set_current_page (0);
        cbox_recommend->signal_changed().connect
              (sigc::mem_fun (this, &Bmp::LastFM::RecommendDialog::recommend_changed));
      }

      void
      RecommendDialog::recommend_changed ()
      {
        int current = cbox_recommend->get_active_row_number();

        e_artist->set_sensitive (false);
        e_album->set_sensitive (false);
        e_title->set_sensitive (false);

        l_artist->set_sensitive (false);
        l_album->set_sensitive (false);
        l_title->set_sensitive (false);

        switch (current)
        {
          case 0:
          {
            e_artist->set_sensitive (true);
            l_artist->set_sensitive (true);
            break;
          }

          case 1:
          {
            e_artist->set_sensitive (true);
            l_artist->set_sensitive (true);

            e_album->set_sensitive (true);
            l_album->set_sensitive (true);
            break;
          }

          case 2:
          {
            e_artist->set_sensitive (true);
            l_artist->set_sensitive (true);

            e_title->set_sensitive (true);
            l_title->set_sensitive (true);
            break;
          }
        }
      }

      void
      RecommendDialog::update_view ()
      {
        using namespace Gtk;
        using namespace Glib;

        notebook->set_current_page (1);
      
        liststore = ListStore::create (friends_cr); 
        liststore->set_default_sort_func
          (sigc::mem_fun (this, &Bmp::LastFM::RecommendDialog::sort_func));
        liststore->set_sort_column (Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID,
                          Gtk::SORT_ASCENDING);
        treeview->set_model (liststore);

        //Get Friends list

        Friends friends;
        get_friends (mcs->key_get<std::string>("lastfm", "username"), friends); 

        for (LastFM::Friends::const_iterator i = friends.begin (); i != friends.end(); ++i)
        {
            LastFM::Friend const& f = (*i); 
            TreeModel::iterator m_iter = liststore->append();
   
            RefPtr<Gdk::Pixbuf> image = Util::get_image_from_uri (f.image);

            if (image)
                (*m_iter)[friends_cr.avatar] = image->scale_simple (64, 64, Gdk::INTERP_BILINEAR);

            ustring text;
            text  .append ("<b>")
                  .append (f.username)
                  .append ("</b>\n<i>")
                  .append (f.url)
                  .append ("</i>");
            (*m_iter)[friends_cr.name] = text; 
            (*m_iter)[friends_cr.username] = f.username; 
            while (gtk_events_pending()) gtk_main_iteration();
          }
        cbox_recommend->set_active (0);
        notebook->set_current_page (0);
      }

      int
      RecommendDialog::sort_func (Gtk::TreeModel::iterator const& m_iter1,
                                  Gtk::TreeModel::iterator const& m_iter2)
      {
        using namespace Glib;
      
        ustring name1 ((*m_iter1)[friends_cr.username]);
        ustring name2 ((*m_iter2)[friends_cr.username]);

        return name1.compare (name2);
      }

      void
      RecommendDialog::run (Glib::ustring const& artist,
                            Glib::ustring const& album,
                            Glib::ustring const& title)
      {
        using namespace Gtk;

        e_artist->set_text  (artist);
        e_album->set_text   (album);
        e_title->set_text   (title);
      
        show ();
        update_view ();
        if (Dialog::run() == GTK_RESPONSE_OK)
          {
            try {
                recommend ();
              }
            catch (LastFM::Exception& cxe)
              {
                MessageDialog dialog (cxe.what(), false,
                  MESSAGE_ERROR, BUTTONS_OK, true);
                dialog.run ();
              }
          }
        view_notes->get_buffer()->set_text("");
        hide ();
      }

      void
      RecommendDialog::recommend ()
      {
          using namespace Gtk;
          using namespace Glib;
          using namespace std;

          static ustring param_pre  = "<param><value><string>";
          static ustring param_post = "</string></value></param>";

          string username = mcs->key_get<std::string>("lastfm", "username"); 
          string password = mcs->key_get<std::string>("lastfm", "password"); 
          string passmd5  = 
            Util::md5_hex ((char*)password.c_str(), strlen(password.c_str())); 

          ustring artist = e_artist->get_text ();
          ustring album  = e_album->get_text ();
          ustring title  = e_title->get_text ();

          int type = cbox_recommend->get_active_row_number ();

          ustring challenge = "12345678"; 
          ustring xmldata;

          TreeModel::iterator m_iter =
            treeview->get_selection()->get_selected();
          ustring user_to = (*m_iter)[friends_cr.username];
         
          const char * reply = (passmd5+challenge).c_str();
          string md5reply = Bmp::Util::md5_hex ((char*)reply, strlen(reply));

          switch (type)
          {
              case 0:
                {
                  xmldata = "<?xml version=\"1.0\"?><methodCall><methodName>"
                            "recommendArtist</methodName><params>";
                  xmldata.append (param_pre).append (username)
                      .append (param_post);
                  xmldata.append (param_pre).append (challenge)
                      .append (param_post);
                  xmldata.append (param_pre).append (md5reply)
                      .append (param_post);
                  xmldata.append (param_pre).append (artist)
                      .append (param_post);
                  xmldata.append (param_pre).append (user_to)
                      .append (param_post);
                  xmldata.append (param_pre).append
                      (view_notes->get_buffer()->get_text(false))
                      .append (param_post);
                  xmldata.append ("</params></methodCall>");
                  break;
                }

              case 1:
                {
                  xmldata = "<?xml version=\"1.0\"?><methodCall><methodName>"
                        "recommendAlbum</methodName><params>";
                  xmldata.append (param_pre).append (username)
                      .append (param_post);
                  xmldata.append (param_pre).append (challenge)
                      .append (param_post);
                  xmldata.append (param_pre).append (md5reply)
                      .append (param_post);
                  xmldata.append (param_pre).append (artist)
                      .append (param_post);
                  xmldata.append (param_pre).append (album)
                      .append (param_post);
                  xmldata.append (param_pre).append (user_to)
                      .append (param_post);
                  xmldata.append (param_pre).append
                      (view_notes->get_buffer()->get_text(false))
                      .append (param_post);
                  xmldata.append ("</params></methodCall>");
                  break;
                }

              case 2:
                {
                  xmldata = "<?xml version=\"1.0\"?><methodCall><methodName>"
                        "recommendTrack</methodName><params>";
                  xmldata.append (param_pre).append (username)
                        .append (param_post);
                  xmldata.append (param_pre).append (challenge)
                        .append (param_post);
                  xmldata.append (param_pre).append (md5reply)
                        .append (param_post);
                  xmldata.append (param_pre).append (artist)
                        .append (param_post);
                  xmldata.append (param_pre).append (title)
                        .append (param_post);
                  xmldata.append (param_pre).append (user_to)
                        .append (param_post);
                  xmldata.append (param_pre).append
                     (view_notes->get_buffer()->get_text(false))
                      .append (param_post);
                  xmldata.append ("</params></methodCall>");
                  break;
                }
          }

          Neon::Request request ("ws.audioscrobbler.com",
                                 "/1.0/rw/xmlrpc.php",
                                 Neon::HttpPort(80),
                                 Neon::Request::RESPONSE_IGNORE,
                                 Neon::Request::METHOD_POST);

          request.add_request_header ("Content-Type", "text/xml");
          request.set_request_data (xmldata.c_str(), (size_t)xmldata.size());
          request.dispatch ();

          int result = request.dispatch (); 
          int status = request.get_status ()->code;
          std::string error_message (request.get_error ());
          request.clear ();
          neon_status_check (result, status, error_message);
      }
  }
}
