//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <revision.h>
#include <build.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include <locale.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus.h>

#include <bmp/dbus.hh>
#include <src/paths.hh>

static GMainLoop *mainloop = NULL; 

static gboolean arg_show_version    = FALSE;
static gboolean arg_play_lastfm     = FALSE;
static gboolean arg_add_files       = FALSE;
static gboolean arg_add_uris        = FALSE;
static gboolean arg_playback        = FALSE;
static gboolean arg_no_network      = FALSE;

static GOptionEntry options[] =
  {
    {"version",     'v', 0, G_OPTION_ARG_NONE, &arg_show_version, N_("Display version"), NULL},
    {"add-files",   'a', 0, G_OPTION_ARG_NONE, &arg_add_files,    N_("Add Files to playlist"), NULL},
    {"add-uris",    'u', 0, G_OPTION_ARG_NONE, &arg_add_uris,     N_("Add URIs to playlist"), NULL},
    {"playback",    'p', 0, G_OPTION_ARG_NONE, &arg_playback,     N_("Start Playback after Adding"), NULL},
    {"play-lastfm", 'l', 0, G_OPTION_ARG_NONE, &arg_play_lastfm,  N_("Play LastFM URI"), NULL},
    {"no-network",  'n', 0, G_OPTION_ARG_NONE, &arg_no_network,   N_("Start BMPx in Offline Mode"), NULL},

#if 0
    {"next",         0, 0,   G_OPTION_ARG_NONE, &play_next,   N_("Skip to next Track"), NULL},
    {"prev",         0, 0,   G_OPTION_ARG_NONE, &play_prev,   N_("Skip to previous Track"), NULL},
    {"stop",         0, 0,   G_OPTION_ARG_NONE, &play_stop,   N_("Stop Playback"), NULL},
    {"play",         0, 0,   G_OPTION_ARG_NONE, &play_pause,  N_("Start Playback"), NULL},
    {"pause",        0, 0,   G_OPTION_ARG_NONE, &play_pause,  N_("Pause Playback"), NULL},
#endif

#if 0
    {"repeat",       0, 0,   G_OPTION_ARG_NONE, &repeat,      N_("Change Repeat Mode"), NULL},
    {"shuffle",      0, 0,   G_OPTION_ARG_NONE, &shuffle,     N_("Set Repeat"), NULL},
    {"volume_set",   0, 0,   G_OPTION_ARG_NONE, &volume_set,  N_("Set Repeat"), NULL},
#endif

    { NULL }
  };

static const char *features[] =
{
#ifdef HAVE_SM
      N_("X11 Session Management"),
#endif

#ifdef HAVE_HAL
      N_("HAL support"),
#endif

#ifdef HAVE_ALSA
      N_("ALSA Support (Linux)"),
#endif

#ifdef HAVE_SUNAUDIO
      N_("SUN Audio support")
#endif
};

static void print_paths (void)
{
  const char *user_cache_dir	= g_get_user_cache_dir ();
  const char *user_data_dir	= g_get_user_data_dir ();
  const char *user_config_dir	= g_get_user_config_dir ();

  // FIXME: Replace hardcoded paths

  g_print ("\tFiles used by BMP:\n\n");
  g_print ("\t\t- Configuration file.....:\t%s%s\n", user_config_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S "config.xml");
  g_print ("\t\t- Last.FM queue file.....:\t%s%s\n", user_config_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S "lastfm.lqm");
  g_print ("\t\t- Playlist...............:\t%s%s\n", user_config_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S "playlist.xspf");
  g_print ("\t\t- Sqlite3 Main library...:\t%s%s\n", user_data_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S "library.mlib");
  g_print ("\t\t- Sqlite3 HAL library....:\t%s%s\n", user_data_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S "hal.mlib");

  g_print ("\n\n");

  g_print ("\tPaths used by BMP:\n\n");
  g_print ("\t\t- Album-Art cache...:\t%s%s\n", user_cache_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S BMP_BASENAME_COVER_THUMB_DIR);
  g_print ("\t\t- Lyrics cache......:\t%s%s\n", user_cache_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S BMP_BASENAME_LYRIC_CACHE_DIR);
  g_print ("\t\t- Logfile directory.:\t%s%s\n", user_config_dir, G_DIR_SEPARATOR_S BMP_XDG_BASEDIR G_DIR_SEPARATOR_S "logs");
  g_print ("\n");
}

static void print_version (void)
{
  gint i;

  g_print ("\n    %s %s",
           PACKAGE,
           VERSION);

  if (strlen(RV_REVISION))
      g_print (" (%s-R%s)",
               RV_LAST_CHANGED_DATE,
               RV_REVISION);

  g_print (_("\n    Copyright (c) 2003-2006 BMP Project <http://www.beep-media-player.org>\n\n"));

  g_print (_("    Built the %s on %s with:\n"),
           BUILD_DATE,
           BUILD_ARCH);

  g_print (_("\n\tOptional Features built in:\n\n"));

  for (i = 0; i < G_N_ELEMENTS (features); i++)
      g_print ("\t\t* %s\n", _(features[i]));

  g_print ("\n");

  print_paths ();
}

static void setup_i18n (void)
{
  setlocale (LC_ALL, "");

  bindtextdomain (PACKAGE, LOCALE_DIR);
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);
}

static void app_startup_completed (DBusGProxy *proxy, gpointer data)
{
  g_main_loop_quit (mainloop);
}

static void display_startup_crash ()
{
    // We assume BMPx started up, but crashed again
    if (!g_getenv ("DISPLAY"))
      {
        char *message = g_strconcat
          (_("\n       BMPx seems to have crashed.\n\n       Please try starting it from a terminal using '"),
           PREFIX,
           _("/libexec/beep-media-player-2-bin --no-log'\n       for further information on what could have caused the crash\n"),
           _("       and report it to our IRC channel, #bmp on irc.freenode.net\n\n"), NULL);
        g_printf (message);
        g_free (message);
      }
    else
      {
        gtk_init (0, NULL);
        GtkWidget *dialog = gtk_message_dialog_new_with_markup (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                            "<big><b>ERROR</b></big>\n\n<b>BMPx tried to start up, but crashed</b>\n\n"
                            "Please try starting it up using the following commandline <i>from a terminal</i> "
                            "for further information on what could have caused the crash, and report it to "
                            "our IRC channel: <b>#bmp</b> on <b>irc.freenode.net</b>, or file a bug at "
                            "<b>http://bugs.beep-media-player.org</b>\n\n"
                            PREFIX "/libexec/beep-media-player-2-bin --no-log\n\n");

        gtk_window_set_title (GTK_WINDOW (dialog), _("BMPx Crashed - BMP"));
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
      }
    exit (EXIT_FAILURE);
}

static void name_owner_changed (DBusGProxy *proxy, char *name, char *old_owner, char *new_owner, gpointer data)
{
  if (!name || (name && strcmp (name, BMP_DBUS_SERVICE))) return;

  if (strlen (old_owner) && !strlen (new_owner))
    { 
      display_startup_crash ();
    }
}

static void print_error (DBusError *error)
{
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "#1 DBus Error: (%s): %s", error->name, error->message);
}

static void print_gerror (GError **error)
{
  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "Error: %s", (*error)->message);
  g_error_free (*error);
  *error = NULL;
}

int main (int n_args, char **args)
{
  DBusGConnection   *dbus           = NULL;
  DBusConnection    *c_bus          = NULL;
  DBusGProxy        *o_bus          = NULL,
                    *o_bmp          = NULL;
  GOptionContext    *option_context = NULL;
  GError            *error          = NULL;
  gboolean           startup_only   = FALSE;
  DBusError          dbus_error;


  g_type_init ();
  dbus_g_type_specialized_init ();
  mainloop = g_main_loop_new (NULL, FALSE); 
  setup_i18n ();

  option_context = g_option_context_new(" - Run BMP and/or perform actions on BMP");
  g_option_context_add_main_entries (option_context, options, PACKAGE);

  if (!g_option_context_parse (option_context, &n_args, &args, &error))
  {
    g_printerr (_("\nInvalid argument: '%s'\nPlease run '%s --help' for help on usage\n\n"), error->message, args[0]);
    g_error_free (error);
    return EXIT_FAILURE;
  }

  if (arg_show_version)
  {
    print_version ();
    return EXIT_SUCCESS;
  }

  if (n_args == 1)
    startup_only = TRUE;
  else
    startup_only = FALSE;

  dbus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

  if (!dbus)
    {
      if (error)
        g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL, "#2 DBus Error: %s", error->message);

      if (!g_getenv ("DISPLAY"))
        {
          g_printerr (_("%s: Couldn't connect to session bus: %s\n\n"), args[0], error->message);
          return EXIT_FAILURE;
        }

      gtk_init (&n_args, &args);
      const char * template = _("<big><b>BMP/DBus Error</b></big>\n\nBMP can not be started trough DBus activation.\n"
                               "The following error occured trying to start up BMP:\n\n'<b>%s</b>'\n\n"
                               "DBus might not be running at all or have even crashed, please consult further "
                               "help with this problem from DBus or BMP support.");
  
      char * message = NULL;

      if (error)
        {
          message = g_strdup_printf (template, error->message);
        }
      else
        {
          message = g_strdup_printf (template, _("(Unknown error. Perhaps DBus is not running at all?)"));
        }

      GtkWidget *dialog = gtk_message_dialog_new_with_markup
                                 (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, message); 
      
      g_free (message);

      if (error)
        {
          g_error_free (error);
        }

      gtk_window_set_title (GTK_WINDOW (dialog), _("BMP/DBus Error - BMP"));
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return EXIT_FAILURE;      
    }

  dbus_error_init (&dbus_error);

  c_bus = dbus_g_connection_get_connection (dbus);
  dbus_connection_setup_with_g_main (c_bus, g_main_context_default()); 

  o_bus = dbus_g_proxy_new_for_name (dbus, "org.freedesktop.DBus", "/org/freedesktop/DBus", "org.freedesktop.DBus");
  dbus_g_proxy_add_signal (o_bus, "NameOwnerChanged", G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
  dbus_g_proxy_connect_signal (o_bus, "NameOwnerChanged", G_CALLBACK (name_owner_changed), NULL, NULL);

  gboolean was_running = TRUE;

  o_bmp = dbus_g_proxy_new_for_name_owner (dbus,  BMP_DBUS_SERVICE,
                                                  BMP_DBUS_PATH,
                                                  BMP_DBUS_INTERFACE, &error);

  if (!o_bmp)
    was_running = FALSE;

  if (error)
    {
      g_message ("#1 Error: %s", error->message);  
      g_error_free (error);
      error = NULL;
    }
    
  o_bmp = dbus_g_proxy_new_for_name (dbus,  BMP_DBUS_SERVICE,
                                            BMP_DBUS_PATH,
                                            BMP_DBUS_INTERFACE);

  dbus_g_proxy_call (o_bmp, "Startup", &error,
                     G_TYPE_INT,
                     arg_no_network ? ((int)1) : ((int)0),
                     G_TYPE_INVALID,
                     G_TYPE_INVALID);

  if (error)
    {
      g_message ("#2 Error: %s", error->message);  
      g_error_free (error);
      error = NULL;
      display_startup_crash ();
    }

  if (!was_running)
  {
    dbus_g_proxy_add_signal (o_bmp, "StartupComplete", G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (o_bmp, "StartupComplete", G_CALLBACK (app_startup_completed), NULL, NULL);
    g_main_loop_run (mainloop);
  }

  // Start sentinel
  if (!was_running)
  {
    uint32_t reply = 0;
    dbus_bus_start_service_by_name (c_bus,
                                  BMP_DBUS_SENTINEL_SERVICE,
                                  (uint32_t)0,
                                  &reply,
                                  &dbus_error);                   
  if (dbus_error_is_set (&dbus_error))
    {
      print_error (&dbus_error);
      dbus_error_free (&dbus_error);
      return EXIT_FAILURE;
    }
  }

  // Now BMP is running -> process args
  if (startup_only)
  {
    if (was_running)
    {
      if (!dbus_g_proxy_call (o_bmp, "UiRaise", &error,
                            G_TYPE_INVALID,
                            G_TYPE_INVALID))
        {
          if (error) print_gerror (&error);
          abort ();
        }
    }
    return EXIT_SUCCESS;
  }

  if (arg_play_lastfm)
    {
        if (n_args < 2) goto END;
        if (!dbus_g_proxy_call (o_bmp, "LastFmPlayUri", &error,
                                G_TYPE_STRING, args[1],
                                G_TYPE_INVALID,
                                G_TYPE_INVALID))
          {
            if (error) print_gerror (&error);
            abort ();
          }
    }
  else if (arg_add_uris)
    {
        gchar **uri_list;
        gint    n;

        uri_list = g_new (gchar*, n_args+1);

        for (n = 1; n < n_args; n++)
          {
            uri_list[n-1] = g_strdup (args[n]);
          }
        uri_list[n-1] = NULL;

        if (!dbus_g_proxy_call (o_bmp, "PlaylistAddUriList", &error,
                                G_TYPE_STRV, uri_list,
                                G_TYPE_BOOLEAN, arg_playback,
                                G_TYPE_INVALID,
                                G_TYPE_INVALID))
          {
            if (error) print_gerror (&error);
            abort ();
          }
        g_strfreev (uri_list);
    }
  else if (arg_add_files)
    {
        gchar **uri_list;
        gint    n;
        uri_list = g_new (gchar*, n_args+1);
        for (n = 1; n < n_args; n++)
          {
            GError  *error = NULL;
            gchar   *filename,
                    *uri;

            if (args[n][0] != '/')
              {
                filename = g_build_filename (g_getenv("PWD"), args[n], NULL);
              }
            else
              {
                filename = g_strdup (args[n]);
              }

            uri = g_filename_to_uri (filename, NULL, &error);
            g_free (filename);
            if (error)
              {
                g_error_free (error);
                error = NULL;
              }
            uri_list[n-1] = uri;
          }
        uri_list[n-1] = NULL;
        if (!dbus_g_proxy_call (o_bmp, "PlaylistAddUriList", &error,
                                G_TYPE_STRV, uri_list,
                                G_TYPE_BOOLEAN, arg_playback,
                                G_TYPE_INVALID,
                                G_TYPE_INVALID))
          {
            if (error) print_gerror (&error);
            abort ();
          }
        g_strfreev (uri_list);
    }

  END:
  return EXIT_SUCCESS;
}
