/* Bluefish HTML Editor
 * gdk_pixbuf_save.c - imlib can save, gdk_pixbuf cannot... :(
 *
 * Copyright (C) 2001 Olivier Sessink
 *
 * Using code from a gdk_pixbuf patch from davidw@apache.org
 * Copyright (C) David Welton
 * http://people.debian.org/~davidw/patches/gdk-pixbuf-save.patch
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include "default_include.h"

#ifdef USE_GDK_PIXBUF
#include <png.h>
#include <jpeglib.h>
#include <jerror.h>
#include <gdk/gdk.h>
#include <stdio.h>	/* fopen() */
#include <stdlib.h> /* free() */
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "bluefish.h"
#include "images.h"

static int gdk_pixbuf_png_image_save(FILE * f, GdkPixbuf * pixbuf);
static int gdk_pixbuf_jpeg_image_save(FILE * f, GdkPixbuf * pixbuf);

int create_thumbnail(gchar *filename, GdkPixbuf * pb, gint w, gint h) {
	FILE *fd;
	gint retval=0;

	if (!filename || !pb) {
		return 0;
	}
	fd = fopen(filename, "wb");
	if (fd) {
		GdkPixbuf *tmp_im;
	 	tmp_im = gdk_pixbuf_scale_simple(pb, w, h, GDK_INTERP_BILINEAR);
		if (strcmp(main_v->props.cfg_thumbnailtype, "jpg") ==0) {
			retval = gdk_pixbuf_png_image_save(fd, tmp_im);
		} else {
			retval = gdk_pixbuf_jpeg_image_save(fd, tmp_im);
		}
		gdk_pixbuf_unref (tmp_im);
	}
	return retval;
}

static int gdk_pixbuf_png_image_save(FILE * f, GdkPixbuf * pixbuf) {
	png_structp png_ptr;
	png_infop info_ptr;
	guchar *ptr;
	guchar *pixels;
	int x, y, j;
	png_bytep row_ptr, data = NULL;
	png_color_8 sig_bit;
	int w, h, rowstride;
	int has_alpha;
	int bpc;

	bpc = gdk_pixbuf_get_bits_per_sample(pixbuf);
	w = gdk_pixbuf_get_width(pixbuf);
	h = gdk_pixbuf_get_height(pixbuf);
	rowstride = gdk_pixbuf_get_rowstride(pixbuf);
	has_alpha = gdk_pixbuf_get_has_alpha(pixbuf);
	pixels = gdk_pixbuf_get_pixels(pixbuf);

	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
									  NULL, NULL, NULL);
	if (!png_ptr) {
		fclose(f);
		return 0;
	}
	info_ptr = png_create_info_struct(png_ptr);
	if (info_ptr == NULL) {
		fclose(f);
		png_destroy_write_struct(&png_ptr, (png_infopp) NULL);
		return 0;
	}
	if (setjmp(png_ptr->jmpbuf)) {
		fclose(f);
		png_destroy_write_struct(&png_ptr, (png_infopp) NULL);
		return 0;
	}
	png_init_io(png_ptr, f);
	if (has_alpha) {
		png_set_IHDR(png_ptr, info_ptr, w, h, bpc,
					 PNG_COLOR_TYPE_RGB_ALPHA, PNG_INTERLACE_NONE,
					 PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);
#ifdef WORDS_BIGENDIAN
		png_set_swap_alpha(png_ptr);
#else
		png_set_bgr(png_ptr);
#endif
	} else {
		png_set_IHDR(png_ptr, info_ptr, w, h, bpc,
					 PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
					 PNG_COMPRESSION_TYPE_BASE, PNG_FILTER_TYPE_BASE);
		data = malloc(w * 3 * sizeof(char));
	}
	sig_bit.red = bpc;
	sig_bit.green = bpc;
	sig_bit.blue = bpc;
	sig_bit.alpha = bpc;
	png_set_sBIT(png_ptr, info_ptr, &sig_bit);
	png_write_info(png_ptr, info_ptr);
	png_set_shift(png_ptr, &sig_bit);
	png_set_packing(png_ptr);

	ptr = pixels;
	for (y = 0; y < h; y++) {
		if (has_alpha)
			row_ptr = (png_bytep) ptr;
		else {
			for (j = 0, x = 0; x < w; x++)
				memcpy(&(data[x * 3]), &(ptr[x * 3]), 3);

			row_ptr = (png_bytep) data;
		}
		png_write_rows(png_ptr, &row_ptr, 1);
		ptr += rowstride;
	}
	if (data)
		free(data);
	png_write_end(png_ptr, info_ptr);
	png_destroy_write_struct(&png_ptr, (png_infopp) NULL);
	return 0;
}

struct error_handler_data {
  struct jpeg_error_mgr pub;    /* "public" fields */
  jmp_buf setjmp_buffer;        /* for return to caller */
};

static int gdk_pixbuf_jpeg_image_save(FILE * f, GdkPixbuf * pixbuf) {
	struct jpeg_compress_struct cinfo;
	guchar *buf;
	guchar *ptr;
	guchar *pixels;
	JSAMPROW *jbuf;
	int y = 0;
	int i, j;
	int w, h = 0;
	int rowstride = 0;
	struct              error_handler_data jerr;

	rowstride = gdk_pixbuf_get_rowstride(pixbuf);

	w = gdk_pixbuf_get_width(pixbuf);
	h = gdk_pixbuf_get_height(pixbuf);

	/* no image data? abort */
	pixels = gdk_pixbuf_get_pixels(pixbuf);
	if (!pixels)
		return 0;
	/* allocate a small buffer to convert image data */
	buf = malloc(w * 3 * sizeof(guchar));
	if (!buf)
		return 0;

	cinfo.err = jpeg_std_error(&(jerr.pub));
	if (sigsetjmp(jerr.setjmp_buffer, 1)) {
		jpeg_destroy_compress(&cinfo);
		free(buf);
		fclose(f);
		return 1;
	}

	/* setup compress params */
	jpeg_create_compress(&cinfo);
	jpeg_stdio_dest(&cinfo, f);
	cinfo.image_width = w;
	cinfo.image_height = h;
	cinfo.input_components = 3;
	cinfo.in_color_space = JCS_RGB;


	/* set up jepg compression parameters */
	jpeg_set_defaults(&cinfo);
	jpeg_start_compress(&cinfo, TRUE);
	/* get the start pointer */
	ptr = pixels;
	/* go one scanline at a time... and save */
	i = 0;
	while (cinfo.next_scanline < cinfo.image_height) {
		/* convert scanline from ARGB to RGB packed */
		for (j = 0; j < w; j++)
			memcpy(&(buf[j * 3]), &(ptr[i * rowstride + j * 3]), 3);

		/* write scanline */
		jbuf = (JSAMPROW *) (&buf);
		jpeg_write_scanlines(&cinfo, jbuf, 1);
		i++;
		y++;
	}
	/* finish off */
	jpeg_finish_compress(&cinfo);
	free(buf);
	return 0;
}
#endif /* USE_GDK_PIXBUF */
