open Printf

open Jane.Std

open Bin_prot

TYPE_CONV_PATH "Foo"

let n_els = 20
let buf_size = 100000
let buf_size_1 = buf_size - 1
let buf = Bigstring.of_string (String.make buf_size '|')

let clear_buf () =
  for i = 0 to buf_size_1 do
    buf.{i} <- '|';
  done

let test_noclear n_msgs name f n =
  let t1 = Unix.gettimeofday () in
  let len = f buf n_msgs n in
  let t2 = Unix.gettimeofday () in
  printf "%s: %.2f  |%s|\n%!"
    name (t2 -. t1) (String.escaped (Bigstring.to_string ~len buf))

let test_clear n_msgs name f n =
  clear_buf ();
  test_noclear n_msgs name f n

(**)

let c_write_int buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_int !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_int sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_int buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_int buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_int buf ~pos:0 arg

let c_read_int buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_int sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_int sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_int buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_int buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_int buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_string buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_string !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_string sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_string buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_string buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_string buf ~pos:0 arg

let c_read_string buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_string sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_string sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_string buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_string buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_string buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_float buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_float !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_float sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_float buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_float buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_float buf ~pos:0 arg

let c_read_float buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_float sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_float sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_float buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_float buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_float buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_int32 buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_int32 !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_int32 sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_int32 buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_int32 buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_int32 buf ~pos:0 arg

let c_read_int32 buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_int32 sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_int32 sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_int32 buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_int32 buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_int32 buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_int64 buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_int64 !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_int64 sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_int64 buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_int64 buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_int64 buf ~pos:0 arg

let c_read_int64 buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_int64 sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_int64 sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_int64 buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_int64 buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_int64 buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_nativeint buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_nativeint !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_nativeint sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_nativeint buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_nativeint buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_nativeint buf ~pos:0 arg

let c_read_nativeint buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_nativeint sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_nativeint sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_nativeint buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_nativeint buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_nativeint buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_variant buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_variant_tag !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_variant_tag sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_variant buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_variant_tag buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_variant_tag buf ~pos:0 arg

let c_read_variant buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_variant_tag sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_variant_tag sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_variant buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_variant_tag buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_variant_tag buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_list buf n_msgs (bin_write_el, arg) =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_list bin_write_el !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_list bin_write_el sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_list buf n_msgs (bin_write_el, arg) =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_list bin_write_el buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_list bin_write_el buf ~pos:0 arg

let c_read_list buf n_msgs (bin_read_el, correct) =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_list bin_read_el sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_list bin_read_el sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_list buf n_msgs (bin_read_el, correct) =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_list bin_read_el buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_list bin_read_el buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_array buf n_msgs (bin_write_el, arg) =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr  = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_array bin_write_el !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_array bin_write_el sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_array buf n_msgs (bin_write_el, arg) =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_array bin_write_el buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_array bin_write_el buf ~pos:0 arg

let c_read_array buf n_msgs (bin_read_el, correct) =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_array bin_read_el sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_array bin_read_el sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_array buf n_msgs (bin_read_el, correct) =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_array bin_read_el buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_array bin_read_el buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let c_write_float_array buf n_msgs arg =
  for i = 1 to n_msgs do
    let sptr = Unsafe_common.get_sptr buf ~pos:0 in
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ref = ref sptr in
    (try
      for j = 1 to n_els do
        sptr_ref := Unsafe_write_c.bin_write_float_array !sptr_ref eptr arg;
      done;
    with Exit -> raise Exit);
    ignore (buf == buf);
  done;
  let sptr = Unsafe_common.get_sptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let cur = Unsafe_write_c.bin_write_float_array sptr eptr arg in
  Unsafe_common.get_buf_pos ~start:sptr ~cur

let ml_write_float_array buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (Write_ml.bin_write_float_array buf ~pos:0 arg);
      done;
    with Exit -> raise Exit);
  done;
  Write_ml.bin_write_float_array buf ~pos:0 arg

let c_read_float_array buf n_msgs correct =
  for i = 1 to n_msgs do
    let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
    let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
    (try
      for j = 1 to n_els do
        ignore (Unsafe_read_c.bin_read_float_array sptr_ptr eptr);
      done;
      ignore (Unsafe_common.dealloc_sptr_ptr buf sptr_ptr);
    with Exit -> raise Exit);
  done;
  let sptr_ptr = Unsafe_common.alloc_sptr_ptr buf ~pos:0 in
  let eptr = Unsafe_common.get_eptr buf ~pos:buf_size_1 in
  let res = Unsafe_read_c.bin_read_float_array sptr_ptr eptr in
  assert (res = correct);
  Unsafe_common.dealloc_sptr_ptr buf sptr_ptr

let ml_read_float_array buf n_msgs correct =
  for i = 1 to n_msgs do
    let pos_ref = ref 0 in
    (try
      for j = 1 to n_els do
        ignore (Read_ml.bin_read_float_array buf ~pos_ref);
      done;
    with Exit -> raise Exit);
  done;
  let pos_ref = ref 0 in
  let res = Read_ml.bin_read_float_array buf ~pos_ref in
  assert (res = correct);
  !pos_ref

(**)

let marshal_write buf n_msgs arg =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (
          Sysprog.Bigstring_marshal.marshal_blit
            ~flags:[Our_marshal.Jane_marshal.No_sharing] arg ~pos:0 buf);
      done;
    with Exit -> raise Exit);
  done;
  Sysprog.Bigstring_marshal.marshal_blit
    ~flags:[Our_marshal.Jane_marshal.No_sharing] arg ~pos:0 buf

let marshal_read buf n_msgs correct =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (
          Sysprog.Bigstring_marshal.unmarshal_next ~pos:0 buf);
      done;
    with Exit -> raise Exit);
  done;
  let res, pos = Sysprog.Bigstring_marshal.unmarshal_next ~pos:0 buf in
  assert (res = correct);
  pos

(**)

type t = float array
with sexp

let sexp_write_float_array _buf n_msgs ar =
  for i = 1 to n_msgs do
    (try
      for j = 1 to n_els do
        ignore (sexp_of_t ar);
      done;
    with Exit -> raise Exit);
  done

let () = Sexplib.Conv.default_string_of_float := string_of_float

(**)

let bar () =
  print_string
    "----------------------------------------------------------------------\n"

let test_int n_msgs n =
  test_clear n_msgs ("ml_write_int " ^ string_of_int n) ml_write_int n;
  test_clear n_msgs ("c_write_int  " ^ string_of_int n) c_write_int n;
  test_noclear n_msgs ("c_read_int   " ^ string_of_int n) c_read_int n;
  test_noclear n_msgs ("ml_read_int  " ^ string_of_int n) ml_read_int n;
  bar ()

let test_ints n =
  test_int n 0x0000007f;
  test_int n 0x00007fff;
  if Sys.word_size = 32 then test_int n 0x3fffffff
  else (
    test_int n 0x7fffffff;
    test_int n (int_of_string "0x3fffffffffffffff"));
  test_int n (-0x00000080);
  test_int n (-0x00008000);
  if Sys.word_size = 32 then test_int n (-0x40000000)
  else (
    test_int n (int_of_string "-0x0000000080000000");
    test_int n (int_of_string "-0x4000000000000000"))

(**)

let test_int32 n_msgs n =
  test_clear n_msgs ("ml_write_int32 " ^ Int32.to_string n) ml_write_int32 n;
  test_clear n_msgs ("c_write_int32  " ^ Int32.to_string n) c_write_int32 n;
  test_noclear n_msgs ("c_read_int32   " ^ Int32.to_string n) c_read_int32 n;
  test_noclear n_msgs ("ml_read_int32  " ^ Int32.to_string n) ml_read_int32 n;
  bar ()

let test_int32s n =
  test_int32 n   0x0000007fl;
  test_int32 n   0x00007fffl;
  test_int32 n   0x7fffffffl;
  test_int32 n (-0x00000080l);
  test_int32 n (-0x00008000l);
  test_int32 n ( 0x80000000l)  (* otherwise parser fails *)

(**)

let test_int64 n_msgs n =
  test_clear n_msgs ("ml_write_int64 " ^ Int64.to_string n) ml_write_int64 n;
  test_clear n_msgs ("c_write_int64  " ^ Int64.to_string n) c_write_int64 n;
  test_noclear n_msgs ("c_read_int64   " ^ Int64.to_string n) c_read_int64 n;
  test_noclear n_msgs ("ml_read_int64  " ^ Int64.to_string n) ml_read_int64 n;
  bar ()

let test_int64s n =
  test_int64 n   0x000000000000007fL;
  test_int64 n   0x0000000000007fffL;
  test_int64 n   0x000000007fffffffL;
  test_int64 n   0x7fffffffffffffffL;
  test_int64 n (-0x0000000000000080L);
  test_int64 n (-0x0000000000008000L);
  test_int64 n (-0x0000000080000000L);
  test_int64 n ( 0x8000000000000000L)  (* otherwise parser fails *)

(**)

let test_nativeint n_msgs n =
  test_clear
    n_msgs ("ml_write_nativeint " ^ Nativeint.to_string n) ml_write_nativeint n;
  test_clear
    n_msgs ("c_write_nativeint  " ^ Nativeint.to_string n) c_write_nativeint n;
  test_noclear
    n_msgs ("c_read_nativeint   " ^ Nativeint.to_string n) c_read_nativeint n;
  test_noclear
    n_msgs ("ml_read_nativeint  " ^ Nativeint.to_string n) ml_read_nativeint n;
  bar ()

let test_nativeints n =
  test_nativeint n   0x000000000000007fn;
  test_nativeint n   0x0000000000007fffn;
  test_nativeint n   0x000000007fffffffn;
  if Sys.word_size = 64 then
    test_nativeint n (Nativeint.of_string "0x7fffffffffffffff");
  test_nativeint n (-0x0000000000000080n);
  test_nativeint n (-0x0000000000008000n);
  test_nativeint n (Nativeint.of_string "-0x0000000080000000");
  if Sys.word_size = 64 then
    test_nativeint n (Nativeint.of_string "0x8000000000000000")

(**)

let () =
  let n = 100000 in

  let n_ints = 1000000 in
  test_ints n_ints;

  test_int32s n_ints;
  test_int64s n_ints;
  test_nativeints n_ints;

  let int_array = Array.create 10 0x3fffffff in
  test_clear n         "     ml_write_int_array" ml_write_array (Write_ml.bin_write_int, int_array);
  test_clear n         "      c_write_int_array" c_write_array (Unsafe_write_c.bin_write_int, int_array);
  test_noclear n "       c_read_int_array" c_read_array (Unsafe_read_c.bin_read_int, int_array);
  test_noclear n "      ml_read_int_array" ml_read_array (Read_ml.bin_read_int, int_array);
  test_clear n         "marshal_write_int_array" marshal_write int_array;
  test_noclear n " marshal_read_int_array" marshal_read int_array;

  bar ();

  let bool_array = Array.create 10 true in
  test_clear n         "     ml_write_bool_array" ml_write_array (Write_ml.bin_write_bool, bool_array);
  test_clear n         "      c_write_bool_array" c_write_array (Unsafe_write_c.bin_write_bool, bool_array);
  test_noclear n "       c_read_bool_array" c_read_array (Unsafe_read_c.bin_read_bool, bool_array);
  test_noclear n "      ml_read_bool_array" ml_read_array (Read_ml.bin_read_bool, bool_array);
  test_clear n         "marshal_write_bool_array" marshal_write bool_array;
  test_noclear n " marshal_read_bool_array" marshal_read bool_array;

  bar ();

  let int64_array =
    [|
       0x000000000000007fL;
       0x0000000000007fffL;
       0x000000007fffffffL;
       0x7fffffffffffffffL;
      -0x0000000000000080L;
      -0x0000000000008000L;
      -0x0000000080000000L;
       0x8000000000000000L;
    |]
  in
  test_clear n         "     ml_write_int64_array" ml_write_array (Write_ml.bin_write_int64, int64_array);
  test_clear n         "      c_write_int64_array" c_write_array (Unsafe_write_c.bin_write_int64, int64_array);
  test_noclear n "       c_read_int64_array" c_read_array (Unsafe_read_c.bin_read_int64, int64_array);
  test_noclear n "      ml_read_int64_array" ml_read_array (Read_ml.bin_read_int64, int64_array);
  test_clear n         "marshal_write_int64_array" marshal_write int64_array;
  test_noclear n " marshal_read_int64_array" marshal_read int64_array;

  bar ();

  let float_array = Array.create 50 1. in
  test_clear n         "     ml_write_float_array" ml_write_float_array float_array;
  test_clear n         "      c_write_float_array" c_write_float_array float_array;
  test_noclear n "       c_read_float_array" c_read_float_array float_array;
  test_noclear n "      ml_read_float_array" ml_read_float_array float_array;
  test_clear n         "marshal_write_float_array" marshal_write float_array;
  test_noclear n " marshal_read_float_array" marshal_read float_array;
