(* File: utils.ml

    Copyright (C) 2007-

      Jane Street Holding, LLC
      Author: Markus Mottl
      email: mmottl\@janestcapital.com
      WWW: http://www.janestcapital.com/ocaml

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Utils: utility functions for user convenience *)

open Bigarray
open Common
open Read_ml
open Write_ml
open Size

let bin_dump ?(with_size = false) (sizer, writer) v =
  let buf, pos, pos_len =
    let len = sizer v in
    if with_size then
      let len_len = bin_size_int len in
      let tot_len = len + len_len in
      let buf = create_buf tot_len in
      let pos = bin_write_nat0 buf ~pos:0 (Nat0.unsafe_of_int len) in
      buf, pos, pos + len
    else
      let buf = create_buf len in
      buf, 0, len
  in
  let pos = writer buf ~pos v in
  if pos = pos_len then buf
  else failwith "Bin_prot.Utils.bin_dump: size changed during writing"

module Read_buf = struct
  type state =
    | Empty
    | Read_len of int  (* how many bytes of the length have been read *)
    | Read_data of int * int  (* [(pos, remaining)] *)

  type t =
    {
      mutable state : state;
      mutable data_buf : buf;
    }

  let max_int_size = 9

  let create () =
    {
      state = Empty;
      data_buf = create_buf max_int_size;
    }

  let alloc_buf rbuf buf_size =
    let dst = create_buf buf_size in
    rbuf.data_buf <- dst;
    dst

  let enforce_buf_size rbuf buf_size =
    let buf_size = max buf_size max_int_size in
    let src = rbuf.data_buf in
    Array1.dim src <= buf_size ||
      match rbuf.state with
      | Empty -> ignore (alloc_buf rbuf buf_size); true
      | Read_len len ->
          let dst = alloc_buf rbuf buf_size in
          for i = 0 to len - 1 do dst.{i} <- src.{i} done;
          true
      | Read_data (pos, remaining) ->
          pos + remaining <= buf_size &&
            let dst = alloc_buf rbuf buf_size in
            unsafe_blit_buf ~src_pos:0 ~src ~dst_pos:0 ~dst ~len:pos;
            true

  let purge rbuf = rbuf.state <- Empty

  let get_size rbuf = Array1.dim rbuf.data_buf

  let get_buffered_data_size rbuf =
    match rbuf.state with
    | Empty -> 0
    | Read_len size | Read_data (size, _) -> size

  let maybe_read_size buf ~pos_ref =
    try
      let psize = Read_ml.bin_read_nat0 buf ~pos_ref in
      Some (Nat0.to_int psize)
    with Buffer_short -> None

  let check_max_size ?max_size len =
    match max_size with
    | None -> ()
    | Some max_size ->
        if max_size < len then failwith "Bin_prot.Utils.next: max_size exceeded"

  let have_len ?max_size reader src ~pos_ref rbuf msg_len =
    check_max_size ?max_size msg_len;
    let src_pos = !pos_ref in
    let dim = Array1.dim src in
    let len = dim - src_pos in
    if msg_len > len then
      let dst =
        let data_buf = rbuf.data_buf in
        if Array1.dim data_buf < msg_len then
          let new_data_buf = create_buf msg_len in
          rbuf.data_buf <- new_data_buf;
          new_data_buf
        else data_buf
      in
      unsafe_blit_buf ~src_pos ~src ~dst_pos:0 ~dst ~len;
      rbuf.state <- Read_data (len, msg_len - len);
      pos_ref := dim;
      None
    else (
      rbuf.state <- Empty;
      Some (reader src ~pos_ref))

  let next_empty ?max_size reader src ~pos_ref rbuf =
    let pos = !pos_ref in  (* not after maybe_read_size (side effect!) *)
    match maybe_read_size src ~pos_ref with
    | Some len -> have_len ?max_size reader src ~pos_ref rbuf len
    | None ->
        let dim = Array1.dim src in
        pos_ref := dim;
        let data_buf = rbuf.data_buf in
        for i = pos to dim - 1 do data_buf.{i - pos} <- src.{i} done;
        rbuf.state <- Read_len (dim - pos);
        None

  let next_read_len ?max_size reader src ~pos_ref rbuf data_buf_pos =
    let pos = !pos_ref in
    let dim = Array1.dim src in
    let max_chars = min (max_int_size - data_buf_pos) (dim - pos) in
    let data_buf = Array1.sub rbuf.data_buf 0 (data_buf_pos + max_chars) in
    for i = 0 to max_chars - 1 do
      data_buf.{data_buf_pos + i} <- src.{pos + i};
    done;
    pos_ref := 0;
    match maybe_read_size data_buf ~pos_ref with
    | Some len ->
        pos_ref := pos + !pos_ref - data_buf_pos;
        have_len ?max_size reader src ~pos_ref rbuf len
    | None ->
        pos_ref := dim;
        rbuf.state <- Read_len (data_buf_pos + max_chars);
        None

  let next_read_data reader src ~pos_ref rbuf dst_pos len =
    let pos = !pos_ref in
    let dim = Array1.dim src in
    let avail = dim - pos in
    if len > avail then (
      unsafe_blit_buf ~src_pos:pos ~src ~dst_pos ~dst:rbuf.data_buf ~len:avail;
      rbuf.state <- Read_data (dst_pos + avail, len - avail);
      pos_ref := dim;
      None)
    else
      let msg_len = dst_pos + len in
      let dst =
        let data_buf = rbuf.data_buf in
        if Array1.dim data_buf > msg_len then Array1.sub data_buf 0 msg_len
        else data_buf
      in
      rbuf.state <- Empty;
      unsafe_blit_buf ~src_pos:pos ~src ~dst_pos ~dst ~len;
      let dst_pos_ref = ref 0 in
      let res = reader dst ~pos_ref:dst_pos_ref in
      if !dst_pos_ref <> msg_len then
        failwith
          "Bin_prot.Utils.next_read_data: protocol lied about length of value";
      pos_ref := pos + len;
      Some res

  let next ?max_size reader src ~pos_ref rbuf =
    match rbuf.state with
    | Empty -> next_empty ?max_size reader src ~pos_ref rbuf
    | Read_len pos -> next_read_len ?max_size reader src ~pos_ref rbuf pos
    | Read_data (pos, len) ->
        check_max_size ?max_size (pos + len);
        next_read_data reader src ~pos_ref rbuf pos len
end


(* Conversion of binable types *)

module type Make_binable_spec = sig
  type t
  type binable

  val to_binable : t -> binable
  val of_binable : binable -> t

  val bin_size_binable : binable Size.sizer
  val bin_write_binable : binable Map_to_safe.writer
  val bin_write_binable_ : binable Unsafe_write_c.writer
  val bin_read_binable : binable Read_ml.reader
  val bin_read_binable_ : binable Unsafe_read_c.reader
  val bin_read_binable__ : (int -> binable) Unsafe_read_c.reader
  val bin_sw_arg_binable : binable Map_to_safe.sw_arg
end

module Make_binable (Bin_spec : Make_binable_spec) = struct
  open Bin_spec

  type t = Bin_spec.t

  let bin_size_t t = bin_size_binable (to_binable t)
  let bin_write_t buf ~pos t = bin_write_binable buf ~pos (to_binable t)
  let bin_write_t_ sptr eptr t = bin_write_binable_ sptr eptr (to_binable t)
  let bin_read_t buf ~pos_ref = of_binable (bin_read_binable buf ~pos_ref)
  let bin_read_t_ sptr_ptr eptr = of_binable (bin_read_binable_ sptr_ptr eptr)

  let bin_read_t__ sptr_ptr eptr n =
    of_binable (bin_read_binable__ sptr_ptr eptr n)

  let bin_sw_arg_t = bin_size_t, bin_write_t
end
